# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")] param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Write-IntelEthernetDebugDump
{
    [CmdletBinding(DefaultParameterSetName = 'ByAdapter')]
    Param(
        [parameter(Mandatory=$true, ParameterSetName="ByName")]
        [ValidateNotNullOrEmpty()]
        [String]
        $Name,

        [parameter(Mandatory=$true, Position=0, ValueFromPipeline=$true, ParameterSetName="ByAdapter")]
        [ValidateNotNullOrEmpty()]
        [Object]
        $Adapter,

        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false,  Position=1, ParameterSetName="ByAdapter")]
        [SupportsWildcards()]
        [ValidateNotNullOrEmpty()]
        [String[]]
        $Cluster = '*',

        [string]
        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false, Position=2, ParameterSetName="ByAdapter")]
        [ValidateNotNullOrEmpty()]
        $Path = '',

        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false, ParameterSetName="ByAdapter")]
        [switch]
        $Force = $false,

        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false, ParameterSetName="ByAdapter")]
        [switch]
        $Append = $false
    )
    Begin
    {
        $script:ErrorMessagesWrite = @()
        $script:WarningMessagesWrite = @()

        GetIntelEthernetDevices
        $script:SupportedAdapters = @(GetSupportedAdaptersByDriver @('icea'))
        $Clusters = $Cluster
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesWrite += $Messages.AdapterNotFound -f $Name
                break
            }

            $AdapterName = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesWrite)

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesWrite)))
            {
                break
            }

            $AdapterName = GetSupportedAdapters $AdapterName ([ref]$script:ErrorMessagesWrite)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (([string]::IsNullOrEmpty($Path)))
            {
                $Timestamp = Get-Date -Format "MMddyyyy"
                $DefaultName = ($AdapterName -replace '\s+', '_') + "_$Timestamp`_dump.bin"
                $UseDefaultPath = $true
            }

            $LogPath = ''
            if (-not (ValidatePathParams ([ref]$LogPath) $UseDefaultPath $DefaultName ([ref]$script:ErrorMessagesWrite)))
            {
                break
            }

            $ClusterIDs = ValidateClusterParam $Clusters $AdapterName
            if ($ClusterIDs.Count)
            {
                $Output = [System.Collections.ArrayList]::new()

                # First Dump the 24 Byte Metadata Cluster
                $Metadata = CreateMetadataCluster $AdapterName
                if ($null -eq $Metadata)
                {
                    foreach ($ClusterID in $ClusterIDs)
                    {
                        $script:ErrorMessagesWrite += $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
                    }
                    break
                }

                $Output.AddRange($Metadata)
                $bWriteFile = $false

                $bUseTableMethod = $true
                foreach ($ClusterID in $ClusterIDs)
                {
                    $ClusterID = [int]$ClusterID
                    $Dump = $null
                    if ($bUseTableMethod)
                    {
                        $Dump = ReadClusterTable $ClusterID
                    }

                    if (-Not $Dump)
                    {
                        $bUseTableMethod = $false
                        $Dump = ReadCluster $ClusterID
                    }

                    if ($Dump)
                    {
                        $Output.AddRange($Dump)
                        $bWriteFile = $true
                        $Dump = $null # hint to Garbage Collector that memory can be freed
                    }
                    else
                    {
                        $bUseTableMethod = $true

                        $TmpMsg = $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
                        if (-Not ($TmpMsg -in $script:ErrorMessagesWrite))
                        {
                            $script:ErrorMessagesWrite += $TmpMsg
                        }
                    }
                }

                if ($bWriteFile)
                {
                    WriteToFile $LogPath $Output $Append
                }
                $Output = $null # hint to Garbage Collector that memory can be freed
            }
            else
            {
                break
            }

        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesWrite)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesWrite)
        {
            Write-Error $ErrorMessage
        }
        # trigger Garbage Collection
        [System.GC]::Collect()
    }
}


Function CreateMetadataCluster($AdapterName)
{
    $MetadataClusterOutput = [System.Collections.ArrayList]::new()
    $EETrackIDs = @(Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EetrackId -Property Id -ErrorAction SilentlyContinue)
    if ($EETrackIDs)
    {
        $EETrackID = ($EETrackIDs.Where({$_.InstanceName -eq $AdapterName})).Id

        if ($EETrackID)
        {
            $MetadataClusterOutput.AddRange([System.BitConverter]::GetBytes(0x80000001))  # Metadata Cluster ID
            $MetadataClusterOutput.AddRange([System.BitConverter]::GetBytes($EETrackID))
            $MetadataClusterOutput.AddRange([byte[]]::new(16)) # Reserved / not used
        }
    }

    $MetadataClusterOutput
}


Function WriteToFile($Path, $Output, $Append)
{
    if ($Append)
    {
        Add-Content $Path -Value $Output -Encoding Byte
    }
    else
    {
        Set-Content $Path -Value $Output -Encoding Byte
    }

    Resolve-Path -Path $Path -ErrorAction SilentlyContinue
}


Function ValidateClusterParam($Clusters, $AdapterName)
{
    $ValidClusterIDs = @()
    $InvalidClusters = @()

    # Throw an error for Invalid Clusters, but still run if there are any valid ones
    foreach ($Cluster in $Clusters)
    {
        $ClusterID = $DebugClusterMap.GetEnumerator().Where({$_.Value -like $Cluster}).Name
        if ($ClusterID.Count -gt 0) # Check for corresponding Keys
        {
            $ValidClusterIDs += $ClusterID
            continue
        }

        # Check if they sent in the numerical Cluster ID
        if($Cluster -match "^\d+$")
        {
            if ($DebugClusterMap.[int]$Cluster) # Check for corresponding Values
            {
                $ValidClusterIDs += $Cluster
                continue
            }
        }

        $InvalidClusters += $Cluster
    }

    $InvalidClusters = $InvalidClusters | Sort-Object | Get-Unique
    foreach ($Cluster in $InvalidClusters)
    {
        $script:ErrorMessagesWrite += $Messages.InvalidCluster -f $AdapterName, $Cluster
    }

    $ValidClusterIDs = $ValidClusterIDs | Sort-Object | Get-Unique
    $ValidClusterIDs
}

function ReadClusterTable($ClusterID)
{
    $i = 0

    $TableID = 0
    $TableIndex = 0
    $ClusterOutput = [System.Collections.ArrayList]::new()

    Write-Verbose "`[IN`] $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tTableID $($TableID)`tTableIndex $($TableIndex)"
    $Dump = GetTable $ClusterID $TableID $TableIndex

    if ($null -ne $Dump)
    {
        $ClusterOutput.AddRange($Dump.Table[0..($Dump.TableSize - 1)])
    }

    while ($null -ne $Dump -and $Dump.NextClusterId -eq $ClusterID -and $Dump.NextClusterId -ne 0xFFFF -and $Dump.MoreDataAvailable -eq 1)
    {
        Write-Progress -Activity $Messages.WritingDebugDump -PercentComplete $i

        $TableId = $Dump.NextTableId
        $TableIndex = $Dump.NextTableIndex

        if ($i++ -ge 100)
        {
            $i = 0
        }

        Write-Verbose "`[IN`] $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tTableID $($TableID)`tTableIndex $($TableIndex)"
        $Dump = GetTable $ClusterID $TableID $TableIndex

        if ($null -ne $Dump)
        {
            $ClusterOutput.AddRange($Dump.Table[0..($Dump.TableSize - 1)])
        }
    }

    return $ClusterOutput
}

function GetTable($ClusterId, $TableId, $TableIndex)
{
    $params = @{
        Type        = [uint32]1
        Version     = [uint32]1
        Size        = [uint32]24
        ClusterId   = [uint32]$ClusterId
        TableId     = [uint32]$TableId
        TableIndex  = [uint32]$TableIndex
    }

    $Result = InvokeCimMethod "IntlLan_GetFWClusterTable" $AdapterName "WmiGetFWClusterTable" $Params

    if ($null -ne $Result -and $Result.FwErrorCode -ne 0)
    {
        $TmpMsg = $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
        if ($script:ErrorMessagesWrite -NotContains $TmpMsg)
        {
            $script:ErrorMessagesWrite += $TmpMsg
        }
        Write-Verbose "`[OUT`] $($DebugClusterMap.$ClusterId) Cluster `($ClusterId`)`tTableID $($TableId)`tTableIndex $($TableIndex)"

        return $null
    }

    return $Result
}

Function BuildTableOutput($ClusterID, $TableID, $TableIndex, $TableBufferSize, $TableBuffer)
{
    $Output = [System.Collections.ArrayList]::new()

    $Output.AddRange([System.BitConverter]::GetBytes($ClusterID))
    $Output.AddRange([System.BitConverter]::GetBytes($TableID))
    $Output.AddRange([System.BitConverter]::GetBytes($TableBufferSize))
    $Output.AddRange([System.BitConverter]::GetBytes($TableIndex))
    $Output.AddRange([byte[]]::new(8)) # Reserved Bytes
    $Output.AddRange($TableBuffer)

    $Output
}

Function ReadCluster($ClusterID)
{
    $i = 0
    $j = 0

    $TableID = 0
    $TableIndex = 0
    $ClusterOutput = [System.Collections.ArrayList]::new()

    $FinalTableID = 0xFF
    $FinalTableID_FWBug = 0xFFFF
    $FinalTableIndex = [uint32]0xFFFFFFFFL

    while ($TableID -ne $FinalTableID -and $TableID -ne $FinalTableID_FWBug)
    {
        Write-Progress -Activity $Messages.WritingDebugDump -PercentComplete $j
        # Call the OID for the current buffer and the next iteration info
        Write-Verbose "`[IN`] Buffer $i : $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tTableID $TableID`tTableIndex $TableIndex"
        $Dump = GetDump $ClusterID $TableID $TableIndex
        $Output = BuildTableOutput $ClusterID $TableID $TableIndex $Dump.TableBufferSize $Dump.TableBuffer
        $ClusterOutput.AddRange($Output)

        if ($Dump.NextTableID -eq -1 -or ($TableID -eq $Dump.NextTableID -and $TableIndex -eq $Dump.NextTableIndex))
        {
            return $ClusterOutput
        }

        if ($TableID -ne $Dump.NextTableID -or
            $TableIndex -eq $FinalTableIndex)
        {
            $TableIndex = 0
            $TableID = $Dump.NextTableID
        }
        else
        {
            $TableIndex = $Dump.NextTableIndex
        }

        $i++

        if ($j++ -ge 100)
        {
            $j = 0
        }
    }

    $ClusterOutput
}


function GetDump($ClusterID, $TableID, $TableIndex)
{
    $TableBuffer = [System.Collections.ArrayList]::new()
    $TableBufferSize = 0

    $MaxOIDBufs = 8
    for ($BufIndex = 0; $BufIndex -lt $MaxOIDBufs; $BufIndex++)
    {
        # Build Input Buffer: BufIndex, ClusterId, TableId, TableIndex starting at the 4th byte
        $InBuf = [byte[]]::new(1024)
        [System.BitConverter]::GetBytes($BufIndex).CopyTo($InBuf, 4)
        [System.BitConverter]::GetBytes($ClusterID).CopyTo($InBuf, 8)
        [System.BitConverter]::GetBytes($TableID).CopyTo($InBuf, 12)
        [System.BitConverter]::GetBytes($TableIndex).CopyTo($InBuf, 16)

        $params = @{
            Type        = [uint32]1
            Version     = [uint32]1
            Size        = [uint32]1048 # Header size (24 bytes) + Data Size (1024)
            Operation   = [uint32]0x22 # Operation = 0x22 is INTEL_GEN_DATA_OP_GET_DUMP
            OpVersion   = [uint32]1
            DataSize    = [uint32]1024
            InBuf       = $InBuf
        }

        # first call WmiSetGenData
        $Result = InvokeCimMethod "IntlLan_SetGenData" $AdapterName "WmiSetGenData" $Params

        # if the Result is INTEL_GEN_DATA_STATUS_CALL_GET_GEN_DATA - read GenData
        if ($Result.OutStatus -eq 1)
        {
            $Result = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetGenData -Filter "InstanceName = '$AdapterName'"
            if ($null -eq $Result)
            {
                $TmpMsg = $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
                if ($script:ErrorMessagesWrite -NotContains $TmpMsg)
                {
                    $script:ErrorMessagesWrite += $TmpMsg
                }
                $TableID = -1
                break
            }

            # DumpHeader consists of (Little Endian):
            #   UInt32 BufIndex
            #   UInt32 TableId
            #   UInt32 TableIndex
            #   UInt32 TableSize
            #   UInt32 FwError
            #   UInt32 BufSize
            #   Byte DumpBuf[512]

            $OIDHeaderSize = 48
            $DumpParametersHeaderSize = 48

            $GenDataLength = $Result.GenData.Length
            $DumpHeaderLength = $GenDataLength - $OIDHeaderSize
            $DumpHeader = [byte[]]::new($DumpHeaderLength)
            $DumpBufLength = $DumpHeaderLength - $DumpParametersHeaderSize
            $DumpBufContent = [byte[]]::new($DumpBufLength)

            # Strip OID response header
            [System.Array]::ConstrainedCopy($Result.GenData, 24, $DumpHeader, 0, $DumpHeaderLength)

            # Strip DumpParameters header
            [System.Array]::ConstrainedCopy($DumpHeader, 24, $DumpBufContent, 0, $DumpBufLength)

            $BufIndex        = [bitconverter]::ToUInt32($DumpHeader, 0)
            $NextTableId     = [bitconverter]::ToUInt32($DumpHeader, 4)
            $NextTableIndex  = [bitconverter]::ToUInt32($DumpHeader, 8)
            $TableSize       = [bitconverter]::ToUInt32($DumpHeader, 12)
            $FwError         = [bitconverter]::ToUInt32($DumpHeader, 16)
            $BufSize         = [bitconverter]::ToUInt32($DumpHeader, 20)

            if ($BufIndex -eq 0)
            {
                Write-Verbose "`[OUT`] $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tNextTableID $($NextTableID)`tNextTableIndex $($NextTableIndex)"

                $TableID = $NextTableID
                $TableIndex = $NextTableIndex
                $TableBufferSize = $TableSize
            }

            if (-Not $BufSize)
            {
                break
            }

            $TableBuffer.AddRange($DumpBufContent[0..($BufSize-1)])
        }
        else
        {
            $TmpMsg = $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
            if ($script:ErrorMessagesWrite -NotContains $TmpMsg)
            {
                $script:ErrorMessagesWrite += $TmpMsg
            }
            $TableID = -1

            Write-Verbose "`[OUT`] Buffer $($BufIndex) : $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tNextTableID $($NextTableID)`tNextTableIndex $($NextTableIndex)`tFwError: $($FwError)"
            break
        }
    }

    return [PsCustomObject] @{
        NextTableId = $TableID
        NextTableIndex = $TableIndex
        TableBuffer = $TableBuffer
        TableBufferSize = $TableBufferSize
    }
}

# SIG # Begin signature block
# MIIocAYJKoZIhvcNAQcCoIIoYTCCKF0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBGykbbyQjnuLmT
# q8ku78KmAgC4vo2Jy2DvSH95k0P4sqCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1DCCFdACAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEIFrxIbmwUgsOTnUnmb1n0f8e
# 6pyIzBEKEEQ4btq4JdZIMA0GCSqGSIb3DQEBAQUABIIBgI/Ou70xRqpIGQ2UQrwd
# mGhKVwJAEW0VjyPtOUoOnzIV0x2tDBUr2z8uURR6ieZYui2NPt6IIpaKQIaDuyrG
# zruqlc/a4XD5ntIJJOdNDdjzSjUL45bmWf63WX+pZweO7WKp5QG0/gFSYXV9Goyg
# EOuKDcRov8ymFtAwEXpz/CHN9NyZonfvLKCS+Cf6csgqqzOJrhFdMT0nMT68QQK+
# XQgHmIpVZfl35tCHKkz2+g2zvrRjh4prqky/O8SHrZu+cSbOOrDcQ2m2Ah5aNZUW
# v9K8+VKNhBB9C/Kbstg1Asbj9lH/UQoRMU7Rm9ud73WtjR9FhvQfdHBVb4f68SnT
# Mgky3wcyBe6neeuEKZ6WTMqaaHsy1lU1jg6+sd1ZWixjHBKY/2yfI43v4NU+iXR0
# qOkkHVPx00H1fpFQ5/LYsvLsl+pG/qWIwJriYEt+AIgsXI9lkfogoKKkX8T+Durd
# uqLZMV6/aIxctlyjCEv2POZUhhAca7if8kfHNr/ACq+jZKGCE04wghNKBgorBgEE
# AYI3AwMBMYITOjCCEzYGCSqGSIb3DQEHAqCCEycwghMjAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwge8GCyqGSIb3DQEJEAEEoIHfBIHcMIHZAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIEciOOGng5o1a8KEzVO71wFm1v8I0nEkFQ63GyZAvGy9
# AhR7Hus/bXP/2CsLucgyS2TrEZpjFhgPMjAyNDAzMTgxODI0MDdaoG6kbDBqMQsw
# CQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNp
# Z25lciAjNKCCDekwggb1MIIE3aADAgECAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCSqG
# SIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNo
# ZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAeFw0yMzA1
# MDMwMDAwMDBaFw0zNDA4MDIyMzU5NTlaMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQI
# EwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMM
# I1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICM0MIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEApJMoUkvPJ4d2pCkcmTjA5w7U0RzsaMsBZOSK
# zXewcWWCvJ/8i7u7lZj7JRGOWogJZhEUWLK6Ilvm9jLxXS3AeqIO4OBWZO2h5YEg
# ciBkQWzHwwj6831d7yGawn7XLMO6EZge/NMgCEKzX79/iFgyqzCz2Ix6lkoZE1ys
# /Oer6RwWLrCwOJVKz4VQq2cDJaG7OOkPb6lampEoEzW5H/M94STIa7GZ6A3vu03l
# PYxUA5HQ/C3PVTM4egkcB9Ei4GOGp7790oNzEhSbmkwJRr00vOFLUHty4Fv9Gbsf
# PGoZe267LUQqvjxMzKyKBJPGV4agczYrgZf6G5t+iIfYUnmJ/m53N9e7UJ/6GCVP
# E/JefKmxIFopq6NCh3fg9EwCSN1YpVOmo6DtGZZlFSnF7TMwJeaWg4Ga9mBmkFgH
# gM1Cdaz7tJHQxd0BQGq2qBDu9o16t551r9OlSxihDJ9XsF4lR5F0zXUS0Zxv5F4N
# m+x1Ju7+0/WSL1KF6NpEUSqizADKh2ZDoxsA76K1lp1irScL8htKycOUQjeIIISo
# h67DuiNye/hU7/hrJ7CF9adDhdgrOXTbWncC0aT69c2cPcwfrlHQe2zYHS0RQlNx
# dMLlNaotUhLZJc/w09CRQxLXMn2YbON3Qcj/HyRU726txj5Ve/Fchzpk8WBLBU/v
# uS/sCRMCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7RNV8G
# EgRVMB0GA1UdDgQWBBQDDzHIkSqTvWPz0V1NpDQP0pUBGDAOBgNVHQ8BAf8EBAMC
# BsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBKBgNVHSAE
# QzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3Rp
# Z28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDegNYYzaHR0cDovL2Ny
# bC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3JsMHQGCCsG
# AQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5zZWN0aWdvLmNvbS9T
# ZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEATJtlWPrgec/vFcMy
# bd4zket3WOLrvctKPHXefpRtwyLHBJXfZWlhEwz2DJ71iSBewYfHAyTKx6XwJt/4
# +DFlDeDrbVFXpoyEUghGHCrC3vLaikXzvvf2LsR+7fjtaL96VkjpYeWaOXe8vrqR
# ZIh1/12FFjQn0inL/+0t2v++kwzsbaINzMPxbr0hkRojAFKtl9RieCqEeajXPawh
# j3DDJHk6l/ENo6NbU9irALpY+zWAT18ocWwZXsKDcpCu4MbY8pn76rSSZXwHfDVE
# Ha1YGGti+95sxAqpbNMhRnDcL411TCPCQdB6ljvDS93NkiZ0dlw3oJoknk5fTtOP
# D+UTT1lEZUtDZM9I+GdnuU2/zA2xOjDQoT1IrXpl5Ozf4AHwsypKOazBpPmpfTXQ
# MkCgsRkqGCGyyH0FcRpLJzaq4Jgcg3Xnx35LhEPNQ/uQl3YqEqxAwXBbmQpA+oBt
# lGF7yG65yGdnJFxQjQEg3gf3AdT4LhHNnYPl+MolHEQ9J+WwhkcqCxuEdn17aE+N
# t/cTtO2gLe5zD9kQup2ZLHzXdR+PEMSU5n4k5ZVKiIwn1oVmHfmuZHaR6Ej+yFUK
# 7SnDH944psAU+zI9+KmDYjbIw74Ahxyr+kpCHIkD3PVcfHDZXXhO7p9eIOYJanwr
# CKNI9RX8BE/fzSEceuX1jhrUuUAwggbsMIIE1KADAgECAhAwD2+s3WaYdHypRjan
# eC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VS
# VFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0
# aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAxMTgyMzU5NTlaMH0x
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMc
# U2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQAD
# ggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nPoD5T77NawHbWGgSl
# zkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEFkmZvIavVcRB5h/RG
# Ks3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQFcN/nqJc1zjxFrJw
# 06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1VdEXOzQhe5cBlK9/
# gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpWcdEtlEzIjDzTFKKc
# vSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ffnz82Pg4wZytGtzWv
# lr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV6xlV/9A5VC0m7kUa
# Om/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7wW33pdmmTIBxKlyx3
# GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFkyYiSWcuoRsDJ8qb/
# fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXDTYLQjdprt9v3GEBX
# c/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70atZ2h3pVBeqeDAgMB
# AAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNV
# HQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/BAQDAgGGMBIGA1Ud
# EwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAG
# BgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNv
# bS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDB2BggrBgEF
# BQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNlcnRydXN0LmNvbS9V
# U0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcwAYYZaHR0cDovL29j
# c3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEAbVSBpTNdFuG1U4GR
# dd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNviffmfAO9Aw63T4uRW
# +VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzVWHi/SHbhRGdtj7BD
# EA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJqO0I3fBZ4OZOTlJux
# 3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytCiiGS/qhvWiVwYoFz
# Y16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6NIxi5o8JHHfBd1qFz
# VwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9JkWtgqIsJLnfE5Xkze
# SD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw0NP8srMftpmWJvQ8
# eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUgOw9Bqh/poUGd/rG4
# oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20OAkrSQeq/eww2vGn
# L8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8XwgDCp3rrWiAoa6Ke60
# WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQsMIIEKAIBATCBkTB9MQswCQYDVQQGEwJH
# QjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3Jk
# MRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNB
# IFRpbWUgU3RhbXBpbmcgQ0ECEDlMJeF8oG0nqGXiO9kdItQwDQYJYIZIAWUDBAIC
# BQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUx
# DxcNMjQwMzE4MTgyNDA3WjA/BgkqhkiG9w0BCQQxMgQwZaz7Zvq2UY8N4Ddetpfk
# elRW6wHk6MvsXPP4XRsyJbU57MAyNoOnajLAdhE15ON5MIHtBgsqhkiG9w0BCRAC
# DDGB3TCB2jCB1zAWBBSuYq91Cgy9R9ZGH3Vo4ryM58pPlDCBvAQUAtZbleKDcMFX
# AJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpO
# ZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVT
# RVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBAQUA
# BIICADvLKD8iv67pdvKrw1DKA+JgEXzYWQa/3POdUwFu2SKsMgQBj6rovIzIML+X
# uzAI/L8yuqAlVrKYmNkm7d8Pzlo4A6bJa4d9UgKe3GmOvCDjnyFauoJZ1Jgf2eSd
# BClFvJ8UQhz7S89tyiYwI9G/Ic8Zl0o39aQ2l1jk3DnMxjKEmKIR1ODYCgUc4qDP
# pxHm/BTPwe8qAhL+w2qw7qzhiHy0CKXv9BQdjTuXl/px39E+1325G8krjd4h7cEk
# znNJrj2PnG5N2HBFsB7JwGA4MajJJ9FlmveDreAe0+eC24AR0Nb10HmndA15W9Is
# pkRdD4MM1DbCfX7Py3iXRHAa3AFHhJ4eICEAzEtQc3uKPkFK0kmj77XeDOU+3Otj
# bYDSGnvF7xp8/RkOms4JwnRzgsl2LSddgQNdALWAOfl9WRiEB6rqi32ItBXt7wCE
# alaRZMHWZfBslaaLJ8douSswrNTjtYTXifL8T73YToEBQbnFGl0qXYBN3+l8zGCp
# rWrh5zKke8G1bv4Qs0MeMrHlTZWFO0wKaQW+HLiblL776YRIpsOPxcFYo2sKbn1C
# ol3QOOXAwNLYQtezL9ulP9xa+SfHaDaz/he0+9acJGNnzJ2+5PHr498CD0kpjIAX
# i1auxjwcgIMiHNBSG879R0nN6qHYZHS0QLuLWnYq9rErRpzJ
# SIG # End signature block
