# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Invoke-IntelEthernetIdentifyPort
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null,
    [parameter(Mandatory = $false, Position = 1)]
    [ValidateRange(1, 120)]
    [int]
    $Seconds = 0
    )
    Begin
    {
        GetIntelEthernetDevices
        $AdapterName = $Name
        $script:ErrorMessagesInvoke = @()
        $script:WarningMessagesInvoke = @()
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesInvoke += $Messages.AdapterNotFound -f $Name
                break
            }

            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $AdapterName $Adapter ([ref]$script:ErrorMessagesInvoke)

            $AdapterName = GetSupportedAdapters $PreProcessedAdapterNames ([ref]$ErrorMessagesInvoke)

            if (-not [string]::IsNullOrEmpty($AdapterName))
            {
                $ValidateResult = ValidateIdentifyPortParams $input $AdapterName
                if ($false -eq $ValidateResult)
                {
                    break
                }

                $TmpStatusMsg = CheckDeviceError $AdapterName
                if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                {
                    $script:ErrorMessagesInvoke += $TmpStatusMsg
                }

                try
                {
                    $BlinkResult = ToggleBlinkLED $AdapterName
                    if ($null -eq $BlinkResult)
                    {
                        $script:ErrorMessagesInvoke += $Messages.OperationFailed -f $AdapterName, "Invoke"
                        break
                    }

                    if ($false -eq $BlinkResult.ToggleLED)
                    {
                        # try again in case ToggleLED was already on and we just turned it off
                        ToggleBlinkLED $AdapterName | Out-Null
                    }
                    Start-Sleep -Seconds $Seconds
                }
                finally
                {
                    # Now call again to turn it off
                    ToggleBlinkLED $AdapterName | Out-Null
                }
            }
        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesInvoke)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesInvoke)
        {
            Write-Error $ErrorMessage
        }
    }
}

function ToggleBlinkLED($AdapterName)
{
    # Using Get-CimInstance queries all instances of the class. Using system management queries the specified adapter instance
    $BlinkObj = New-Object System.Management.ManagementObject("root\wmi:IntlLan_BlinkLED.instancename='$AdapterName'")
    if ($null -ne $BlinkObj)
    {
        # BlinkObj needs to be called or read (i.e. $BlinkObj.ToggleLED) to invoke the System.ManagementObject, but we do not want to display output
        # (i.e. piping to Out-Null to avoid displaying). $BlinkObj | Out-Null does not work for blinking the LED
        $BlinkObj.ToggleLED | Out-Null
    }
    return $BlinkObj
}

function ValidateIdentifyPortParams($PipelineInput, $AdapterName)
{
    $Result = $false
    do
    {
        if (-not (ValidateSingleAdapter $PipelineInput $AdapterName ([ref]$script:ErrorMessagesInvoke)))
        {
            break
        }

        $NetAdapterObject = Get-NetAdapter -InterfaceDescription $AdapterName -ErrorAction SilentlyContinue
        if ($NetAdapterObject.State -eq [int][NetAdapterState]::Disabled -or $true -eq $NetAdapterObject.OperationalStatusDownLowPowerState)
        {
            $script:ErrorMessagesInvoke += $Messages.IdentifyPortDisabledOrLowPower
            break
        }

        if ($Seconds -eq 0)
        {
            $script:ErrorMessagesInvoke += $Messages.InvalidParams
            break
        }

        $Result = $true
    } while (0)

    return $Result
}

# SIG # Begin signature block
# MIIocAYJKoZIhvcNAQcCoIIoYTCCKF0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAInWFay9e/tFD8
# 0uhSjDO92fdNzGLNIc/5WaBy9H/wjaCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1DCCFdACAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEILNY/tX1h4Rrc1Z5BsLiweyP
# ziZL+P3e7bxslwVBq+lWMA0GCSqGSIb3DQEBAQUABIIBgLA0tIr+bGbqh6dvKmAd
# uxo9oEmVOV0yx1XJYaKayBPXS3mguTEy2trFsCc072lTtqXdjxgMl2zsXgLvFL+H
# JlzSEWJR7m8/daqwwAoLhTwqNTzikfCtt3m0ni7UFmy2nfmHu72tQM5M7joPKuwl
# l5D+JJrQMwjScXNQ8Cw37viKlfp0xCwxZ6ILlN0rtPeSVH9GV4gH1ceKYbWp/ftX
# v4FNKZO4Wi6aV16teqAguQuLA+rTz7QGjogoy8XwAaTOFRopBeFkX5snymsrAlUX
# cOZdPZ7FJ/iJYUcNCeKQ9Ew6Bxu4+7D2KJSBwbR0/R2R4MItmqhuFTyx5pNGoXxF
# /KL0UI3QGPnvWP1mg9EHvAby7mHyfYdEzx5YNEuZqs7R/gl2wvLriFcL22rmc6WT
# 9qM0ijdb1hm2MPKor0AGyH8KpyoshisVB4ocxMVt4bClGS9zEiDxHLm+QHKW/YCA
# YKJpTwNCrlVNkVzXMI81gO2VxsOQ+Or7Tg/fgUVIatV5/KGCE04wghNKBgorBgEE
# AYI3AwMBMYITOjCCEzYGCSqGSIb3DQEHAqCCEycwghMjAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwge8GCyqGSIb3DQEJEAEEoIHfBIHcMIHZAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIAfz0Hc7C4XLuVJPRaAavOeSALujIGlPXgRE+ysdsXcH
# AhRXxNALwPy90oJ/lat/WTtuGTCLDhgPMjAyNDAzMTgxODI0MDRaoG6kbDBqMQsw
# CQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNp
# Z25lciAjNKCCDekwggb1MIIE3aADAgECAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCSqG
# SIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNo
# ZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAeFw0yMzA1
# MDMwMDAwMDBaFw0zNDA4MDIyMzU5NTlaMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQI
# EwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMM
# I1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICM0MIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEApJMoUkvPJ4d2pCkcmTjA5w7U0RzsaMsBZOSK
# zXewcWWCvJ/8i7u7lZj7JRGOWogJZhEUWLK6Ilvm9jLxXS3AeqIO4OBWZO2h5YEg
# ciBkQWzHwwj6831d7yGawn7XLMO6EZge/NMgCEKzX79/iFgyqzCz2Ix6lkoZE1ys
# /Oer6RwWLrCwOJVKz4VQq2cDJaG7OOkPb6lampEoEzW5H/M94STIa7GZ6A3vu03l
# PYxUA5HQ/C3PVTM4egkcB9Ei4GOGp7790oNzEhSbmkwJRr00vOFLUHty4Fv9Gbsf
# PGoZe267LUQqvjxMzKyKBJPGV4agczYrgZf6G5t+iIfYUnmJ/m53N9e7UJ/6GCVP
# E/JefKmxIFopq6NCh3fg9EwCSN1YpVOmo6DtGZZlFSnF7TMwJeaWg4Ga9mBmkFgH
# gM1Cdaz7tJHQxd0BQGq2qBDu9o16t551r9OlSxihDJ9XsF4lR5F0zXUS0Zxv5F4N
# m+x1Ju7+0/WSL1KF6NpEUSqizADKh2ZDoxsA76K1lp1irScL8htKycOUQjeIIISo
# h67DuiNye/hU7/hrJ7CF9adDhdgrOXTbWncC0aT69c2cPcwfrlHQe2zYHS0RQlNx
# dMLlNaotUhLZJc/w09CRQxLXMn2YbON3Qcj/HyRU726txj5Ve/Fchzpk8WBLBU/v
# uS/sCRMCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7RNV8G
# EgRVMB0GA1UdDgQWBBQDDzHIkSqTvWPz0V1NpDQP0pUBGDAOBgNVHQ8BAf8EBAMC
# BsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBKBgNVHSAE
# QzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3Rp
# Z28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDegNYYzaHR0cDovL2Ny
# bC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3JsMHQGCCsG
# AQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5zZWN0aWdvLmNvbS9T
# ZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEATJtlWPrgec/vFcMy
# bd4zket3WOLrvctKPHXefpRtwyLHBJXfZWlhEwz2DJ71iSBewYfHAyTKx6XwJt/4
# +DFlDeDrbVFXpoyEUghGHCrC3vLaikXzvvf2LsR+7fjtaL96VkjpYeWaOXe8vrqR
# ZIh1/12FFjQn0inL/+0t2v++kwzsbaINzMPxbr0hkRojAFKtl9RieCqEeajXPawh
# j3DDJHk6l/ENo6NbU9irALpY+zWAT18ocWwZXsKDcpCu4MbY8pn76rSSZXwHfDVE
# Ha1YGGti+95sxAqpbNMhRnDcL411TCPCQdB6ljvDS93NkiZ0dlw3oJoknk5fTtOP
# D+UTT1lEZUtDZM9I+GdnuU2/zA2xOjDQoT1IrXpl5Ozf4AHwsypKOazBpPmpfTXQ
# MkCgsRkqGCGyyH0FcRpLJzaq4Jgcg3Xnx35LhEPNQ/uQl3YqEqxAwXBbmQpA+oBt
# lGF7yG65yGdnJFxQjQEg3gf3AdT4LhHNnYPl+MolHEQ9J+WwhkcqCxuEdn17aE+N
# t/cTtO2gLe5zD9kQup2ZLHzXdR+PEMSU5n4k5ZVKiIwn1oVmHfmuZHaR6Ej+yFUK
# 7SnDH944psAU+zI9+KmDYjbIw74Ahxyr+kpCHIkD3PVcfHDZXXhO7p9eIOYJanwr
# CKNI9RX8BE/fzSEceuX1jhrUuUAwggbsMIIE1KADAgECAhAwD2+s3WaYdHypRjan
# eC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VS
# VFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0
# aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAxMTgyMzU5NTlaMH0x
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMc
# U2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQAD
# ggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nPoD5T77NawHbWGgSl
# zkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEFkmZvIavVcRB5h/RG
# Ks3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQFcN/nqJc1zjxFrJw
# 06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1VdEXOzQhe5cBlK9/
# gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpWcdEtlEzIjDzTFKKc
# vSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ffnz82Pg4wZytGtzWv
# lr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV6xlV/9A5VC0m7kUa
# Om/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7wW33pdmmTIBxKlyx3
# GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFkyYiSWcuoRsDJ8qb/
# fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXDTYLQjdprt9v3GEBX
# c/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70atZ2h3pVBeqeDAgMB
# AAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNV
# HQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/BAQDAgGGMBIGA1Ud
# EwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAG
# BgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNv
# bS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDB2BggrBgEF
# BQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNlcnRydXN0LmNvbS9V
# U0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcwAYYZaHR0cDovL29j
# c3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEAbVSBpTNdFuG1U4GR
# dd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNviffmfAO9Aw63T4uRW
# +VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzVWHi/SHbhRGdtj7BD
# EA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJqO0I3fBZ4OZOTlJux
# 3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytCiiGS/qhvWiVwYoFz
# Y16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6NIxi5o8JHHfBd1qFz
# VwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9JkWtgqIsJLnfE5Xkze
# SD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw0NP8srMftpmWJvQ8
# eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUgOw9Bqh/poUGd/rG4
# oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20OAkrSQeq/eww2vGn
# L8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8XwgDCp3rrWiAoa6Ke60
# WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQsMIIEKAIBATCBkTB9MQswCQYDVQQGEwJH
# QjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3Jk
# MRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNB
# IFRpbWUgU3RhbXBpbmcgQ0ECEDlMJeF8oG0nqGXiO9kdItQwDQYJYIZIAWUDBAIC
# BQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUx
# DxcNMjQwMzE4MTgyNDA0WjA/BgkqhkiG9w0BCQQxMgQwLMSja9ByzgJOIAqP/Oii
# 5XQWQvriO+QA48Gtg7VlIzdMNin/bELZ4t2zh7YlcyUGMIHtBgsqhkiG9w0BCRAC
# DDGB3TCB2jCB1zAWBBSuYq91Cgy9R9ZGH3Vo4ryM58pPlDCBvAQUAtZbleKDcMFX
# AJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpO
# ZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVT
# RVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBAQUA
# BIICAJJX1lyeP0QZnl9YKyEUH1T6hPtfiQGmwWXhg63bkxsl/iZ+nTFy+Q6p7svF
# BQLifouMxVDBUYC5ngH4k1yD43rr0Df5fNEIRxctAUXY/tOaE5BFQXu9zpb9Wt0Q
# n3HRRHIkxyf0Wf5u9Yi8+WbLWNbekQWyeMoohF6fyiSZBwkM92kS0ZjZ4WWty3ib
# RvlVZIOx6hjZUfvV6Ty7IkIKZrMAihclFHl3WBYb+D6a4eMV8fuyHj+nw/laxjm7
# EL9lh5v2iQQZw/VlMNdcN5It+Sqx1tKYGdR4w2g5tdnab3gXhS2vLWnF2r/K8bxN
# PGuQjsAAhG7GhuJxUjLr7Ln9Isu94cX8QaBhnTOQ+wQX7sJejkvGkP7ImoSdn/As
# m8Y+IVBKdqyM06IuFyywv9xAcF+keGZSYCFrxpbRVJWoo4W4VP8yaLSXXvj5veWk
# DMzSIRc1li+DZi+3S+M4zFG5EapFz7VF4cz7cIAf9CZT5Z02Ab4JdHAVZ9O/X5FX
# 6vqlUijsXwpiCw+ic5ZDPuD5BEiB5iqR5Xni9D/bWEbQ8/c9jVX+OfEDLGAdE2qi
# owbGuG4ua9EUjp3wiG9ePje9YCBGqn5Yb8fzbEvqTuowU5IWHXYmK7OqrCsif5ab
# Go5R8eIzCFYIFWidGodxiiJiLPyTUPYxwAXRsieraKXx1y/b
# SIG # End signature block
