# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetSetting
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $DisplayName = '',
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object]
    $DisplayValue = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $RegistryKeyword = '',
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object]
    $RegistryValue = $null,
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Setting = $null,
    [parameter(Mandatory = $false)]
    [switch]
    $Reset
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesSet = @()
        $script:WarningMessagesSet = @()
        $FinalObject = @()
        GetIntelEthernetDevices @('icei*', 'i40ei*')
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
        $script:MSNetAdvProperty = @(Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue)
    }
    Process
    {
        $Adapters = $Adapter

        if (IsSettingObject $Adapters)
        {
            $Setting = $Adapters
            $Adapters = $null
        }

        do
        {
            if (!(ValidateParams $DisplayName $DisplayValue $RegistryKeyword $RegistryValue $Setting $AdapterName $Adapters))
            {
                break
            }

            if ($null -ne $Setting)
            {
                $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Setting.Name $null ([ref]$script:ErrorMessagesSet)
                $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet) | Get-Unique)
            }
            else
            {
                $PreProcessedAdapterNames = ValidateSetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesSet)
                $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))
            }

            try
            {
                $PreErrorActionPreference = $global:ErrorActionPreference
                $global:ErrorActionPreference = 'SilentlyContinue'
                foreach ($a in $AdapterNamesArray)
                {
                    $TmpStatusMsg = CheckDeviceError $a
                    if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                    {
                        $script:ErrorMessagesSet += $TmpStatusMsg
                    }

                    $script:isProfileSetting = $false
                    $AdapterSettings = @()
                    if ($null -ne $Setting)
                    {
                        $AdapterSettings = GetSetSettings $null $Setting.RegistryKeyword $a
                    }
                    else
                    {
                        $AdapterSettings = GetSetSettings $DisplayName $RegistryKeyword $a
                    }

                    if ($AdapterSettings.RegistryKeyword -Contains 'NetworkAddress' -and -not $Reset)
                    {
                        if ($false -eq (ValidateLAASetting $DisplayValue $RegistryValue))
                        {
                            break
                        }
                    }

                    if ($script:isProfileSetting)
                    {
                        $FinalObject += SetProfileSettings $a $DisplayValue $RegistryValue
                    }

                    foreach ($adapterSetting in $AdapterSettings)
                    {
                        if ($null -ne $DisplayValue)
                        {
                            Set-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName -DisplayValue $DisplayValue
                            $FinalObject += ValidateSetting $adapterSetting $DisplayValue $null $a
                        }
                        elseif ($null -ne $RegistryValue)
                        {
                            Set-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName -RegistryValue $RegistryValue
                            $FinalObject += ValidateSetting $adapterSetting $null $RegistryValue $a
                        }
                        elseif ($Reset)
                        {
                            Reset-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName
                            $FinalObject += ValidateSetting $adapterSetting $adapterSetting.DefaultDisplayValue $null $a
                        }
                    }
                }
            }
            finally
            {
                $global:ErrorActionPreference = $PreErrorActionPreference
            }
        } while ($false)
    }
    End
    {
        $FinalObject

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function ValidateParams($DisplayName, $DisplayValue, $RegistryKeyword, $RegistryValue, $Setting, $AdapterName, $Adapters)
{
    $bContinue = $true

    do
    {
        # if DisplayName, RegistryKeyword, & Setting are not used, or both DisplayName & RegistryKeyword are used - stop execution
        if (([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword) -and $null -eq $Setting) -or
            (-not [string]::IsNullOrEmpty($DisplayName) -and -not [string]::IsNullOrEmpty($RegistryKeyword)))
        {
            $bContinue = $false
            break
        }

        # if both DisplayValue & RegistryValue are not used & Reset isn't used, or both DisplayValue & RegistryValue are used - stop exection
        if (([string]::IsNullOrEmpty($DisplayValue) -and [string]::IsNullOrEmpty($RegistryValue) -and -not $Reset) -or
            (-not [string]::IsNullOrEmpty($DisplayValue) -and -not [string]::IsNullOrEmpty($RegistryValue)))
        {
            $bContinue = $false
            break
        }

        # if either DisplayValue & RegistryValue are used and $Reset is used - stop execution
        if (-not [string]::IsNullOrEmpty($DisplayValue) -or -not [string]::IsNullOrEmpty($RegistryValue) -and $Reset)
        {
            $bContinue = $false
            break
        }

        # if Setting is used and either DisplayName, RegistryKeyword are used - stop execution
        if ($null -ne $Setting -and (-not [string]::IsNullOrEmpty($DisplayName) -or -not [string]::IsNullOrEmpty($RegistryKeyword)))
        {
            $bContinue = $false
            break
        }

        # if Setting is used and either $AdapterName or $Adapters is used - stop execution
        if ($null -ne $Setting -and (-not [string]::IsNullOrEmpty($AdapterName) -or $Adapters))
        {
            $bContinue = $false
            break
        }

    } while ($false)

    if ($false -eq $bContinue)
    {
        $script:ErrorMessagesSet += $Messages.InvalidParams
    }

    return $bContinue
}

function GetSetSettings($DisplayName, $RegistryKeyword, $AdapterName)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            if ($TmpDisplayName -eq 'Profile')
            {
                $script:isProfileSetting = $true
            }
            else
            {
                $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName})

                if (-not $TmpSetting)
                {
                    $script:ErrorMessagesSet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
                }
                else
                {
                    $SettingArray += $TmpSetting
                }
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            if ($TmpRegistryKeyword -eq 'PerformanceProfile')
            {
                $script:isProfileSetting = $true
            }
            else
            {
                $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword})

                if (-not $TmpSetting)
                {
                    $script:ErrorMessagesSet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
                }
                else
                {
                    $SettingArray += $TmpSetting
                }
            }
        }
    }

    return $SettingArray
}

function ValidateSetting($Setting, $DisplayValue = $null, $RegistryValue = $null, $AdapterName)
{
    $IntelEthernetSetting = Get-IntelEthernetSetting -Name $a -DisplayName $Setting.DisplayName -WarningAction:SilentlyContinue
    # setting type 'int'
    if ($null -ne $IntelEthernetSetting.PSTypeNames -and $IntelEthernetSetting.PSTypeNames[0] -eq "IntelEthernetSettingInt")
    {
        if ($null -ne $DisplayValue)
        {
            $SettingIntValue = $DisplayValue
        }
        else
        {
            $SettingIntValue = $RegistryValue
        }

        $bValidIntValue = $false
        # check if numeric value
        if ($SettingIntValue -match "^[\d]+$")
        {
            if ([int]$SettingIntValue -ge $IntelEthernetSetting.Min -and [int]$SettingIntValue -le $IntelEthernetSetting.Max)
            {
                if (($SettingIntValue % $IntelEthernetSetting.Step) -eq 0)
                {
                    $bValidIntValue = $true
                }
            }
        }

        if (-not $bValidIntValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $SettingIntValue, $Setting.RegistryKeyword
        }
        elseif ($SettingIntValue -ne $IntelEthernetSetting.DisplayValue)
        {
            $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.RegistryKeyword
        }
        else
        {
            return $IntelEthernetSetting
        }
    }
    else
    {
        $bValidEnumValue = $false
        if ($null -ne $DisplayValue)
        {
            if ($IntelEthernetSetting.DescriptionMap -notcontains $DisplayValue -and $IntelEthernetSetting.RegistryKeyword -ne "NetworkAddress")
            {
                $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $DisplayValue, $Setting.RegistryKeyword
            }
            elseif ($DisplayValue.ToString() -ne $IntelEthernetSetting.DisplayValue)
            {
                $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.DisplayName
            }
            else
            {
                $bValidEnumValue = $true
            }
        }
        elseif ($null -ne $RegistryValue)
        {
            if ($IntelEthernetSetting.PossibleValues -notcontains $RegistryValue -and $IntelEthernetSetting.RegistryKeyword -ne "NetworkAddress")
            {
                $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $RegistryValue, $Setting.RegistryKeyword
            }
            elseif (($RegistryValue.ToString() -ne $IntelEthernetSetting.RegistryValue))
            {
                $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.DisplayName
            }
            else
            {
                $bValidEnumValue = $true
            }
        }

        if ($bValidEnumValue)
        {
            return $IntelEthernetSetting
        }
    }
}

function ValidateLAASetting($DisplayValue, $RegistryValue)
{
    $Valid = $False
    do
    {
        if (-not [string]::IsNullOrEmpty($DisplayValue))
        {
            $Value = $DisplayValue
        }
        else
        {
            $Value = $RegistryValue
        }
        # first check if the string has correct length
        if ($Value.Length -eq $NETWORK_ADDRESS_LEN)
        {
            # filter out all non-hexadecimal numbers
            try
            {
                $ValueInt = [Convert]::ToUInt64($Value, 16)
            }
            catch
            {
                break
            }
            # check for reserved and multicast addresses
            if ($ValueInt -band 0x010000000000 -or $ValueInt -eq 0 -or $ValueInt -eq 0xFFFFFFFFFFFF)
            {
                break
            }
            $Valid = $True
        }
    } while($false)

    if ($false -eq $Valid)
    {
        $script:ErrorMessagesSet += $Messages.InvalidSettingValueGeneric
    }
    return $Valid
}

function IsSettingObject($PipedObject)
{
    $bSettingObject = $false
    if ($null -ne $PipedObject -and $null -ne $PipedObject[0].PSTypeNames -and
        ($PipedObject[0].PSTypeNames[0]).Contains("IntelEthernetSetting"))
    {
        $bSettingObject = $true
    }

    return $bSettingObject
}

function SetProfileSettings($AdapterName, $DisplayValue, $RegistryValue)
{
    $ProfileSetting = InitializeProfileSetting $AdapterName

    if ($null -ne $DisplayValue)
    {
        if ($ProfileSetting.DescriptionMap -notcontains $DisplayValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $DisplayValue, $ProfileSetting.RegistryKeyword
        }
        else
        {
            $ProfileRegValue = $PROFILE_VALUE_TO_NAME.GetEnumerator().Where({ $_.Value -eq $DisplayValue}).Name
        }
    }
    elseif ($null -ne $RegistryValue)
    {
        if ($ProfileSetting.PossibleValues -notcontains $RegistryValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $RegistryValue, $ProfileSetting.RegistryKeyword
        }
        else
        {
            $ProfileRegValue = $RegistryValue
        }
    }
    elseif ($Reset)
    {
        $ProfileRegValue = $ProfileSetting.DefaultValue
    }

    if ($null -ne $ProfileRegValue)
    {
        SetAdapterPropertyInRegistry $AdapterName 'PerformanceProfile' $ProfileRegValue

        $DriverFamily = $script:SupportedAdapters.Where({ $_.Name -eq $AdapterName }).Service
        $ProfileCmdletValName = $PROFILE_VALUE_TO_NAME[$ProfileRegValue.ToString()]
        $tmp = GetProfileSettingsFromXml $ProfileCmdletValName $DriverFamily
        $tmp.GetEnumerator().ForEach(
            {
                Set-NetAdapterAdvancedProperty -InterfaceDescription $AdapterName -RegistryKeyword $_.Key -RegistryValue $_.Value
            })

        $script:MSNetAdvProperty = Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue
        return InitializeProfileSetting $AdapterName
    }
}

# SIG # Begin signature block
# MIIocQYJKoZIhvcNAQcCoIIoYjCCKF4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCLN5FwYb/sNfWG
# 5Obua/HWn5GcYyrg0JLLeysq+NyofKCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1TCCFdECAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEINiKB+8wp2n1Rs0gY2wRzNPt
# tiMQzH/P+J7mPNFw/vOhMA0GCSqGSIb3DQEBAQUABIIBgD1qErMehAXGSBBhXHmB
# YIz2PrMmt9rAsvAXLR3/5q2jDyqBMolpI1YyOCH0wki3Y7WliPHz/GzlsRo4vOmS
# K+zv3yff2GAmNaUZwNZSOtG0p22ac4L+yreh3874AIpTupfDLr3pzD6gh+3ECx7Y
# iW5ZO4+wjgXDPUf5bW90FJg4i60VU/Z1qCJlGN/g3NTRKcGfuUYbZi923v8zq9iS
# UXNO9KAnICjPlpqWsbyTLyvbrsO6vm6PG0Co58c3D7S0W+I4v0H2zgKzKDJ/OvMK
# szvkkvFC6CctnlO0ftATZG6gSQfOtOh9HheFNtWG53N9T9lP+ijNcmBsoFIOLqk9
# rqaMdhruPP1yTJwhxTUEv6qJX2DCTJxzwBud8p2MQCxk1oYzVFMN00wCVZbZRnyR
# uZWDb33UGaFSAPZVV3QyxOx6DWuk9U7jIbWQ0gjVveeTTC1nmCSIVggV8jYGd2+I
# bkEJ3OYklkfv4/v5nB9xQVrM04NF+35It+9fMz3DB32XoaGCE08wghNLBgorBgEE
# AYI3AwMBMYITOzCCEzcGCSqGSIb3DQEHAqCCEygwghMkAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwgfAGCyqGSIb3DQEJEAEEoIHgBIHdMIHaAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIJ11Mo1LX5NzhCshrkMchH/J5X9lkObcPmQ8GK4Gv3+O
# AhUAgT+x5PqWp+vSV3Ri5wT2AaynSrMYDzIwMjQwMzE4MTgyNDA1WqBupGwwajEL
# MAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBT
# aWduZXIgIzSggg3pMIIG9TCCBN2gAwIBAgIQOUwl4XygbSeoZeI72R0i1DANBgkq
# hkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5j
# aGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwHhcNMjMw
# NTAzMDAwMDAwWhcNMzQwODAyMjM1OTU5WjBqMQswCQYDVQQGEwJHQjETMBEGA1UE
# CBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# DCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjNDCCAiIwDQYJKoZI
# hvcNAQEBBQADggIPADCCAgoCggIBAKSTKFJLzyeHdqQpHJk4wOcO1NEc7GjLAWTk
# is13sHFlgryf/Iu7u5WY+yURjlqICWYRFFiyuiJb5vYy8V0twHqiDuDgVmTtoeWB
# IHIgZEFsx8MI+vN9Xe8hmsJ+1yzDuhGYHvzTIAhCs1+/f4hYMqsws9iMepZKGRNc
# rPznq+kcFi6wsDiVSs+FUKtnAyWhuzjpD2+pWpqRKBM1uR/zPeEkyGuxmegN77tN
# 5T2MVAOR0Pwtz1UzOHoJHAfRIuBjhqe+/dKDcxIUm5pMCUa9NLzhS1B7cuBb/Rm7
# HzxqGXtuuy1EKr48TMysigSTxleGoHM2K4GX+hubfoiH2FJ5if5udzfXu1Cf+hgl
# TxPyXnypsSBaKaujQod34PRMAkjdWKVTpqOg7RmWZRUpxe0zMCXmloOBmvZgZpBY
# B4DNQnWs+7SR0MXdAUBqtqgQ7vaNereeda/TpUsYoQyfV7BeJUeRdM11EtGcb+Re
# DZvsdSbu/tP1ki9ShejaRFEqoswAyodmQ6MbAO+itZadYq0nC/IbSsnDlEI3iCCE
# qIeuw7ojcnv4VO/4ayewhfWnQ4XYKzl021p3AtGk+vXNnD3MH65R0Hts2B0tEUJT
# cXTC5TWqLVIS2SXP8NPQkUMS1zJ9mGzjd0HI/x8kVO9urcY+VXvxXIc6ZPFgSwVP
# 77kv7AkTAgMBAAGjggGCMIIBfjAfBgNVHSMEGDAWgBQaofhhGSAPw0F3RSiO0TVf
# BhIEVTAdBgNVHQ4EFgQUAw8xyJEqk71j89FdTaQ0D9KVARgwDgYDVR0PAQH/BAQD
# AgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwSgYDVR0g
# BEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6Ly9zZWN0
# aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1UdHwQ9MDswOaA3oDWGM2h0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNybDB0Bggr
# BgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQuc2VjdGlnby5jb20v
# U2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggIBAEybZVj64HnP7xXD
# Mm3eM5Hrd1ji673LSjx13n6UbcMixwSV32VpYRMM9gye9YkgXsGHxwMkysel8Cbf
# +PgxZQ3g621RV6aMhFIIRhwqwt7y2opF87739i7Efu347Wi/elZI6WHlmjl3vL66
# kWSIdf9dhRY0J9Ipy//tLdr/vpMM7G2iDczD8W69IZEaIwBSrZfUYngqhHmo1z2s
# IY9wwyR5OpfxDaOjW1PYqwC6WPs1gE9fKHFsGV7Cg3KQruDG2PKZ++q0kmV8B3w1
# RB2tWBhrYvvebMQKqWzTIUZw3C+NdUwjwkHQepY7w0vdzZImdHZcN6CaJJ5OX07T
# jw/lE09ZRGVLQ2TPSPhnZ7lNv8wNsTow0KE9SK16ZeTs3+AB8LMqSjmswaT5qX01
# 0DJAoLEZKhghssh9BXEaSyc2quCYHIN158d+S4RDzUP7kJd2KhKsQMFwW5kKQPqA
# bZRhe8huuchnZyRcUI0BIN4H9wHU+C4RzZ2D5fjKJRxEPSflsIZHKgsbhHZ9e2hP
# jbf3E7TtoC3ucw/ZELqdmSx813UfjxDElOZ+JOWVSoiMJ9aFZh35rmR2kehI/shV
# Cu0pwx/eOKbAFPsyPfipg2I2yMO+AIccq/pKQhyJA9z1XHxw2V14Tu6fXiDmCWp8
# KwijSPUV/ARP380hHHrl9Y4a1LlAMIIG7DCCBNSgAwIBAgIQMA9vrN1mmHR8qUY2
# p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5l
# dyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNF
# UlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNh
# dGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/IJC9nIAhVJO5z6A+U++zWsB21hoE
# pc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnkuyWPCE5u2hj8BBZJmbyGr1XEQeYf0
# RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnTv/M5u7mkI0Ks0BXDf56iXNc48Ray
# cNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwPDU+zkQGObp0ndVXRFzs0IXuXAZSv
# f4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN9Q0/29DqhYyKVnHRLZRMyIw80xSi
# nL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gpGaQpL0bj3duRX58/Nj4OMGcrRrc1
# r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMSHv2qyghYh3ekFesZVf/QOVQtJu5F
# GjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdonigbSlofe6GsO8Ft96XZpkyAcSpcs
# dxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179g6JTZ8qx+o2hZMmIklnLqEbAyfKm
# /31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkUAx4p+hulIq6lw02C0I3aa7fb9xhA
# V3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLvbxhQiddPnTO9GrWdod6VQXqngwID
# AQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/WqorSs9UgOHYm8Cd8rIDZsswHQYD
# VR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRVMA4GA1UdDwEB/wQEAwIBhjASBgNV
# HRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAG1UgaUzXRbhtVOB
# kXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJWZApbGHLtVDb4n35nwDvQMOt0+Lk
# VvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc87QRD9XVw+K81Vh4v0h24URnbY+w
# QxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYIuM1cfwGWvnJSajtCN3wWeDmTk5Sb
# sdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8VmBvbz1kteraMrQoohkv6ob1olcGKB
# c2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncEvAQ2ted3y5wujSMYuaPCRx3wXdah
# c1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSmTqo9CfUFbLfSZFrYKiLCS53xOV5M
# 3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/W0NdZT1QgW6r8NDT/LKzH7aZlib0
# PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYybxzrF06Uw3hlIDsPQaof6aFBnf6x
# uKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/qEZ0eBsambTJdtDgJK0kHqv3sMNrx
# py/Pt/360KOE2See+wFmd7lWEOEgbsausfm2usg1XTN2jvF8IAwqd661ogKGuinu
# tFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELDCCBCgCAQEwgZEwfTELMAkGA1UEBhMC
# R0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9y
# ZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJT
# QSBUaW1lIFN0YW1waW5nIENBAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCWCGSAFlAwQC
# AgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkF
# MQ8XDTI0MDMxODE4MjQwNVowPwYJKoZIhvcNAQkEMTIEMPp728MiFIMIov1uREdq
# Vr7lCLZEmPxjsnPMaJyqTpwJTAnXXK66NfLNag6o9h+kNTCB7QYLKoZIhvcNAQkQ
# Agwxgd0wgdowgdcwFgQUrmKvdQoMvUfWRh91aOK8jOfKT5QwgbwEFALWW5Xig3DB
# VwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQEF
# AASCAgCKoZmAI/TyR5Z54SeozGxPczqwiJ4SSN01Y/cgjaxeUrC22iHgv2Tk2vGU
# 7FeeXrJupqrnErRiQfA43J3rS24xSIgLsSw8JctyiRFkKUDuC0E1N3+SUOA8eu0J
# IiFaz1BniqaTLZTigNZbb5suJ/GnviWIMLt45M7xbeqeKkjI7bBGO1NHcUcM69g1
# cNyMl9w+2+SjbAtLUm7WFArjwsItXkZTSkoCfM5iWxi3u2ZOF/g2lkDV5gkwfn6d
# SBVKHhMcEx/hbU+ahk4x39VmatfxxTYGrtwL0UPzW0sr6JuidJSGyIkPNWlr7BWO
# oNLgJr6XT6lgTBSnETJBPXzBuIR4SRiPpuKWqupFFQ9xvVZgnJ0hlqgkudzZzTx0
# MscdIXAQugggOXR6mMKv1fQEFD36Jv1pWt6Xxea4xIsTUwTIjvkctluXyTCtKa21
# fQWsj/vK6md+rExpGL2Pj3xlMDx9tyO92I+851ncDgAKdj91C9sCf7zi7G1Xm/Ad
# erthSv2CszcI6HQBJkiwLJXVFe2pATABykQVoif4R8eoIr0RaIieJQKVQ3KKqXs2
# BxNHeOFWbV/C+fvZ4T2UGPdzXCmP7RMXe2XOHAtZ/KNL65jWruZFWeQE6o8vHL5g
# zonGZNG+UeXeL1kh4B8LJU13i31Ga02VCacH481ZsPGOUvvyow==
# SIG # End signature block
