# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernet
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $FinalObject = @()
        GetIntelEthernetDevices @('icei*', 'i40ei*')
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
        GetIntelDriverInfo
        $script:MSNetHwInfo = @(Get-NetAdapterHardwareInfo -ErrorAction SilentlyContinue)
        $script:MSNetAdvProperty = @(Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue)
    }
    Process
    {
        $Adapters = $Adapter
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNamesArray)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
            }

            $SupportedSpeeds = GetAdapterSupportedSpeeds $a
            $AdapterStatuses = GetAdapterStatus $a
            $BusTypes = GetAdapterBusType $a
            $Capabilities = GetAdapterCapabilities $a $SupportedSpeeds
            $ConnectionNames = GetAdapterConnectionName $a
            $DDPPackageNameAndVersions = GetAdapterDDPPackageNameAndVersion $a
            $DDPPackageTrackIds = GetAdapterDDPPackageTrackId $a
            $DeviceStatuses = GetAdapterDeviceStatus $a $SupportedSpeeds
            $DriverVersion = GetDriverVersion $a
            $EEEStatuses = GetAdapterEEEStatus $a
            $ETrackIDs = GetAdapterETrackId $a
            $FullDuplex = GetFullDuplex $a
            $LinkSpeedDuplex = GetLinkSpeedDuplex $a
            $MaxSpeeds = GetAdapterMaxSpeed $a $SupportedSpeeds
            $MediaTypes = GetAdapterMediaType $a
            $MiniPortNames = GetAdapterMiniPortName $a
            $NVMVersions = GetAdapterNVMVersion $a
            $NegotiatedLinkSpeed = GetAdapterNegotiatedLinkSpeed $a
            $NegotiatedLinkWidth = GetAdapterNegotiatedLinkWidth $a
            $NetlistVersions = GetAdapterNetlistVersion $a
            $OemFwVersions = GetOemFwVersion $a
            $OriginalDisplayNames = GetOriginalDisplayName $a
            $PCIDeviceIDs = GetAdapterPCIDeviceID $a
            $PartNumbers = GetAdapterPartNumber $a
            $PciLocations = GetAdapterPCILocation $a
            $AdapterPort = GetAdapterPort $a
            $SanMacAddresses = GetAdapterSanMacAddress $a

            # Assemble it all together in PSCustomObject
            $FinalObject += [PsCustomObject] @{
                PSTypeName          = 'IntelEthernetAdapter'
                AdapterStatus       = $AdapterStatuses
                BusType             = $BusTypes.BusType
                BusTypeString       = $BusTypes.BusTypeString
                Capabilities        = $Capabilities
                ConnectionName      = $ConnectionNames
                DDPPackageName      = $DDPPackageNameAndVersions.Name
                DDPPackageTrackId   = $DDPPackageTrackIds
                DDPPackageVersion   = $DDPPackageNameAndVersions.Version
                DeviceStatus        = $DeviceStatuses.DeviceStatus
                DeviceStatusString  = $DeviceStatuses.DeviceStatusString
                DriverVersion       = $DriverVersion
                EEE                 = $EEEStatuses.EEEStatus
                EEEString           = $EEEStatuses.EEEStatusString
                ETrackID            = $ETrackIDs
                FWVersion           = $OemFwVersions
                FullDuplex          = $FullDuplex
                Location            = $PciLocations
                MaxSpeed            = $MaxSpeeds
                MediaType           = $MediaTypes.MediaType
                MediaTypeString     = $MediaTypes.MediaTypeString
                MiniPortInstance    = $MiniPortNames.Instance
                MiniPortName        = $MiniPortNames.Name
                NVMVersion          = $NVMVersions
                Name                = $a
                NegotiatedLinkSpeed       = $NegotiatedLinkSpeed.NegotiatedLinkSpeed
                NegotiatedLinkSpeedString = $NegotiatedLinkSpeed.NegotiatedLinkSpeedString
                NegotiatedLinkWidth       = $NegotiatedLinkWidth.NegotiatedLinkWidth
                NegotiatedLinkWidthString = $NegotiatedLinkWidth.NegotiatedLinkWidthString
                NetlistVersion      = $NetlistVersions
                OriginalDisplayName = $OriginalDisplayNames
                PCIDeviceID         = $PCIDeviceIDs
                PartNumber          = $PartNumbers
                PortNumber          = $AdapterPort.Port
                PortNumberString    = $AdapterPort.PortString
                SANMacAddress       = $SanMacAddresses
                Speed               = $LinkSpeedDuplex.Speed
                SpeedString         = $LinkSpeedDuplex.SpeedString
                }
        }
    }
    End
    {
        $FinalObject | Sort-Object -Property Location

        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function GetAdapterBusType($AdapterName)
{
    $BusType       = 0
    $BusTypeString = $Messages.Unknown

    foreach ($Bus in $script:BusTypesArray)
    {
        if (($null -ne $Bus) -and ($Bus.InstanceName -eq $AdapterName))
        {
            $BusType       = [int]$Bus.BusType
            $BusTypeString = $BusTypeMap[[int]$Bus.BusType]
            break
        }
    }

    return [PsCustomObject] @{
        BusType       = $BusType
        BusTypeString = $BusTypeString }
}

function GetAdapterConnectionName($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceAlias
}

function GetAdapterDDPPackageNameAndVersion($AdapterName)
{
    $Name    = $Messages.NotSupported
    $Version = $Messages.NotSupported

    foreach ($DDPPkgName in $script:DDPPkgNamesArray)
    {
        if (($null -ne $DDPPkgName) -and ($DDPPkgName.InstanceName -eq $AdapterName))
        {
            if ($DDPPkgName.Name -and ($null -ne $DDPPkgName.Major) -and ($null -ne $DDPPkgName.Minor))
            {
                $Name = [System.Text.Encoding]::ASCII.GetString($DDPPkgName.Name).Trim(0)
                $Version = $DDPPkgName.Major.ToString() + "." + $DDPPkgName.Minor.ToString()
            }
            break
        }
    }
    return [PsCustomObject] @{
        Name    = $Name
        Version = $Version }
}


function GetAdapterDDPPackageTrackId($AdapterName)
{
    $TrackId = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetTrackId" $AdapterName "WmiGetTrackId" $Params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true) -and ($Result.Track_Id))
    {
        $TrackId = '0x{0:X}' -f $Result.Track_Id
    }
    return $TrackId
}


function GetAdapterDeviceStatus($AdapterName, $SupportedSpeeds)
{
    $DeviceStatus = 0
    $DeviceStatusString = $Messages.Unknown

    $AdapterNames = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})

    foreach($TmpAdapter in $AdapterNames)
    {
        if ("Up" -eq $TmpAdapter.Status)
        {
            $MaxSpeed = GetAdapterMaxSpeed $AdapterName $SupportedSpeeds
            $CurrentSpeed = (GetLinkSpeedDuplex $AdapterName).Speed
            if ($CurrentSpeed -lt $MaxSpeed)
            {
                $DeviceStatus = 4
                $DeviceStatusString = $Messages.LinkUpNotMax
            }
            else
            {
                $DeviceStatus = 1
                $DeviceStatusString = $Messages.LinkUp
            }
        }
        elseif ("Disconnected" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 2
            $DeviceStatusString = $Messages.LinkDown
        }
        elseif ("Disabled" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 0
            $DeviceStatusString = $Messages.Disabled
        }
    }

    return [PsCustomObject] @{
        DeviceStatus       = $DeviceStatus
        DeviceStatusString = $DeviceStatusString}

}

function GetAdapterETrackId($AdapterName)
{
    $strETrackId = $Messages.NotSupported
    foreach ($ETrackId in $script:ETrackIdsArray)
    {
        if (($null -ne $ETrackId) -and ($ETrackId.InstanceName -eq $AdapterName))
        {
            if ($ETrackId.Id)
            {
                $strETrackId = '0x{0:X}' -f $ETrackId.Id
            }
            break
        }
    }
    return $strETrackId
}


function GetAdapterNVMVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($NVMVersion in $script:NVMVersionsArray)
    {
        if (($null -ne $NVMVersion) -and ($NVMVersion.InstanceName -eq $AdapterName))
        {
            if ($NVMVersion.Version)
            {
                $Version = (($NVMVersion.Version -band 0xffff) -shr 8).ToString() + "." + (($NVMVersion.Version -band 0xff)).ToString("X2")
            }
            break
        }
    }
    return $Version
}

function GetAdapterNetlistVersion($AdapterName)
{
    $NetlistVersion = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetNVMNetlistInfo" $AdapterName "WmiGetNVMNetlistInfo" $params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true) -and ($Result.VersionStr))
    {
        $NetlistVersion = [System.Text.Encoding]::ASCII.GetString($Result.VersionStr).Trim(0)
    }
    return $NetlistVersion
}

function GetAdapterPartNumber($AdapterName)
{
    $PartNumberString = $Messages.NotSupported
    foreach ($PartNumber in $script:PartNumbersArray)
    {
        if (($null -ne $PartNumber) -and ($PartNumber.InstanceName -eq $AdapterName))
        {
            if ($PartNumber.PartNumberString)
            {
                $PartNumberString = [System.Text.Encoding]::ASCII.GetString($PartNumber.PartNumberString).Trim(0)
            }
            break
        }
    }
    return $PartNumberString
}

function GetAdapterSanMacAddress($AdapterName)
{
    $MacAddress = $Messages.NotSupported

    foreach ($SanMacAddress in $script:SanMacAddressesArray)
    {
        if (($null -ne $SanMacAddress) -and ($SanMacAddress.InstanceName -eq $AdapterName))
        {
            if ($SanMacAddress.SanMacAddr)
            {
                $MacAddress = ""
                for ($i = 0; $i -lt 6; $i++)
                {
                    # convert to string hex representation; X - hex, 2 - add leading zeroes if needed
                    $MacAddress += $SanMacAddress.SanMacAddr[$i].ToString("X2")
                }
            }

            break
        }
    }
    return $MacAddress
}

function GetAdapterMediaType($AdapterName)
{
    $MediaType = [PsCustomObject] @{
        MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
        MediaTypeString = $Messages.Unknown }

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    switch ($Service)
    {
        {($_ -eq 'icea') -or ($_ -eq 'scea')}   { $MediaType = GetAdapterMediaTypeIce $AdapterName; break }
        {($_ -eq 'i40ea') -or ($_ -eq 'i40eb')} { $MediaType = GetAdapterMediaTypeI40e $AdapterName; break }
    }
    return $MediaType
}

function GetAdapterMediaTypeIce($AdapterName)
{
    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
    $MediaTypeString = $Messages.Unknown

    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 8))
        {
            # Interpreting PhyInfo array values:
            # 0 == PHY type
            # 1 == link info <-- Bit 0 (value of 1 means has link)
            # 2 == an_info
            # 3 == ext_info
            # 4 == module type [0]
            # 5 == module type [1]
            # 6 == module type [2]
            # 7 == media interface  <-- 1=Backplane, 2=QSFP, 3=SFP
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkInfo = $PhyInfo.PhyInfo[1];
            $MediaInterface = $PhyInfo.PhyInfo[7];

            if ($LinkInfo -band 0x01)
            {
                switch ($PhyType)
                {
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL25GAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2AOC_ACC}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_AOC_ACC
                        $MediaTypeString = $Messages.AOCACC
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL5GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL10GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseKR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR
                        $MediaTypeString = $Messages.BackplaneKR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR1
                        $MediaTypeString = $Messages.BackplaneKR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2
                        $MediaTypeString = $Messages.BackplaneKR2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR2PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2_PAM4
                        $MediaTypeString = $Messages.BackplaneKR2PAM4
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseKR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseKR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4
                        $MediaTypeString = $Messages.BackplaneKR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR4PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4_PAM4
                        $MediaTypeString = $Messages.BackplaneKR4PAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_PAM4
                        $MediaTypeString = $Messages.BackplaneKRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_S
                        $MediaTypeString = $Messages.BackplaneKRS
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseKX,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseKX}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KX
                        $MediaTypeString = $Messages.BackplaneKX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIC2C,
                    [int][CVL_PHY_TYPE]::CVL25GAUIC2C,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUI,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Chip_to_Chip
                        $MediaTypeString = $Messages.ChiptoChip
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseT,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL10GBaseT,
                    [int][CVL_PHY_TYPE]::CVL25GBaseT}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100BaseTX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_TX
                        $MediaTypeString = $Messages.CopperTX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL10GSFIDA)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach
                        $MediaTypeString = $Messages.DirectAttach
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseCP2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CP2
                        $MediaTypeString = $Messages.DirectAttachCP2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR
                        $MediaTypeString = $Messages.DirectAttachCR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR1
                        $MediaTypeString = $Messages.DirectAttachCR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR2
                        $MediaTypeString = $Messages.DirectAttachCR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseCR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseCR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR4
                        $MediaTypeString = $Messages.DirectAttachCR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_PAM4
                        $MediaTypeString = $Messages.DirectAttachCRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_S
                        $MediaTypeString = $Messages.DirectAttachCRS
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseDR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_DR
                        $MediaTypeString = $Messages.FiberDR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseFR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_FR
                        $MediaTypeString = $Messages.FiberFR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseLR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR
                        $MediaTypeString = $Messages.FiberLR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseLR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseLR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR4
                        $MediaTypeString = $Messages.FiberLR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseLX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LX
                        $MediaTypeString = $Messages.FiberLX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseSR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR
                        $MediaTypeString = $Messages.FiberSR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseSR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR2
                        $MediaTypeString = $Messages.FiberSR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseSR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseSR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR4
                        $MediaTypeString = $Messages.FiberSR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseSX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SX
                        $MediaTypeString = $Messages.FiberSX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL2Point5gBaseX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_X
                        $MediaTypeString = $Messages.FiberX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL100MSGMII,
                    [int][CVL_PHY_TYPE]::CVL1GSGMII}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
            else
            {
                switch ($MediaInterface)
                {
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BACKPLANE)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BACKPLANE
                        $MediaTypeString = $Messages.Backplane
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_QSFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_QSFP
                        $MediaTypeString = $Messages.QSFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SFP
                        $MediaTypeString = $Messages.SFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BASE_T)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BASE_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SGMII)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_FIBER)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_FIBER
                        $MediaTypeString = $Messages.Fiber
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterMediaTypeI40e($AdapterName)
{
    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
    $MediaTypeString = $Messages.Unknown

    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 13))
        {
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkType = ([uint32]$PhyInfo.PhyInfo[11] -shl 24) + ([uint32]$PhyInfo.PhyInfo[10] -shl 16) + ([uint32]$PhyInfo.PhyInfo[9] -shl 8) + [uint32]$PhyInfo.PhyInfo[8];
            $LinkTypeExt = $PhyInfo.PhyInfo[12];

            if ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_EMPTY)
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_KR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_KR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                $MediaTypeString = $Messages.Backplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SFP_Cu) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                $MediaTypeString = $Messages.SFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_LR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_LR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_LR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                $MediaTypeString = $Messages.LRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SR) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_SR4) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_SR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                $MediaTypeString = $Messages.SRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_100BASE_TX) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_2_5GBASE_T) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_5GBASE_T))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                $MediaTypeString = $Messages.Copper
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                $MediaTypeString = $Messages.QSFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KX4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_KX))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                $MediaTypeString = $Messages.KXBackplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_XAUI) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_XLAUI))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                $MediaTypeString = $Messages.XAUI
            }
            elseif (($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_AOC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_ACC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_CR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                $MediaTypeString = $Messages.SFP28DirectAttach
            }
            # old FW or HW different than XL710
            elseif ($LinkTypeExt -eq 0 -and $LinkType -eq 0)
            {
                if (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_KR4) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KR) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_KR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                    $MediaTypeString = $Messages.Backplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SFPP_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                    $MediaTypeString = $Messages.SFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_LR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_LR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_LR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                    $MediaTypeString = $Messages.LRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_SR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_SR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                    $MediaTypeString = $Messages.SRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_100BASE_TX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                    $MediaTypeString = $Messages.Copper
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                    $MediaTypeString = $Messages.QSFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KX4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_KX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                    $MediaTypeString = $Messages.KXBackplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XAUI) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XLAUI))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                    $MediaTypeString = $Messages.XAUI
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_AOC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_ACC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_CR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                    $MediaTypeString = $Messages.SFP28DirectAttach
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterPCILocation($AdapterName)
{
    $PCILocation = $Messages.NotSupported
    foreach ($HwInfo in $script:MSNetHwInfo)
    {
        if ($AdapterName -eq $HwInfo.ifDesc)
        {
            $PCILocation = $HwInfo.Bus.ToString() + ":" + $HwInfo.Device.ToString()  + ":" + $HwInfo.Function.ToString() + ":" + $HwInfo.Segment.ToString()
            break
        }
    }
    return $PCILocation
}

function GetOriginalDisplayName($AdapterName)
{
    return ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).ifDesc
}

function GetOemFwVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($FwVersion in $script:FwVersionsArray)
    {
        if ($FwVersion.InstanceName -eq $AdapterName)
        {
            # driver can return an array of zeroes - don't attempt to construct a string using it
            if (($null -ne $FwVersion.SingleNvmVersion) -and $FwVersion.SingleNvmVersion[0] -ne 0)
            {
                $Version = [System.Text.Encoding]::ASCII.GetString($FwVersion.SingleNvmVersion).Trim(0)
            }
            break
        }
    }
    return $Version
}

function GetAdapterPCIDeviceID($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).PnPDeviceID
}

function GetAdapterNegotiatedLinkWidth($AdapterName)
{
    $NegotiatedLinkWidth = 0
    $NegotiatedLinkWidthString = $Messages.NotAvailable

    $AdapterHWInfo = $script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})

    if ($AdapterHWInfo)
    {
        $NegotiatedLinkWidth = $AdapterHWInfo.PciExpressCurrentLinkWidth
        $NegotiatedLinkWidthString = "x" + $NegotiatedLinkWidth.ToString()
    }

    return [PsCustomObject] @{
        NegotiatedLinkWidth = $NegotiatedLinkWidth
        NegotiatedLinkWidthString = $NegotiatedLinkWidthString }
}

function GetAdapterNegotiatedLinkSpeed($AdapterName)
{
    $NegotiatedLinkSpeed = ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).PciExpressCurrentLinkSpeedEncoded
    switch ($NegotiatedLinkSpeed)
    {
        0 {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
        1 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed25; break}
        2 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed50; break}
        3 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed80; break}
        default {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
    }
    return [PsCustomObject] @{
        NegotiatedLinkSpeed = $NegotiatedLinkSpeed
        NegotiatedLinkSpeedString = $NegotiatedLinkSpeedString }
}

function GetLinkSpeedDuplex($AdapterName)
{
    $AdapterObj = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})
    $Speed = $AdapterObj.Speed
    $SpeedString = $AdapterObj.LinkSpeed
    $FullDuplex = $AdapterObj.FullDuplex

    if ("Up" -ne $AdapterObj.Status)
    {
        $Speed = 0
        $SpeedString = $Messages.NotAvailable
    }
    elseif ($true -eq $FullDuplex)
    {
        $SpeedString += " " + $Messages.FullDuplex
    }

    return [PsCustomObject] @{
        Speed = $Speed
        SpeedString = $SpeedString }
}

function GetFullDuplex($AdapterName)
{
    $FullDuplex = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).FullDuplex
    if (-not $FullDuplex)
    {
        $FullDuplex = ""
    }
    return $FullDuplex
}

function GetAdapterPort($AdapterName)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName

    switch ($AdapterRegKey.Port)
    {
        0 {$PortNumberString = $Messages.PortA; break}
        1 {$PortNumberString = $Messages.PortB; break}
        2 {$PortNumberString = $Messages.PortC; break}
        3 {$PortNumberString = $Messages.PortD; break}
        default {$PortNumberString = $Messages.NotSupported; break}
    }

    return [PsCustomObject] @{
        Port       = $AdapterRegKey.Port
        PortString = $PortNumberString }
}

function GetDriverVersion($AdapterName)
{
    $DriverVersion = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).DriverVersion
    return $DriverVersion
}

function GetAdapterMiniPortName($AdapterName)
{
    $Name = ($script:PnpDevice.Where({$_.Name -eq $AdapterName})).Service
    $Instance = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    return [PsCustomObject] @{
        Name     = $Name
        Instance = $Instance }
}

function GetAdapterMaxSpeed($AdapterName, $SupportedSpeeds)
{
    if ($SupportedSpeeds.Length -gt 0)
    {
        # array is sorted, so we just return the last element
        return $SupportedSpeeds[-1]
    }
    return 0
}

function GetAdapterSupportedSpeeds($AdapterName)
{
    $SpeedDuplex = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "*SpeedDuplex"})
    if ($SpeedDuplex)
    {
        $RegistryValues = ($SpeedDuplex).ValidRegistryValues
    }

    $SupportedSpeeds = @()

    foreach ($i in $RegistryValues)
    {
        $SupportedSpeeds += $SupportedSpeedsMap[$i]
    }

    return $SupportedSpeeds | Sort-Object
}


function GetAdapterEEEStatus($AdapterName)
{
    $EEEStatus = 0
    $EEEStatusString = $Messages.NotSupported

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    foreach ($EEE in $script:EEELinkStatusArray)
    {
        if (($null -ne $EEE) -and ($EEE.InstanceName -eq $AdapterName))
        {
            if ($EEE.EEELinkStatus -eq $false)
            {
                $EEELinkAdvertisement = 0
                if ($Service -in @("i40ea", "i40eb"))
                {
                    $EEELinkAdvertisement = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "EEELinkAdvertisement"})
                    if ($EEELinkAdvertisement -and $EEELinkAdvertisement.RegistryValue)
                    {
                        $EEELinkAdvertisement = $EEELinkAdvertisement.RegistryValue
                    }
                    else
                    {
                        break
                    }
                }

                if ($EEELinkAdvertisement -gt 0)
                {
                    $EEEStatus = 3 #Not Negotiated
                    $EEEStatusString = $Messages.NotNegotiated
                }
                else
                {
                    $EEEStatus = 1 #Disabled
                    $EEEStatusString = $Messages.Disabled
                }
            }
            else
            {
                $EEEStatus = 2 #Active
                $EEEStatusString = $Messages.Active
            }
        }
    }

    return [PsCustomObject] @{
        EEEStatus       = $EEEStatus
        EEEStatusString = $EEEStatusString }
}

function GetAdapterStatus($AdapterName)
{
    $AdapterStatus = [ADAPTER_STATUS]::Installed -bor [ADAPTER_STATUS]::DriverLoaded -bor [ADAPTER_STATUS]::HasDiag

    $LinkStatus = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).Status
    if ($LinkStatus -eq 'Up')
    {
        $AdapterStatus = $AdapterStatus -bor [ADAPTER_STATUS]::HasLink
    }
    return $AdapterStatus
}

function GetAdapterCapabilities($AdapterName, $SupportedSpeeds)
{
    $Capabilities = @([int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_VENDOR_INTEL)

    foreach ($SupportedSpeed in $SupportedSpeeds)
    {
        switch ($SupportedSpeed)
        {
            10000000     {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10_MBPS; break}
            100000000    {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100_MBPS; break}
            1000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_1000_MBPS; break}
            2500000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_2500_MBPS; break}
            5000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_5000_MBPS; break}
            10000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10000_MBPS; break}
            40000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_40000_MBPS; break}
            25000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_25000_MBPS; break}
            50000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_50000_MBPS; break}
            100000000000 {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100000_MBPS; break}
        }
    }
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_PERFORMANCE_PROFILE
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DIAGNOSTIC_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_FLASH_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_CYPRESS
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_IDENTIFY_ADAPTER_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_NDIS_IOAT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_EXTENDED_DMIX_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DCB
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_JUMBO_FRAMES

    return , ($Capabilities | Sort-Object)
}

function GetIntelDriverInfo()
{
    $script:BusTypesArray        = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_BusType -Property BusType -ErrorAction SilentlyContinue
    $script:DDPPkgNamesArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPackageInfo -Property Name, Major, Minor -ErrorAction SilentlyContinue
    $script:ETrackIdsArray       = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EetrackId -Property Id -ErrorAction SilentlyContinue
    $script:NVMVersionsArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EepromVersion -Property Version -ErrorAction SilentlyContinue
    $script:SanMacAddressesArray = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetSanMacAddress -Property SanMacAddr -ErrorAction SilentlyContinue
    $script:PartNumbersArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_PartNumberString -Property PartNumberString -ErrorAction SilentlyContinue
    $script:PhyInfoArray         = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPhyInfo -Property PhyInfo -ErrorAction SilentlyContinue
    $script:FwVersionsArray      = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetOemProductVer -Property SingleNvmVersion -ErrorAction SilentlyContinue
    $script:EEELinkStatusArray   = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EEELinkStatus -Property EEELinkStatus -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIocQYJKoZIhvcNAQcCoIIoYjCCKF4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCACKIU7lkdvjS75
# O4GGqXKUGpY27jd8+uVCpwn4LBgV1qCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1TCCFdECAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEIDSzypVMQXAxEqxpddk9pnSX
# 3lTnTh+ZOMCPvGmUlh8sMA0GCSqGSIb3DQEBAQUABIIBgEnhfuDlSONWfQbtGlB2
# 8LG8i/K2vGcXchPI8usraaMC+FhDWzqMagS/9Rsgj/Yu2Tcvoulby/9sfV47mYKk
# pv0hGWO7FDdPxAQNtfY4PgRue4ZzRe0wTJONZznu0kGZeB6BGVCSWTg4iS8QayEZ
# Ewz1NetZXQ2Aa3jfENec1aqupac0lp/L3ZjwDT2nE8PuaYGRzTgpUDN3deozJl0E
# VxsI2jLRRJWyeRj41ISpkZUF25Dw+tf/ehLXfFpuPmQA7yyE/LhtcfB3KObYXkoX
# KOMJWxF6c/ERFnEOd0dYdTgFlb/2TJfsbx/X05S5sowkd9xpMHnumH55FVSC/imc
# yHfbPrcv3QEQDOURrkeqtkebl/oNu5mRXKMP+193hgueivKE6Vm9u3JN0P2uUpQh
# qMd9iRI2tWNxnoqOfGr9dMGTLRkbXsZkpphV/DBglBZygM8sDF5Ai1J0fwz33IF6
# JhaD2YGIeqQKPe4t8BZC3N2gBGK5re2qrKIXZRCpeOs+p6GCE08wghNLBgorBgEE
# AYI3AwMBMYITOzCCEzcGCSqGSIb3DQEHAqCCEygwghMkAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwgfAGCyqGSIb3DQEJEAEEoIHgBIHdMIHaAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIJw4mt2jSRdc67nzneoijjsrQ30g39RxZ7zoC9AJxIed
# AhUAvJB8XaCYj0IpjyAp0f4KowVa9woYDzIwMjQwMzE4MTgyMzQ5WqBupGwwajEL
# MAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBT
# aWduZXIgIzSggg3pMIIG9TCCBN2gAwIBAgIQOUwl4XygbSeoZeI72R0i1DANBgkq
# hkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5j
# aGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwHhcNMjMw
# NTAzMDAwMDAwWhcNMzQwODAyMjM1OTU5WjBqMQswCQYDVQQGEwJHQjETMBEGA1UE
# CBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# DCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjNDCCAiIwDQYJKoZI
# hvcNAQEBBQADggIPADCCAgoCggIBAKSTKFJLzyeHdqQpHJk4wOcO1NEc7GjLAWTk
# is13sHFlgryf/Iu7u5WY+yURjlqICWYRFFiyuiJb5vYy8V0twHqiDuDgVmTtoeWB
# IHIgZEFsx8MI+vN9Xe8hmsJ+1yzDuhGYHvzTIAhCs1+/f4hYMqsws9iMepZKGRNc
# rPznq+kcFi6wsDiVSs+FUKtnAyWhuzjpD2+pWpqRKBM1uR/zPeEkyGuxmegN77tN
# 5T2MVAOR0Pwtz1UzOHoJHAfRIuBjhqe+/dKDcxIUm5pMCUa9NLzhS1B7cuBb/Rm7
# HzxqGXtuuy1EKr48TMysigSTxleGoHM2K4GX+hubfoiH2FJ5if5udzfXu1Cf+hgl
# TxPyXnypsSBaKaujQod34PRMAkjdWKVTpqOg7RmWZRUpxe0zMCXmloOBmvZgZpBY
# B4DNQnWs+7SR0MXdAUBqtqgQ7vaNereeda/TpUsYoQyfV7BeJUeRdM11EtGcb+Re
# DZvsdSbu/tP1ki9ShejaRFEqoswAyodmQ6MbAO+itZadYq0nC/IbSsnDlEI3iCCE
# qIeuw7ojcnv4VO/4ayewhfWnQ4XYKzl021p3AtGk+vXNnD3MH65R0Hts2B0tEUJT
# cXTC5TWqLVIS2SXP8NPQkUMS1zJ9mGzjd0HI/x8kVO9urcY+VXvxXIc6ZPFgSwVP
# 77kv7AkTAgMBAAGjggGCMIIBfjAfBgNVHSMEGDAWgBQaofhhGSAPw0F3RSiO0TVf
# BhIEVTAdBgNVHQ4EFgQUAw8xyJEqk71j89FdTaQ0D9KVARgwDgYDVR0PAQH/BAQD
# AgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwSgYDVR0g
# BEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6Ly9zZWN0
# aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1UdHwQ9MDswOaA3oDWGM2h0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNybDB0Bggr
# BgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQuc2VjdGlnby5jb20v
# U2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggIBAEybZVj64HnP7xXD
# Mm3eM5Hrd1ji673LSjx13n6UbcMixwSV32VpYRMM9gye9YkgXsGHxwMkysel8Cbf
# +PgxZQ3g621RV6aMhFIIRhwqwt7y2opF87739i7Efu347Wi/elZI6WHlmjl3vL66
# kWSIdf9dhRY0J9Ipy//tLdr/vpMM7G2iDczD8W69IZEaIwBSrZfUYngqhHmo1z2s
# IY9wwyR5OpfxDaOjW1PYqwC6WPs1gE9fKHFsGV7Cg3KQruDG2PKZ++q0kmV8B3w1
# RB2tWBhrYvvebMQKqWzTIUZw3C+NdUwjwkHQepY7w0vdzZImdHZcN6CaJJ5OX07T
# jw/lE09ZRGVLQ2TPSPhnZ7lNv8wNsTow0KE9SK16ZeTs3+AB8LMqSjmswaT5qX01
# 0DJAoLEZKhghssh9BXEaSyc2quCYHIN158d+S4RDzUP7kJd2KhKsQMFwW5kKQPqA
# bZRhe8huuchnZyRcUI0BIN4H9wHU+C4RzZ2D5fjKJRxEPSflsIZHKgsbhHZ9e2hP
# jbf3E7TtoC3ucw/ZELqdmSx813UfjxDElOZ+JOWVSoiMJ9aFZh35rmR2kehI/shV
# Cu0pwx/eOKbAFPsyPfipg2I2yMO+AIccq/pKQhyJA9z1XHxw2V14Tu6fXiDmCWp8
# KwijSPUV/ARP380hHHrl9Y4a1LlAMIIG7DCCBNSgAwIBAgIQMA9vrN1mmHR8qUY2
# p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5l
# dyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNF
# UlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNh
# dGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/IJC9nIAhVJO5z6A+U++zWsB21hoE
# pc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnkuyWPCE5u2hj8BBZJmbyGr1XEQeYf0
# RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnTv/M5u7mkI0Ks0BXDf56iXNc48Ray
# cNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwPDU+zkQGObp0ndVXRFzs0IXuXAZSv
# f4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN9Q0/29DqhYyKVnHRLZRMyIw80xSi
# nL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gpGaQpL0bj3duRX58/Nj4OMGcrRrc1
# r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMSHv2qyghYh3ekFesZVf/QOVQtJu5F
# GjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdonigbSlofe6GsO8Ft96XZpkyAcSpcs
# dxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179g6JTZ8qx+o2hZMmIklnLqEbAyfKm
# /31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkUAx4p+hulIq6lw02C0I3aa7fb9xhA
# V3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLvbxhQiddPnTO9GrWdod6VQXqngwID
# AQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/WqorSs9UgOHYm8Cd8rIDZsswHQYD
# VR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRVMA4GA1UdDwEB/wQEAwIBhjASBgNV
# HRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAG1UgaUzXRbhtVOB
# kXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJWZApbGHLtVDb4n35nwDvQMOt0+Lk
# VvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc87QRD9XVw+K81Vh4v0h24URnbY+w
# QxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYIuM1cfwGWvnJSajtCN3wWeDmTk5Sb
# sdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8VmBvbz1kteraMrQoohkv6ob1olcGKB
# c2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncEvAQ2ted3y5wujSMYuaPCRx3wXdah
# c1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSmTqo9CfUFbLfSZFrYKiLCS53xOV5M
# 3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/W0NdZT1QgW6r8NDT/LKzH7aZlib0
# PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYybxzrF06Uw3hlIDsPQaof6aFBnf6x
# uKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/qEZ0eBsambTJdtDgJK0kHqv3sMNrx
# py/Pt/360KOE2See+wFmd7lWEOEgbsausfm2usg1XTN2jvF8IAwqd661ogKGuinu
# tFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELDCCBCgCAQEwgZEwfTELMAkGA1UEBhMC
# R0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9y
# ZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJT
# QSBUaW1lIFN0YW1waW5nIENBAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCWCGSAFlAwQC
# AgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkF
# MQ8XDTI0MDMxODE4MjM0OVowPwYJKoZIhvcNAQkEMTIEMGgEjkwplZlTxptZdbox
# rWRZowcGWcDHsgA9PBVKcKfNTd2MBQZY1ZJflCWVpFtr9DCB7QYLKoZIhvcNAQkQ
# Agwxgd0wgdowgdcwFgQUrmKvdQoMvUfWRh91aOK8jOfKT5QwgbwEFALWW5Xig3DB
# VwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQEF
# AASCAgBF666WoD5R04WYVQ4eFzupnn5aZiqnFebukX7Q6e0tpFrBNi+i1+kfKcAN
# 6pnw5Ju8FS6ERcQBjxSeStphJh2sy953809wLxgqRmV7Vo4NR8Vu+52VHTKx+u0i
# 0tuj6Hb6tXfNgPp3cmiro+UrR+UnYuxxqQ5+esswzPw9WUlVkrNUXSQ5Pp76NTi0
# HvStIfq9bejmtKX4y//9rh14PjHMzefeOxlzXKf6YvRnDNQcFd5HM9DvxQHMsC7d
# +plyqTHRk2iLBwbZx6AWD7kouimMF1HlTL0V7HRFSUKHiRgDaRci5zWsicZ46cPZ
# omJQeSP2JRPlOLxlJ++RndDvsY/FR6pqbMU2n5fj19E2VP1cfVRU7CzWe/BiLtHq
# 78cU+k92cLc6E1NB4O/ZY/KAIAEIGyQSgaPuEa8mLqACU2Y8prKly8h4rQa9w3kS
# OtZZS0fceS6FC029sEv73dCNGJJqoURPERefkIC5EeAuYxoWrH9/ghr978vD73zl
# TWOSLt0Fk3c6Dgr8uxIZgVbdx1OxkstB9I5qApZjXOajEyr20HOyozqUq69yC2rw
# 7jDEmwgnJgYykFEK/Hwn7cG8AJTwwHOCid758etcWKlyXBvvkoDHbGOWzAGx51RD
# Gl/jkXKpvQNM+29SRhb3FrfmW4k140NtoExlZSLq1ZK17aM3Fg==
# SIG # End signature block
