#!/bin/sh
# Copyright (c) 2019 - 2023 Intel Corporation
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenFabrics.org BSD license below:
#
#   Redistribution and use in source and binary forms, with or
#   without modification, are permitted provided that the following
#   conditions are met:
#
#    - Redistributions of source code must retain the above
#	copyright notice, this list of conditions and the following
#	disclaimer.
#
#    - Redistributions in binary form must reproduce the above
#	copyright notice, this list of conditions and the following
#	disclaimer in the documentation and/or other materials
#	provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# script for building the rdma driver icrdma.
# A path to ice driver sources should be provided
# as an argument of this script to ensure the driver
# builds correctly.
# if second argument is equal to 'true' the libirdma
# is also updated during building process.

DIR_ICE=$1
update_lib=$2

# verify_kernel_src function attempts to find where is current kernel
#   binary located, to ensure the driver is built against correct kernel
#   code. As a result appropriate kernel source path should be stored in
#   $kern_src variable
#   Note: to ensure the compilation works well one need to remember that
#   /sys is required to point to correct kernel sources. The script will
#   attempt to set it correctly when it finds proper source.
verify_kernel_src() {
	loaded_kernel=`sysctl kern.bootfile | cut -d' ' -f2`
	# std_kernel=`find /usr/obj/usr/src/ -name kernel -type f -maxdepth 6`
	sys_link=`readlink -f /sys`
	sys_link_parent=$(dirname "$sys_link")
	sys_link_kernel=`find /usr/obj/$sys_link_parent -name kernel -type f -maxdepth 6`
	for slk in $sys_link_kernel
	do
		cmp $loaded_kernel $slk
		res=`echo $?`
		if [ "$res" == "0" ]
		then
			kern_src=$sys_link_parent
			echo "current working kernel is in $sys_link_parent"
			return 0
		fi
	done
	echo "/sys link doesn't point to correct kernel"

	avail_kernels=`find /usr/obj/ -name kernel -type f -maxdepth 6`
	for ak in $avail_kernels
	do
		cmp $loaded_kernel $ak
		res=`echo $?`
		if [ "$res" == "0" ]
		then
			usr_obj=$(dirname "${ak}")
			echo "$usr_obj $ak"
			kern_src=`echo "$usr_obj" | sed "s/\/usr\/obj\(.*\)\/\(.*\)\/sys\/\(.*\)/\1/"`
			echo "previously /sys was pointing to $sys_link, updating it to $kern_src/sys"
			rm /sys
			ln -s $kern_src/sys /sys
			return 0
		fi
	done
	if [ "$kern_src" == "" ]
	then
		kern_src="/usr/src"
		echo "ERROR, proper kernel was not found, setting /usr/src"
	fi
}

verify_kernel_src
sed -i '' "s+KSRC = .*+KSRC = $kern_src+g" src/Makefile

cd src/
make clean
make ICE_DIR=$DIR_ICE COPTS=-g
if [ "$?" -ne "0" ]
then
	exit 1
fi
make install

if [ "$update_lib" == "true" ]
then
	cd ..
	mkdir -p $kern_src/lib/ofed/libirdma
	cp lib_Makefile $kern_src/lib/ofed/libirdma/Makefile
	if [ -e $kern_src/contrib/ofed/libirdma ]
	then
		mv $kern_src/contrib/ofed/libirdma $kern_src/contrib/ofed/bkp_libirdma
		cp -r libirdma $kern_src/contrib/ofed/
	else
		cp -r libirdma $kern_src/contrib/ofed/
	fi
	cd $kern_src/lib/ofed/libirdma
	make clean ; make -j4 ; make install
fi

