# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'Global:FormatEnumerationLimit')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
                continue
            }

            $Local:IntelLogConfigurations = @{}
            $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
            $Local:IntelLogConfigModules = @{}
            $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"
            $Local:IntelLogConfigLevels = @{}
            $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

            if ($IntelLogConfigLevels.Count -eq 0)
            {
                $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
            }

            $DisplayConfigNames = @()
            if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Get" $a))
            {
                if ($script:ErrorMessagesGet.Count -gt 0)
                {
                    break
                }
                continue
            }

            $DisplayModules = @()
            $DisplayLevels = @()
            $DriverPassed = $null

            if ($DisplayConfigNames)
            {
                $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"
                if (!(ValidateMethodOutput $MethodOutput "Get"))
                {
                    continue
                }

                GetConfigLogs $DisplayConfigNames ([ref]$DisplayModules) ([ref]$DisplayLevels) $MethodOutput
            }
            else
            {
                $DriverPassed = $ModuleNames.Where({$DRIVER_MODULE_NAME -like $_})

                if (-Not $DriverPassed -and $SubModule)
                {
                    $script:ErrorMessagesGet += $Messages.InvalidParams
                    break
                }
                elseif ($SubModule)
                {
                    $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesGet)
                    if ($SubModuleSum -eq -1)
                    {
                        break
                    }
                }

                if ($DriverPassed -or -Not $ModuleNames)
                {
                    $LogConfigs += GetDriverLogConfig $DriverPassed $SubModuleSum $a ([ref]$script:ErrorMessagesGet)
                }

                $MethodOutput = $null
                if ($IntelLogConfigModules.Count -gt 0)
                {
                    $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"

                    if (!(ValidateMethodOutput $MethodOutput "Get"))
                    {
                        continue
                    }
                }

                # ValidateGetModuleParam will error if Driver is passed and not present here.
                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                ValidateGetModuleParam ([ref]$DisplayModules) ([ref]$DisplayLevels) $ModuleNames $MethodOutput $a
            }

            if ($DisplayModules -and $DisplayLevels)
            {
                $LogConfigs += GatherOutput $a $DisplayModules $DisplayLevels $DriverPassed
            }
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }
        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false,Position=2)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String]
    $Level = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $ModuleNames = $Module
        $LevelName = $Level
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:WarningMessagesSet = @()
        $script:ErrorMessagesSet = @()
        $LogConfigs = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        do
        {
            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesSet)
            $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))

            if ([string]::IsNullOrEmpty($ConfigurationNames) -and ([string]::IsNullOrEmpty($ModuleNames)))
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigOrModule
                break
            }

            foreach($a in $AdapterNames)
            {
                $TmpStatusMsg = CheckDeviceError $a
                if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                {
                    $script:ErrorMessagesSet += $TmpStatusMsg
                    continue
                }

                $DisplayModules = @()
                $DisplayLevels = @()
                $InputModulesId = [uint32[]]::new(64)
                $InputLevelId = [uint32[]]::new(64)

                $Local:IntelLogConfigurations = @{}
                $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
                $Local:IntelLogConfigModules = @{}
                $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"

                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                $ModuleCount = 0
                $DisplayConfigNames = @()
                if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Set" $a))
                {
                    break
                }

                $bDriverPassed = $false
                $InputConfigurationIDs = $null
                if ($DisplayConfigNames)
                {
                    $InputConfigurationIDs = $IntelLogConfigurations[$DisplayConfigNames]
                }
                else
                {
                    ValidateSetModuleParam ([ref]$InputModulesId) ([ref]$DisplayModules) $ModuleNames $a

                    if ($DisplayModules -contains $DRIVER_MODULE_NAME)
                    {
                        $bDriverPassed = $true
                    }
                    elseif ($SubModule)
                    {
                        $script:ErrorMessagesSet += $Messages.InvalidParams
                        break
                    }

                    $ModuleCount = $DisplayModules.Count
                }

                if ($ModuleCount -ne 0)
                {
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $ModuleCount $a $bDriverPassed))
                    {
                        break
                    }

                    if ($bDriverPassed)
                    {
                        # Driver module is not included in Wmi method params
                        $ModuleCount -= 1

                        if ($InputLevelId[0] -lt 3)
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidDriverLevel
                            break
                        }

                        if ($SubModule)
                        {
                            $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesSet)
                            if ($SubModuleSum -eq -1)
                            {
                                break
                            }
                        }

                        if (-not (SetSubModuleRegistryValue $a $SubModuleSum))
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                            break
                        }

                        $LevelName = GetLevelName($InputLevelId[0])

                        if (-Not (SetDriverLogLevel $a $LevelName))
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                            break
                        }
                    }

                    if ($ModuleCount -gt 0)
                    {
                        $params = @{Count    = [uint32]$ModuleCount;
                                ModuleID = $InputModulesId;
                                Loglevel = $InputLevelId}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogLevels" $params
                    }

                    if ($bDriverPassed)
                    {
                        # Driver is set, display it. Avoid call to Validate if no other modules passed.
                        if ($ModuleCount -eq 0 -or (ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules -SubModule $SubModuleSum
                        }
                        else
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule $SubModuleSum
                        }
                    }
                    elseif (ValidateMethodOutput $MethodOutput "Set")
                    {
                        $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules
                    }
                }
                elseif ($null -ne $InputConfigurationIDs)
                {
                    $count = 1
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $count $a))
                    {
                        break
                    }

                    $bContinue = $true
                    foreach ($ConfigID in $InputConfigurationIDs)
                    {
                        $params = @{ConfigurationID = [uint32]$ConfigID;
                                    Level = $InputLevelId[0]}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogConfiguration" $params

                        if (!(ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $bContinue = $false
                            break
                        }
                    }

                    if (!$bContinue)
                    {
                        break
                    }
                    $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Configuration $DisplayConfigNames
                }
            }
        } while ($false)
    }
    End
    {
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Start-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [ValidateNotNullOrEmpty()]
    [string]
    $Path = '',
    [parameter(Mandatory=$false)]
    [switch]
    $Force,
    [parameter(Mandatory=$false)]
    [switch]
    $Append
    )
    Begin
    {
        $SubModuleSum = 0xFFFF
        # Set LogmanLevel to Normal by default
        $LogmanLevel = 4
        $script:ErrorMessagesStart = @()
        $script:WarningMessagesStart = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesStart += $Messages.AdapterNotFound -f $Name
                break
            }

            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStart)
            $AdapterName = GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesStart)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $TmpStatusMsg = CheckDeviceError $AdapterName
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesStart += $TmpStatusMsg
                break
            }

            $UseDefaultPath = $false
            if (([string]::IsNullOrEmpty($Path)))
            {
                $UseDefaultPath = $true
            }

            $LogmanGUID = GetLogmanGUID $AdapterName ([ref]$script:ErrorMessagesStart)
            $LogName = GetLogName $AdapterName ([ref]$script:ErrorMessagesStart)

            if ($null -eq $LogmanGUID -or $null -eq $LogName)
            {
                break
            }

            if (LogmanProcessRunning $AdapterName $LogName $LogmanGUID)
            {
                break
            }

            $LogmanPath = ''
            if (-not (ValidatePathParams ([ref]$LogmanPath) $UseDefaultPath $LogName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $DriverLogLevel = GetDriverLogLevel $AdapterName
            if ($DriverLogLevel)
            {
                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $AdapterName "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                # Driver log levels are normal levels value + 1
                $LogmanLevel = $IntelLogConfigLevels[$DriverLogLevel] + 1

                $LogMetadataName = "$LogName.txt"

                $LogMetadataPath = ''
                if (-Not $UseDefaultPath)
                {
                    if ([IO.Path]::GetExtension($Path) -eq ".txt")
                    {
                        $Path += ".txt"
                    }
                    else
                    {
                        $Path = [IO.Path]::ChangeExtension($Path, ".txt")
                    }
                }

                $iBeforeErrorCount = $script:ErrorMessagesStart.Count
                if (-not (ValidatePathParams ([ref]$LogMetadataPath) $UseDefaultPath $LogMetadataName ([ref]$script:ErrorMessagesStart)))
                {
                    if ($iBeforeErrorCount -lt $script:ErrorMessagesStart.Count -and $script:ErrorMessagesStart[-1] -eq $Messages.LogmanFileExists -f $AdapterName)
                    {
                        # Driver Metadata File is always replaced if it exists.
                        $script:ErrorMessagesStart = $script:ErrorMessagesStart | Select-Object -First ($script:ErrorMessagesStart.Count - 1)
                    }
                    else
                    {
                        break
                    }
                }

                $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
                if ($null -eq $AdapterRegistryPath)
                {
                    $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
                    break
                }

                $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
                if ($SubModuleRegistryValue)
                {
                    $SubModuleSum = $SubModuleRegistryValue.SubModule

                    if (-Not (CreateDriverLogMetadataFile $AdapterName))
                    {
                        break
                    }
                }
            }

            LogmanStart $LogmanPath $LogName $LogmanGUID $SubModuleSum $LogmanLevel
            if (ValidateLogmanStart $AdapterName)
            {
                $LogmanPath = Resolve-Path -Path $LogmanPath -ErrorAction SilentlyContinue
                Write-Output $LogmanPath
            }

            break
        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesStart)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStart)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Stop-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null
    )
    Begin
    {
        $script:ErrorMessagesStop = @()
        $script:WarningMessagesStop = @()
        $bErrorObtained = $false
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStop)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesStop))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesStop += $TmpStatusMsg
                continue
            }

            if (-Not $bErrorObtained)
            {
                $LogName = GetLogName $a ([ref]$script:ErrorMessagesStop)
            }
            else
            {
                $LogName = GetLogName $a ([ref]@())
            }

            if ($null -eq $LogName)
            {
                $bErrorObtained = $true
                continue
            }

            LogmanStop $LogName
            if (-Not $bErrorObtained)
            {
                if (-Not (ValidateLogmanStop $a ([ref]$script:ErrorMessagesStop)))
                {
                    $bErrorObtained = $true
                }
            }
        }
    }
    End
    {
        foreach ($WarningMessage in $script:WarningMessagesStop)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStop)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Disable-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesDisable = @()
        $script:WarningMessagesDisable = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesDisable)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesDisable))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesDisable += $TmpStatusMsg
                continue
            }

            $DriverConfig = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule @($SUB_MODULE_MAP.Keys)[0] -Level "Normal"
            if (-Not $DriverConfig)
            {
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiDisableFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Disable"))
                {
                    continue
                }
            }

            # Will pass Driver as module, so returns IntelDriverLogConfiguration
            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*" -SubModule @($SUB_MODULE_MAP.Keys)[0]
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesDisable)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesDisable)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Reset-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesReset = @()
        $script:WarningMessagesReset = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesReset)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesReset))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesReset += $TmpStatusMsg
                continue
            }

            # Set driver level back to Normal
            $null = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiResetFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Reset"))
                {
                    continue
                }
            }

            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*"
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesReset)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesReset)
        {
            Write-Error $ErrorMessage
        }
    }
}

Function GatherOutput($AdapterName, $DisplayModules, $DisplayLevels, $DriverPassed)
{
    $LogConfigs = @()
    $ValidLevels = $IntelLogConfigLevels.Keys
    $LogConfigCount = $DisplayModules.Count
    for ($i = 0; $i -lt $LogConfigCount; $i++)
    {
        if ($DriverPassed)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                SubModule  = "";
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
        else
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
    }

    return $LogConfigs
}

Function GetModuleName($ModuleId)
{
    return $IntelLogConfigModules.GetEnumerator().Where({$_.Value -eq $ModuleId}).Name
}

Function GetLevelName($LevelId)
{
    return $IntelLogConfigLevels.GetEnumerator().Where({$_.Value -eq $LevelId}).Name
}

Function GetDriverLogLevel($AdapterName)
{
    $DriverRegistryValue = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $DriverRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    }

    if ($null -ne $DriverRegistryValue)
    {
        $DriverRegistryValue = $DriverRegistryValue.DriverLogLevel
    }

    return $DriverRegistryValue
}

Function GetDriverLogConfig($DriverPassed, $SubModuleSum, $AdapterName, [ref]$ErrorMessages)
{
    $LogConfigs = @()
    $InactiveSubModuleArray = @()
    $ActiveSubModuleArray = @()

    $DriverLevel = GetDriverLogLevel $AdapterName
    if (-Not $DriverLevel)
    {
        # Set to Normal by default
        $DriverLevel = @($DRIVER_LOG_LEVEL_MAP.Keys)[1]
        if (-Not (SetDriverLogLevel $AdapterName $DriverLevel))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }
    }

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -eq $AdapterRegistryPath)
    {
        $ErrorMessages.Value += $Messages.NoCmdletSupport -f $AdapterName
        break
    }

    $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    if (-Not $SubModuleRegistryValue -or [string]::IsNullOrEmpty($SubModuleRegistryValue.SubModule))
    {
        # If driver passed but set has not been used, enable default driver logging
        if (-not (SetSubModuleRegistryValue $a $SUBMODULE_DEFAULT_SUM))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }

        $RegistrySubModuleSum = $SUBMODULE_DEFAULT_SUM
    }
    else
    {
        $RegistrySubModuleSum = $SubModuleRegistryValue.SubModule
    }

    foreach ($DisplayName in $SUB_MODULE_MAP.Keys)
    {
        if ($SUB_MODULE_MAP[$DisplayName] -band $SubModuleSum)
        {
            if ($SUB_MODULE_MAP[$DisplayName] -band $RegistrySubModuleSum)
            {
                $ActiveSubModuleArray += $DisplayName
            }
            else
            {
                $InactiveSubModuleArray += $DisplayName
            }
        }
    }

    if ($DriverPassed)
    {
        foreach ($ActiveSubModule in $ActiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $ActiveSubModule;
                                            Level       = $DriverLevel;
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
        foreach ($InactiveSubModule in $InactiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $InactiveSubModule;
                                            Level       = @($DRIVER_LOG_LEVEL_MAP.Keys)[-1]; # Disabled
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
    }
    else
    {
        $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                                        Name            = $AdapterName;
                                        Module          = $DRIVER_MODULE_NAME;
                                        SubModule       = @($ActiveSubModuleArray);
                                        Level           = $DriverLevel;
                                        ValidSubModules = $SUB_MODULE_MAP.Keys;
                                        ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
    }

    return $LogConfigs
}

Function GetSubModuleBitmask($SubModuleArray, [ref]$ErrorMessages)
{
    $iSubModuleSum = 0

    foreach ($SubModule in $SubModuleArray)
    {
        if ($SubModule -match "^0[xX][A-Fa-f0-9]+$" -or $SubModule -match "^[1-9]\d*$")
        {
            if ([int]$SubModule -le $SUM_SUBMODULES)
            {
                $iSubModuleSum = $iSubModuleSum -bor $SubModule
            }
            else
            {
                $iSubModuleSum = $SUM_SUBMODULES
                continue
            }
        }
        elseif ($SUB_MODULE_MAP.Keys -Contains $SubModule)
        {
            $iSubModuleSum = $iSubModuleSum -bor $SUB_MODULE_MAP[$SubModule]
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidSubModule -f $a, $SubModule
            $iSubModuleSum = -1
            break
        }
    }

    return $iSubModuleSum
}

Function GetLogmanGUID($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_WPP_GUID; break }
            'i40eb' { return $I40EB_WPP_GUID; break }
            'icea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $ICEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            'scea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $SCEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function GetLogName($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_LOGNAME; break }
            'i40eb' { return $I40EB_LOGNAME; break }
            'icea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $ICEA_LOGNAME
                }

                return $LogName
                break
            }
            'scea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $SCEA_LOGNAME
                }

                return $LogName
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function SetDriverLogLevel($AdapterName, $DriverLevel)
{
    $bSuccess = $false
    $SetDriverLogLevel = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetDriverLogLevel = Set-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -Type "String" -Value $DriverLevel -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetDriverLogLevel -and -Not [string]::IsNullOrEmpty($SetDriverLogLevel.DriverLogLevel))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function SetSubModuleRegistryValue($AdapterName, $SubModuleValue)
{
    $bSuccess = $false
    $SetSubModule = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetSubModule = Set-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -Value $SubModuleValue -Type "DWORD" -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetSubModule -and -Not [string]::IsNullOrEmpty($SetSubModule.SubModule))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function CreateDriverLogMetadataFile($AdapterName)
{
    $strDriverName = ($script:SupportedAdapters.Where({ $_.Name -eq $AdapterName })).Service

    $strOSSpecificDriverName = (GetOSSpecificDriverName $strDriverName)
    if (-Not $strOSSpecificDriverName)
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    $DriverFolder = GetDriverInfParentFolder $strOSSpecificDriverName "Start"
    $DriverMetaData = Get-Item "$DriverFolder\$strOSSpecificDriverName.sys" -ErrorAction SilentlyContinue

    if ($DriverMetaData)
    {
        $DriverVersionInfo = $DriverMetaData.VersionInfo
        $DriverLength = "$strOSSpecificDriverName.sys Length: " + $DriverMetaData.Length
        $DriverChecksum = Get-FileHash $DriverMetaData

        try
        {
            Set-Content $LogMetadataPath -Value $DriverVersionInfo -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverLength -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverChecksum -ErrorAction Stop
        }
        catch
        {
            $script:ErrorMessagesStart += $Messages.PathIncorrect
        }
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    return $true
}

Function ValidateMethodOutput($MethodOutput, $Verb)
{
    Write-Verbose -Message "MethodOutput: $MethodOutput verb: $Verb"

    do
    {
        $Result = $true
        if ($null -eq $MethodOutput)
        {
            switch ($Verb)
            {
                Get {$script:WarningMessagesGet += $Messages.NoCmdletSupport -f $a; break}
                Set {$script:WarningMessagesSet += $Messages.NoCmdletSupport -f $a; break}
                Start {$script:WarningMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
                Stop {$script:WarningMessagesStop += $Messages.NoCmdletSupport -f $a; break}
                Disable {$script:WarningMessagesDisable += $Messages.NoCmdletSupport -f $a; break}
                Reset {$script:WarningMessagesReset += $Messages.NoCmdletSupport -f $a; break}
            }

            $Result = $false
            break
        }

        if ($MethodOutput.OutStatus -ne 0)
        {
            switch ($Verb)
            {
                Get {$script:ErrorMessagesGet += $Messages.OperationFailed -f $a, $Verb; break}
                Set {$script:ErrorMessagesSet += $Messages.OperationFailed -f $a, $Verb; break}
                Start {$script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, $Verb; break}
                Stop {$script:ErrorMessagesStop += $Messages.OperationFailed -f $a, $Verb; break}
                Disable {$script:ErrorMessagesDisable += $Messages.OperationFailed -f $a, $Verb; break}
                Reset {$script:ErrorMessagesReset += $Messages.OperationFailed -f $a, $Verb; break}
            }

            $Result = $false
        }
    } while ($false)

    Return $Result
}

Function ValidateGetModuleParam([ref]$DisplayModules, [ref]$DisplayLevels, $ModuleNames, $MethodOutput, $AdapterName)
{
    if ($ModuleNames)
    {
        foreach ($ModuleName in $ModuleNames)
        {
            $ModuleIds = $null
            $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
            $ModuleIDCount = $ModuleIDNames.Count

            if ($ModuleIDCount -gt 0)
            {
                $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })

                if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
                {
                    $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                    foreach ($ModuleId in $ModuleIds)
                    {
                        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
                        {
                            if ($ModuleId -eq $MethodOutput.ModuleID[$i])
                            {
                                $DisplayModules.Value += GetModuleName($ModuleId)
                                $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
                                break
                            }
                        }
                    }
                }
            }
            else
            {
                #if user passes in an array of modules and one (or more) is spelled incorrectly,
                #show error msg for that 1 and continue on getting the modules for the rest
                $script:ErrorMessagesGet += $Messages.InvalidModule -f $AdapterName, $ModuleName
            }
        }
    }
    else
    {
        GetAllModulesLevel $MethodOutput $DisplayModules $DisplayLevels
    }
}

Function GetAllModulesLevel($MethodOutput, [ref]$DisplayModules, [ref]$DisplayLevels)
{
    if ($MethodOutput)
    {
        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$i])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
            # driver returns an array of 64 but only 31 modules populated
            if ($i -eq 31)
            {
                break
            }
        }
    }
}

Function ValidateLevelParam([ref]$InputLevelId, [ref]$DisplayLevels, $LevelName, $Count, $AdapterName, $DriverPassed)
{
    do
    {
        $NewLevelId = $null
        $Result = $true

        $LevelMatchArray = $IntelLogConfigLevels.GetEnumerator().Where({$_.Name -like $LevelName})
        if ($LevelMatchArray)
        {
            $NewLevelId = @($LevelMatchArray.Value)
        }
        # Handle default case of "Normal" if Level is not passed
        elseif ($DriverPassed -and -not $LevelName)
        {
            $NewLevelId = @($IntelLogConfigLevels["Normal"])
        }

        if ($null -eq $NewLevelId -or 0 -eq $NewLevelId.Count)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevel -f $AdapterName, $LevelName, $ModuleNames[0]
            $Result = $false
            break
        }

        if ($NewLevelId.count -gt 1)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevelWildcard
            $Result = $false
            break
        }

        for ($i = 0; $i -lt $Count; $i++)
        {
            $InputLevelId.Value[$i] = $NewLevelId[0]
            $DisplayLevels.Value += GetLevelName($NewLevelId[0])
        }

    } while ($false)

    Return $Result
}

Function ValidateSetModuleParam([ref]$InputModulesId, [ref]$DisplayModules, $ModuleNames, $AdapterName)
{
    $count = 0
    foreach ($ModuleName in $ModuleNames)
    {
        $ModuleIds = $null
        $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
        $ModuleIDCount = $ModuleIDNames.Count

        if ($ModuleIDCount -gt 0)
        {
            $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })
            if (-Not $ModuleIDNames -or $ModuleIDCount -gt $ModuleIDNames.Count)
            {
                $DisplayModules.Value += $DRIVER_MODULE_NAME
            }

            if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
            {
                $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                foreach ($ModuleId in $ModuleIds)
                {
                    $InputModulesId.Value[$count] = $ModuleId
                    $DisplayModules.Value += GetModuleName($ModuleId)
                    $count++
                }
            }
        }
        else
        {
            #if user passes in an array of modules and one (or more) is spelled incorrectly,
            #show error msg for that 1 and continue on getting the modules for the rest
            $script:ErrorMessagesSet += $Messages.InvalidModule -f $AdapterName, $ModuleName
        }
    }
}

Function LogmanProcessRunning($AdapterName, $LogName, $GUID)
{
    $Result = $false

    $LogmanQueryString = logman query $LogName -ets
    if ($LogmanQueryString -match ($GUID))
    {
        $script:ErrorMessagesStart += $Messages.LogmanRunning -f $AdapterName
        $Result = $true
    }

    return $Result
}

Function LogmanStart($LogmanPath, $LogName, $GUID, $SubModuleSum, $Level)
{
    if ($Append)
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level -a
    }
    else
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level
    }
}

Function LogmanStop($LogName)
{
    $null = logman stop $LogName -ets
}


Function ValidateLogmanStart($AdapterName)
{
    $Result = $true
    # -2147024629, -2147024773, -2147024893, -2147024891 error codes mean directory name is invalid
    # or filename, directory name, or volume label syntax is incorrect or permission is denied.
    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2147024629 -or $LASTEXITCODE -eq -2147024773 -or $LASTEXITCODE -eq -2147024893 -or $LASTEXITCODE -eq -2147024891)
    {
        $script:ErrorMessagesStart += $Messages.PathIncorrect
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, "Start"
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStop($AdapterName, [ref]$ErrorMessages)
{
    $Result = $true

    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2144337918)
    {
        $ErrorMessages.Value += $Messages.NoLogsStarted -f $AdapterName
        $Result = $false
    }
    else
    {
        $ErrorMessages.Value += $Messages.OperationFailed -f $AdapterName, "Stop"
        $Result = $false
    }
    Return $Result
}

Function ValidateConfigurationParam($ConfigurationNames, $ModuleNames, [ref]$DisplayConfigNames, $Verb, $a)
{
    $Result = $true

    do
    {
        if ($ConfigurationNames)
        {
            if ($ModuleNames)
            {
                switch ($Verb)
                {
                    Set { $script:ErrorMessagesSet += $Messages.InvalidParamsConfigAndModule; break }
                    Get { $script:ErrorMessagesGet += $Messages.InvalidParamsConfigAndModule; break }
                }
                $Result = $false
                break
            }

            if ($IntelLogConfigurations.Count -eq 0)
            {
                if ($Verb -eq "Set")
                {
                    $script:ErrorMessagesSet += $Messages.NoConfigurationSupport -f $a
                }
                elseif ($Verb -eq "Get")
                {
                    $script:WarningMessagesGet += $Messages.NoConfigurationSupport -f $a
                }

                $Result = $false
                break
            }

            foreach ($ConfigName in $ConfigurationNames)
            {
                $TmpDisplayConfigNames = $IntelLogConfigurations.Keys -Like $ConfigName

                if ($TmpDisplayConfigNames.Count -gt 0)
                {
                    $DisplayConfigNames.Value += $TmpDisplayConfigNames
                }
                else
                {
                    #if user passes in an array of configs and one (or more) is spelled incorrectly,
                    #show error msg for that 1 and continue on getting the config for the rest
                    switch ($Verb)
                    {
                        Set { $script:ErrorMessagesSet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                        Get { $script:ErrorMessagesGet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                    }

                    $Result = $false
                }
            }
        }
    } while ($false)

    return $Result
}

Function GetConfigLogs($DisplayConfigNames, [ref]$DisplayModules, [ref]$DisplayLevels, $MethodOutput)
{
    $ConfigurationID = $IntelLogConfigurations[$DisplayConfigNames]
    foreach ($ConfigID in $ConfigurationID)
    {
        $params = @{ConfigurationID = [uint32]$ConfigID}
        $GetConfigMethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevelsForConfiguration" $params

        for ($i = 0; $i -lt $GetConfigMethodOutput.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$GetConfigMethodOutput.ModuleID[$i]])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$GetConfigMethodOutput.ModuleID[$i]])
        }
    }
}

function GetConfigHashTable($AdapterName, $ConfigPath)
{
    $ConfigHashTable = @{}

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $ConfigPath
    if ($null -ne $FWLogConfigPath)
    {
        $LogConfigs = Get-Item $FWLogConfigPath -ErrorAction SilentlyContinue
        if ($null -ne $LogConfigs)
        {
            $LogConfigs = $LogConfigs.Property
            foreach ($Config in $LogConfigs)
            {
                $ConfigHashTable.add($Config, (Get-ItemProperty -Path $FWLogConfigPath -Name $Config -ErrorAction SilentlyContinue).$Config)
            }
        }
    }

    return $ConfigHashTable
}

function GetAdapterRegistryConfigPath($AdapterName, $Path)
{
    $ConfigPath = $null
    $AdapterObject = $script:MSNetAdapters.Where({$_.ifDesc -eq $AdapterName})
    $972Key = @(Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -Name "NetCfgInstanceId" -ErrorAction SilentlyContinue)
    $Adapter972Key = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterObject.InterfaceGuid})
    if ($Adapter972Key)
    {
        $ConfigPath = $Adapter972Key.PSPath + $Path
    }

    return $ConfigPath
}

function GetItemPropertyValue($AdapterName, $Path, $PropertyName)
{
    $PropertyItem = $null

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $Path
    if ($null -ne $FWLogConfigPath)
    {
        $PropertyItem = Get-ItemProperty -Path $FWLogConfigPath -Name $PropertyName -ErrorAction SilentlyContinue
    }

    $PropertyValue = $null
    if ($null -ne $PropertyItem)
    {
        $PropertyValue = $PropertyItem.$PropertyName
    }

    return $PropertyValue
}

# SIG # Begin signature block
# MIIocgYJKoZIhvcNAQcCoIIoYzCCKF8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCWHYOF1gGMOvcg
# 7t6MGE6DwANnu0CF3XTMI6S6JZ5sMKCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQCmOovGKSxCae+j/hQscnOWMA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjMwMTI1MDAwMDAwWhcNMjQwMTI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDGlXgQt1127m+5LYJmaFBzYTdi9cRozQjdZt/+
# +l4e/XT+sHZemhE0o/x+nydO/Ai7lhA6+Rs0Juw3vSXdGjKZLJ2G19nKCeICKfyK
# /rTX7GkdhwZnX8yQJEWU2VuZ/Ad9MtybHiZY3tIlpauWvNmfILMXWL+8DZOzVifw
# KbQgObHaYERAsfT2ZJ0HYp8lo5XepZy8SCeY/kRShw37vMeqNnuJtjnoB81/sZmI
# bZfLL4XxYn3ESyeeXgGhpplv3DkBUFM6YdD7lNXN7yp7h1gCJUA648U9EMOoQAHi
# ewLOPQSX/xgNqHfk3FUbcwZp07/zLrHtKvDNdPFrgkTBz81xGzVHOi48Lh6FBGW7
# UFEq/jI5TOk/AcM5eDvsizy4dA6J0AJ0o3lez+jAP2tDccEt7bn1aaamcjFpDOol
# MktQz0H07XqmRgVT3wHGHiDswDNUCxbk3ervsNG1afAPcFh5Hx4up8hpJoF7YTOf
# 6OHm8k3dRtjF1DC78oNYgqkpuNMCAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBRvq1HVd47vo0fOeEIdOUOyHE5Z
# ojAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBAJan0xi9l18PVDDZYZmI6kX64OJRsJvMbeAOd/iCwfoO1W2d
# +vin3je7xjz7TX1tERtLQGbRiraogMGVMco6+BSXJn0mGwpPcxsaRLpNbcbfXPHx
# niq3yO3gbSOrUU1Z9kScqlrqgbaAMKWnwlhy8MYbo9H2wIXCtrZ9kCuwoI9iilKS
# jjQqWs4085hIBzp3Kc32V4fLkuKGH1KYeTRMm8hP4+V0D+vBVFSu7PFh5t8p93y+
# Y/n7icZkyci6GUMUbVi8YK2z5KDr/ohPWS3a+DP+gAtjafTITgucRazljocMCIIg
# eWtL1Gwq0hvcpQOn+pdNGg8vUIXuVtziS9NmJKVfHEnOXtvte0ep9ziC5yaE8ui0
# 3e0ouY6xuhqPY5+mh9Wve5BX5ju1ewEUuImurs0xjtNBr3bscylS3AkssWz0LBko
# gR/aT9luMyUm/rC/2JLMrDlRTuiWergSd1ffFeLGnQBWtvd0snyFK3ZlVbJelDJP
# 47S4Vj1s45eyKo2DYTGCFdUwghXRAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRAKY6i8YpLEJp76P+FCxyc5YwDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgNVssg8lWdvCmZ3eLXFcP
# 8hfiKa5lRYUdxSoTmu8mAW0wDQYJKoZIhvcNAQEBBQAEggGAAFZHaJx7wx8S/e8X
# d4fdYo81rVPEkscb4H738ELVWjVfIJKWgmjBJ1+wlRTDbPKLr3bxGmWdNJcl5DdC
# Kl2kCugOg1+x5Y025PJXUJQXej3zsvzI7mh6EdQC4l0P9uDaNr8YD20+7dXUsieW
# h/iBB1NTUPy5T3uxxj403WJaLIhJpXtMVNZ47M4n8arfQpy1VTzEtAMZkYQjkcZk
# WyamdoUMuDT/30SaREjoD6wK22JcJvhBDKmj/q4CwmCV74PY2RdMJb6JFA6xZJuD
# 3rZ3v5uKsAsTJk7fNHOMdNIHfcW2LpAhCkdqPpJ+Ot9dzkEq0lPwT/GEdTwdkPkm
# Y8aep0PsUboMlVTPgp1FiHvV4nJ4vLH6utEIoONx0UC/3sSOZDisaSYnfRvDOjG2
# Kz4fZwLvoruiMjhLM1xAkgm5EltkkDL2KvmnCoAyo3axn24DC93VAkNw9y2FqHaH
# iYvJ18cc0VDxtLrVxgQQB8eg5CIGa0ydZSscMlyeunfxiLGxoYITTjCCE0oGCisG
# AQQBgjcDAwExghM6MIITNgYJKoZIhvcNAQcCoIITJzCCEyMCAQMxDzANBglghkgB
# ZQMEAgIFADCB7wYLKoZIhvcNAQkQAQSggd8EgdwwgdkCAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQglBNgovLCStGwPJnBUiWElfJRIcZZbe0z2zOt6EBS
# n5ECFHBCH1qZKSANVIdZqxoJABDHq7F1GA8yMDIzMTEwNzE5MDA1NVqgbqRsMGox
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# U2lnbmVyICM0oIIN6TCCBvUwggTdoAMCAQICEDlMJeF8oG0nqGXiO9kdItQwDQYJ
# KoZIhvcNAQEMBQAwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFu
# Y2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMB4XDTIz
# MDUwMzAwMDAwMFoXDTM0MDgwMjIzNTk1OVowajELMAkGA1UEBhMCR0IxEzARBgNV
# BAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UE
# AwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBTaWduZXIgIzQwggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQCkkyhSS88nh3akKRyZOMDnDtTRHOxoywFk
# 5IrNd7BxZYK8n/yLu7uVmPslEY5aiAlmERRYsroiW+b2MvFdLcB6og7g4FZk7aHl
# gSByIGRBbMfDCPrzfV3vIZrCftcsw7oRmB780yAIQrNfv3+IWDKrMLPYjHqWShkT
# XKz856vpHBYusLA4lUrPhVCrZwMlobs46Q9vqVqakSgTNbkf8z3hJMhrsZnoDe+7
# TeU9jFQDkdD8Lc9VMzh6CRwH0SLgY4anvv3Sg3MSFJuaTAlGvTS84UtQe3LgW/0Z
# ux88ahl7brstRCq+PEzMrIoEk8ZXhqBzNiuBl/obm36Ih9hSeYn+bnc317tQn/oY
# JU8T8l58qbEgWimro0KHd+D0TAJI3VilU6ajoO0ZlmUVKcXtMzAl5paDgZr2YGaQ
# WAeAzUJ1rPu0kdDF3QFAaraoEO72jXq3nnWv06VLGKEMn1ewXiVHkXTNdRLRnG/k
# Xg2b7HUm7v7T9ZIvUoXo2kRRKqLMAMqHZkOjGwDvorWWnWKtJwvyG0rJw5RCN4gg
# hKiHrsO6I3J7+FTv+GsnsIX1p0OF2Cs5dNtadwLRpPr1zZw9zB+uUdB7bNgdLRFC
# U3F0wuU1qi1SEtklz/DT0JFDEtcyfZhs43dByP8fJFTvbq3GPlV78VyHOmTxYEsF
# T++5L+wJEwIDAQABo4IBgjCCAX4wHwYDVR0jBBgwFoAUGqH4YRkgD8NBd0UojtE1
# XwYSBFUwHQYDVR0OBBYEFAMPMciRKpO9Y/PRXU2kNA/SlQEYMA4GA1UdDwEB/wQE
# AwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMEoGA1Ud
# IARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8vc2Vj
# dGlnby5jb20vQ1BTMAgGBmeBDAEEAjBEBgNVHR8EPTA7MDmgN6A1hjNodHRwOi8v
# Y3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcmwwdAYI
# KwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnNlY3RpZ28uY29t
# L1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6
# Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBMm2VY+uB5z+8V
# wzJt3jOR63dY4uu9y0o8dd5+lG3DIscEld9laWETDPYMnvWJIF7Bh8cDJMrHpfAm
# 3/j4MWUN4OttUVemjIRSCEYcKsLe8tqKRfO+9/YuxH7t+O1ov3pWSOlh5Zo5d7y+
# upFkiHX/XYUWNCfSKcv/7S3a/76TDOxtog3Mw/FuvSGRGiMAUq2X1GJ4KoR5qNc9
# rCGPcMMkeTqX8Q2jo1tT2KsAulj7NYBPXyhxbBlewoNykK7gxtjymfvqtJJlfAd8
# NUQdrVgYa2L73mzECqls0yFGcNwvjXVMI8JB0HqWO8NL3c2SJnR2XDegmiSeTl9O
# 048P5RNPWURlS0Nkz0j4Z2e5Tb/MDbE6MNChPUitemXk7N/gAfCzKko5rMGk+al9
# NdAyQKCxGSoYIbLIfQVxGksnNqrgmByDdefHfkuEQ81D+5CXdioSrEDBcFuZCkD6
# gG2UYXvIbrnIZ2ckXFCNASDeB/cB1PguEc2dg+X4yiUcRD0n5bCGRyoLG4R2fXto
# T4239xO07aAt7nMP2RC6nZksfNd1H48QxJTmfiTllUqIjCfWhWYd+a5kdpHoSP7I
# VQrtKcMf3jimwBT7Mj34qYNiNsjDvgCHHKv6SkIciQPc9Vx8cNldeE7un14g5glq
# fCsIo0j1FfwET9/NIRx65fWOGtS5QDCCBuwwggTUoAMCAQICEDAPb6zdZph0fKlG
# Nqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpO
# ZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVT
# RVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoXDTM4MDExODIzNTk1OVow
# fTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4G
# A1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQD
# ExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyAIVSTuc+gPlPvs1rAdtYa
# BKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhObtoY/AQWSZm8hq9VxEHmH
# 9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5pCNCrNAVw3+eolzXOPEW
# snDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EBjm6dJ3VV0Rc7NCF7lwGU
# r3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWMilZx0S2UTMiMPNMU
# opy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G493bkV+fPzY+DjBnK0a3
# Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3pBXrGVX/0DlULSbu
# RRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH3uhrDvBbfel2aZMgHEqX
# LHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqNoWTJiJJZy6hGwMny
# pv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfobpSKupcNNgtCN2mu32/cY
# QFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInXT50zvRq1naHelUF6p4MC
# AwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0G
# A1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNVHQ8BAf8EBAMCAYYwEgYD
# VR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAI
# MAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3Qu
# Y29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHkuY3JsMHYGCCsG
# AQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2NydC51c2VydHJ1c3QuY29t
# L1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUFBzABhhlodHRwOi8v
# b2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBtVIGlM10W4bVT
# gZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ2+J9+Z8A70DDrdPi
# 5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V1cPivNVYeL9IduFEZ22P
# sEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8Blr5yUmo7Qjd8Fng5k5OU
# m7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZLXq2jK0KKIZL+qG9aJXBi
# gXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXnd8ucLo0jGLmjwkcd8F3W
# oXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1BWy30mRa2Coiwkud8Tle
# TN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9UIFuq/DQ0/yysx+2mZYm
# 9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4ZSA7D0GqH+mhQZ3+
# sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbGpm0yXbQ4CStJB6r97DDa
# 8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7xfCAMKneutaIChrop
# 7rRaALGMq+P5CslUXdS5anSevUiumDGCBCwwggQoAgEBMIGRMH0xCzAJBgNVBAYT
# AkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZv
# cmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBS
# U0EgVGltZSBTdGFtcGluZyBDQQIQOUwl4XygbSeoZeI72R0i1DANBglghkgBZQME
# AgIFAKCCAWswGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJ
# BTEPFw0yMzExMDcxOTAwNTVaMD8GCSqGSIb3DQEJBDEyBDDfqh3ivETGA7RWsImv
# 9QMlvzGRmfLnbTpJndn56YuX6Mkz1g/0XIwW6SS2KyN2ro4wge0GCyqGSIb3DQEJ
# EAIMMYHdMIHaMIHXMBYEFK5ir3UKDL1H1kYfdWjivIznyk+UMIG8BBQC1luV4oNw
# wVcAlfqI+SPdk3+tjzCBozCBjqSBizCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Ck5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUg
# VVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlm
# aWNhdGlvbiBBdXRob3JpdHkCEDAPb6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEB
# BQAEggIAfIDG9eh0RB9nbs9CUGKRuz7Q9rQRLNEdUfG0VenIGKeL8MkHr8b6m3vI
# 8Y1rFsLHdZDTE3/ruT8Hb7/2ZOMMOd6d78EN5gKvInwo/s8kAh+yXAPqSBGuyMkj
# G/uy/yBacWEUugHQ73YKH5iomXHTN9VK6WXote2Q+P/2V5iE75StT1rjQy73OwBc
# vaRlNWVZJ0WsiQr3MthjY7bpQfBLXbC1NzX6i3wlgwdUvs3p/QuSYsbA1Az1zzUw
# QpsR3hvIKbaeeNG1gxpralAxqb4UwPws9q31GFEJCgh4sEf71/CN4AAaGftPafGu
# W4BjGYJ6iSq5H2TEuEb8ebycriWBJj35XqpzRgVm0G/K7HXc0N5td70IJ7owC2So
# AuPGBBKROe0gIK2hVa1+J/IHbYwbnMCYJk55aPMtG84zxf5ecjPuc79ygCCoBQMF
# 6RWj4upr1SEcGQaxWjw7Na3iA1tXx3c0bhyqyPJ9E3ASfUhXd7CAvS1F0dVBY/Lg
# n8+ser/w5ci8h4XohB8MKRVOV68uijzV4PeJlbBEHagOhL10ecAljlVJquMKSpTe
# ARvjudk++Fjm0Fr76YaByxpqVPzeYVz61qi/PNoHNKmUe7erntiKPPN1HkYK6GhK
# 1i5/3A+LzlfI0QuMef95p7SRLO7a/ZVlg4Z3JKGkuFYEDoJbYNw=
# SIG # End signature block
