# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2023 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseDeclaredVarsMoreThanAssignments", "")] Param()

#region CSharp_code
$Dflat = @"
using System;
using System.Runtime.InteropServices;

namespace IntelEthernetCmdlets
{
    class Util
    {
        public static bool CheckBit(UInt16 Value, UInt16 Bit)
        {
            if ((Value & Bit) == Bit)
            {
                return true;
            }
            return false;
        }
    }

    public class DcbConstants
    {
        public const Int32 MAX_TRAFFIC_CLASS = 8;
        public const Int32 MAX_USER_PRIORITY = 8;
        public const Int32 DCBX_MAX_APPS = 32;
        public const Int32 DCBX_APPS_SIZE = DCBX_MAX_APPS * 4;
    }

    public class DcbDataRegFlagValue
    {
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_CAP = 0x0001;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_EN = 0x0002;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_FW_INC = 0x0004;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_NPAR = 0x0008;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_ISCSI = 0x0010;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_MAIN_VSI = 0x0020;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_FCOE_VSI = 0x0040;
    }

    public class DcbxModeValue
    {
        public const Byte DCBX_MODE_CEE = 0x1;
        public const Byte DCBX_MODE_IEEE = 0x2;
    }

    public enum RegVer
    {
        VER_INVALID,
        I40E,
        ICE
    }

    public class IntlLan_GetDcbData
    {
        public IntlLan_GetDcbData()
        {
            LocalConfig = new IntlLan_GetDcbDataDcbxConfig();
            LocalConfig.ConfigName = "LocalConfig";

            RemoteConfig = new IntlLan_GetDcbDataDcbxConfig();
            RemoteConfig.ConfigName = "RemoteConfig";

            DesiredConfig = new IntlLan_GetDcbDataDcbxConfig();
            DesiredConfig.ConfigName = "DesiredConfig";
        }
        public String Name { get; set; }
        public UInt16 Flags { get; set; }
        public Boolean DcbCapable { get; set; }
        public Boolean DcbEnabled { get; set; }

        public UInt16 DcbxStatus { get; set; }

        public IntlLan_GetDcbDataDcbxConfig LocalConfig { get; set; }
        public IntlLan_GetDcbDataDcbxConfig RemoteConfig { get; set; }
        public IntlLan_GetDcbDataDcbxConfig DesiredConfig { get; set; }

        public IIntlLan_GetDcbDataRegisters DcbRegs { get; set; }
        public RegVer RegVer { get; set; }
    }

    public class IntlLan_GetDcbDataDcbxConfig
    {
        public IntlLan_GetDcbDataDcbxConfig()
        {
            EtsCfg = new IntlLan_GetDcbDataEts();
            EtsCfg.TableName = "EtsCfg";
            EtsRec = new IntlLan_GetDcbDataEts();
            EtsRec.TableName = "EtsRec";
            Pfc = new IntlLan_GetDcbDataPfc();
        }
        public String Name { get; set; }
        public Byte DcbxMode { get; set; }
        public Byte AppMode { get; set; }
        public UInt32 CeeTlvStatus { get; set; }
        public UInt32 NumApps { get; set; }

        public String ConfigName { get; set; }

        public IntlLan_GetDcbDataEts EtsCfg { get; set; }
        public IntlLan_GetDcbDataEts EtsRec { get; set; }
        public IntlLan_GetDcbDataPfc Pfc { get; set; }
        public IntlLan_GetDcbDataAppPriority[] AppPriority { get; set; }
    }

    public class IntlLan_GetDcbDataEts
    {
        public IntlLan_GetDcbDataEts()
        {
            UP = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            BW = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            TSA = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            PfcEnableBits = new Boolean[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public String TableName { get; set; }
        public Boolean Willing { get; set; }
        public Byte CBS { get; set; }
        public Byte MaxTCs { get; set; }
        public Boolean[] PfcEnableBits { get; set; }
        public Byte[] UP { get; set; }
        public Byte[] BW { get; set; }
        public Byte[] TSA { get; set; }
    }

    public class IntlLan_GetDcbDataPfc
    {
        public IntlLan_GetDcbDataPfc()
        {
            PfcEnableBits = new Boolean[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public Boolean Willing { get; set; }
        public Byte MBC { get; set; }
        public Byte PFCcap { get; set; }
        public Byte PfcEnable { get; set; }
        public Boolean[] PfcEnableBits { get; set; }
    }

    public class IntlLan_GetDcbDataAppPriority
    {
        public Byte Priority { get; set; }
        public Byte Selector { get; set; }
        public UInt16 ProtocolId { get; set; }
    }

    public interface IIntlLan_GetDcbDataRegisters { }

    public class IntlLan_GetDcbDataRegistersI40e : IIntlLan_GetDcbDataRegisters
    {
        public IntlLan_GetDcbDataRegistersI40e()
        {
            I40E_PRT_TCTUPR = new UInt32[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public UInt32 I40E_PRTDCB_GENC { get; set; }
        public UInt32 I40E_PRTDCB_TC2PFC { get; set; }
        public UInt32 I40E_PRTDCB_MFLCN { get; set; }
        public UInt32 I40E_PRTDCB_TUP2TC { get; set; }
        public UInt32 I40E_PRTDCB_RUP2TC { get; set; }
        public UInt32[] I40E_PRT_TCTUPR { get; set; }
        public UInt32 I40E_VSI_TUPR_MainVsi { get; set; }
        public UInt32 I40E_VSI_RUPR_MainVsi { get; set; }
        public UInt32 I40E_VSI_TUPR_FCoEVsi { get; set; }
        public UInt32 I40E_VSI_RUPR_FCoEVsi { get; set; }
    }

    public class IntlLan_GetDcbDataRegistersIce : IIntlLan_GetDcbDataRegisters
    {
        public IntlLan_GetDcbDataRegistersIce()
        {
            PrtTcTuprVal = new UInt32[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public UInt32 PrtDcbGencVal { get; set; }
        public UInt32 GlDcbTc2PfcVal { get; set; }
        public UInt32 PrtMacHsecCtlTxPauseEnableVal { get; set; }
        public UInt32 PrtMacHsecCtlRxPauseEnableVal { get; set; }
        public UInt32 PrtDcbTup2TcVal { get; set; }
        public UInt32 PrtDcbRup2TcVal { get; set; }
        public UInt32[] PrtTcTuprVal { get; set; }
        public UInt32 VsiTuprMainVsiVal { get; set; }
        public UInt32 VsiRuprMainVsiVal { get; set; }
    }

    public unsafe interface IGetDcb
    {
        IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes);
    }

    #region 100G DCB
    public unsafe class GetDcbIce : IGetDcb
    {
        private const Int32 ICE_DSCP_NUM_VAL = 64;
        private const Int32 DSCP_MAPPED_SIZE = 8;


        private struct ice_dcb_ets_cfg
        {
            public Byte willing;
            public Byte cbs;
            public Byte maxtcs;
            public fixed Byte prio_table[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tcbwtable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tsatable[DcbConstants.MAX_TRAFFIC_CLASS];
        }


        private struct ice_dcb_pfc_cfg
        {
            public Byte willing;
            public Byte mbc;
            public Byte pfccap;
            public Byte pfcenable;
        }


        private struct ice_dcb_app_priority_table
        {
            public UInt16 prot_id;
            public Byte priority;
            public Byte selector;
        }


        private struct ice_dcbx_cfg
        {
            public Byte dcbx_mode;
            public Byte app_mode;
            public UInt32 numapps;
            public UInt32 tlv_status;
            public ice_dcb_ets_cfg etscfg;
            public ice_dcb_ets_cfg etsrec;
            public ice_dcb_pfc_cfg pfc;
            public fixed Byte app[DcbConstants.DCBX_APPS_SIZE];
        }


        private struct DcbRegisters
        {
            public UInt32 PrtDcbGencVal;
            public UInt32 GlDcbTc2PfcVal;
            public UInt32 PrtMacHsecCtlTxPauseEnableVal;
            public UInt32 PrtMacHsecCtlRxPauseEnableVal;
            public UInt32 PrtDcbTup2TcVal;
            public UInt32 PrtDcbRup2TcVal;
            public fixed UInt32 PrtTcTuprVal[8];
            public UInt32 VsiTuprMainVsiVal;
            public UInt32 VsiRuprMainVsiVal;
        }


        private struct DcbGetGenData
        {
            public UInt16 Flags;
            public UInt16 DcbxStatus;
            public ice_dcbx_cfg LocalConfig;
            public ice_dcbx_cfg RemoteConfig;
            public ice_dcbx_cfg DesiredConfig;
            public DcbRegisters DcbRegs;
        }

        public IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes)
        {
            var pData = GCHandle.Alloc(bytes, GCHandleType.Pinned);
            var result = (DcbGetGenData)Marshal.PtrToStructure(pData.AddrOfPinnedObject(), typeof(DcbGetGenData));
            pData.Free();
            DcbGetGenData* pDcbData = (DcbGetGenData*)&result;
            return CreatePsObject(strPortName, *pDcbData);
        }

        private IntlLan_GetDcbData CreatePsObject(String strPortName, DcbGetGenData DcbData)
        {
            IntlLan_GetDcbData GetDcbData = new IntlLan_GetDcbData
            {
                Name = strPortName,
                Flags = DcbData.Flags,
                DcbCapable = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_CAP)),
                DcbEnabled = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_EN)),
                DcbxStatus = DcbData.DcbxStatus,
            };

            CopyDcbxConfig(strPortName, DcbData.LocalConfig, GetDcbData.LocalConfig);
            CopyDcbxConfig(strPortName, DcbData.RemoteConfig, GetDcbData.RemoteConfig);
            CopyDcbxConfig(strPortName, DcbData.DesiredConfig, GetDcbData.DesiredConfig);

            GetDcbData.DcbRegs = new IntlLan_GetDcbDataRegistersIce();
            GetDcbData.RegVer = RegVer.ICE;
            CopyDcbRegs(strPortName, DcbData.DcbRegs, GetDcbData.DcbRegs);
            return GetDcbData;
        }


        private void CopyDcbxConfig(String strPortName, ice_dcbx_cfg FromDcbxConfig, IntlLan_GetDcbDataDcbxConfig ToDcbxConfig)
        {
            ToDcbxConfig.Name = strPortName;
            ToDcbxConfig.DcbxMode = FromDcbxConfig.dcbx_mode;
            ToDcbxConfig.AppMode = FromDcbxConfig.app_mode;
            ToDcbxConfig.CeeTlvStatus = FromDcbxConfig.tlv_status;
            ToDcbxConfig.NumApps = FromDcbxConfig.numapps;
            if (ToDcbxConfig.NumApps > DcbConstants.DCBX_MAX_APPS)
            {
                ToDcbxConfig.NumApps = DcbConstants.DCBX_MAX_APPS;
            }

            CopyPfcConfig(strPortName, FromDcbxConfig.pfc, ToDcbxConfig.Pfc);

            CopyEtsConfig(strPortName, FromDcbxConfig.etscfg, ToDcbxConfig.EtsCfg);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsCfg.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            CopyEtsConfig(strPortName, FromDcbxConfig.etsrec, ToDcbxConfig.EtsRec);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsRec.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            ToDcbxConfig.AppPriority = new IntlLan_GetDcbDataAppPriority[ToDcbxConfig.NumApps];

            CopyApps(FromDcbxConfig.app, ToDcbxConfig.NumApps, ToDcbxConfig.AppPriority);
        }


        private void CopyEtsConfig(String strPortName, ice_dcb_ets_cfg FromEtsConfig, IntlLan_GetDcbDataEts ToEtsConfig)
        {
            ToEtsConfig.Name = strPortName;
            ToEtsConfig.Willing = (FromEtsConfig.willing == 1);
            ToEtsConfig.CBS = FromEtsConfig.cbs;
            ToEtsConfig.MaxTCs = FromEtsConfig.maxtcs;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToEtsConfig.UP[i] = FromEtsConfig.prio_table[i];
                ToEtsConfig.BW[i] = FromEtsConfig.tcbwtable[i];
                ToEtsConfig.TSA[i] = FromEtsConfig.tsatable[i];
            }
        }


        private void CopyPfcConfig(String strPortName, ice_dcb_pfc_cfg FromPfcConfig, IntlLan_GetDcbDataPfc ToPfcConfig)
        {
            ToPfcConfig.Name = strPortName;
            ToPfcConfig.Willing = (FromPfcConfig.willing == 1);
            ToPfcConfig.MBC = FromPfcConfig.mbc;
            ToPfcConfig.PFCcap = FromPfcConfig.pfccap;
            ToPfcConfig.PfcEnable = FromPfcConfig.pfcenable;
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToPfcConfig.PfcEnableBits[i] = ((ToPfcConfig.PfcEnable & (1 << i)) == (1 << i));
            }
        }


        private void CopyApps(Byte* FromApps, UInt32 NumApps, IntlLan_GetDcbDataAppPriority[] ToApps)
        {
            ice_dcb_app_priority_table* AppTable = (ice_dcb_app_priority_table*)FromApps;
            for (int i = 0; i < NumApps; i++)
            {
                // Since this array's elements are a class, and classes are ref types, we need
                // to explicitly allocate the elements.
                ToApps[i] = new IntlLan_GetDcbDataAppPriority();

                if (AppTable->prot_id > 0)
                {
                    ToApps[i].Priority = AppTable->priority;
                    ToApps[i].Selector = AppTable->selector;
                    ToApps[i].ProtocolId = AppTable->prot_id;
                }

                AppTable++;
            }
        }


        private void CopyDcbRegs(String strPortName, DcbRegisters FromDcbRegs, IIntlLan_GetDcbDataRegisters ToDcbRegs)
        {
            IntlLan_GetDcbDataRegistersIce toReg = (IntlLan_GetDcbDataRegistersIce)ToDcbRegs;

            toReg.Name = strPortName;
            toReg.PrtDcbGencVal = FromDcbRegs.PrtDcbGencVal;
            toReg.GlDcbTc2PfcVal = FromDcbRegs.GlDcbTc2PfcVal;
            toReg.PrtMacHsecCtlTxPauseEnableVal = FromDcbRegs.PrtMacHsecCtlTxPauseEnableVal;
            toReg.PrtMacHsecCtlRxPauseEnableVal = FromDcbRegs.PrtMacHsecCtlRxPauseEnableVal;
            toReg.PrtDcbTup2TcVal = FromDcbRegs.PrtDcbTup2TcVal;
            toReg.PrtDcbRup2TcVal = FromDcbRegs.PrtDcbRup2TcVal;
            toReg.VsiTuprMainVsiVal = FromDcbRegs.VsiTuprMainVsiVal;
            toReg.VsiRuprMainVsiVal = FromDcbRegs.VsiRuprMainVsiVal;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                toReg.PrtTcTuprVal[i] = FromDcbRegs.PrtTcTuprVal[i];
            }
        }
    }
    #endregion

    #region 40G DCB
    public unsafe class GetDcbI40e : IGetDcb
    {
        private struct i40e_dcb_ets_config
        {
            public Byte willing;
            public Byte cbs;
            public Byte maxtcs;
            public fixed Byte prioritytable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tcbwtable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tsatable[DcbConstants.MAX_TRAFFIC_CLASS];
        }

        private struct i40e_dcb_pfc_config
        {
            public Byte willing;
            public Byte mbc;
            public Byte pfccap;
            public Byte pfcenable;
        }

        private struct i40e_dcb_app_priority_table
        {
            public Byte priority;
            public Byte selector;
            public UInt16 protocolid;
        }

        private struct i40e_dcbx_config
        {
            public Byte dcbx_mode;
            public Byte app_mode;
            public UInt32 numapps;
            public UInt32 tlv_status;
            public i40e_dcb_ets_config etscfg;
            public i40e_dcb_ets_config etsrec;
            public i40e_dcb_pfc_config pfc;
            public fixed Byte app[DcbConstants.DCBX_APPS_SIZE];
        }

        private struct DcbRegisters
        {
            public UInt32 I40E_PRTDCB_GENC;
            public UInt32 I40E_PRTDCB_TC2PFC;
            public UInt32 I40E_PRTDCB_MFLCN;
            public UInt32 I40E_PRTDCB_TUP2TC;
            public UInt32 I40E_PRTDCB_RUP2TC;
            public fixed UInt32 I40E_PRT_TCTUPR[8];
            public UInt32 I40E_VSI_TUPR_MainVsi;
            public UInt32 I40E_VSI_RUPR_MainVsi;
            public UInt32 I40E_VSI_TUPR_FCoEVsi;
            public UInt32 I40E_VSI_RUPR_FCoEVsi;
        }

        private struct DcbGetGenData
        {
            public UInt16 Flags;
            public UInt16 DcbxStatus;
            public i40e_dcbx_config LocalConfig;
            public i40e_dcbx_config RemoteConfig;
            public i40e_dcbx_config DesiredConfig;
            public DcbRegisters DcbRegs;
        }

        public IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes)
        {
            var pData = GCHandle.Alloc(bytes, GCHandleType.Pinned);
            var result = (DcbGetGenData)Marshal.PtrToStructure(pData.AddrOfPinnedObject(), typeof(DcbGetGenData));
            pData.Free();
            DcbGetGenData* pDcbData = (DcbGetGenData*)&result;
            return CreatePsObject(strPortName, *pDcbData);
        }

        private IntlLan_GetDcbData CreatePsObject(String strPortName, DcbGetGenData DcbData)
        {
            IntlLan_GetDcbData GetDcbData = new IntlLan_GetDcbData
            {
                Name = strPortName,
                Flags = DcbData.Flags,
                DcbCapable = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_CAP)),
                DcbEnabled = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_EN)),
                DcbxStatus = DcbData.DcbxStatus,
            };

            CopyDcbxConfig(strPortName, DcbData.LocalConfig, GetDcbData.LocalConfig);
            CopyDcbxConfig(strPortName, DcbData.RemoteConfig, GetDcbData.RemoteConfig);
            CopyDcbxConfig(strPortName, DcbData.DesiredConfig, GetDcbData.DesiredConfig);

            GetDcbData.DcbRegs = new IntlLan_GetDcbDataRegistersI40e();
            GetDcbData.RegVer = RegVer.I40E;
            CopyDcbRegs(strPortName, DcbData.DcbRegs, GetDcbData.DcbRegs);
            return GetDcbData;
        }

        private void CopyDcbxConfig(String strPortName, i40e_dcbx_config FromDcbxConfig, IntlLan_GetDcbDataDcbxConfig ToDcbxConfig)
        {
            ToDcbxConfig.Name = strPortName;
            ToDcbxConfig.DcbxMode = FromDcbxConfig.dcbx_mode;
            ToDcbxConfig.AppMode = FromDcbxConfig.app_mode;
            ToDcbxConfig.CeeTlvStatus = FromDcbxConfig.tlv_status;
            ToDcbxConfig.NumApps = FromDcbxConfig.numapps;
            if (ToDcbxConfig.NumApps > DcbConstants.DCBX_MAX_APPS)
            {
                ToDcbxConfig.NumApps = DcbConstants.DCBX_MAX_APPS;
            }

            CopyPfcConfig(strPortName, FromDcbxConfig.pfc, ToDcbxConfig.Pfc);

            CopyEtsConfig(strPortName, FromDcbxConfig.etscfg, ToDcbxConfig.EtsCfg);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsCfg.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            CopyEtsConfig(strPortName, FromDcbxConfig.etsrec, ToDcbxConfig.EtsRec);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsRec.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            ToDcbxConfig.AppPriority = new IntlLan_GetDcbDataAppPriority[ToDcbxConfig.NumApps];

            CopyApps(FromDcbxConfig.app, ToDcbxConfig.NumApps, ToDcbxConfig.AppPriority);
        }


        private void CopyEtsConfig(String strPortName, i40e_dcb_ets_config FromEtsConfig, IntlLan_GetDcbDataEts ToEtsConfig)
        {
            ToEtsConfig.Name = strPortName;
            ToEtsConfig.Willing = (FromEtsConfig.willing == 1);
            ToEtsConfig.CBS = FromEtsConfig.cbs;
            ToEtsConfig.MaxTCs = FromEtsConfig.maxtcs;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToEtsConfig.UP[i] = FromEtsConfig.prioritytable[i];
                ToEtsConfig.BW[i] = FromEtsConfig.tcbwtable[i];
                ToEtsConfig.TSA[i] = FromEtsConfig.tsatable[i];
            }
        }


        private void CopyPfcConfig(String strPortName, i40e_dcb_pfc_config FromPfcConfig, IntlLan_GetDcbDataPfc ToPfcConfig)
        {
            ToPfcConfig.Name = strPortName;
            ToPfcConfig.Willing = (FromPfcConfig.willing == 1);
            ToPfcConfig.MBC = FromPfcConfig.mbc;
            ToPfcConfig.PFCcap = FromPfcConfig.pfccap;
            ToPfcConfig.PfcEnable = FromPfcConfig.pfcenable;
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToPfcConfig.PfcEnableBits[i] = ((ToPfcConfig.PfcEnable & (1 << i)) == (1 << i));
            }
        }


        private void CopyApps(Byte* FromApps, UInt32 NumApps, IntlLan_GetDcbDataAppPriority[] ToApps)
        {
            i40e_dcb_app_priority_table* AppTable = (i40e_dcb_app_priority_table*)FromApps;
            for (int i = 0; i < NumApps; i++)
            {
                // Since this array's elements are a class, which and classes are ref types, we need
                // to explicitly allocate the elements.
                ToApps[i] = new IntlLan_GetDcbDataAppPriority();

                if (AppTable->protocolid > 0)
                {
                    ToApps[i].Priority = AppTable->priority;
                    ToApps[i].Selector = AppTable->selector;
                    ToApps[i].ProtocolId = AppTable->protocolid;
                }

                AppTable++;
            }
        }


        private void CopyDcbRegs(String strPortName, DcbRegisters FromDcbRegs, IIntlLan_GetDcbDataRegisters ToDcbRegs)
        {
            IntlLan_GetDcbDataRegistersI40e toReg = (IntlLan_GetDcbDataRegistersI40e)ToDcbRegs;

            toReg.Name = strPortName;
            toReg.I40E_PRTDCB_GENC = FromDcbRegs.I40E_PRTDCB_GENC;
            toReg.I40E_PRTDCB_TC2PFC = FromDcbRegs.I40E_PRTDCB_TC2PFC;
            toReg.I40E_PRTDCB_MFLCN = FromDcbRegs.I40E_PRTDCB_MFLCN;
            toReg.I40E_PRTDCB_TUP2TC = FromDcbRegs.I40E_PRTDCB_TUP2TC;
            toReg.I40E_PRTDCB_RUP2TC = FromDcbRegs.I40E_PRTDCB_RUP2TC;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                toReg.I40E_PRT_TCTUPR[i] = FromDcbRegs.I40E_PRT_TCTUPR[i];
            }

            toReg.I40E_VSI_TUPR_MainVsi = FromDcbRegs.I40E_VSI_TUPR_MainVsi;
            toReg.I40E_VSI_RUPR_MainVsi = FromDcbRegs.I40E_VSI_RUPR_MainVsi;
            toReg.I40E_VSI_TUPR_FCoEVsi = FromDcbRegs.I40E_VSI_TUPR_FCoEVsi;
            toReg.I40E_VSI_RUPR_FCoEVsi = FromDcbRegs.I40E_VSI_RUPR_FCoEVsi;
        }
    }
    #endregion
}
"@
#endregion CSharp_code

# SIG # Begin signature block
# MIIocwYJKoZIhvcNAQcCoIIoZDCCKGACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCM5Jlr7XMXI4xy
# ovQHeqTjsb8STCSzX/UJOOHIVbKPIqCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQCmOovGKSxCae+j/hQscnOWMA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjMwMTI1MDAwMDAwWhcNMjQwMTI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDGlXgQt1127m+5LYJmaFBzYTdi9cRozQjdZt/+
# +l4e/XT+sHZemhE0o/x+nydO/Ai7lhA6+Rs0Juw3vSXdGjKZLJ2G19nKCeICKfyK
# /rTX7GkdhwZnX8yQJEWU2VuZ/Ad9MtybHiZY3tIlpauWvNmfILMXWL+8DZOzVifw
# KbQgObHaYERAsfT2ZJ0HYp8lo5XepZy8SCeY/kRShw37vMeqNnuJtjnoB81/sZmI
# bZfLL4XxYn3ESyeeXgGhpplv3DkBUFM6YdD7lNXN7yp7h1gCJUA648U9EMOoQAHi
# ewLOPQSX/xgNqHfk3FUbcwZp07/zLrHtKvDNdPFrgkTBz81xGzVHOi48Lh6FBGW7
# UFEq/jI5TOk/AcM5eDvsizy4dA6J0AJ0o3lez+jAP2tDccEt7bn1aaamcjFpDOol
# MktQz0H07XqmRgVT3wHGHiDswDNUCxbk3ervsNG1afAPcFh5Hx4up8hpJoF7YTOf
# 6OHm8k3dRtjF1DC78oNYgqkpuNMCAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBRvq1HVd47vo0fOeEIdOUOyHE5Z
# ojAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBAJan0xi9l18PVDDZYZmI6kX64OJRsJvMbeAOd/iCwfoO1W2d
# +vin3je7xjz7TX1tERtLQGbRiraogMGVMco6+BSXJn0mGwpPcxsaRLpNbcbfXPHx
# niq3yO3gbSOrUU1Z9kScqlrqgbaAMKWnwlhy8MYbo9H2wIXCtrZ9kCuwoI9iilKS
# jjQqWs4085hIBzp3Kc32V4fLkuKGH1KYeTRMm8hP4+V0D+vBVFSu7PFh5t8p93y+
# Y/n7icZkyci6GUMUbVi8YK2z5KDr/ohPWS3a+DP+gAtjafTITgucRazljocMCIIg
# eWtL1Gwq0hvcpQOn+pdNGg8vUIXuVtziS9NmJKVfHEnOXtvte0ep9ziC5yaE8ui0
# 3e0ouY6xuhqPY5+mh9Wve5BX5ju1ewEUuImurs0xjtNBr3bscylS3AkssWz0LBko
# gR/aT9luMyUm/rC/2JLMrDlRTuiWergSd1ffFeLGnQBWtvd0snyFK3ZlVbJelDJP
# 47S4Vj1s45eyKo2DYTGCFdYwghXSAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRAKY6i8YpLEJp76P+FCxyc5YwDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgu19QzRfWaNZNOIRgoJ1f
# zLiPI0ZMlgiQKGGU6ay0LCkwDQYJKoZIhvcNAQEBBQAEggGAhqZlLC7LGw47Ypqj
# SbGNTC3LubOnhjHaodi1GIVDvbVoU+Yo4npz+q9GXppKExNaZsfXNDbQnSEq0a7R
# OW3s3zz7y1ekV9HUFYxxQpz6ZXXkIa3vuno5YHOac0ee/B9KqIGaEOrZc/eHWYg9
# SC0UwlATnnvkGgdk660/RHqIC3kPGV6LPj+ri6mv4mQm6Lnrkg4eJ3RndCYoZnhY
# VHJ7t0abkbkftI189A6w9ujFjy7DJ4zuLLMvyi8lLgdMJ7NE86GISoxSIlm+i7wV
# 6tn2MaxoeTRFyl2u7UTfp/j4RdgoR4EmhYxcOyYmdOtIpTxiZKHOxL8FFfmAv6mN
# /rQKhWohI7vWUYLH832Qbxk3oVUFAsIklh0SyGCT9rTs2GEvV2baOyyJfaVC1gVH
# OwcsopVJp7Q59E8HA4noQx+ILqtQZvPYYjqpOPmKhiK1FmdHvICGGJJA6HMcvAt2
# sadu4hn0fk+ZXFLc2bv5nU61LW1fpxeviBKp0ftDGd3PqVZRoYITTzCCE0sGCisG
# AQQBgjcDAwExghM7MIITNwYJKoZIhvcNAQcCoIITKDCCEyQCAQMxDzANBglghkgB
# ZQMEAgIFADCB8AYLKoZIhvcNAQkQAQSggeAEgd0wgdoCAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQg9lGg0NZRfYS0LyhBdRa+Qlcp4zrTr21aTEFUbl9c
# YH8CFQC88bengcfqPAEoL7dgNIV+5ysxoxgPMjAyMzExMDcxOTAwNTNaoG6kbDBq
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IFNpZ25lciAjNKCCDekwggb1MIIE3aADAgECAhA5TCXhfKBtJ6hl4jvZHSLUMA0G
# CSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1h
# bmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAeFw0y
# MzA1MDMwMDAwMDBaFw0zNDA4MDIyMzU5NTlaMGoxCzAJBgNVBAYTAkdCMRMwEQYD
# VQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNV
# BAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICM0MIICIjANBgkq
# hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEApJMoUkvPJ4d2pCkcmTjA5w7U0RzsaMsB
# ZOSKzXewcWWCvJ/8i7u7lZj7JRGOWogJZhEUWLK6Ilvm9jLxXS3AeqIO4OBWZO2h
# 5YEgciBkQWzHwwj6831d7yGawn7XLMO6EZge/NMgCEKzX79/iFgyqzCz2Ix6lkoZ
# E1ys/Oer6RwWLrCwOJVKz4VQq2cDJaG7OOkPb6lampEoEzW5H/M94STIa7GZ6A3v
# u03lPYxUA5HQ/C3PVTM4egkcB9Ei4GOGp7790oNzEhSbmkwJRr00vOFLUHty4Fv9
# GbsfPGoZe267LUQqvjxMzKyKBJPGV4agczYrgZf6G5t+iIfYUnmJ/m53N9e7UJ/6
# GCVPE/JefKmxIFopq6NCh3fg9EwCSN1YpVOmo6DtGZZlFSnF7TMwJeaWg4Ga9mBm
# kFgHgM1Cdaz7tJHQxd0BQGq2qBDu9o16t551r9OlSxihDJ9XsF4lR5F0zXUS0Zxv
# 5F4Nm+x1Ju7+0/WSL1KF6NpEUSqizADKh2ZDoxsA76K1lp1irScL8htKycOUQjeI
# IISoh67DuiNye/hU7/hrJ7CF9adDhdgrOXTbWncC0aT69c2cPcwfrlHQe2zYHS0R
# QlNxdMLlNaotUhLZJc/w09CRQxLXMn2YbON3Qcj/HyRU726txj5Ve/Fchzpk8WBL
# BU/vuS/sCRMCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7R
# NV8GEgRVMB0GA1UdDgQWBBQDDzHIkSqTvWPz0V1NpDQP0pUBGDAOBgNVHQ8BAf8E
# BAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBKBgNV
# HSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3Nl
# Y3RpZ28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDegNYYzaHR0cDov
# L2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3JsMHQG
# CCsGAQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5zZWN0aWdvLmNv
# bS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUFBzABhhdodHRw
# Oi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEATJtlWPrgec/v
# FcMybd4zket3WOLrvctKPHXefpRtwyLHBJXfZWlhEwz2DJ71iSBewYfHAyTKx6Xw
# Jt/4+DFlDeDrbVFXpoyEUghGHCrC3vLaikXzvvf2LsR+7fjtaL96VkjpYeWaOXe8
# vrqRZIh1/12FFjQn0inL/+0t2v++kwzsbaINzMPxbr0hkRojAFKtl9RieCqEeajX
# Pawhj3DDJHk6l/ENo6NbU9irALpY+zWAT18ocWwZXsKDcpCu4MbY8pn76rSSZXwH
# fDVEHa1YGGti+95sxAqpbNMhRnDcL411TCPCQdB6ljvDS93NkiZ0dlw3oJoknk5f
# TtOPD+UTT1lEZUtDZM9I+GdnuU2/zA2xOjDQoT1IrXpl5Ozf4AHwsypKOazBpPmp
# fTXQMkCgsRkqGCGyyH0FcRpLJzaq4Jgcg3Xnx35LhEPNQ/uQl3YqEqxAwXBbmQpA
# +oBtlGF7yG65yGdnJFxQjQEg3gf3AdT4LhHNnYPl+MolHEQ9J+WwhkcqCxuEdn17
# aE+Nt/cTtO2gLe5zD9kQup2ZLHzXdR+PEMSU5n4k5ZVKiIwn1oVmHfmuZHaR6Ej+
# yFUK7SnDH944psAU+zI9+KmDYjbIw74Ahxyr+kpCHIkD3PVcfHDZXXhO7p9eIOYJ
# anwrCKNI9RX8BE/fzSEceuX1jhrUuUAwggbsMIIE1KADAgECAhAwD2+s3WaYdHyp
# RjaneC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAxMTgyMzU5NTla
# MH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAO
# BgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UE
# AxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEB
# BQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nPoD5T77NawHbW
# GgSlzkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEFkmZvIavVcRB5
# h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQFcN/nqJc1zjx
# FrJw06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1VdEXOzQhe5cB
# lK9/gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpWcdEtlEzIjDzT
# FKKcvSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ffnz82Pg4wZytG
# tzWvlr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV6xlV/9A5VC0m
# 7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7wW33pdmmTIBxK
# lyx3GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFkyYiSWcuoRsDJ
# 8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXDTYLQjdprt9v3
# GEBXc/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70atZ2h3pVBeqeD
# AgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAd
# BgNVHQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/BAQDAgGGMBIG
# A1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAow
# CDAGBgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0
# LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDB2Bggr
# BgEFBQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNlcnRydXN0LmNv
# bS9VU0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcwAYYZaHR0cDov
# L29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEAbVSBpTNdFuG1
# U4GRdd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNviffmfAO9Aw63T
# 4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzVWHi/SHbhRGdt
# j7BDEA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJqO0I3fBZ4OZOT
# lJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytCiiGS/qhvWiVw
# YoFzY16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6NIxi5o8JHHfBd
# 1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9JkWtgqIsJLnfE5
# XkzeSD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw0NP8srMftpmW
# JvQ8eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUgOw9Bqh/poUGd
# /rG4oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20OAkrSQeq/eww
# 2vGnL8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8XwgDCp3rrWiAoa6
# Ke60WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQsMIIEKAIBATCBkTB9MQswCQYDVQQG
# EwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxm
# b3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28g
# UlNBIFRpbWUgU3RhbXBpbmcgQ0ECEDlMJeF8oG0nqGXiO9kdItQwDQYJYIZIAWUD
# BAICBQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0B
# CQUxDxcNMjMxMTA3MTkwMDUzWjA/BgkqhkiG9w0BCQQxMgQwVv/C6QXIgQEUEUfl
# pN7cSPXXI8R6lzbum4LBIV2D5+pHrv2CWJ9R3EwoU5d7/ZfwMIHtBgsqhkiG9w0B
# CRACDDGB3TCB2jCB1zAWBBSuYq91Cgy9R9ZGH3Vo4ryM58pPlDCBvAQUAtZbleKD
# cMFXAJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhl
# IFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEB
# AQUABIICABPfDBVIPC+N6SxvNARKgzfvO4C5xGKAfmGbEP7dpI9jdDhapivlaXLM
# z2IMqyO54D4tAYZjXxobKZr/L/d4tkJeYNE8BvjhKPwThtmiqmKm/WeZ4O+nXaHZ
# Fdi6SpxS4kirnV36TK8MIAT97Abf8l2BDgnkiDH7F+o9lAEEXb/4rm+0iXTiLgTx
# a30qXJKVLbQTKGZ1Lu5SWy07JNWxdmaKVnOwgC2bwLMESbNG+5ZC6XdrHXAkH4nL
# w3VAJwTQHu19jLyxVIar7vJgcXmeJSAM9jbkfbUZs9ZDMbb5LfKhdE3V82VE/eTJ
# JwF0lr0eK6haajLXQdXPHlZ/aPaVi7XfOSDzvZiTHlBp8I96ObcX+6dFtw6cy4WR
# Wqbgfq46hDk+sh+Hn2duGX9WqQcVqkvMkebV6uBcpoBEKnkDSlWGzeHDKH3IoUdZ
# g7EADCilqR1uZTzp3o+EyLTvhDZbnFNhyqe1GnzLUo1FoRQ+74MpjYDokpzAUbAC
# LlVn23aOag9OQNSwdLQPGF6JU2DcYJ6nfTzt74vZStykdewEFfGKhooSDDab8hI+
# /tvXSyFjYkMC8AqwbxfME25FqCvK6MPHeVLgFINyLWBXsv7V+eZYITJoHJaPBc3z
# 31ciaTZ4jkIIPuc9jKg80lFzxofUH8cpEqNpTR5RMnN1ivLYhsTr
# SIG # End signature block
