/******************************************************************************

  Copyright (c) 2013-2019, Intel Corporation
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   3. Neither the name of the Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived from
      this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

******************************************************************************/
/*$FreeBSD$*/
#ifndef _VIRTCHNL_2_H_
#define _VIRTCHNL_2_H_

/* Type of virtual port */
enum virtchnl_vport_type {
	VIRTCHNL_VPORT_TYPE_DEFAULT     = 0,
};

/* Type of queue model */
enum virtchnl_queue_model {
	/* In the single queue model, the same TX descriptor queue is used by
	 * SW to post descriptors to HW and by HW to post completed descriptors
	 * to SW.
	 * Likewise, the same RX descriptor queue is used by HW to post
	 * completions to SW and by SW to post buffers to HW.
	 */
	VIRTCHNL_QUEUE_MODEL_SINGLE     = 0,
	/* In the split queue model, HW uses TX completion queues to post
	 * descriptor/buffer completions to SW, while SW uses TX descriptor
	 * queues to post descriptors to HW.
	 * Likewise, HW posts descriptor completions to the RX descriptor
	 * queue, while SW uses RX buffer queues to post buffers to HW.
	 */
	VIRTCHNL_QUEUE_MODEL_SPLIT      = 1,
};

/* Checksum offload capability flags */
enum virtchnl_checksum_caps {
	VIRTCHNL_CAP_TX_CSUM_L3_IPV4            = BIT(0),
	VIRTCHNL_CAP_TX_CSUM_L4_IPV4_TCP        = BIT(1),
	VIRTCHNL_CAP_TX_CSUM_L4_IPV4_UDP        = BIT(2),
	VIRTCHNL_CAP_TX_CSUM_L4_IPV4_SCTP       = BIT(3),
	VIRTCHNL_CAP_TX_CSUM_L4_IPV6_TCP        = BIT(4),
	VIRTCHNL_CAP_TX_CSUM_L4_IPV6_UDP        = BIT(5),
	VIRTCHNL_CAP_TX_CSUM_L4_IPV6_SCTP       = BIT(6),
	VIRTCHNL_CAP_TX_CSUM_GENERIC            = BIT(7),
	VIRTCHNL_CAP_RX_CSUM_L3_IPV4            = BIT(8),
	VIRTCHNL_CAP_RX_CSUM_L4_IPV4_TCP        = BIT(9),
	VIRTCHNL_CAP_RX_CSUM_L4_IPV4_UDP        = BIT(10),
	VIRTCHNL_CAP_RX_CSUM_L4_IPV4_SCTP       = BIT(11),
	VIRTCHNL_CAP_RX_CSUM_L4_IPV6_TCP        = BIT(12),
	VIRTCHNL_CAP_RX_CSUM_L4_IPV6_UDP        = BIT(13),
	VIRTCHNL_CAP_RX_CSUM_L4_IPV6_SCTP       = BIT(14),
	VIRTCHNL_CAP_RX_CSUM_GENERIC            = BIT(15),
};

/* Segmentation offload capability flags */
enum virtchnl_segmentation_caps {
	VIRTCHNL_CAP_SEG_IPV4_TCP               = BIT(0),
	VIRTCHNL_CAP_SEG_IPV4_UDP               = BIT(1),
	VIRTCHNL_CAP_SEG_IPV4_SCTP              = BIT(2),
	VIRTCHNL_CAP_SEG_IPV6_TCP               = BIT(3),
	VIRTCHNL_CAP_SEG_IPV6_UDP               = BIT(4),
	VIRTCHNL_CAP_SEG_IPV6_SCTP              = BIT(5),
	VIRTCHNL_CAP_SEG_GENERIC                = BIT(6),
};

/* RSS Flow type capability flags */
enum virtchnl_rss_flow_type_caps {
	VIRTCHNL_CAP_RSS_IPV4_TCP               = BIT(0),
	VIRTCHNL_CAP_RSS_IPV4_UDP               = BIT(1),
	VIRTCHNL_CAP_RSS_IPV4_SCTP              = BIT(2),
	VIRTCHNL_CAP_RSS_IPV4_OTHER             = BIT(3),
	VIRTCHNL_CAP_RSS_IPV6_TCP               = BIT(4),
	VIRTCHNL_CAP_RSS_IPV6_UDP               = BIT(5),
	VIRTCHNL_CAP_RSS_IPV6_SCTP              = BIT(6),
	VIRTCHNL_CAP_RSS_IPV6_OTHER             = BIT(7),
	VIRTCHNL_CAP_RSS_IPV4_AH                = BIT(8),
	VIRTCHNL_CAP_RSS_IPV4_ESP               = BIT(9),
	VIRTCHNL_CAP_RSS_IPV4_AH_ESP            = BIT(10),
	VIRTCHNL_CAP_RSS_IPV6_AH                = BIT(11),
	VIRTCHNL_CAP_RSS_IPV6_ESP               = BIT(12),
	VIRTCHNL_CAP_RSS_IPV6_AH_ESP            = BIT(13),
};

/* Header split capability flags */
enum virtchnl_header_split_caps {
	/* for prepended metadata */
	VIRTCHNL_CAP_RX_HSPLIT_AT_L2            = BIT(0),

	/* all VLANs go into header buffer */
	VIRTCHNL_CAP_RX_HSPLIT_AT_L3            = BIT(1),

	VIRTCHNL_CAP_RX_HSPLIT_AT_L4V4          = BIT(2),
	VIRTCHNL_CAP_RX_HSPLIT_AT_L4V6          = BIT(3),
};

/* Receive Side Coalescing offload capability flags */
enum virtchnl_rsc_caps {
	VIRTCHNL_CAP_RSC_IPV4_TCP               = BIT(0),
	VIRTCHNL_CAP_RSC_IPV4_SCTP              = BIT(1),
	VIRTCHNL_CAP_RSC_IPV6_TCP               = BIT(2),
	VIRTCHNL_CAP_RSC_IPV6_SCTP              = BIT(3),
};

/* Other capability flags
 * SPLITQ_QSCHED: Queue based scheduling using split queue model
 * TX_VLAN: VLAN tag insertion
 * RX_VLAN: VLAN tag stripping
 */
enum virtchnl_other_caps {
	VIRTCHNL_CAP_RDMA                       = BIT(0),
	VIRTCHNL_CAP_SRIOV                      = BIT(1),
	VIRTCHNL_CAP_MACFILTER                  = BIT(2),
	VIRTCHNL_CAP_FLOW_DIRECTOR              = BIT(3),
	VIRTCHNL_CAP_SPLITQ_QSCHED              = BIT(4),
	VIRTCHNL_CAP_CRC                        = BIT(5),
	VIRTCHNL_CAP_ADQ                        = BIT(6),
	VIRTCHNL_CAP_WB_ON_ITR                  = BIT(7),
	VIRTCHNL_CAP_PROMISC                    = BIT(8),
	VIRTCHNL_CAP_LINK_SPEED                 = BIT(9),
	VIRTCHNL_CAP_INLINE_IPSEC               = BIT(10),
	VIRTCHNL_CAP_LARGE_NUM_QUEUES           = BIT(11),
	/* require additional info */
	VIRTCHNL_CAP_VLAN                       = BIT(12),
	VIRTCHNL_CAP_PTP                        = BIT(13),
	VIRTCHNL_CAP_ADV_RSS                    = BIT(15),
	VIRTCHNL_CAP_FDIR                       = BIT(16),
	VIRTCHNL_CAP_RX_FLEX_DESC		= BIT(17),
	VIRTCHNL_CAP_PTYPE			= BIT(18),
};

/* VIRTCHNL_OP_GET_CAPS
 * Dataplane driver sends this message to CP to negotiate capabilities and
 * provides a virtchnl_get_capabilities structure with its desired capabilities,
 * max_sriov_vfs and num_allocated_vectors.
 * CP responds with a virtchnl_get_capabilities structure updated
 * with allowed capabilities and the other fields as below.
 * If PF sets max_sriov_vfs as 0, CP will respond with max number of VFs
 * that can be created by this PF. For any other value 'n', CP responds
 * with max_sriov_vfs set to min(n, x) where x is the max number of VFs
 * allowed by CP's policy. max_sriov_vfs is not applicable for VFs.
 * If dataplane driver sets num_allocated_vectors as 0, CP will respond with 1
 * which is default vector associated with the default mailbox. For any other
 * value 'n', CP responds with a value <= n based on the CP's policy of
 * max number of vectors for a PF.
 * CP will respond with the vector ID of mailbox allocated to the PF in
 * mailbox_vector_id and the number of itr index registers in itr_idx_map.
 * It also responds with default number of vports that the dataplane driver
 * should comeup with in default_num_vports and maximum number of vports that
 * can be supported in max_vports
 */
struct virtchnl_get_capabilities {
	/* see enum virtchnl_checksum_caps */
	u32 csum_caps;

	/* see enum virtchnl_segmentation_caps */
	u32 seg_caps;

	/* see enum virtchnl_rss_flow_type_caps */
	u32 rss_caps;

	/* see enum virtchnl_header_split_caps */
	u32 hsplit_caps;

	/* see enum virtchnl_rsc_caps */
	u32 rsc_caps;

	/* see enum virtchnl_other_caps */
	u32 other_caps;

	/* DYN_CTL register offset and vector id for mailbox provided by CP */
	u32 mailbox_dyn_ctl;
	u16 mailbox_vector_id;
	u16 num_allocated_vectors;

	/* The PF sends the maximum VFs it is requesting. The CP responds with
	 * the maximum VFs granted.
	 */
	u16 max_sriov_vfs;

	/* maximum number of vports that can be supported */
	u16 max_vports;
	/* default number of vports driver should allocate on load */
	u16 default_num_vports;

	/* Max header length HW can parse/checksum, in bytes */
	u16 max_tx_hdr_size;

	/* Max number of scatter gather buffers that can be sent per TX
	 * packet without needing to be linearized
	 */
	u8 max_sg_bufs_per_tx_pkt;

	/* see enum virtchnl_itr_idx definition */
	u8 itr_idx_map;
	u8 pad[6];
};

VIRTCHNL_CHECK_STRUCT_LEN(48, virtchnl_get_capabilities);

struct virtchnl_queue_reg_chunk {
	/* see enum virtchnl_queue_type */
	s32 type;
	u16 start_queue_id;
	u16 num_queues;

	/* Queue tail register offset and spacing provided by CP */
	u32 qtail_reg_start;
	u16 qtail_reg_spacing;

	u8 pad[6];
};

VIRTCHNL_CHECK_STRUCT_LEN(20, virtchnl_queue_reg_chunk);

/* structure to specify several chunks of contiguous queues */
struct virtchnl_queue_reg_chunks {
	u16 num_chunks;
	u16 pad;
	struct virtchnl_queue_reg_chunk chunks[1];
};

VIRTCHNL_CHECK_STRUCT_LEN(24, virtchnl_queue_reg_chunks);

/* TX descriptor ID flags */
enum virtchnl_tx_desc_ids {
	VIRTCHNL_TXDID_DATA                             = BIT(0),
	VIRTCHNL_TXDID_CTX                              = BIT(1),
	VIRTCHNL_TXDID_REINJECT_CTX                     = BIT(2),
	VIRTCHNL_TXDID_FLEX_DATA                        = BIT(3),
	VIRTCHNL_TXDID_FLEX_CTX                         = BIT(4),
	VIRTCHNL_TXDID_FLEX_TSO_CTX                     = BIT(5),
	VIRTCHNL_TXDID_FLEX_TSYN_L2TAG1                 = BIT(6),
	VIRTCHNL_TXDID_FLEX_L2TAG1_L2TAG2               = BIT(7),
	VIRTCHNL_TXDID_FLEX_TSO_L2TAG2_PARSTAG_CTX      = BIT(8),
	VIRTCHNL_TXDID_FLEX_HOSTSPLIT_SA_TSO_CTX        = BIT(9),
	VIRTCHNL_TXDID_FLEX_HOSTSPLIT_SA_CTX            = BIT(10),
	VIRTCHNL_TXDID_FLEX_L2TAG2_CTX                  = BIT(11),
	VIRTCHNL_TXDID_FLEX_FLOW_SCHED                  = BIT(12),
	VIRTCHNL_TXDID_FLEX_HOSTSPLIT_TSO_CTX           = BIT(13),
	VIRTCHNL_TXDID_FLEX_HOSTSPLIT_CTX               = BIT(14),
	VIRTCHNL_TXDID_DESC_DONE                        = BIT(15),
};

#define VIRTCHNL_ETH_LENGTH_OF_ADDRESS  6

/* VIRTCHNL_OP_CREATE_VPORT
 * PF sends this message to CP to create a vport by filling in the first 8
 * fields of virtchnl_create_vport structure (vport type, tx, rx queue models
 * and desired number of queues and vectors). CP responds with the updated
 * virtchnl_create_vport structure containing the number of assigned queues,
 * vectors, vport id, max mtu, default mac addr followed by chunks which in turn
 * will have an array of num_chunks entries of virtchnl_queue_chunk structures.
 */
struct virtchnl_create_vport {
	/* see enum virtchnl_vport_type vport */
	s16 vport_type;

	/* see enum virtchnl_queue_model */
	s16 txq_model;

	/* see enum virtchnl_queue_model */
	s16 rxq_model;
	u16 num_tx_q;
	/* valid only if txq_model is split Q */
	u16 num_tx_complq;
	u16 num_rx_q;
	/* valid only if rxq_model is split Q */
	u16 num_rx_bufq;
	u16 vport_id;
	u16 max_mtu;
	u8 default_mac_addr[VIRTCHNL_ETH_LENGTH_OF_ADDRESS];
	/* see enum virtchnl_rx_desc_ids */
	s64 rx_desc_ids;
	/* see enum virtchnl_tx_desc_ids */
	s16 tx_desc_ids;
	u16 pad;

	/* see enum virtchnl_rss_algorithm */
	s32 rss_algorithm;
	u16 rss_key_size;
	u16 rss_lut_size;
	u16 qset_handle;
	u16 pad2;

	/* see enum virtchnl_header_split_caps */
	s32 rx_split_pos;

	u32 ingress_pasid;
	u32 ingress_hdr_pasid;
	u32 ingress_buf_pasid;
	u32 egress_pasid;
	u32 egress_hdr_passid;
	u32 egress_buf_pasid;
	u32 pad3;
	struct virtchnl_queue_reg_chunks chunks;
};

VIRTCHNL_CHECK_STRUCT_LEN(104, virtchnl_create_vport);

/* VIRTCHNL_OP_DESTROY_VPORT
 * VIRTCHNL_OP_ENABLE_VPORT
 * VIRTCHNL_OP_DISABLE_VPORT
 * PF sends this message to CP to destroy, enable or disable a vport by filling
 * in the vport_id in virtchnl_vport structure.
 * CP responds with the status of the requested operation.
 */
struct virtchnl_vport {
	u16 vport_id;
};

VIRTCHNL_CHECK_STRUCT_LEN(2, virtchnl_vport);

/* TX Queue Scheduling Modes - Queue mode is the legacy mode i.e. inorder
 * completions where descriptors and buffers are completed at the same time.
 * Flow scheduling mode allows for out of order packet processing where
 * descriptors are cleaned in order, but buffers can be completed out of order.
 */
enum virtchnl_txq_sched_mode {
	VIRTCHNL_TXQ_SCHED_MODE_QUEUE           = 0,
	VIRTCHNL_TXQ_SCHED_MODE_FLOW            = 1,
};

/* TX Queue feature flags */
enum virtchnl_txq_flags {
	/* Enable rule miss completion type; packet completion for a packet
	 * sent on exception path; only relevant in flow scheduling mode
	 */
	VIRTCHNL_TXQ_ENABLE_MISS_COMPL          = BIT(0),
};

/* Tx queue config info */
struct virtchnl_txq_info_v2 {
	u64 dma_ring_addr;

	/* see enum virtchnl_queue_type */
	s32 type;

	u16 queue_id;

	/* see enum virtchnl_queue_model */
	s16 model;

	/* see enum virtchnl_txq_sched_mode */
	s16 sched_mode;

	/* see enum virtchnl_txq_flags */
	s16 qflags;
	u16 ring_len;

	/* valid only if queue model is split and type is tx */
	u16 tx_compl_queue_id;
	/* valid only if queue type is CONFIG_TX */
	u16 peer_rx_queue_id;

	u32 pad;
};

VIRTCHNL_CHECK_STRUCT_LEN(32, virtchnl_txq_info_v2);

/* VIRTCHNL_OP_CONFIG_TX_QUEUES
 * PF sends this message to set up parameters for one or more TX queues.
 * This message contains an array of num_qinfo instances of virtchnl_txq_info_v2
 * structures. CP configures requested queues and returns a status code. If
 * num_qinfo specified is greater than the number of queues associated with the
 * vport, an error is returned and no queues are configured.
 */
struct virtchnl_config_tx_queues {
	u16 vport_id;
	u16 num_qinfo;
	u32 rsvd;
	struct virtchnl_txq_info_v2 qinfo[1];
};

VIRTCHNL_CHECK_STRUCT_LEN(40, virtchnl_config_tx_queues);

/* RX Queue Feature flags */
enum virtchnl_rxq_flags {
	VIRTCHNL_RXQ_RSC                        = BIT(0),
	VIRTCHNL_RXQ_HDR_SPLIT                  = BIT(1),

	/* When set, packet descriptors are flushed by HW immediately after
	 * processing each packet.
	 */
	VIRTCHNL_RXQ_IMMEDIATE_WRITE_BACK       = BIT(2),
	VIRTCHNL_RX_DESC_SIZE_16BYTE            = BIT(3),
	VIRTCHNL_RX_DESC_SIZE_32BYTE            = BIT(4)
};

/* Rx queue config info */
struct virtchnl_rxq_info_v2 {
	/* see enum virtchnl_rx_desc_ids */
	s64 desc_ids;
	u64 dma_ring_addr;

	/* see enum virtchnl_queue_type */
	s32 type;
	u16 queue_id;

	/* see enum virtchnl_queue_model */
	s16 model;

	u32 data_buffer_size;
	u32 max_pkt_size;

	u16 ring_len;
	u16 hdr_buffer_size;
	u8 buffer_notif_stride;
	u8 pad2[3];

	/* Applicable only for RX buffer queues */
	u64 dma_head_wb_addr;

	/* Applicable only for RX completion queues */
	/* see enum virtchnl_rxq_flags */
	s16 qflags;

	u16 rsc_low_watermark;

	/* valid only in split queue model */
	u16 rx_bufq1_id;
	/* valid only in split queue model */
	u16 rx_bufq2_id;

	u32 pad3[2];
};

VIRTCHNL_CHECK_STRUCT_LEN(64, virtchnl_rxq_info_v2);

/* VIRTCHNL_OP_CONFIG_RX_QUEUES
 * PF sends this message to set up parameters for one or more RX queues.
 * This message contains an array of num_qinfo instances of virtchnl_rxq_info_v2
 * structures. CP configures requested queues and returns a status code.
 * If the number of queues specified is greater than the number of queues
 * associated with the vport, an error is returned and no queues are configured.
 */
struct virtchnl_config_rx_queues {
	u16 vport_id;
	u16 num_qinfo;
	u32 pad;
	struct virtchnl_rxq_info_v2 qinfo[1];
};

VIRTCHNL_CHECK_STRUCT_LEN(72, virtchnl_config_rx_queues);

/* VIRTCHNL_OP_ADD_QUEUES
 * PF sends this message to request additional TX/RX queues beyond the ones
 * that were assigned via CREATE_VPORT request. virtchnl_add_queues structure is
 * used to specify the number of each type of queues.
 * CP responds with the same structure with the actual number of queues assigned
 * followed by num_chunks of virtchnl_queue_chunk structures.
 */
struct virtchnl_add_queues {
	u16 vport_id;
	u16 num_tx_q;
	u16 num_tx_complq;
	u16 num_rx_q;
	u16 num_rx_bufq;
	u16 pad[3];
	struct virtchnl_queue_reg_chunks chunks;
};

VIRTCHNL_CHECK_STRUCT_LEN(40, virtchnl_add_queues);

/* Structure to specify a chunk of contiguous interrupt vectors */
struct virtchnl_vector_chunk {
	u16 start_vector_id;
	u16 start_evv_id;
	u16 num_vectors;
	u16 pad1;

	/* Register offsets and spacing provided by CP.
	 * dynctl registers are used for enabling/disabling/re-enabling
	 * interrupts and updating interrupt rates in the hotpath. Any changes
	 * to interrupt rates in the dynctl registers will be reflected in the
	 * itrn registers.
	 * itrn registers are used to update interrupt rates for specific
	 * interrupt indices without modifying the state of the interrupt.
	 */
	u32 dynctl_reg_start;
	u32 dynctl_reg_spacing;

	u32 itrn_reg_start;
	u32 itrn_reg_spacing;
	u32 pad2;
};

VIRTCHNL_CHECK_STRUCT_LEN(28, virtchnl_vector_chunk);

/* Structure to specify several chunks of contiguous interrupt vectors */
struct virtchnl_vector_chunks {
	u16 num_vchunks;
	u16 pad;
	struct virtchnl_vector_chunk vchunks[1];
};

VIRTCHNL_CHECK_STRUCT_LEN(32, virtchnl_vector_chunks);

/* VIRTCHNL_OP_ALLOC_VECTORS
 * PF sends this message to request additional interrupt vectors beyond the
 * ones that were assigned via GET_CAPS request. virtchnl_alloc_vectors
 * structure is used to specify the number of vectors requested. CP responds
 * with the same structure with the actual number of vectors assigned followed
 * by virtchnl_vector_chunks structure identifying the vector ids.
 */
struct virtchnl_alloc_vectors {
	u16 num_vectors;
	u16 pad;
	struct virtchnl_vector_chunks vchunks;
};

VIRTCHNL_CHECK_STRUCT_LEN(36, virtchnl_alloc_vectors);

/* VIRTCHNL_OP_DEALLOC_VECTORS
 * PF sends this message to release the vectors.
 * PF sends virtchnl_vector_chunks struct to specify the vectors it is giving
 * away. CP performs requested action and returns status.
 */

/* VIRTCHNL_OP_GET_RSS_LUT
 * VIRTCHNL_OP_SET_RSS_LUT
 * PF sends this message to get or set RSS lookup table. Only supported if
 * both PF and CP drivers set the VIRTCHNL_CAP_RSS bit during configuration
 * negotiation. Uses the virtchnl_rss_lut_v2 structure
 */
struct virtchnl_rss_lut_v2 {
	u16 vport_id;
	u16 lut_entries;
	u16 lut[1]; /* RSS lookup table */
};

VIRTCHNL_CHECK_STRUCT_LEN(6, virtchnl_rss_lut_v2);

/* VIRTCHNL_OP_GET_RSS_KEY
 * PF sends this message to get RSS key. Only supported if both PF and CP
 * drivers set the VIRTCHNL_CAP_RSS bit during configuration negotiation. Uses
 * the virtchnl_rss_key structure
 */

/* VIRTCHNL_OP_GET_RSS_HASH
 * VIRTCHNL_OP_SET_RSS_HASH
 * PF sends these messages to get and set the hash filter enable bits for RSS.
 * By default, the CP sets these to all possible traffic types that the
 * hardware supports. The PF can query this value if it wants to change the
 * traffic types that are hashed by the hardware.
 * Only supported if both PF and CP drivers set the VIRTCHNL_CAP_RSS bit
 * during configuration negotiation.
 */
struct virtchnl_rss_hash {
	u64 hash;
	u16 vport_id;
	u16 pad[3];
};

VIRTCHNL_CHECK_STRUCT_LEN(16, virtchnl_rss_hash);

/* VIRTCHNL_OP_CREATE_SRIOV_VFS
 * VIRTCHNL_OP_DESTROY_SRIOV_VFS
 * This message is used to let the CP know how many SRIOV VFs need to be
 * created. The actual allocation of resources for the VFs in terms of VSI,
 * Queues and Interrupts is done by CP. When this call completes, the APF driver
 * calls pci_enable_sriov to let the OS instantiate the SRIOV PCIE devices.
 */
struct virtchnl_sriov_vfs_info {
	u16 num_vfs;
};

VIRTCHNL_CHECK_STRUCT_LEN(2, virtchnl_sriov_vfs_info);

/* VIRTCHNL_OP_GET_PTYPES
 * This opcode can be used only if VIRTCHNL_CAP_PTYPE capability is set in
 * virtchnl_get_capabilities.
 * virtchnl_ptype_lookup contains the packet types and associated index to it.
 * These indices are for reference as the right index which is associated to the
 * packet type depends on package loaded. PF/VF sends VIRTCHNL_OP_GET_PTYPES
 * to CP to get the ptypes supported by the hardware and CP responds with ptype
 * indices that are generated by the package.
 * Ex: from virtchnl_ptype_lookup,
 * packet type 'L2, NONE, NOF, NONE, NONE, NOF, TS, PAY2' has an index 2, but
 * it might be someother value based on the package loaded. Packet type in the
 * lookup table are static but the indices are dynamic.
 */
struct virtchnl_get_ptypes {
	u16 ptypes[VIRTCHNL_RX_MAX_PTYPE];
};

VIRTCHNL_CHECK_STRUCT_LEN(2048, virtchnl_get_ptypes);
#endif /* _VIRTCHNL_2_H_ */
