// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2018 - 2021 Intel Corporation */
#include "ice_idc.h"
#include "main.h"
#include "ws.h"
#include "icrdma_hw.h"

#define IRDMA_MAX_DEVICES 32
static void *irdma_devices[IRDMA_MAX_DEVICES];

static inline int irdma_find_pci_dev_idx(struct pci_dev *pdev)
{
	int i;

	for (i = 0; i < IRDMA_MAX_DEVICES; ++i) {
		if (irdma_devices[i] == pdev)
			return i;
	}

	return -1;
}

static inline int irdma_add_pci_dev_and_ret_idx(struct pci_dev *pdev)
{
	int i;

	for (i = 0; i < IRDMA_MAX_DEVICES; ++i) {
		if (!irdma_devices[i]) {
			irdma_devices[i] = pdev;
			return i;
		}
	}

	return -1;
}

int irdma_set_ibdev_name(struct irdma_pci_f *rf, struct pci_dev *pdev)
{
	int idx;

	idx = irdma_find_pci_dev_idx(pdev);
	if (idx < 0) {
		idx = irdma_add_pci_dev_and_ret_idx(pdev);
		if (idx < 0)
			return -ENOMEM;
	}
	sprintf(rf->ib_dev_name, "irdma%d", idx);

	return 0;
}

/**
 * irdma_lan_register_qset - Register qset with LAN driver
 * @vsi: vsi structure
 * @tc_node: Traffic class node
 */
static enum irdma_status_code irdma_lan_register_qset(struct irdma_sc_vsi *vsi,
						      struct irdma_ws_node *tc_node)
{
	struct irdma_device *iwdev = vsi->back_vsi;
	struct ice_peer_obj *peer_info = iwdev->rf->priv_peer_info.peer_info;
	struct ice_res rdma_qset_res = {};
	int ret;

	rdma_qset_res.cnt_req = 1;
	rdma_qset_res.res_type = ICE_RDMA_QSETS_TXSCHED;
	rdma_qset_res.res[0].res.qsets.qs_handle = tc_node->qs_handle;
	rdma_qset_res.res[0].res.qsets.tc = tc_node->traffic_class;
	rdma_qset_res.res[0].res.qsets.vsi_id = vsi->vsi_idx;
	ret = peer_info->ops->alloc_res(peer_info, &rdma_qset_res, 0);
	if (ret) {
		irdma_dbg(vsi->dev,
			  "WS: LAN alloc_res for rdma qset failed.\n");
		return IRDMA_ERR_REG_QSET;
	}

	tc_node->l2_sched_node_id = rdma_qset_res.res[0].res.qsets.teid;
	vsi->qos[tc_node->user_pri].l2_sched_node_id =
		rdma_qset_res.res[0].res.qsets.teid;

	return 0;
}

/**
 * irdma_lan_unregister_qset - Unregister qset with LAN driver
 * @vsi: vsi structure
 * @tc_node: Traffic class node
 */
static void irdma_lan_unregister_qset(struct irdma_sc_vsi *vsi,
				      struct irdma_ws_node *tc_node)
{
	struct irdma_device *iwdev = vsi->back_vsi;
	struct ice_peer_obj *peer_info = iwdev->rf->priv_peer_info.peer_info;
	struct ice_res rdma_qset_res = {};

	rdma_qset_res.res_allocated = 1;
	rdma_qset_res.res_type = ICE_RDMA_QSETS_TXSCHED;
	rdma_qset_res.res[0].res.qsets.vsi_id = vsi->vsi_idx;
	rdma_qset_res.res[0].res.qsets.teid = tc_node->l2_sched_node_id;
	rdma_qset_res.res[0].res.qsets.qs_handle = tc_node->qs_handle;

	if (peer_info->ops->free_res(peer_info, &rdma_qset_res))
		irdma_dbg(vsi->dev,
		          "WS: LAN free_res for rdma qset failed.\n");
}

/**
 * irdma_prep_tc_change - Prepare for TC changes
 * @peer_info: parent lan device information structure with data/ops
 */
static void irdma_prep_tc_change(struct ice_peer_obj *peer_info)
{
	struct irdma_device *iwdev;

	iwdev = irdma_get_device(peer_info->netdev);
	if (!iwdev)
		return;

	if (iwdev->vsi.tc_change_pending)
		goto done;

	iwdev->vsi.tc_change_pending = true;
	irdma_sc_suspend_resume_qps(&iwdev->vsi, IRDMA_OP_SUSPEND);

	/* Wait for all qp's to suspend */
	wait_event_timeout(iwdev->suspend_wq,
			   !atomic_read(&iwdev->vsi.qp_suspend_reqs),
			   IRDMA_EVENT_TIMEOUT);
	irdma_ws_reset(&iwdev->vsi);
done:
	irdma_put_device(iwdev);
}

static void irdma_log_invalid_mtu(u16 mtu, struct irdma_sc_dev *dev)
{
	if (mtu < IRDMA_MIN_MTU_IPV4)
		dev_warn(idev_to_dev(dev),
		         "MTU setting [%d] too low for RDMA traffic. Minimum MTU is 576 for IPv4\n",
		         mtu);
	else if (mtu < IRDMA_MIN_MTU_IPV6)
		dev_warn(idev_to_dev(dev),
		         "MTU setting [%d] too low for RDMA traffic. Minimum MTU is 1280 for IPv6\\n",
		         mtu);
}

static void irdma_get_qos_info(struct irdma_l2params *l2params, struct ice_qos_params *qos_info)
{
	int i;

	l2params->num_tc = qos_info->num_tc;
	l2params->num_apps = qos_info->num_apps;
	l2params->vsi_prio_type = qos_info->vsi_priority_type;
	l2params->vsi_rel_bw = qos_info->vsi_relative_bw;
	for (i = 0; i < l2params->num_tc; i++) {
		l2params->tc_info[i].egress_virt_up =
			qos_info->tc_info[i].egress_virt_up;
		l2params->tc_info[i].ingress_virt_up =
			qos_info->tc_info[i].ingress_virt_up;
		l2params->tc_info[i].prio_type = qos_info->tc_info[i].prio_type;
		l2params->tc_info[i].rel_bw = qos_info->tc_info[i].rel_bw;
		l2params->tc_info[i].tc_ctx = qos_info->tc_info[i].tc_ctx;
	}
	for (i = 0; i < IRDMA_MAX_USER_PRIORITY; i++)
		l2params->up2tc[i] = qos_info->up2tc[i];
	if (qos_info->pfc_mode == IDC_QOS_MODE_DSCP) {
		l2params->dscp_mode = true;
		memcpy(l2params->dscp_map, qos_info->dscp_map, sizeof(l2params->dscp_map));
	}
}

/**
 * irdma_event_handler - Called by LAN driver to notify events
 * @peer_info: parent lan device information structure with data/ops
 * @event: event from LAN driver
 */
static void irdma_event_handler(struct ice_peer_obj *peer_info,
				struct ice_event *event)
{
	struct irdma_l2params l2params = {};
	struct irdma_device *iwdev;

	iwdev = irdma_get_device(peer_info->netdev);
	if (!iwdev)
		return;

	if (*event->type & BIT(ICE_EVENT_LINK_CHANGE)) {
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: LINK_CHANGE event\n");
	} else if (*event->type & BIT(ICE_EVENT_MTU_CHANGE)) {
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: new MTU = %d\n",
			  event->info.mtu);
		if (iwdev->vsi.mtu != event->info.mtu) {
			l2params.mtu = event->info.mtu;
			l2params.mtu_changed = true;
			irdma_log_invalid_mtu(l2params.mtu, &iwdev->rf->sc_dev);
			irdma_change_l2params(&iwdev->vsi, &l2params);
		}
	} else if (*event->type & BIT(ICE_EVENT_TC_CHANGE)) {
		if (!iwdev->vsi.tc_change_pending)
			goto done;

		l2params.tc_changed = true;
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: TC Change\n");
		iwdev->dcb = event->info.port_qos.num_tc > 1;

		irdma_get_qos_info(&l2params, &event->info.port_qos);

		irdma_check_fc_for_tc_update(&iwdev->vsi, &l2params);
		irdma_change_l2params(&iwdev->vsi, &l2params);
	} else if (*event->type & BIT(ICE_EVENT_CRIT_ERR)) {
		ibdev_warn(&iwdev->ibdev, "ICE OICR event notification: oicr = 0x%08x\n",
			   event->info.reg);
		if (event->info.reg & IRDMAPFINT_OICR_PE_CRITERR_M) {
			u32 pe_criterr;

			pe_criterr = readl(iwdev->rf->sc_dev.hw_regs[IRDMA_GLPE_CRITERR]);
#define IRDMA_Q1_RESOURCE_ERR 0x0001024d
			if (pe_criterr != IRDMA_Q1_RESOURCE_ERR) {
				ibdev_err(&iwdev->ibdev, "critical PE Error, GLPE_CRITERR=0x%08x\n",
					  pe_criterr);
				iwdev->rf->reset = true;
			} else {
				ibdev_warn(&iwdev->ibdev, "Q1 Resource Check\n");
			}
		}
		if (event->info.reg & IRDMAPFINT_OICR_HMC_ERR_M) {
			ibdev_err(&iwdev->ibdev, "HMC Error\n");
			iwdev->rf->reset = true;
		}
		if (event->info.reg & IRDMAPFINT_OICR_PE_PUSH_M) {
			ibdev_err(&iwdev->ibdev, "PE Push Error\n");
			iwdev->rf->reset = true;
		}
		if (iwdev->rf->reset)
			iwdev->rf->gen_ops.request_reset(iwdev->rf);
	}
done:
	irdma_put_device(iwdev);
}

/**
 * irdma_remove - GEN_2 driver remove
 * @pdev: platform device
 */
int irdma_remove(struct platform_device *pdev)
{
	struct ice_peer_obj_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct ice_peer_obj *peer_info = pdata->peer_obj;
	struct irdma_handler *hdl;

	hdl = irdma_find_handler(peer_info->pdev);
	if (!hdl)
		return 0;

	peer_info->ops->peer_unregister(peer_info);

	irdma_deinit_rf(&hdl->rf);

	pr_debug("INIT: Gen2 device remove success peer_info=%p\n", peer_info);

	return 0;
}

/**
 * irdma_open - client interface operation open for RDMA device
 * @peer_info: parent lan device information structure with data/ops
 *
 * Called by the lan driver during the processing of client
 * register.
 */
static int irdma_open(struct ice_peer_obj *peer_info)
{
	struct irdma_handler *hdl;
	struct irdma_device *iwdev;
	struct irdma_sc_dev *dev;
	struct ice_event events = {};
	struct irdma_pci_f *rf;
	struct irdma_priv_peer_info *priv_peer_info;
	struct irdma_l2params l2params = {};
	int ret;

	hdl = irdma_find_handler(peer_info->pdev);
	if (!hdl)
		return -ENODEV;

	rf = &hdl->rf;
	if (rf->init_state != CEQ0_CREATED)
		return -EINVAL;

	iwdev = ib_alloc_device(irdma_device, ibdev);
	if (!iwdev)
		return -ENOMEM;

	priv_peer_info = &rf->priv_peer_info;
	priv_peer_info->pf_vsi_num = peer_info->pf_vsi_num;
	dev = &hdl->rf.sc_dev;

	iwdev->hdl = hdl;
	iwdev->rf = rf;
	iwdev->roce_cwnd = IRDMA_ROCE_CWND_DEFAULT;
	iwdev->roce_ackcreds = IRDMA_ROCE_ACKCREDS_DEFAULT;
	iwdev->rcv_wnd = IRDMA_CM_DEFAULT_RCV_WND_SCALED;
	iwdev->rcv_wscale = IRDMA_CM_DEFAULT_RCV_WND_SCALE;
	iwdev->netdev = peer_info->netdev;
	INIT_LIST_HEAD(&iwdev->ah_list);
	mutex_init(&iwdev->ah_list_lock);
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
	iwdev->iwarp_ecn_en = true;
	iwdev->iwarp_rtomin = 5;
	iwdev->up_up_map = IRDMA_DEFAULT_UP_UP_MAP;
#endif
	if (rf->protocol_used == IRDMA_ROCE_PROTOCOL_ONLY) {
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
		iwdev->roce_rtomin = IRDMA_ROCE_RTOMIN_DEFAULT;
#endif
		iwdev->roce_dcqcn_en = rf->dcqcn_ena;
		iwdev->roce_mode = true;
	}
	l2params.mtu = peer_info->netdev->mtu;

	irdma_get_qos_info(&l2params, &peer_info->initial_qos_info);

	iwdev->vsi_num = peer_info->pf_vsi_num;
	peer_info->ops->update_vsi_filter(peer_info, ICE_RDMA_FILTER_BOTH, true);

	if (irdma_rt_init_hw(rf, iwdev, &l2params)) {
		ib_dealloc_device(&iwdev->ibdev);
		return -EIO;
	}

	ret = irdma_ib_register_device(iwdev);
	if (ret) {
		irdma_rt_deinit_hw(iwdev);
		ib_dealloc_device(&iwdev->ibdev);
		return ret;
	}

	events.reporter = peer_info;
	set_bit(ICE_EVENT_LINK_CHANGE, events.type);
	set_bit(ICE_EVENT_MTU_CHANGE, events.type);
	set_bit(ICE_EVENT_TC_CHANGE, events.type);
	set_bit(ICE_EVENT_CRIT_ERR, events.type);

	peer_info->ops->reg_for_notification(peer_info, &events);
	irdma_dbg(dev, "INIT: Gen2 VSI[%d] open success peer_info=%p\n",
		  peer_info->pf_vsi_num, peer_info);

	return 0;
}

void irdma_cleanup_dead_qps(struct irdma_sc_vsi *vsi)
{
	struct irdma_sc_qp *qp = NULL;
	struct irdma_qp *iwqp;
	struct irdma_pci_f *rf;
	u8 i;

	for (i = 0; i < IRDMA_MAX_USER_PRIORITY; i++) {
		qp = irdma_get_qp_from_list(&vsi->qos[i].qplist, qp);
		while (qp) {
			if (qp->qp_uk.qp_type == IRDMA_QP_TYPE_UDA) {
				qp = irdma_get_qp_from_list(&vsi->qos[i].qplist, qp);
				continue;
			}

			iwqp = qp->qp_uk.back_qp;
			rf = iwqp->iwdev->rf;
			dma_free_coherent(rf->hw.device,
					  iwqp->q2_ctx_mem.size,
					  iwqp->q2_ctx_mem.va,
					  iwqp->q2_ctx_mem.pa);
			dma_free_coherent(rf->hw.device,
					  iwqp->kqp.dma_mem.size,
					  iwqp->kqp.dma_mem.va,
					  iwqp->kqp.dma_mem.pa);
			kfree(iwqp->kqp.sq_wrid_mem);
			kfree(iwqp->kqp.rq_wrid_mem);
			qp = irdma_get_qp_from_list(&vsi->qos[i].qplist, qp);
			kfree(iwqp);
		}
	}
}

/**
 * irdma_close - client interface operation close for iwarp/uda device
 * @peer_info: parent lan device information structure with data/ops
 * @reason: reason for closing
 *
 * Called by the lan driver during the processing of client unregister
 * Destroy and clean up the driver resources
 */
static void irdma_close(struct ice_peer_obj *peer_info, enum ice_close_reason reason)
{
	struct irdma_handler *hdl;
	struct irdma_device *iwdev;
	struct irdma_pci_f *rf;

	hdl = irdma_find_handler(peer_info->pdev);
	if (!hdl)
		return;

	rf = &hdl->rf;
	iwdev = list_first_entry_or_null(&rf->vsi_dev_list, struct irdma_device,
					 list);
	if (!iwdev)
		return;
	if (iwdev->up_map_en) {
		struct irdma_up_info up_map_info = {};

		*((u64 *)up_map_info.map) = IRDMA_DEFAULT_UP_UP_MAP;
		up_map_info.use_cnp_up_override = false;
		up_map_info.cnp_up_override = 0;
		up_map_info.hmc_fcn_idx = iwdev->rf->sc_dev.hmc_fn_id;
		irdma_cqp_up_map_cmd(&iwdev->rf->sc_dev, IRDMA_OP_SET_UP_MAP,
				     &up_map_info);
	}
	if (reason == ICE_REASON_GLOBR_REQ || reason == ICE_REASON_CORER_REQ ||
	    reason == ICE_REASON_PFR_REQ || rf->reset) {
		iwdev->reset = true;
		rf->reset = true;
	}

	irdma_ib_unregister_device(iwdev);
#ifndef IB_DEALLOC_DRIVER_SUPPORT
	irdma_cleanup_dead_qps(&iwdev->vsi);
	irdma_rt_deinit_hw(iwdev);
	ib_dealloc_device(&iwdev->ibdev);
#endif
	if (!rf->reset)
		peer_info->ops->update_vsi_filter(peer_info, ICE_RDMA_FILTER_BOTH, false);
	if (rf->reset && try_module_get(THIS_MODULE)) {
		struct reset_work *work;

		work = kzalloc(sizeof(*work), GFP_ATOMIC);
		if (!work) {
			module_put(THIS_MODULE);
			dev_warn(idev_to_dev(&rf->sc_dev),
				 "IRDMA reset recovery failed. Please rmmod/modprobe irdma.\n");
			return;
		}
		work->rst_to = rf->rst_to;
		work->pdev = rf->pdev;
		INIT_DELAYED_WORK(&work->work, irdma_reset_task);
		irdma_remove(rf->pdev);
		schedule_delayed_work(&work->work, work->rst_to * HZ);
	}

	pr_debug("INIT: Gen2 VSI[%d] close complete peer_info=%p\n",
		 peer_info->pf_vsi_num, peer_info);
}

static enum irdma_status_code irdma_vc_receive(struct ice_peer_obj *ldev,
					       u32 vf_id, u8 *msg, u16 len)
{
	struct irdma_handler *hdl;
	struct irdma_sc_dev *dev;

	if (!len || !msg)
		return IRDMA_ERR_PARAM;

	hdl = irdma_find_handler(ldev->pdev);
	if (!hdl)
		return IRDMA_ERR_PARAM;

	dev = &hdl->rf.sc_dev;
	if (dev->vchnl_if->vchnl_recv) {
		enum irdma_status_code ret;

		ret = dev->vchnl_if->vchnl_recv(dev, (u16)vf_id, msg, len);
		if (!dev->privileged) {
			atomic_dec(&dev->vchnl_msgs);
			wake_up(&dev->wq_vchnl);
		}
		return ret;
	}

	return IRDMA_ERR_DEVICE_NOT_SUPPORTED;
}

static enum irdma_status_code irdma_vc_send(struct irdma_sc_dev *dev,
					    u16 vf_id, u8 *msg, u16 len)
{
	struct ice_peer_obj *ldev =
		((struct irdma_pci_f *)dev->back_dev)->priv_peer_info.peer_info;

	if (ldev->ops->vc_send) {
		ldev->ops->vc_send(ldev, vf_id, msg, len);

		return 0;
	}

	return IRDMA_ERR_BAD_PTR;
}

static struct irdma_vchnl_if irdma_vchnl_if = {
	.vchnl_recv = irdma_vchnl_recv_pf,
	.vchnl_send = irdma_vc_send,
};

static const struct ice_peer_ops irdma_peer_ops = {
	.close = irdma_close,
	.event_handler = irdma_event_handler,
	.open = irdma_open,
	.prep_tc_change = irdma_prep_tc_change,
	.vc_receive = irdma_vc_receive,
};

static struct ice_peer_drv irdma_peer_drv = {
	.driver_id = ICE_PEER_RDMA_DRIVER,
	.name = KBUILD_MODNAME,
	.ver.major = ICE_PEER_MAJOR_VER,
	.ver.minor = ICE_PEER_MINOR_VER,
};

/**
 * icrdma_request_reset - Request a reset
 * @rf: RDMA PCI function
 */
static void icrdma_request_reset(struct irdma_pci_f *rf)
{
	struct ice_peer_obj *peer_info = rf->priv_peer_info.peer_info;

	dev_warn(idev_to_dev(&rf->sc_dev), "Requesting a reset\n");
	peer_info->ops->request_reset(peer_info, ICE_PEER_PFR);
}

/**
 * irdma_probe - GEN_2 driver probe
 * @pdev: platform device
 *
 * Create device resources, set up queues, pble and hmc objects.
 * Return 0 if successful, otherwise return error
 */
int irdma_probe(struct platform_device *pdev)
{
	struct ice_peer_obj_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct ice_peer_obj *peer_info = pdata->peer_obj;
	struct irdma_handler *hdl;
	struct irdma_pci_f *rf;
	struct irdma_sc_dev *dev;
	struct irdma_priv_peer_info *priv_peer_info;
	int err;

	pr_info("probe: peer_info=%p, peer_info->dev.pdev.bus->number=%d, peer_info->netdev=%p\n",
		peer_info, peer_info->pdev->bus->number, peer_info->netdev);
	if (peer_info->ver.major != ICE_PEER_MAJOR_VER ||
	    peer_info->ver.minor != ICE_PEER_MINOR_VER) {
		pr_err("version mismatch:\n");
		pr_err("expected major ver %d, caller specified major ver %d\n",
		       ICE_PEER_MAJOR_VER, peer_info->ver.major);
		pr_err("expected minor ver %d, caller specified minor ver %d\n",
		       ICE_PEER_MINOR_VER, peer_info->ver.minor);
		return -EINVAL;
	}

	hdl = irdma_find_handler(peer_info->pdev);
	if (hdl)
		return -EBUSY;

	hdl = kzalloc(sizeof(*hdl), GFP_KERNEL);
	if (!hdl)
		return -ENOMEM;

	rf = &hdl->rf;
	err = irdma_set_ibdev_name(rf, peer_info->pdev);
	if (err) {
		kfree(hdl);
		return err;
	}
	priv_peer_info = &rf->priv_peer_info;
	rf->pdev = pdev;
	rf->hdl = hdl;
	dev = &rf->sc_dev;
	dev->back_dev = rf;
	rf->gen_ops.request_reset = icrdma_request_reset;
	rf->check_fc = irdma_check_fc_for_qp;
	priv_peer_info->peer_info = peer_info;
	rf->rdma_ver = IRDMA_GEN_2;
	rf->rsrc_profile = IRDMA_HMC_PROFILE_DEFAULT;
	rf->rst_to = IRDMA_RST_TIMEOUT_HZ;
	dev->pci_rev = peer_info->pdev->revision;
	rf->default_vsi.vsi_idx = peer_info->pf_vsi_num;
	/* save information from peer_info to priv_peer_info*/
	priv_peer_info->fn_num = PCI_FUNC(peer_info->pdev->devfn);
	rf->hw.hw_addr = peer_info->hw_addr;
	rf->pcidev = peer_info->pdev;
	rf->netdev = peer_info->netdev;
	priv_peer_info->msix_count = peer_info->msix_count;
	priv_peer_info->msix_entries = peer_info->msix_entries;
	irdma_set_rf_user_cfg_params(rf);

	irdma_add_handler(hdl);
	if (irdma_ctrl_init_hw(rf)) {
		err = -EIO;
		goto err_ctrl_init;
	}
	if (irdma_set_attr_from_fragcnt(&rf->sc_dev, rf->fragcnt_limit))
		dev_warn(idev_to_dev(dev),
			 "device limit update failed for fragment count %d\n",
			 rf->fragcnt_limit);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_pf_init(hdl);
#endif
	priv_peer_info->ftype = peer_info->ftype;
	dev->vchnl_if = &irdma_vchnl_if;
	if (dev->privileged) {
		rf->gen_ops.register_qset = irdma_lan_register_qset;
		rf->gen_ops.unregister_qset = irdma_lan_unregister_qset;
	}
	peer_info->peer_ops = &irdma_peer_ops;
	peer_info->peer_drv = &irdma_peer_drv;

	err = peer_info->ops->peer_register(peer_info);
	if (err)
		goto err_peer_reg;

	irdma_dbg(dev, "INIT: Gen2 device probe success peer_info=%p\n",
		  peer_info);

	return 0;

err_peer_reg:
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_pf_exit(rf->hdl);
#endif
	irdma_ctrl_deinit_hw(rf);
err_ctrl_init:
	irdma_del_handler(rf->hdl);
	kfree(rf->hdl);

	return err;
}

/*
 * irdma_lan_vsi_ready - check to see if lan reset done
 * @pdev: platform device
 */
bool irdma_lan_vsi_ready(struct platform_device *pdev)
{
	struct ice_peer_obj_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct ice_peer_obj *peer_info = pdata->peer_obj;

	return peer_info->ops->is_vsi_ready(peer_info) ? true : false;
}

/**
 * irdma_reset_task: worker for reset recovery
 * @work: work_struct pointer
 */
void irdma_reset_task(struct work_struct *work)
{
	struct reset_work *rst_work = container_of(to_delayed_work(work),
						   struct reset_work, work);
	struct platform_device *pdev = rst_work->pdev;

	/* Reset Recovery */
	if (!irdma_lan_vsi_ready(pdev))
		goto reschd;

	irdma_probe(pdev);
	kfree(rst_work);
	module_put(THIS_MODULE);
	return;

reschd:
	if (!rst_work->rst_to) {
		pr_err("RF rebuild after reset timed out\n");
		kfree(rst_work);
		module_put(THIS_MODULE);
	} else {
		schedule_delayed_work(&rst_work->work, --rst_work->rst_to * HZ);
	}
}

/* TODO: Remove when iidc merged */
int irdma_suspend(struct platform_device *pdev, pm_message_t state)
{
	return irdma_remove(pdev);
}

