/******************************************************************************
 *                    Copyright 2014-2017 aQuantia Corporation
 *                    Confidential and Proprietary
 *
 * $File: diag_driver.c $
 *
 * $Revision: $
 *
 * $DateTime: $
 *
 * $Author:   $
 *
 ******************************************************************************/
#include "aqdiag.h"

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/dma-mapping.h>
#include <linux/uaccess.h>
#include <linux/bitops.h>
#include <linux/version.h>

static LIST_HEAD(aqdiag_devices);
static DEFINE_MUTEX(devicelist_lock);
static char drv_name[] = "aqdiag";
static struct cdev aqdiag_cdev;
static int aqdiag_major;
static DEFINE_IDR(minor_idr);
static DEFINE_MUTEX(minor_lock);

static int aqdiag_alloc_mem(struct aqdiag_dev *dev, struct aqdiag_alloc_mem __user *ureq)
{
	int ret = -ENOMEM;
	int idx;
	struct aqdiag_memreg *memreg;
	struct aqdiag_alloc_mem req;

	if (copy_from_user(&req, ureq, sizeof(req)))
		return -EFAULT;

	memreg = kzalloc(sizeof(*memreg), GFP_KERNEL);
	if (!memreg) {
		dev_err(&dev->dev, "Can't alloc memreg\n");
		goto out;
	}
	INIT_LIST_HEAD(&memreg->list);
	memreg->dev = dev;
	kobject_init(&memreg->kobj, &memreg_type);

	memreg->size = req.size;
	req.size = ALIGN(req.size, PAGE_SIZE);
	memreg->vaddr = dma_alloc_coherent(&dev->pdev->dev, req.size, &memreg->paddr, GFP_KERNEL);
	if (!memreg->vaddr) {
		dev_err(&dev->dev, "Can't alloc DMA memory (size %u)", req.size);
		goto err_dma_alloc;
	}
	memreg->real_size = req.size;


#if LINUX_VERSION_CODE <= KERNEL_VERSION(3,15,0)
	do {
		ret = -ENOMEM;
		if (!idr_pre_get(&dev->memreg_idr, GFP_KERNEL))
			goto err_idr;

		mutex_lock(&dev->memreg_lock);
		if (test_bit(AQDSTATE_REMOVING, &dev->state))
			ret = -ENODEV;
		else
			ret = idr_get_new(&dev->memreg_idr, memreg, &idx);
		mutex_unlock(&dev->memreg_lock);
	} while (ret == -EAGAIN);
	if (ret)
		goto err_idr;
	memreg->index = req.index = idx;
#else
	idr_preload(GFP_KERNEL);
	mutex_lock(&dev->memreg_lock);
	ret = idr_alloc(&dev->memreg_idr, memreg,
		0, 0, GFP_KERNEL);
	mutex_unlock(&dev->memreg_lock);
	idr_preload_end();
	if (ret < 0)
		goto err_idr;
	memreg->index = req.index = idx = ret;
#endif

	//memreg->index = req.index = (int)((memreg->paddr >> 12) & 0xffffffff);
	ret = aqdiag_publish_memreg(memreg);
	if (ret) {
		dev_err(&dev->dev, "Can't publish memreg to sysfs: %d", ret);
		goto err_publish;
	}

	if(copy_to_user(ureq, &req, sizeof(req))) {
		ret = -EFAULT;
		goto err_copy_to_user;
	}

	mutex_lock(&dev->memreg_lock);
	list_add(&memreg->list, &dev->mem_regions);
	mutex_unlock(&dev->memreg_lock);

	return 0;

err_copy_to_user:
	aqdiag_hide_memreg(memreg);
err_publish:
	mutex_lock(&dev->memreg_lock);
	idr_remove(&dev->memreg_idr, idx);
	mutex_unlock(&dev->memreg_lock);
err_idr:
	dma_free_coherent(&dev->pdev->dev, memreg->real_size,
			  memreg->vaddr, memreg->paddr);
err_dma_alloc:
	kobject_put(&memreg->kobj);
out:
	return ret;
}

// dev->memreg_lock must be held
static void __aqdiag_unlink_mem(struct aqdiag_memreg *memreg)
{
	idr_remove(&memreg->dev->memreg_idr, memreg->index);
	list_del(&memreg->list);
}

/*
static void aqdiag_unlink_mem(struct aqdiag_memreg *memreg)
{
	mutex_lock(&memreg->dev->memreg_lock);
	__aqdiag_unlink_mem(memreg);
	mutex_unlock(&memreg->dev->memreg_lock);
}
*/

static void aqdiag_free_mem(struct aqdiag_memreg *memreg)
{
	aqdiag_hide_memreg(memreg);

	dma_free_coherent(&memreg->dev->pdev->dev, memreg->real_size,
			  memreg->vaddr, memreg->paddr);
	kobject_put(&memreg->kobj);
}

static void aqdiag_put_minor(int minor)
{
	mutex_lock(&minor_lock);
	idr_remove(&minor_idr, minor);
	mutex_unlock(&minor_lock);
}

static int aqdiag_get_minor(struct aqdiag_dev *dev)
{
	int ret;
	int minor;

#if LINUX_VERSION_CODE <= KERNEL_VERSION(3,15,0)
	do {
		if (!idr_pre_get(&minor_idr, GFP_KERNEL))
			return -ENOMEM;

		mutex_lock(&minor_lock);
		ret = idr_get_new(&minor_idr, dev, &minor);
		mutex_unlock(&minor_lock);
	} while (ret == -EAGAIN);

	if (ret)
		return ret;
#else
	idr_preload(GFP_KERNEL);
	mutex_lock(&minor_lock);
	ret = idr_alloc(&minor_idr, dev,
		0, 0, GFP_KERNEL);
	mutex_unlock(&minor_lock);
	idr_preload_end();
	if (ret < 0)
		return ret;
	minor = ret;
#endif

	if (minor > AQDIAG_MAX_DEVS - 1) {
		aqdiag_put_minor(minor);
		return -ENOSPC;
	}

	dev->minor = minor;
	return 0;
}

static void aqdiag_dev_release(struct device *kdev)
{
	struct aqdiag_dev *dev = container_of(kdev, struct aqdiag_dev, dev);
	kfree(dev);
}

static struct class aqdiag_class = {
	.name = drv_name,
	.dev_release = aqdiag_dev_release,
};

#if !defined(RHEL_RELEASE) && LINUX_VERSION_CODE <= KERNEL_VERSION(3,12,0)
// not defined in older linux, so define it ourselves from newer linux sources
static inline int dma_set_mask_and_coherent(struct device *dev, u64 mask)
{
	int rc = dma_set_mask(dev, mask);
	if (rc == 0)
		dma_set_coherent_mask(dev, mask);
	return rc;
}
#endif

static int aqdiag_probe(struct pci_dev *pdev, const struct pci_device_id *id)
{
	struct aqdiag_dev *dev;
	struct device *kdev;
	int ret;

	ret = pci_enable_device_mem(pdev);
	if (ret)
		return ret;

	if (dma_set_mask_and_coherent(&pdev->dev, DMA_BIT_MASK(64))
	    && (ret = dma_set_mask_and_coherent(&pdev->dev, DMA_BIT_MASK(32)))) {
		dev_err(&pdev->dev, "Set DMA mask failed: 0x%x\n", ret);
		goto err_dma;
	}

	ret = pci_request_regions(pdev, drv_name);
	if (ret) {
		dev_err(&pdev->dev, "Request regions failed: 0x%x\n", ret);
		goto err_regions;
	}

	pci_set_master(pdev);

	dev = kzalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev) {
		ret = -ENOMEM;
		goto err_alloc;
	}
	idr_init(&dev->memreg_idr);
	mutex_init(&dev->memreg_lock);
	INIT_LIST_HEAD(&dev->mem_regions);
	dev->pdev = pdev;
	kdev = &dev->dev;
	device_initialize(kdev);

	ret = aqdiag_get_minor(dev);
	if (ret)
		goto err_minor;

	ret = aqdiag_init_interrupts(dev);
	if (ret) {
		dev_err(&pdev->dev, "Can't init interrupts: %d\n", ret);
		goto err_ints;
	}

	kdev->class = &aqdiag_class;
	kdev->parent = &pdev->dev;
	kdev->devt = MKDEV(aqdiag_major, dev->minor);
	ret = dev_set_name(kdev, "aqdiag%d", dev->minor);
	if (ret)
		goto err_device_create;
	ret = device_add(kdev);
	if (ret) {
		dev_err(&pdev->dev, "Create device failed: %d\n", ret);
		goto err_device_create;
	}

	ret = aqdiag_create_attrs(dev);
	if (ret)
		goto err_attrs;

	mutex_lock(&devicelist_lock);
	list_add(&dev->devices, &aqdiag_devices);
	mutex_unlock(&devicelist_lock);

	pci_set_drvdata(pdev, dev);

	return 0;

err_attrs:
	device_del(kdev);
err_device_create:
	aqdiag_free_irqs(dev);
err_ints:
	aqdiag_put_minor(dev->minor);
err_minor:
	put_device(kdev);
err_alloc:
	pci_release_regions(pdev);
err_regions:
err_dma:
	pci_disable_device(pdev);
	return ret;
}

static void aqdiag_remove(struct pci_dev *pdev)
{
	struct aqdiag_dev *dev = pci_get_drvdata(pdev);
	struct aqdiag_memreg *memreg, *tmp;
	struct list_head memregs;

	if (!dev)
		return;

	set_bit(AQDSTATE_REMOVING, &dev->state);

	mutex_lock(&devicelist_lock);
	list_del(&dev->devices);
	mutex_unlock(&devicelist_lock);

	aqdiag_del_attrs(dev);
	aqdiag_free_irqs(dev);

	device_del(&dev->dev);
	aqdiag_put_minor(dev->minor);

	mutex_lock(&dev->memreg_lock);
#if LINUX_VERSION_CODE <= KERNEL_VERSION(3,15,0)
	idr_remove_all(&dev->memreg_idr);
#endif
	idr_destroy(&dev->memreg_idr);
	list_replace_init(&dev->mem_regions, &memregs);
	mutex_unlock(&dev->memreg_lock);
	list_for_each_entry_safe(memreg, tmp, &memregs, list)
		aqdiag_free_mem(memreg);

	put_device(&dev->dev);

	pci_release_regions(pdev);
	pci_disable_device(pdev);
}

static int aqdiag_open(struct inode *ino, struct file *file)
{
	int ret;
	struct aqdiag_dev *dev;

	mutex_lock(&minor_lock);
	dev = idr_find(&minor_idr, iminor(ino));
	if (dev)
		get_device(&dev->dev);
	mutex_unlock(&minor_lock);

	if (!dev) {
		ret = -ENODEV;
		goto out;
	}

	file->private_data = dev;

	return nonseekable_open(ino, file);

out:
	return ret;
}

static int aqdiag_release(struct inode *ino, struct file *file)
{
	struct aqdiag_dev *dev = file->private_data;

	put_device(&dev->dev);
	return 0;
}

static long aqdiag_ioctl (struct file *file, unsigned int cmd, unsigned long arg)
{
	struct aqdiag_dev *dev = file->private_data;
	struct aqdiag_memreg *memreg;

	switch (cmd) {
	int ret;

	case AQDIAG_ALLOC_MEM:
		return aqdiag_alloc_mem(dev, (struct aqdiag_alloc_mem __user *) arg);

	case AQDIAG_FREE_MEM:
		mutex_lock(&dev->memreg_lock);
		memreg = idr_find(&dev->memreg_idr, (int)arg);
		if (memreg)
			__aqdiag_unlink_mem(memreg);
		mutex_unlock(&dev->memreg_lock);
		if (!memreg)
			return -ENOENT;
		aqdiag_free_mem(memreg);
		return 0;

	case AQDIAG_REQ_INT:
		ret = aqdiag_req_int(dev, (struct aqdiag_req_int __user *) arg);
		if (ret == -ERESTARTSYS)
			ret = -EINTR; // Never restart if interrupted by a signal
		return ret;

	case AQDIAG_INT_CANCEL:
		aqdiag_cancel_req(dev);
		return 0;
	default:
		return -ENOTTY;
	}
}

#ifdef CONFIG_COMPAT
#include <asm/compat.h>

static long
aqdiag_compat_ioctl(struct file *file, unsigned int cmd,
                          unsigned long arg)
{
    switch (_IOC_NR(cmd)) {
	case _IOC_NR(AQDIAG_ALLOC_MEM):
	case _IOC_NR(AQDIAG_FREE_MEM):
	case _IOC_NR(AQDIAG_REQ_INT):
	case _IOC_NR(AQDIAG_INT_CANCEL):
		/* Fix up pointer size (usually 4 -> 8 in 32-on-64-bit case */
		if (_IOC_SIZE(cmd) == sizeof(compat_uptr_t)) {
			cmd &= ~IOCSIZE_MASK;
			cmd |= sizeof(void *) << IOCSIZE_SHIFT;
		}
		break;
    }

    return aqdiag_ioctl(file, cmd, (unsigned long)compat_ptr(arg));
}
#endif

static struct file_operations aqdiag_fops = {
	.owner = THIS_MODULE,
	.open = aqdiag_open,
	.release = aqdiag_release,
	.unlocked_ioctl = aqdiag_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = aqdiag_compat_ioctl,
#endif
};

static int init_aqdiag_class(void)
{
	int ret;
	dev_t aqdiag_dev;
	struct cdev *cdev = &aqdiag_cdev;

	ret = alloc_chrdev_region(&aqdiag_dev, 0, AQDIAG_MAX_DEVS, drv_name);
	if (ret)
		goto out;

	cdev_init(cdev, &aqdiag_fops);
	cdev->owner = THIS_MODULE;
	kobject_set_name(&cdev->kobj, "%s", drv_name);

	ret = cdev_add(cdev, aqdiag_dev, AQDIAG_MAX_DEVS);
	if (ret)
		goto err_reg;

	aqdiag_major = MAJOR(aqdiag_dev);

	ret = class_register(&aqdiag_class);
	if (ret) {
		printk(KERN_ERR "Failed to register class\n");
		goto err_cdev;
	}

	return 0;

err_cdev:
	cdev_del(cdev);
err_reg:
	unregister_chrdev_region(aqdiag_dev, AQDIAG_MAX_DEVS);
out:
	return ret;
}

static void release_aqdiag_class(void)
{
	class_unregister(&aqdiag_class);
	cdev_del(&aqdiag_cdev);
	unregister_chrdev_region(MKDEV(aqdiag_major, 0), AQDIAG_MAX_DEVS);
}

static struct pci_device_id aqdiag_ids[] = {
	{ PCI_DEVICE(0x1d6a, 0x0001), },
	{ PCI_DEVICE(0x1d6a, 0xd107), },
	{ PCI_DEVICE(0x1d6a, 0xd108), },
	{ PCI_DEVICE(0x1d6a, 0xd100), },
	{ PCI_DEVICE(0x1d6a, 0x07b1), },
	{ PCI_DEVICE(0x1d6a, 0x08b1), },
	{ PCI_DEVICE(0x1d6a, 0x09b1), },
	{ PCI_DEVICE(0x1d6a, 0x11b1), },
	{ PCI_DEVICE(0x1d6a, 0x12b1), },
	{ PCI_DEVICE(0x1d6a, 0x00b1), },
	{ PCI_DEVICE(0x1d6a, 0x87b1), },
	{ PCI_DEVICE(0x1d6a, 0x88b1), },
	{ PCI_DEVICE(0x1d6a, 0x89b1), },
	{ PCI_DEVICE(0x1d6a, 0x91b1), },
	{ PCI_DEVICE(0x1d6a, 0x92b1), },
	{ PCI_DEVICE(0x1d6a, 0x80b1), },
	{ 0, },
};
MODULE_DEVICE_TABLE(pci, aqdiag_ids);

static struct pci_driver aqdiag_driver = {
	.name = drv_name,
	.id_table = aqdiag_ids,
	.probe = aqdiag_probe,
	.remove = aqdiag_remove
};

static int __init aqdiag_init(void)
{
	int ret;

	ret = init_aqdiag_class();
	if (ret)
		return ret;

	ret = pci_register_driver(&aqdiag_driver);
	if (ret) {
		release_aqdiag_class();
		return ret;
	}

	return 0;
}

static void __exit aqdiag_exit(void)
{
	pci_unregister_driver(&aqdiag_driver);
	release_aqdiag_class();
	idr_destroy(&minor_idr);
}

module_init(aqdiag_init);
module_exit(aqdiag_exit);

MODULE_LICENSE("GPL");
