# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernet
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $FinalObject = @()
        GetIntelEthernetDevices @('icei*', 'i40ei*')
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
        GetIntelDriverInfo
        $script:MSNetHwInfo = @(Get-NetAdapterHardwareInfo -ErrorAction SilentlyContinue)
        $script:MSNetAdvProperty = @(Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue)
    }
    Process
    {
        $Adapters = $Adapter
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNamesArray)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
            }

            $SupportedSpeeds = GetAdapterSupportedSpeeds $a
            $AdapterStatuses = GetAdapterStatus $a
            $BusTypes = GetAdapterBusType $a
            $Capabilities = GetAdapterCapabilities $a $SupportedSpeeds
            $ConnectionNames = GetAdapterConnectionName $a
            $DDPPackageNameAndVersions = GetAdapterDDPPackageNameAndVersion $a
            $DDPPackageTrackIds = GetAdapterDDPPackageTrackId $a
            $DeviceStatuses = GetAdapterDeviceStatus $a $SupportedSpeeds
            $DriverVersion = GetDriverVersion $a
            $EEEStatuses = GetAdapterEEEStatus $a
            $ETrackIDs = GetAdapterETrackId $a
            $FullDuplex = GetFullDuplex $a
            $LinkSpeedDuplex = GetLinkSpeedDuplex $a
            $MaxSpeeds = GetAdapterMaxSpeed $a $SupportedSpeeds
            $MediaTypes = GetAdapterMediaType $a
            $MiniPortNames = GetAdapterMiniPortName $a
            $NVMVersions = GetAdapterNVMVersion $a
            $NegotiatedLinkSpeed = GetAdapterNegotiatedLinkSpeed $a
            $NegotiatedLinkWidth = GetAdapterNegotiatedLinkWidth $a
            $NetlistVersions = GetAdapterNetlistVersion $a
            $OemFwVersions = GetOemFwVersion $a
            $OriginalDisplayNames = GetOriginalDisplayName $a
            $PCIDeviceIDs = GetAdapterPCIDeviceID $a
            $PartNumbers = GetAdapterPartNumber $a
            $PciLocations = GetAdapterPCILocation $a
            $AdapterPort = GetAdapterPort $a
            $SanMacAddresses = GetAdapterSanMacAddress $a

            # Assemble it all together in PSCustomObject
            $FinalObject += [PsCustomObject] @{
                PSTypeName          = 'IntelEthernetAdapter'
                AdapterStatus       = $AdapterStatuses
                BusType             = $BusTypes.BusType
                BusTypeString       = $BusTypes.BusTypeString
                Capabilities        = $Capabilities
                ConnectionName      = $ConnectionNames
                DDPPackageName      = $DDPPackageNameAndVersions.Name
                DDPPackageTrackId   = $DDPPackageTrackIds
                DDPPackageVersion   = $DDPPackageNameAndVersions.Version
                DeviceStatus        = $DeviceStatuses.DeviceStatus
                DeviceStatusString  = $DeviceStatuses.DeviceStatusString
                DriverVersion       = $DriverVersion
                EEE                 = $EEEStatuses.EEEStatus
                EEEString           = $EEEStatuses.EEEStatusString
                ETrackID            = $ETrackIDs
                FWVersion           = $OemFwVersions
                FullDuplex          = $FullDuplex
                Location            = $PciLocations
                MaxSpeed            = $MaxSpeeds
                MediaType           = $MediaTypes.MediaType
                MediaTypeString     = $MediaTypes.MediaTypeString
                MiniPortInstance    = $MiniPortNames.Instance
                MiniPortName        = $MiniPortNames.Name
                NVMVersion          = $NVMVersions
                Name                = $a
                NegotiatedLinkSpeed       = $NegotiatedLinkSpeed.NegotiatedLinkSpeed
                NegotiatedLinkSpeedString = $NegotiatedLinkSpeed.NegotiatedLinkSpeedString
                NegotiatedLinkWidth       = $NegotiatedLinkWidth.NegotiatedLinkWidth
                NegotiatedLinkWidthString = $NegotiatedLinkWidth.NegotiatedLinkWidthString
                NetlistVersion      = $NetlistVersions
                OriginalDisplayName = $OriginalDisplayNames
                PCIDeviceID         = $PCIDeviceIDs
                PartNumber          = $PartNumbers
                PortNumber          = $AdapterPort.Port
                PortNumberString    = $AdapterPort.PortString
                SANMacAddress       = $SanMacAddresses
                Speed               = $LinkSpeedDuplex.Speed
                SpeedString         = $LinkSpeedDuplex.SpeedString
                }
        }
    }
    End
    {
        $FinalObject | Sort-Object -Property Location

        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function GetAdapterBusType($AdapterName)
{
    $BusType       = 0
    $BusTypeString = $Messages.Unknown

    foreach ($Bus in $script:BusTypesArray)
    {
        if (($null -ne $Bus) -and ($Bus.InstanceName -eq $AdapterName))
        {
            $BusType       = [int]$Bus.BusType
            $BusTypeString = $BusTypeMap[[int]$Bus.BusType]
            break
        }
    }

    return [PsCustomObject] @{
        BusType       = $BusType
        BusTypeString = $BusTypeString }
}

function GetAdapterConnectionName($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceAlias
}

function GetAdapterDDPPackageNameAndVersion($AdapterName)
{
    $Name    = $Messages.NotSupported
    $Version = $Messages.NotSupported

    foreach ($DDPPkgName in $script:DDPPkgNamesArray)
    {
        if (($null -ne $DDPPkgName) -and ($DDPPkgName.InstanceName -eq $AdapterName))
        {
            if ($DDPPkgName.Name -and ($null -ne $DDPPkgName.Major) -and ($null -ne $DDPPkgName.Minor))
            {
                $Name = [System.Text.Encoding]::ASCII.GetString($DDPPkgName.Name).Trim(0)
                $Version = $DDPPkgName.Major.ToString() + "." + $DDPPkgName.Minor.ToString()
            }
            break
        }
    }
    return [PsCustomObject] @{
        Name    = $Name
        Version = $Version }
}


function GetAdapterDDPPackageTrackId($AdapterName)
{
    $TrackId = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetTrackId" $AdapterName "WmiGetTrackId" $Params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true) -and ($Result.Track_Id))
    {
        $TrackId = '0x{0:X}' -f $Result.Track_Id
    }
    return $TrackId
}


function GetAdapterDeviceStatus($AdapterName, $SupportedSpeeds)
{
    $DeviceStatus = 0
    $DeviceStatusString = $Messages.Unknown

    $AdapterNames = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})

    foreach($TmpAdapter in $AdapterNames)
    {
        if ("Up" -eq $TmpAdapter.Status)
        {
            $MaxSpeed = GetAdapterMaxSpeed $AdapterName $SupportedSpeeds
            $CurrentSpeed = (GetLinkSpeedDuplex $AdapterName).Speed
            if ($CurrentSpeed -lt $MaxSpeed)
            {
                $DeviceStatus = 4
                $DeviceStatusString = $Messages.LinkUpNotMax
            }
            else
            {
                $DeviceStatus = 1
                $DeviceStatusString = $Messages.LinkUp
            }
        }
        elseif ("Disconnected" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 2
            $DeviceStatusString = $Messages.LinkDown
        }
        elseif ("Disabled" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 0
            $DeviceStatusString = $Messages.Disabled
        }
    }

    return [PsCustomObject] @{
        DeviceStatus       = $DeviceStatus
        DeviceStatusString = $DeviceStatusString}

}

function GetAdapterETrackId($AdapterName)
{
    $strETrackId = $Messages.NotSupported
    foreach ($ETrackId in $script:ETrackIdsArray)
    {
        if (($null -ne $ETrackId) -and ($ETrackId.InstanceName -eq $AdapterName))
        {
            if ($ETrackId.Id)
            {
                $strETrackId = '0x{0:X}' -f $ETrackId.Id
            }
            break
        }
    }
    return $strETrackId
}


function GetAdapterNVMVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($NVMVersion in $script:NVMVersionsArray)
    {
        if (($null -ne $NVMVersion) -and ($NVMVersion.InstanceName -eq $AdapterName))
        {
            if ($NVMVersion.Version)
            {
                $Version = (($NVMVersion.Version -band 0xffff) -shr 8).ToString() + "." + (($NVMVersion.Version -band 0xff)).ToString("X2")
            }
            break
        }
    }
    return $Version
}

function GetAdapterNetlistVersion($AdapterName)
{
    $NetlistVersion = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetNVMNetlistInfo" $AdapterName "WmiGetNVMNetlistInfo" $params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true) -and ($Result.VersionStr))
    {
        $NetlistVersion = [System.Text.Encoding]::ASCII.GetString($Result.VersionStr).Trim(0)
    }
    return $NetlistVersion
}

function GetAdapterPartNumber($AdapterName)
{
    $PartNumberString = $Messages.NotSupported
    foreach ($PartNumber in $script:PartNumbersArray)
    {
        if (($null -ne $PartNumber) -and ($PartNumber.InstanceName -eq $AdapterName))
        {
            if ($PartNumber.PartNumberString)
            {
                $PartNumberString = [System.Text.Encoding]::ASCII.GetString($PartNumber.PartNumberString).Trim(0)
            }
            break
        }
    }
    return $PartNumberString
}

function GetAdapterSanMacAddress($AdapterName)
{
    $MacAddress = $Messages.NotSupported

    foreach ($SanMacAddress in $script:SanMacAddressesArray)
    {
        if (($null -ne $SanMacAddress) -and ($SanMacAddress.InstanceName -eq $AdapterName))
        {
            if ($SanMacAddress.SanMacAddr)
            {
                $MacAddress = ""
                for ($i = 0; $i -lt 6; $i++)
                {
                    # convert to string hex representation; X - hex, 2 - add leading zeroes if needed
                    $MacAddress += $SanMacAddress.SanMacAddr[$i].ToString("X2")
                }
            }

            break
        }
    }
    return $MacAddress
}

function GetAdapterMediaType($AdapterName)
{
    $MediaType = [PsCustomObject] @{
        MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
        MediaTypeString = $Messages.Unknown }

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    switch ($Service)
    {
        {($_ -eq 'icea') -or ($_ -eq 'scea')}   { $MediaType = GetAdapterMediaTypeIce $AdapterName; break }
        {($_ -eq 'i40ea') -or ($_ -eq 'i40eb')} { $MediaType = GetAdapterMediaTypeI40e $AdapterName; break }
    }
    return $MediaType
}

function GetAdapterMediaTypeIce($AdapterName)
{
    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
    $MediaTypeString = $Messages.Unknown

    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 8))
        {
            # Interpreting PhyInfo array values:
            # 0 == PHY type
            # 1 == link info <-- Bit 0 (value of 1 means has link)
            # 2 == an_info
            # 3 == ext_info
            # 4 == module type [0]
            # 5 == module type [1]
            # 6 == module type [2]
            # 7 == media interface  <-- 1=Backplane, 2=QSFP, 3=SFP
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkInfo = $PhyInfo.PhyInfo[1];
            $MediaInterface = $PhyInfo.PhyInfo[7];

            if ($LinkInfo -band 0x01)
            {
                switch ($PhyType)
                {
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL25GAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2AOC_ACC}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_AOC_ACC
                        $MediaTypeString = $Messages.AOCACC
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL5GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL10GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseKR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR
                        $MediaTypeString = $Messages.BackplaneKR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR1
                        $MediaTypeString = $Messages.BackplaneKR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2
                        $MediaTypeString = $Messages.BackplaneKR2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR2PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2_PAM4
                        $MediaTypeString = $Messages.BackplaneKR2PAM4
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseKR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseKR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4
                        $MediaTypeString = $Messages.BackplaneKR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR4PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4_PAM4
                        $MediaTypeString = $Messages.BackplaneKR4PAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_PAM4
                        $MediaTypeString = $Messages.BackplaneKRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_S
                        $MediaTypeString = $Messages.BackplaneKRS
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseKX,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseKX}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KX
                        $MediaTypeString = $Messages.BackplaneKX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIC2C,
                    [int][CVL_PHY_TYPE]::CVL25GAUIC2C,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUI,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Chip_to_Chip
                        $MediaTypeString = $Messages.ChiptoChip
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseT,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL10GBaseT,
                    [int][CVL_PHY_TYPE]::CVL25GBaseT}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100BaseTX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_TX
                        $MediaTypeString = $Messages.CopperTX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL10GSFIDA)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach
                        $MediaTypeString = $Messages.DirectAttach
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseCP2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CP2
                        $MediaTypeString = $Messages.DirectAttachCP2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR
                        $MediaTypeString = $Messages.DirectAttachCR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR1
                        $MediaTypeString = $Messages.DirectAttachCR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR2
                        $MediaTypeString = $Messages.DirectAttachCR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseCR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseCR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR4
                        $MediaTypeString = $Messages.DirectAttachCR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_PAM4
                        $MediaTypeString = $Messages.DirectAttachCRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_S
                        $MediaTypeString = $Messages.DirectAttachCRS
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseDR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_DR
                        $MediaTypeString = $Messages.FiberDR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseFR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_FR
                        $MediaTypeString = $Messages.FiberFR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseLR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR
                        $MediaTypeString = $Messages.FiberLR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseLR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseLR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR4
                        $MediaTypeString = $Messages.FiberLR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseLX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LX
                        $MediaTypeString = $Messages.FiberLX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseSR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR
                        $MediaTypeString = $Messages.FiberSR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseSR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR2
                        $MediaTypeString = $Messages.FiberSR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseSR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseSR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR4
                        $MediaTypeString = $Messages.FiberSR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseSX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SX
                        $MediaTypeString = $Messages.FiberSX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL2Point5gBaseX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_X
                        $MediaTypeString = $Messages.FiberX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL100MSGMII,
                    [int][CVL_PHY_TYPE]::CVL1GSGMII}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
            else
            {
                switch ($MediaInterface)
                {
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BACKPLANE)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BACKPLANE
                        $MediaTypeString = $Messages.Backplane
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_QSFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_QSFP
                        $MediaTypeString = $Messages.QSFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SFP
                        $MediaTypeString = $Messages.SFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BASE_T)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BASE_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SGMII)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_FIBER)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_FIBER
                        $MediaTypeString = $Messages.Fiber
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterMediaTypeI40e($AdapterName)
{
    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
    $MediaTypeString = $Messages.Unknown

    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 13))
        {
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkType = ([uint32]$PhyInfo.PhyInfo[11] -shl 24) + ([uint32]$PhyInfo.PhyInfo[10] -shl 16) + ([uint32]$PhyInfo.PhyInfo[9] -shl 8) + [uint32]$PhyInfo.PhyInfo[8];
            $LinkTypeExt = $PhyInfo.PhyInfo[12];

            if ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_EMPTY)
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_KR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_KR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                $MediaTypeString = $Messages.Backplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SFP_Cu) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                $MediaTypeString = $Messages.SFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_LR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_LR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_LR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                $MediaTypeString = $Messages.LRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SR) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_SR4) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_SR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                $MediaTypeString = $Messages.SRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_100BASE_TX) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_2_5GBASE_T) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_5GBASE_T))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                $MediaTypeString = $Messages.Copper
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                $MediaTypeString = $Messages.QSFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KX4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_KX))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                $MediaTypeString = $Messages.KXBackplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_XAUI) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_XLAUI))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                $MediaTypeString = $Messages.XAUI
            }
            elseif (($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_AOC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_ACC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_CR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                $MediaTypeString = $Messages.SFP28DirectAttach
            }
            # old FW or HW different than XL710
            elseif ($LinkTypeExt -eq 0 -and $LinkType -eq 0)
            {
                if (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_KR4) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KR) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_KR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                    $MediaTypeString = $Messages.Backplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SFPP_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                    $MediaTypeString = $Messages.SFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_LR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_LR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_LR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                    $MediaTypeString = $Messages.LRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_SR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_SR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                    $MediaTypeString = $Messages.SRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_100BASE_TX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                    $MediaTypeString = $Messages.Copper
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                    $MediaTypeString = $Messages.QSFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KX4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_KX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                    $MediaTypeString = $Messages.KXBackplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XAUI) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XLAUI))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                    $MediaTypeString = $Messages.XAUI
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_AOC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_ACC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_CR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                    $MediaTypeString = $Messages.SFP28DirectAttach
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterPCILocation($AdapterName)
{
    $PCILocation = $Messages.NotSupported
    foreach ($HwInfo in $script:MSNetHwInfo)
    {
        if ($AdapterName -eq $HwInfo.ifDesc)
        {
            $PCILocation = $HwInfo.Bus.ToString() + ":" + $HwInfo.Device.ToString()  + ":" + $HwInfo.Function.ToString() + ":" + $HwInfo.Segment.ToString()
            break
        }
    }
    return $PCILocation
}

function GetOriginalDisplayName($AdapterName)
{
    return ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).ifDesc
}

function GetOemFwVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($FwVersion in $script:FwVersionsArray)
    {
        if ($FwVersion.InstanceName -eq $AdapterName)
        {
            # driver can return an array of zeroes - don't attempt to construct a string using it
            if (($null -ne $FwVersion.SingleNvmVersion) -and $FwVersion.SingleNvmVersion[0] -ne 0)
            {
                $Version = [System.Text.Encoding]::ASCII.GetString($FwVersion.SingleNvmVersion).Trim(0)
            }
            break
        }
    }
    return $Version
}

function GetAdapterPCIDeviceID($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).PnPDeviceID
}

function GetAdapterNegotiatedLinkWidth($AdapterName)
{
    $NegotiatedLinkWidth = 0
    $NegotiatedLinkWidthString = $Messages.NotAvailable

    $AdapterHWInfo = $script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})

    if ($AdapterHWInfo)
    {
        $NegotiatedLinkWidth = $AdapterHWInfo.PciExpressCurrentLinkWidth
        $NegotiatedLinkWidthString = "x" + $NegotiatedLinkWidth.ToString()
    }

    return [PsCustomObject] @{
        NegotiatedLinkWidth = $NegotiatedLinkWidth
        NegotiatedLinkWidthString = $NegotiatedLinkWidthString }
}

function GetAdapterNegotiatedLinkSpeed($AdapterName)
{
    $NegotiatedLinkSpeed = ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).PciExpressCurrentLinkSpeedEncoded
    switch ($NegotiatedLinkSpeed)
    {
        0 {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
        1 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed25; break}
        2 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed50; break}
        3 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed80; break}
        default {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
    }
    return [PsCustomObject] @{
        NegotiatedLinkSpeed = $NegotiatedLinkSpeed
        NegotiatedLinkSpeedString = $NegotiatedLinkSpeedString }
}

function GetLinkSpeedDuplex($AdapterName)
{
    $AdapterObj = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})
    $Speed = $AdapterObj.Speed
    $SpeedString = $AdapterObj.LinkSpeed
    $FullDuplex = $AdapterObj.FullDuplex

    if ("Up" -ne $AdapterObj.Status)
    {
        $Speed = 0
        $SpeedString = $Messages.NotAvailable
    }
    elseif ($true -eq $FullDuplex)
    {
        $SpeedString += " " + $Messages.FullDuplex
    }

    return [PsCustomObject] @{
        Speed = $Speed
        SpeedString = $SpeedString }
}

function GetFullDuplex($AdapterName)
{
    $FullDuplex = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).FullDuplex
    if (-not $FullDuplex)
    {
        $FullDuplex = ""
    }
    return $FullDuplex
}

function GetAdapterPort($AdapterName)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName

    switch ($AdapterRegKey.Port)
    {
        0 {$PortNumberString = $Messages.PortA; break}
        1 {$PortNumberString = $Messages.PortB; break}
        2 {$PortNumberString = $Messages.PortC; break}
        3 {$PortNumberString = $Messages.PortD; break}
        default {$PortNumberString = $Messages.NotSupported; break}
    }

    return [PsCustomObject] @{
        Port       = $AdapterRegKey.Port
        PortString = $PortNumberString }
}

function GetDriverVersion($AdapterName)
{
    $DriverVersion = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).DriverVersion
    return $DriverVersion
}

function GetAdapterMiniPortName($AdapterName)
{
    $Name = ($script:PnpDevice.Where({$_.Name -eq $AdapterName})).Service
    $Instance = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    return [PsCustomObject] @{
        Name     = $Name
        Instance = $Instance }
}

function GetAdapterMaxSpeed($AdapterName, $SupportedSpeeds)
{
    if ($SupportedSpeeds.Length -gt 0)
    {
        # array is sorted, so we just return the last element
        return $SupportedSpeeds[-1]
    }
    return 0
}

function GetAdapterSupportedSpeeds($AdapterName)
{
    $SpeedDuplex = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "*SpeedDuplex"})
    if ($SpeedDuplex)
    {
        $RegistryValues = ($SpeedDuplex).ValidRegistryValues
    }

    $SupportedSpeeds = @()

    foreach ($i in $RegistryValues)
    {
        $SupportedSpeeds += $SupportedSpeedsMap[$i]
    }

    return $SupportedSpeeds | Sort-Object
}


function GetAdapterEEEStatus($AdapterName)
{
    $EEEStatus = 0
    $EEEStatusString = $Messages.NotSupported

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    foreach ($EEE in $script:EEELinkStatusArray)
    {
        if (($null -ne $EEE) -and ($EEE.InstanceName -eq $AdapterName))
        {
            if ($EEE.EEELinkStatus -eq $false)
            {
                $EEELinkAdvertisement = 0
                if ($Service -in @("i40ea", "i40eb"))
                {
                    $EEELinkAdvertisement = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "EEELinkAdvertisement"})
                    if ($EEELinkAdvertisement -and $EEELinkAdvertisement.RegistryValue)
                    {
                        $EEELinkAdvertisement = $EEELinkAdvertisement.RegistryValue
                    }
                    else
                    {
                        break
                    }
                }

                if ($EEELinkAdvertisement -gt 0)
                {
                    $EEEStatus = 3 #Not Negotiated
                    $EEEStatusString = $Messages.NotNegotiated
                }
                else
                {
                    $EEEStatus = 1 #Disabled
                    $EEEStatusString = $Messages.Disabled
                }
            }
            else
            {
                $EEEStatus = 2 #Active
                $EEEStatusString = $Messages.Active
            }
        }
    }

    return [PsCustomObject] @{
        EEEStatus       = $EEEStatus
        EEEStatusString = $EEEStatusString }
}

function GetAdapterStatus($AdapterName)
{
    $AdapterStatus = [ADAPTER_STATUS]::Installed -bor [ADAPTER_STATUS]::DriverLoaded -bor [ADAPTER_STATUS]::HasDiag

    $LinkStatus = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).Status
    if ($LinkStatus -eq 'Up')
    {
        $AdapterStatus = $AdapterStatus -bor [ADAPTER_STATUS]::HasLink
    }
    return $AdapterStatus
}

function GetAdapterCapabilities($AdapterName, $SupportedSpeeds)
{
    $Capabilities = @([int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_VENDOR_INTEL)

    foreach ($SupportedSpeed in $SupportedSpeeds)
    {
        switch ($SupportedSpeed)
        {
            10000000     {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10_MBPS; break}
            100000000    {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100_MBPS; break}
            1000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_1000_MBPS; break}
            2500000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_2500_MBPS; break}
            5000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_5000_MBPS; break}
            10000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10000_MBPS; break}
            40000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_40000_MBPS; break}
            25000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_25000_MBPS; break}
            50000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_50000_MBPS; break}
            100000000000 {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100000_MBPS; break}
        }
    }
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_PERFORMANCE_PROFILE
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DIAGNOSTIC_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_FLASH_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_CYPRESS
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_IDENTIFY_ADAPTER_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_NDIS_IOAT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_EXTENDED_DMIX_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DCB
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_JUMBO_FRAMES

    return , ($Capabilities | Sort-Object)
}

function GetIntelDriverInfo()
{
    $script:BusTypesArray        = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_BusType -Property BusType -ErrorAction SilentlyContinue
    $script:DDPPkgNamesArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPackageInfo -Property Name, Major, Minor -ErrorAction SilentlyContinue
    $script:ETrackIdsArray       = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EetrackId -Property Id -ErrorAction SilentlyContinue
    $script:NVMVersionsArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EepromVersion -Property Version -ErrorAction SilentlyContinue
    $script:SanMacAddressesArray = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetSanMacAddress -Property SanMacAddr -ErrorAction SilentlyContinue
    $script:PartNumbersArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_PartNumberString -Property PartNumberString -ErrorAction SilentlyContinue
    $script:PhyInfoArray         = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPhyInfo -Property PhyInfo -ErrorAction SilentlyContinue
    $script:FwVersionsArray      = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetOemProductVer -Property SingleNvmVersion -ErrorAction SilentlyContinue
    $script:EEELinkStatusArray   = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EEELinkStatus -Property EEELinkStatus -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIt8QYJKoZIhvcNAQcCoIIt4jCCLd4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCACKIU7lkdvjS75
# O4GGqXKUGpY27jd8+uVCpwn4LBgV1qCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCG1QwghtQAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgNLPKlUxBcDESrGl12T2m
# dJfeVOdOH5k4wI+8aZSWHywwDQYJKoZIhvcNAQEBBQAEggGABvUHnCVcsuDxelW1
# vXQWuqWsS7jY1g46MNR4XeqepPqQ3SLubAnxXfYmAzak0CRyQO5zLGpy/1/77THV
# 1YcJYnaSEX5u7HTdrx72eYrjyue/FeG3CuX/gAQvxWMXX7NNXCQRVhR8YwUPgC1E
# oI0rrGdtMCyfde3W5bt3tyh9DZQDr3SS2I31IdOncyMr8g4YQXDgnMi68l967J3R
# LYFSh3ngxIQbmDO9w8qTZ9FIR44rSIFF7+cNylp811QyWXAJVB2B4MVIvvAYA7vr
# 2oDJBQYgQgVgH9vyLCKL2GRLXcd7+KHxIt5QioAxa8QDHZFmf22IqI4MDUj1aBAb
# lIlcY1aji5fP1dAxnhqk8mNhJk3Sj5Yl3AyxyDdUTCZFQWzngfXhlHofLc69sDqh
# MEsmxr15YngEWbRrR1pZyMWnfgZSCqPuZ0DACrjmZTRyHA9jH4rXqM6Ne65MMY++
# VEpj9UOVxz2NIJb0QsUgSrjhUtAUpJV0OmOLQSAnN2gmpc8koYIYzTCCGMkGCisG
# AQQBgjcDAwExghi5MIIYtQYJKoZIhvcNAQcCoIIYpjCCGKICAQMxDzANBglghkgB
# ZQMEAgIFADCB8wYLKoZIhvcNAQkQAQSggeMEgeAwgd0CAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQg1BYSDiHksIJGFR5NvWxlob5BB5Gz71Yv8r72l9QT
# 7SoCFEguKOrfVjeOn8eQ+pbQcu2Hghj9GA8yMDI0MDkxNzIxMzExNlqgcqRwMG4x
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBp
# bmcgU2lnbmVyIFIzNaCCEv8wggZdMIIExaADAgECAhA6UmoshM5V5h1l/MwS2OmJ
# MA0GCSqGSIb3DQEBDAUAMFUxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdv
# IExpbWl0ZWQxLDAqBgNVBAMTI1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcg
# Q0EgUjM2MB4XDTI0MDExNTAwMDAwMFoXDTM1MDQxNDIzNTk1OVowbjELMAkGA1UE
# BhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEwMC4GA1UEAxMnU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBTaWdu
# ZXIgUjM1MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAjdFn9MFIm739
# OEk6TWGBm8PY3EWlYQQ2jQae45iWgPXUGVuYoIa1xjTGIyuw3suUSBzKiyG0/c/Y
# n++d5mG6IyayljuGT9DeXQU9k8GWWj2/BPoamg2fFctnPsdTYhMGxM06z1+Ft0Ba
# v8ybww21ii/faiy+NhiUM195+cFqOtCpJXxZ/lm9tpjmVmEqpAlRpfGmLhNdkqiE
# uDFTuD1GsV3jvuPuPGKUJTam3P53U4LM0UCxeDI8Qz40Qw9TPar6S02XExlc8X1Y
# siE6ETcTz+g1ImQ1OqFwEaxsMj/WoJT18GG5KiNnS7n/X4iMwboAg3IjpcvEzw4A
# ZCZowHyCzYhnFRM4PuNMVHYcTXGgvuq9I7j4ke281x4e7/90Z5Wbk92RrLcS35hO
# 30TABcGx3Q8+YLRy6o0k1w4jRefCMT7b5mTxtq5XPmKvtgfPuaWPkGZ/tbxInyND
# A7YgOgccULjp4+D56g2iuzRCsLQ9ac6AN4yRbqCYsG2rcIQ5INTyI2JzA2w1vsAH
# PRbUTeqVLDuNOY2gYIoKBWQsPYVoyzaoBVU6O5TG+a1YyfWkgVVS9nXKs8hVti3V
# pOV3aeuaHnjgC6He2CCDL9aW6gteUe0AmC8XCtWwpePx6QW3ROZo8vSUe9AR7mMd
# u5+FzTmW8K13Bt8GX/YBFJO7LWzwKAUCAwEAAaOCAY4wggGKMB8GA1UdIwQYMBaA
# FF9Y7UwxeqJhQo1SgLqzYZcZojKbMB0GA1UdDgQWBBRo76QySWm2Ujgd6kM5LPQU
# ap4MhTAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAK
# BggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUF
# BwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwSgYDVR0fBEMw
# QTA/oD2gO4Y5aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGlt
# ZVN0YW1waW5nQ0FSMzYuY3JsMHoGCCsGAQUFBwEBBG4wbDBFBggrBgEFBQcwAoY5
# aHR0cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5n
# Q0FSMzYuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAN
# BgkqhkiG9w0BAQwFAAOCAYEAsNwuyfpPNkyKL/bJT9XvGE8fnw7Gv/4SetmOkjK9
# hPPa7/Nsv5/MHuVus+aXwRFqM5Vu51qfrHTwnVExcP2EHKr7IR+m/Ub7PamaeWfl
# e5x8D0x/MsysICs00xtSNVxFywCvXx55l6Wg3lXiPCui8N4s51mXS0Ht85fkXo3a
# uZdo1O4lHzJLYX4RZovlVWD5EfwV6Ve1G9UMslnm6pI0hyR0Zr95QWG0MpNPP0u0
# 5SHjq/YkPlDee3yYOECNMqnZ+j8onoUtZ0oC8CkbOOk/AOoV4kp/6Ql2gEp3bNC7
# DOTlaCmH24DjpVgryn8FMklqEoK4Z3IoUgV8R9qQLg1dr6/BjghGnj2XNA8ujta2
# JyoxpqpvyETZCYIUjIs69YiDjzftt37rQVwIZsfCYv+DU5sh/StFL1x4rgNj2t8G
# ccUfa/V3iFFW9lfIJWWsvtlC5XOOOQswr1UmVdNWQem4LwrlLgcdO/YAnHqY52Qw
# nBLiAuUnuBeshWmfEb5oieIYMIIGFDCCA/ygAwIBAgIQeiOu2lNplg+RyD5c9Mfj
# PzANBgkqhkiG9w0BAQwFADBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5n
# IFJvb3QgUjQ2MB4XDTIxMDMyMjAwMDAwMFoXDTM2MDMyMTIzNTk1OVowVTELMAkG
# A1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAxMjU2Vj
# dGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYwggGiMA0GCSqGSIb3DQEB
# AQUAA4IBjwAwggGKAoIBgQDNmNhDQatugivs9jN+JjTkiYzT7yISgFQ+7yavjA6B
# g+OiIjPm/N/t3nC7wYUrUlY3mFyI32t2o6Ft3EtxJXCc5MmZQZ8AxCbh5c6WzeJD
# B9qkQVa46xiYEpc81KnBkAWgsaXnLURoYZzksHIzzCNxtIXnb9njZholGw9djnjk
# TdAA83abEOHQ4ujOGIaBhPXG2NdV8TNgFWZ9BojlAvflxNMCOwkCnzlH4oCw5+4v
# 1nssWeN1y4+RlaOywwRMUi54fr2vFsU5QPrgb6tSjvEUh1EC4M29YGy/SIYM8ZpH
# admVjbi3Pl8hJiTWw9jiCKv31pcAaeijS9fc6R7DgyyLIGflmdQMwrNRxCulVq8Z
# pysiSYNi79tw5RHWZUEhnRfs/hsp/fwkXsynu1jcsUX+HuG8FLa2BNheUPtOcgw+
# vHJcJ8HnJCrcUWhdFczf8O+pDiyGhVYX+bDDP3GhGS7TmKmGnbZ9N+MpEhWmbiAV
# PbgkqykSkzyYVr15OApZYK8CAwEAAaOCAVwwggFYMB8GA1UdIwQYMBaAFPZ3at0/
# /QET/xahbIICL9AKPRQlMB0GA1UdDgQWBBRfWO1MMXqiYUKNUoC6s2GXGaIymzAO
# BgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggr
# BgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwTAYDVR0fBEUwQzBBoD+gPYY7aHR0
# cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nUm9v
# dFI0Ni5jcmwwfAYIKwYBBQUHAQEEcDBuMEcGCCsGAQUFBzAChjtodHRwOi8vY3J0
# LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3RhbXBpbmdSb290UjQ2LnA3
# YzAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcN
# AQEMBQADggIBABLXeyCtDjVYDJ6BHSVY/UwtZ3Svx2ImIfZVVGnGoUaGdltoX4hD
# skBMZx5NY5L6SCcwDMZhHOmbyMhyOVJDwm1yrKYqGDHWzpwVkFJ+996jKKAXyIIa
# Uf5JVKjccev3w16mNIUlNTkpJEor7edVJZiRJVCAmWAaHcw9zP0hY3gj+fWp8MbO
# ocI9Zn78xvm9XKGBp6rEs9sEiq/pwzvg2/KjXE2yWUQIkms6+yslCRqNXPjEnBnx
# uUB1fm6bPAV+Tsr/Qrd+mOCJemo06ldon4pJFbQd0TQVIMLv5koklInHvyaf6vAT
# JP4DfPtKzSBPkKlOtyaFTAjD2Nu+di5hErEVVaMqSVbfPzd6kNXOhYm23EWm6N2s
# 2ZHCHVhlUgHaC4ACMRCgXjYfQEDtYEK54dUwPJXV7icz0rgCzs9VI29DwsjVZFpO
# 4ZIVR33LwXyPDbYFkLqYmgHjR3tKVkhh9qKV2WCmBuC27pIOx6TYvyqiYbntinmp
# Oqh/QPAnhDgexKG9GX/n1PggkGi9HCapZp8fRwg8RftwS21Ln61euBG0yONM6noD
# 2XQPrFwpm3GcuqJMf0o8LLrFkSLRQNwxPDDkWXhW+gZswbaiie5fd/W2ygcto78X
# CSPfFWveUOSZ5SqK95tBO8aTHmEa4lpJVD7HrTEn9jb1EGvxOb1cnn0CMIIGgjCC
# BGqgAwIBAgIQNsKwvXwbOuejs902y8l1aDANBgkqhkiG9w0BAQwFADCBiDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBD
# aXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVT
# RVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMjEwMzIyMDAw
# MDAwWhcNMzgwMTE4MjM1OTU5WjBXMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1w
# aW5nIFJvb3QgUjQ2MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAiJ3Y
# uUVnnR3d6LkmgZpUVMB8SQWbzFoVD9mUEES0QUCBdxSZqdTkdizICFNeINCSJS+l
# V1ipnW5ihkQyC0cRLWXUJzodqpnMRs46npiJPHrfLBOifjfhpdXJ2aHHsPHggGsC
# i7uE0awqKggE/LkYw3sqaBia67h/3awoqNvGqiFRJ+OTWYmUCO2GAXsePHi+/JUN
# Aax3kpqstbl3vcTdOGhtKShvZIvjwulRH87rbukNyHGWX5tNK/WABKf+Gnoi4cmi
# sS7oSimgHUI0Wn/4elNd40BFdSZ1EwpuddZ+Wr7+Dfo0lcHflm/FDDrOJ3rWqauU
# P8hsokDoI7D/yUVI9DAE/WK3Jl3C4LKwIpn1mNzMyptRwsXKrop06m7NUNHdlTDE
# MovXAIDGAvYynPt5lutv8lZeI5w3MOlCybAZDpK3Dy1MKo+6aEtE9vtiTMzz/o2d
# YfdP0KWZwZIXbYsTIlg1YIetCpi5s14qiXOpRsKqFKqav9R1R5vj3NgevsAsvxsA
# nI8Oa5s2oy25qhsoBIGo/zi6GpxFj+mOdh35Xn91y72J4RGOJEoqzEIbW3q0b2iP
# uWLA911cRxgY5SJYubvjay3nSMbBPPFsyl6mY4/WYucmyS9lo3l7jk27MAe145GW
# xK4O3m3gEFEIkv7kRmefDR7Oe2T1HxAnICQvr9sCAwEAAaOCARYwggESMB8GA1Ud
# IwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0GA1UdDgQWBBT2d2rdP/0BE/8W
# oWyCAi/QCj0UJTAOBgNVHQ8BAf8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB/zATBgNV
# HSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBF
# oEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRp
# ZmljYXRpb25BdXRob3JpdHkuY3JsMDUGCCsGAQUFBwEBBCkwJzAlBggrBgEFBQcw
# AYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# Dr5lQe1oRLjlocXUEYfktzsljOt+2sgXke3Y8UPEooU5y39rAARaAdAxUeiX1ktL
# J3+lgxtoLQhn5cFb3GF2SSZRX8ptQ6IvuD3wz/LNHKpQ5nX8hjsDLRhsyeIiJsms
# 9yAWnvdYOdEMq1W61KE9JlBkB20XBee6JaXx4UBErc+YuoSb1SxVf7nkNtUjPfcx
# uFtrQdRMRi/fInV/AobE8Gw/8yBMQKKaHt5eia8ybT8Y/Ffa6HAJyz9gvEOcF1VW
# XG8OMeM7Vy7Bs6mSIkYeYtddU1ux1dQLbEGur18ut97wgGwDiGinCwKPyFO7Apcm
# VJOtlw9FVJxw/mL1TbyBns4zOgkaXFnnfzg4qbSvnrwyj1NiurMp4pmAWjR+Pb/S
# IduPnmFzbSN/G8reZCL4fvGlvPFk4Uab/JVCSmj59+/mB2Gn6G/UYOy8k60mKcma
# AZsEVkhOFuoj4we8CYyaR9vd9PGZKSinaZIkvVjbH/3nlLb0a7SBIkiRzfPfS9T+
# JesylbHa1LtRV9U/7m0q7Ma2CQ/t392ioOssXW7oKLdOmMBl14suVFBmbzrt5V5c
# QPnwtd3UOTpS9oCG+ZZheiIvPgkDmA8FzPsnfXW5qHELB43ET7HHFHeRPRYrMBKj
# kb8/IN7Po0d0hQoF4TeMM+zYAJzoKQnVKOLg8pZVPT8xggSRMIIEjQIBATBpMFUx
# CzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMT
# I1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0EgUjM2AhA6UmoshM5V5h1l
# /MwS2OmJMA0GCWCGSAFlAwQCAgUAoIIB+TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcN
# AQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI0MDkxNzIxMzExNlowPwYJKoZIhvcNAQkE
# MTIEMKCyS+ITfMIOKFPJ51NwibL954T9IlLhp+YL0UgW/lIQw+iwvYMAwfHejV3z
# 4v5IXzCCAXoGCyqGSIb3DQEJEAIMMYIBaTCCAWUwggFhMBYEFPhgmBmm+4gs9+hS
# l/KhGVIaFndfMIGHBBTGrlTkeIbxfD1VEkiMacNKevnC3TBvMFukWTBXMQswCQYD
# VQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0
# aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2AhB6I67aU2mWD5HIPlz0
# x+M/MIG8BBSFPWMtk4KCYXzQkDXEkd6SwULaxzCBozCBjqSBizCBiDELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5
# MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJU
# cnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDbCsL18Gzrno7PdNsvJ
# dWgwDQYJKoZIhvcNAQEBBQAEggIALiBfb92atIOzCQq5pWk1oBBzyx2Z8BG3ApcU
# oz7Md7KLch+clPnwQcNTJvpdkSRxXU/87UJe+egU9KWi64TcyU6ReCg/KNF0v8+g
# Vs2pg2GQyGaYyo1Z8rYeik1cyAKGoSYzozcH5NiRVrsQNh77SuNNKBs4g0+ek+iB
# LfUFcum/Bg3Q453nvlAkxyRIAYIvnKepHNU7YPR02OwkDo7rMtos2fsahe0BVC6F
# vV3b5Ow9ISDSBFBnpQ7tczdQ4QGiwnbi5Ap6r0N2yipo8mIuqhMNltl/nn859lgl
# qTLBd5TQYEJ6bU6YO1yk2NYslYos7+m8K1hum+N+UkGsJ4ZDTcOqLRtY5flTBKkS
# 2dgn33EpCInD42UnltXIo2XlXfk4TCCAKDJGBw3rGddJZNlCpvTgWL1Cywzddz4A
# DOBK+eIAXsJlfga91veFV3H/C9vlm6vxuTEowIyVky5+M0S4b7cKdkEa+40rlzf0
# WYdQwS/6t2IxfNsAIOnepj3eI/nJZa1Y28KirVt0hcUoz+jupW1JWq8uuVc2IYhg
# hUIUOvGDQKK/l/0/SXInbLuqal7j7XYba5OyEr8J4Q1voY1JqlOsfoj9mGEGOTL0
# 88fuQnG3z0TgQfN+fsKYx4E4SwXr523D9MN9i8icXK+ceqqExCamWo6pWFkWqIGX
# 3ffqbGk=
# SIG # End signature block
