#ifndef va_start
#include <stdarg.h> 
#endif

#define HWLOG_STR_INTERFACE "hwi"
#define HWLOG_STR_GENLOG "glog"


FILE* HW_log = NULL;
#ifndef NOPYTHON
static PyObject *HW_PyLog = NULL;
static PyObject *PyLog_Level[MAX_LOG_LEVEL_COUNT] = { NULL };
static PyObject *PyLog_Flags[MAX_FLAG_COUNT] = { NULL };
#else
static int *HW_PyLog = NULL;
static int *PyLog_Level[MAX_LOG_LEVEL_COUNT] = { NULL };
static int *PyLog_Flags[MAX_FLAG_COUNT] = { NULL };
#endif
uint32_t loggingToFile;

/*! True if HW_log points to a file handle that can be closed (i.e. not stdout). */
#ifdef REGLIBLOGGING

#define HW_LOGGING_MAX_PART_NUMBER_MAPPINGS 1000

/*! A mapping from (adapterNumber, PHY_ID) pair to PartNumber */
typedef struct
{
  uint8_t validMapping;
  uint32_t adapterNumber;
  uint32_t PHY_ID;
  PartNumber pn;
} HW_Logging_PartNumberMapping;

HW_Logging_PartNumberMapping HW_Logging_PartNumbers[HW_LOGGING_MAX_PART_NUMBER_MAPPINGS]; 

/*! Register a mapping from (adapterNumber, PHY_ID) pair to PartNumber.  Return
 * 1 on success, 0 on failure. */
uint32_t HW_Logging_SetPartNumberMapping
(    
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The part number to use when looking up RegLib entries. */
  PartNumber pn
)
{
  uint32_t i;
  HW_Logging_PartNumberMapping* mapping;

  for (i = 0; i < HW_LOGGING_MAX_PART_NUMBER_MAPPINGS; i++)
  {
    mapping = &(HW_Logging_PartNumbers[i]);
    if ((!mapping->validMapping) || 
        (mapping->adapterNumber == adapterNumber && mapping->PHY_ID == PHY_ID))
    {
      mapping->validMapping = 1;
      mapping->adapterNumber = adapterNumber;
      mapping->PHY_ID = PHY_ID;
      mapping->pn = pn;
      return 1;
    }
  }

  return 0;
}

/*! Get the part number that is mapped to this (adapterNumber, PHY_ID) pair. 
 * If none, the INVALID will be returned. */
PartNumber HW_Logging_GetPartNumber
(    
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID
)
{
  uint32_t i;
  HW_Logging_PartNumberMapping* mapping;

  for (i = 0; i < HW_LOGGING_MAX_PART_NUMBER_MAPPINGS; i++)
  {
    mapping = &(HW_Logging_PartNumbers[i]);
    if (mapping->validMapping)  
    {
      if (mapping->adapterNumber == adapterNumber && mapping->PHY_ID == PHY_ID)
        return mapping->pn; /* Found a mapping */
      /*else, Keep looking.. */
    }
    else
      /* We passed the range of valid mappings, so fail. */
      return INVALID;
  }

  /* We looked at all the mappings and didn't find a match, so fail. */
  return INVALID;
}

#endif 
/* #endif REGLIBLOGGING */

/*! Initialize the logging stuff.  Will reset the part number mappings. */
void HW_Logging_Initialize()
{
#ifdef REGLIBLOGGING
  uint32_t i;

  for (i = 0; i < HW_LOGGING_MAX_PART_NUMBER_MAPPINGS; i++)
    HW_Logging_PartNumbers[i].validMapping = 0;
#endif 
}

void HW_Logging_WriteColumnHeaders()
{
  if (HW_log)
  {
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "adapter,phy,op,mmd,address,value,");
  #ifdef REGLIBLOGGING
    #ifdef EXTRA_READ_BEFORE_WRITE
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "BF non-default,BF changed,BF name,BF hi,BF lo,BF default val,BF val,BF prev val");
    #else
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "BF non-default,BF name,BF hi,BF lo,BF default val,BF val");
    #endif 
  #endif 
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "");
  }
}


/*! Enable logging of all operations (MDIO read, MDIO write etc.) to standard out. */
void HW_EnableLoggingStdout ()
{
  HW_DisableLogging(0);

  HW_log = stdout;
  HW_PyLog = NULL;
  loggingToFile = 0;
  HW_Logging_WriteColumnHeaders();
}


/*! Enable logging of all operations (MDIO read, MDIO write etc.) to standard error. */
void HW_EnableLoggingStderr ()
{
  HW_DisableLogging(0);

  HW_log = stderr;
  HW_PyLog = NULL;
  loggingToFile = 0;
  HW_Logging_WriteColumnHeaders();
}


/*! Enable logging of all operations (MDIO read, MDIO write etc.) to Python logger. */
#ifndef NOPYTHON
void HW_EnablePyLogging( PyObject *logger, PyObject *levels)
{
    size_t i, levelCount = 0; 
    HW_DisableLogging( 0 );

    levelCount = PyTuple_GET_SIZE( levels );
    for( i = 0; i < levelCount && i < MAX_LOG_LEVEL_COUNT; i++ )
    {
        PyLog_Level[i] = PyTuple_GET_ITEM( levels, i );
        Py_INCREF( PyLog_Level[i] );
    }

	PyLog_Flags[HWLOG_INTERFACE] = PyString_FromString(HWLOG_STR_INTERFACE);
	PyLog_Flags[HWLOG_DEVICE_CONTROL] = PyString_FromString(HWLOG_STR_DEVICE_CONTROL);
	PyLog_Flags[HWLOG_GENLOG] = PyString_FromString(HWLOG_STR_GENLOG);

    Py_INCREF( logger );
    HW_PyLog = logger;
}
#endif

/*! Enable logging of all operations (MDIO read, MDIO write etc.) to f. */
void HW_EnableLogging (FILE* f)
{
  HW_DisableLogging(0);

  HW_log = f;
  HW_PyLog = NULL;
  loggingToFile = 1;
  HW_Logging_WriteColumnHeaders();
}


/*! Disable logging. */
void HW_DisableLogging (int close)
{
    uint32_t  i;
#ifndef NOPYTHON
    PyObject *pyLog = HW_PyLog;
#endif

    HW_PyLog = NULL;
    HW_log = NULL;
    loggingToFile = 0;

    for( i = 0; i < MAX_LOG_LEVEL_COUNT; i++ )
    {
        if( PyLog_Level[i] == NULL )
        {
            continue;
        }
#ifndef NOPYTHON
        Py_DECREF( PyLog_Level[i] );
#endif
        PyLog_Level[i] = NULL;
    }

    if( close && HW_log && loggingToFile ) fclose( HW_log );
#ifndef NOPYTHON
    if( pyLog ) Py_DECREF( pyLog );
#endif
}

void HW_Log(uint32_t level, uint32_t flags, const char *format, ...)
{
    va_list args;

    // retrieve the variable arguments
    va_start( args, format );

    if( HW_log != NULL )
    {
#ifdef WIN32
        vfprintf_s( HW_log, format, args );
        fprintf_s( HW_log, "\n" );
#else
        vfprintf( HW_log, format, args );
        fprintf( HW_log, "\n" );
#endif
    }
#ifndef NOPYTHON
    else if( HW_PyLog != NULL )
    {
        size_t  len;
        char    buffer[MAX_LOG_LINE_SIZE];
        PyObject *logString, *logTupple, *logLevel, *logFlags;
        PyGILState_STATE gstate;

        //len = _vscprintf( format, args ) // _vscprintf doesn't count
        //    + 1; // terminating '\0'

        //buffer = (char*) malloc( len * sizeof( char ) );

#ifdef WIN32
        len = vsprintf_s( buffer, MAX_LOG_LINE_SIZE, format, args );
#else
        len = vsnprintf( buffer, MAX_LOG_LINE_SIZE, format, args );
#endif
        logString = PyString_FromStringAndSize( buffer, len );
		if (logString != NULL)
		{
			if (level >= MAX_LOG_LEVEL_COUNT)
			{
				level = MAX_LOG_LEVEL_COUNT - 1;
			}

			logLevel = PyLog_Level[level];
			logFlags = PyLog_Flags[flags];
			logTupple = PyTuple_Pack(3, logLevel, logFlags, logString);
			if (logTupple != NULL)
			{
				gstate = PyGILState_Ensure();

				PyObject_CallObject(HW_PyLog, logTupple);

				PyGILState_Release(gstate);

				Py_DECREF(logTupple);
			}
			Py_DECREF(logString);
		}
        //free( buffer );
    }
#endif
    else
    {
    }
}

/*! If logging is enable, print msg (and a newline) to the log. */
void HW_LogMsg (const char* msg)
{
    if( HW_log || HW_PyLog )
  {
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%s", msg);
    fflush(HW_log);
  }
}


/*! If logging is enabled, write a log message describing an MDIO write operation. */
void HW_Logging_MDIOWrite
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
  ,
    /*! The value of the register prior to the write. */
  uint32_t previousData
#endif
)
{
#ifdef REGLIBLOGGING
  PartNumber pn;
  uint32_t defaultRegVal, i;
  RegisterDefinition defaultRegDef, writtenValRegDef;
  BitField* bf; 
  #ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
  #endif
#endif

    if( HW_log || HW_PyLog )
  {
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,MW,%02X,%04X,%04X", adapterNumber, PHY_ID, MMD, address, data);
  #ifdef REGLIBLOGGING
    pn = HW_Logging_GetPartNumber(adapterNumber, PHY_ID);
    if (pn != INVALID)
    {
      /* Check if this register is defined for the relevant part number. */
      getRegisterState(pn, MMD, address, 0, &defaultRegDef);
      if (defaultRegDef.registerDefinitionState == Defined)
      {
        /* Get the default value, and the register definition for the default value. */
        defaultRegVal = getDefaultValue(&defaultRegDef);
        getRegisterState(pn, MMD, address, defaultRegVal, &defaultRegDef);

        /* Get the register definition for the written value. */
        getRegisterState(pn, MMD, address, data, &writtenValRegDef);

      #ifdef EXTRA_READ_BEFORE_WRITE
        /* Get the register definition for the previous value. */
        getRegisterState(pn, MMD, address, previousData, &previousValRefDef);
      #endif

        /* Write a line for each bitfield to the log. */
        for (i = 0; i < writtenValRegDef.numberOfBitFields; i++)
        {
          bf = &(writtenValRegDef.bitField[i]);
        #ifdef EXTRA_READ_BEFORE_WRITE
          previousBf = &(previousValRefDef.bitField[i]);
        #endif  

        #ifdef EXTRA_READ_BEFORE_WRITE
          HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%02X,%04X,%04X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
        #else
          HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%02X,%04X,%04X,%c,%s,%X,%X,0x%X,0x%X",
        #endif
              MMD, 
              address, 
              data,
              (bf->defaultValue == bf->currentValue) ? ' ' : '*',
            #ifdef EXTRA_READ_BEFORE_WRITE
              (previousBf->currentValue == bf->currentValue) ? ' ' : '*',
            #endif
              bf->fieldName,
              bf->startBit,
              bf->stopBit,
              bf->defaultValue,
              bf->currentValue
            #ifdef EXTRA_READ_BEFORE_WRITE
              ,previousBf->currentValue
            #endif
            );
        }
      }
    }
  #endif
  }
}

/*! If logging is enabled, write a log message describing an MDIO read operation. */
void HW_Logging_MDIORead
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being read from to.*/
  uint32_t address,
    /*! The 16-bits of data read with the read operation.*/
  uint32_t data

)
{
#ifdef REGLIBLOGGING
  PartNumber pn;
  uint32_t defaultRegVal, i;
  RegisterDefinition defaultRegDef, readValRegDef;
  BitField* bf;
#endif

  if( HW_log || HW_PyLog )
  {
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,MR,%02X,%04X,%04X", adapterNumber, PHY_ID, MMD, address, data);
  #ifdef REGLIBLOGGING
    pn = HW_Logging_GetPartNumber(adapterNumber, PHY_ID);
    if (pn != INVALID)
    {
      /* Check if this register is defined for the relevant part number. */
      getRegisterState(pn, MMD, address, 0, &defaultRegDef);
      if (defaultRegDef.registerDefinitionState == Defined)
      {
        /* Get the default value, and the register definition for the default value. */
        defaultRegVal = getDefaultValue(&defaultRegDef);
        getRegisterState(pn, MMD, address, defaultRegVal, &defaultRegDef);

        /* Get the register definition for the read value. */
        getRegisterState(pn, MMD, address, data, &readValRegDef);

        /* Write a line for each bitfield to the log. */
        for (i = 0; i < readValRegDef.numberOfBitFields; i++)
        {
          bf = &(readValRegDef.bitField[i]);
        #ifdef EXTRA_READ_BEFORE_WRITE
          HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%02X,%04X,%04X,%c,,%s,%X,%X,0x%X,0x%X,",
        #else
          HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%02X,%04X,%04X,%c,%s,%X,%X,0x%X,0x%X",
        #endif
              MMD, 
              address, 
              data,
              (bf->defaultValue == bf->currentValue) ? ' ' : '*',
              bf->fieldName,
              bf->startBit,
              bf->stopBit,
              bf->defaultValue,
              bf->currentValue);
        }
      }
    }
  #endif
  }
}

/*! If logging is enabled, write a log message describing an asynchronous MDIO read request operation. */
void HW_Logging_BlockMDIORead
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being read from.*/
  uint32_t address
)
{
    if( HW_log || HW_PyLog )
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AR,%02X,%04X,", adapterNumber, PHY_ID, MMD, address);
}

/*! If logging is enabled, write a log message describing an unflushed MDIO write operation. */
void HW_Logging_BlockMDIOWrite
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bit data to write.*/
  uint32_t data
)
{
    if( HW_log || HW_PyLog )
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AW,%02X,%04X,%04X", adapterNumber, PHY_ID, MMD, address, data);
}

/*! If logging is enabled, write a log message describing an asynchronous MDIO read fetch operation. */
void HW_Logging_BlockMDIOFetch
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! If we're logging the execution of a write operation, set to 1.  Otherwise, 0. */
  uint32_t isWrite,
    /*! The 16-bit address being written to.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data read with the fetch operation.*/
  uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
  ,
    /*! The value of the register prior to the write. */
  uint32_t previousData
#endif
)
{
	UNREFERENCED_PARAMETER(address);
	UNREFERENCED_PARAMETER(MMD);
	UNREFERENCED_PARAMETER(isWrite);

#ifdef REGLIBLOGGING
  PartNumber pn;
  uint32_t defaultRegVal, i;
  RegisterDefinition defaultRegDef, dataValRegDef;
  BitField* bf; 
  uint32_t doWriteBitfields;
  #ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
  #endif
#endif
  
    if( HW_log || HW_PyLog )
  {
  #ifndef REGLIBLOGGING
    /* FIXME: We really should be able to log AFW/AFR even if 
     * REGLIBLOGGING is undefined. */
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AF,,,%04X",
        adapterNumber, 
        PHY_ID, 
        /*isWrite ? "AFW" : "AFR",
        MMD, 
        address, */
        data);
    
  #else
    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,%s,%02X,%04X,%04X",
        adapterNumber, 
        PHY_ID, 
        isWrite ? "AFW" : "AFR",
        MMD, 
        address, 
        data);

    /* For AW operations, we always know the written value data and so can  
     * write the bitfields if we know the part number.
     * For AR operations, we only know the register value if 
     * EXTRA_READ_BEFORE_WRITE is defined. */
    pn = HW_Logging_GetPartNumber(adapterNumber, PHY_ID);
    if (pn == INVALID)
      doWriteBitfields = 0;
    else
    {
      if (isWrite) doWriteBitfields = 1;
      else
        #ifdef EXTRA_READ_BEFORE_WRITE
          doWriteBitfields = 1;
        #else
          doWriteBitfields = 0;
        #endif
    }

    if (doWriteBitfields)
    {
      /* Check if this register is defined for the relevant part number. */
      getRegisterState(pn, MMD, address, 0, &defaultRegDef);
      if (defaultRegDef.registerDefinitionState == Defined)
      {
        /* Get the default value, and the register definition for the default value. */
        defaultRegVal = getDefaultValue(&defaultRegDef);
        getRegisterState(pn, MMD, address, defaultRegVal, &defaultRegDef);

        /* Get the register definition for the data value. */
        getRegisterState(pn, MMD, address, data, &dataValRegDef);

      #ifdef EXTRA_READ_BEFORE_WRITE
        /* Get the register definition for the previous value. */
        getRegisterState(pn, MMD, address, previousData, &previousValRefDef);
      #endif

        /* Write a line for each bitfield to the log. */
        for (i = 0; i < dataValRegDef.numberOfBitFields; i++)
        {
          bf = &(dataValRegDef.bitField[i]);
        #ifdef EXTRA_READ_BEFORE_WRITE
          previousBf = &(previousValRefDef.bitField[i]);
        #endif  

        #ifdef EXTRA_READ_BEFORE_WRITE
          HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%02X,%04X,%04X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
        #else
          HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%02X,%04X,%04X,%c,%s,%X,%X,0x%X,0x%X",
        #endif
              MMD, 
              address, 
              data,
              (bf->defaultValue == bf->currentValue) ? ' ' : '*',
            #ifdef EXTRA_READ_BEFORE_WRITE
              (previousBf->currentValue == bf->currentValue) ? ' ' : '*',
            #endif
              bf->fieldName,
              bf->startBit,
              bf->stopBit,
              bf->defaultValue,
              bf->currentValue
            #ifdef EXTRA_READ_BEFORE_WRITE
              ,previousBf->currentValue
            #endif
            );
        }
      }
    }
  #endif
  }
}

/*! If logging is enabled, write a log message describing an MDIO write operation. */
void HW_Logging_Write32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
,
/*! The value of the register prior to the write. */
uint32_t previousData
#endif
)
{
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint32_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, writtenValRegDef;
    BitField* bf;
#ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
#endif
#endif

    if( HW_log || HW_PyLog )
    {
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,MW,%06X,%08X", adapterNumber, DEV_ID, address, data );
#ifdef REGLIBLOGGING
        pn = HW_Logging_GetPartNumber( adapterNumber, DEV_ID );
        if( pn != INVALID )
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState32( pn, address, 0, &defaultRegDef );
            if( defaultRegDef.registerDefinitionState == Defined )
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue( &defaultRegDef );
                getRegisterState32( pn, address, defaultRegVal, &defaultRegDef );

                /* Get the register definition for the written value. */
                getRegisterState32( pn, address, data, &writtenValRegDef );

#ifdef EXTRA_READ_BEFORE_WRITE
                /* Get the register definition for the previous value. */
                getRegisterState32( pn, address, previousData, &previousValRefDef );
#endif

                /* Write a line for each bitfield to the log. */
                for( i = 0; i < writtenValRegDef.numberOfBitFields; i++ )
                {
                    bf = &(writtenValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    previousBf = &(previousValRefDef.bitField[i]);
#endif  

#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
#else
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                             address,
                             data,
                             (bf->defaultValue == bf->currentValue) ? ' ' : '*',
#ifdef EXTRA_READ_BEFORE_WRITE
                             ( previousBf->currentValue == bf->currentValue ) ? ' ' : '*',
#endif
                             bf->fieldName,
                             bf->startBit,
                             bf->stopBit,
                             bf->defaultValue,
                             bf->currentValue
#ifdef EXTRA_READ_BEFORE_WRITE
                             , previousBf->currentValue
#endif
                             );
                }
            }
        }
#endif
    }
}

/*! If logging is enabled, write a log message describing an MDIO read operation. */
void HW_Logging_Read32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being read from to.*/
uint32_t address,
/*! The 32-bits of data read with the read operation.*/
uint32_t data

)
{
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint32_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, readValRegDef;
    BitField* bf;
#endif

    if( HW_log || HW_PyLog )
    {
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,MR,%06X,%08X", adapterNumber, DEV_ID, address, data );
#ifdef REGLIBLOGGING
        pn = HW_Logging_GetPartNumber( adapterNumber, DEV_ID );
        if( pn != INVALID )
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState32( pn, address, 0, &defaultRegDef );
            if( defaultRegDef.registerDefinitionState == Defined )
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue( &defaultRegDef );
                getRegisterState32( pn, address, defaultRegVal, &defaultRegDef );

                /* Get the register definition for the read value. */
                getRegisterState32( pn, address, data, &readValRegDef );

                /* Write a line for each bitfield to the log. */
                for( i = 0; i < readValRegDef.numberOfBitFields; i++ )
                {
                    bf = &(readValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,,%s,%X,%X,0x%X,0x%X,",
#else
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                             address,
                             data,
                             (bf->defaultValue == bf->currentValue) ? ' ' : '*',
                             bf->fieldName,
                             bf->startBit,
                             bf->stopBit,
                             bf->defaultValue,
                             bf->currentValue );
                }
            }
        }
#endif
    }
}

/*! If logging is enabled, write a log message describing an asynchronous MDIO read request operation. */
void HW_Logging_BlockRead32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being read from.*/
uint32_t address
)
{
    if( HW_log || HW_PyLog )
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AR,,%02X,", adapterNumber, DEV_ID, address );
}

/*! If logging is enabled, write a log message describing an unflushed MDIO write operation. */
void HW_Logging_BlockWrite32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 32-bit data to write.*/
uint32_t data
)
{
    if( HW_log || HW_PyLog )
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AW,%06X,%08X", adapterNumber, DEV_ID, address, data );
}

/*! If logging is enabled, write a log message describing an asynchronous MDIO read fetch operation. */
void HW_Logging_BlockFetch32
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! If we're logging the execution of a write operation, set to 1.  Otherwise, 0. */
uint32_t isWrite,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 32-bits of data read with the fetch operation.*/
uint32_t data
#ifdef EXTRA_READ_BEFORE_WRITE
,
/*! The value of the register prior to the write. */
uint32_t previousData
#endif
)
{
#ifdef WIN32
    address;isWrite;
#endif
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint32_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, dataValRegDef;
    BitField* bf;
    uint32_t doWriteBitfields;
#ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
#endif
#endif

    if( HW_log || HW_PyLog )
    {
#ifndef REGLIBLOGGING
        /* FIXME: We really should be able to log AFW/AFR even if
        * REGLIBLOGGING is undefined. */
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AF,,,%08X",
                 adapterNumber,
                 DEV_ID,
                 /*isWrite ? "AFW" : "AFR",
                 MMD,
                 address, */
                 data );

#else
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,%s,%06X,%08X",
                 adapterNumber,
                 DEV_ID,
                 isWrite ? "AFW" : "AFR",
                 address,
                 data );

        /* For AW operations, we always know the written value data and so can
        * write the bitfields if we know the part number.
        * For AR operations, we only know the register value if
        * EXTRA_READ_BEFORE_WRITE is defined. */
        pn = HW_Logging_GetPartNumber( adapterNumber, DEV_ID );
        if( pn == INVALID )
            doWriteBitfields = 0;
        else
        {
            if( isWrite ) doWriteBitfields = 1;
            else
#ifdef EXTRA_READ_BEFORE_WRITE
                doWriteBitfields = 1;
#else
                doWriteBitfields = 0;
#endif
        }

        if( doWriteBitfields )
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState32( pn, address, 0, &defaultRegDef );
            if( defaultRegDef.registerDefinitionState == Defined )
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue( &defaultRegDef );
                getRegisterState32( pn, address, defaultRegVal, &defaultRegDef );

                /* Get the register definition for the data value. */
                getRegisterState32( pn, address, data, &dataValRegDef );

#ifdef EXTRA_READ_BEFORE_WRITE
                /* Get the register definition for the previous value. */
                getRegisterState32( pn, address, previousData, &previousValRefDef );
#endif

                /* Write a line for each bitfield to the log. */
                for( i = 0; i < dataValRegDef.numberOfBitFields; i++ )
                {
                    bf = &(dataValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    previousBf = &(previousValRefDef.bitField[i]);
#endif  

#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
#else
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                             address,
                             data,
                             (bf->defaultValue == bf->currentValue) ? ' ' : '*',
#ifdef EXTRA_READ_BEFORE_WRITE
                             ( previousBf->currentValue == bf->currentValue ) ? ' ' : '*',
#endif
                             bf->fieldName,
                             bf->startBit,
                             bf->stopBit,
                             bf->defaultValue,
                             bf->currentValue
#ifdef EXTRA_READ_BEFORE_WRITE
                             , previousBf->currentValue
#endif
                             );
                }
            }
        }
#endif
    }
}
/*! If logging is enabled, write a log message describing an MDIO write operation. */
void HW_Logging_Write64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
uint64_t data
#ifdef EXTRA_READ_BEFORE_WRITE
,
/*! The value of the register prior to the write. */
uint64_t previousData
#endif
)
{
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint64_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, writtenValRegDef;
    BitField* bf;
#ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
#endif
#endif

    if( HW_log || HW_PyLog )
    {
        HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,MW,%06X,%016llX", adapterNumber, DEV_ID, address, data );
#ifdef REGLIBLOGGING
        pn = HW_Logging_GetPartNumber( adapterNumber, DEV_ID );
        if( pn != INVALID )
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState64( pn, address, 0, &defaultRegDef );
            if( defaultRegDef.registerDefinitionState == Defined )
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue( &defaultRegDef );
                getRegisterState64( pn, address, defaultRegVal, &defaultRegDef );

                /* Get the register definition for the written value. */
                getRegisterState64( pn, address, data, &writtenValRegDef );

#ifdef EXTRA_READ_BEFORE_WRITE
                /* Get the register definition for the previous value. */
                getRegisterState64( pn, address, previousData, &previousValRefDef );
#endif

                /* Write a line for each bitfield to the log. */
                for( i = 0; i < writtenValRegDef.numberOfBitFields; i++ )
                {
                    bf = &(writtenValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    previousBf = &(previousValRefDef.bitField[i]);
#endif  

#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
#else
                    HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                            address,
                            data,
                            (bf->defaultValue == bf->currentValue) ? ' ' : '*',
#ifdef EXTRA_READ_BEFORE_WRITE
                            ( previousBf->currentValue == bf->currentValue ) ? ' ' : '*',
#endif
                            bf->fieldName,
                            bf->startBit,
                            bf->stopBit,
                            bf->defaultValue,
                            bf->currentValue
#ifdef EXTRA_READ_BEFORE_WRITE
                            , previousBf->currentValue
#endif
                            );
                }
            }
        }
#endif
    }
}

/*! If logging is enabled, write a log message describing an MDIO read operation. */
void HW_Logging_Read64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being read from to.*/
uint32_t address,
/*! The 64-bits of data read with the read operation.*/
uint64_t data

)
{
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint64_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, readValRegDef;
    BitField* bf;
#endif

    if( HW_log || HW_PyLog )
    {
        HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,MR,%06X,%016llX", adapterNumber, DEV_ID, address, data );
#ifdef REGLIBLOGGING
        pn = HW_Logging_GetPartNumber( adapterNumber, DEV_ID );
        if( pn != INVALID )
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState64( pn, address, 0, &defaultRegDef );
            if( defaultRegDef.registerDefinitionState == Defined )
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue( &defaultRegDef );
                getRegisterState64( pn, address, defaultRegVal, &defaultRegDef );

                /* Get the register definition for the read value. */
                getRegisterState64( pn, address, data, &readValRegDef );

                /* Write a line for each bitfield to the log. */
                for( i = 0; i < readValRegDef.numberOfBitFields; i++ )
                {
                    bf = &(readValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,,%s,%X,%X,0x%X,0x%X,",
#else
                    HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                            address,
                            data,
                            (bf->defaultValue == bf->currentValue) ? ' ' : '*',
                            bf->fieldName,
                            bf->startBit,
                            bf->stopBit,
                            bf->defaultValue,
                            bf->currentValue );
                }
            }
        }
#endif
    }
}

/*! If logging is enabled, write a log message describing an asynchronous MDIO read request operation. */
void HW_Logging_BlockRead64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being read from.*/
uint32_t address
)
{
    if( HW_log || HW_PyLog )
        HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AR,,%02X,", adapterNumber, DEV_ID, address );
}

/*! If logging is enabled, write a log message describing an unflushed MDIO write operation. */
void HW_Logging_BlockWrite64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 64-bit data to write.*/
uint64_t data
)
{
    if( HW_log || HW_PyLog )
        HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AW,%06X,%016llX", adapterNumber, DEV_ID, address, data );
}

/*! If logging is enabled, write a log message describing an asynchronous MDIO read fetch operation. */
void HW_Logging_BlockFetch64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! If we're logging the execution of a write operation, set to 1.  Otherwise, 0. */
uint32_t isWrite,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 64-bits of data read with the fetch operation.*/
uint64_t data
#ifdef EXTRA_READ_BEFORE_WRITE
,
/*! The value of the register prior to the write. */
uint64_t previousData
#endif
)
{
#ifdef WIN32
    address; isWrite;
#endif
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint64_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, dataValRegDef;
    BitField* bf;
    uint64_t doWriteBitfields;
#ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
#endif
#endif

    if( HW_log || HW_PyLog )
    {
#ifndef REGLIBLOGGING
        /* FIXME: We really should be able to log AFW/AFR even if
        * REGLIBLOGGING is undefined. */
        HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,AF,,,%016llX",
                adapterNumber,
                DEV_ID,
                /*isWrite ? "AFW" : "AFR",
                MMD,
                address, */
                data );

#else
        HW_Log( HWLOG_DEBUG, HWLOG_GENLOG, "%u,%u,%s,%06X,%016llX",
                adapterNumber,
                DEV_ID,
                isWrite ? "AFW" : "AFR",
                address,
                data );

        /* For AW operations, we always know the written value data and so can
        * write the bitfields if we know the part number.
        * For AR operations, we only know the register value if
        * EXTRA_READ_BEFORE_WRITE is defined. */
        pn = HW_Logging_GetPartNumber( adapterNumber, DEV_ID );
        if( pn == INVALID )
            doWriteBitfields = 0;
        else
        {
            if( isWrite ) doWriteBitfields = 1;
            else
#ifdef EXTRA_READ_BEFORE_WRITE
                doWriteBitfields = 1;
#else
                doWriteBitfields = 0;
#endif
        }

        if( doWriteBitfields )
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState64( pn, address, 0, &defaultRegDef );
            if( defaultRegDef.registerDefinitionState == Defined )
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue( &defaultRegDef );
                getRegisterState64( pn, address, defaultRegVal, &defaultRegDef );

                /* Get the register definition for the data value. */
                getRegisterState64( pn, address, data, &dataValRegDef );

#ifdef EXTRA_READ_BEFORE_WRITE
                /* Get the register definition for the previous value. */
                getRegisterState64( pn, address, previousData, &previousValRefDef );
#endif

                /* Write a line for each bitfield to the log. */
                for( i = 0; i < dataValRegDef.numberOfBitFields; i++ )
                {
                    bf = &(dataValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    previousBf = &(previousValRefDef.bitField[i]);
#endif  

#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log( LOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
#else
                    HW_Log( LOG_DEBUG, HWLOG_GENLOG, ",,,%06X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                            address,
                            data,
                            (bf->defaultValue == bf->currentValue) ? ' ' : '*',
#ifdef EXTRA_READ_BEFORE_WRITE
                            ( previousBf->currentValue == bf->currentValue ) ? ' ' : '*',
#endif
                            bf->fieldName,
                            bf->startBit,
                            bf->stopBit,
                            bf->defaultValue,
                            bf->currentValue
#ifdef EXTRA_READ_BEFORE_WRITE
                            , previousBf->currentValue
#endif
                            );
                }
            }
        }
#endif
    }
}

/*! If logging is enabled, write a log message describing an MacConfigSpace write operation. */
void HW_Logging_MacConfigSpaceWrite
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
    uint32_t adapterNumber,
    /*! The 32-bit offset being written to.*/
    uint32_t offset,
    /*! Mask of used bits of data being sent out with the write command.*/
    uint32_t mask,
    /*! The 32-bits of data being sent out with the write command.*/
    uint32_t value
#ifdef EXTRA_READ_BEFORE_WRITE
    ,
    /*! The value of the register prior to the write. */
    uint32_t previousData
#endif
)
{
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint32_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, writtenValRegDef;
    BitField* bf;
#ifdef EXTRA_READ_BEFORE_WRITE
    RegisterDefinition previousValRefDef;
    BitField* previousBf;
#endif
#endif

    if( HW_log || HW_PyLog )
    {
		HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,CW,%08X,%08X,%08X", adapterNumber, offset, mask, value);
#ifdef REGLIBLOGGING
        pn = HW_Logging_GetPartNumber(adapterNumber);
        if (pn != INVALID)
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState(pn, offset, 0, &defaultRegDef);
            if (defaultRegDef.registerDefinitionState == Defined)
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue(&defaultRegDef);
                getRegisterState(pn, offset, defaultRegVal, &defaultRegDef);

                /* Get the register definition for the written value. */
                getRegisterState(pn, offset, value, &writtenValRegDef);

#ifdef EXTRA_READ_BEFORE_WRITE
                /* Get the register definition for the previous value. */
                getRegisterState(pn, offset, previousData, &previousValRefDef);
#endif

                /* Write a line for each bitfield to the log. */
                for (i = 0; i < writtenValRegDef.numberOfBitFields; i++)
                {
                    bf = &(writtenValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    previousBf = &(previousValRefDef.bitField[i]);
#endif  

#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%08X,%08X,%c,%c,%s,%X,%X,0x%X,0x%X,0x%X",
#else
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%08X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                        offset,
                        value,
                        (bf->defaultValue == bf->currentValue) ? ' ' : '*',
#ifdef EXTRA_READ_BEFORE_WRITE
                        (previousBf->currentValue == bf->currentValue) ? ' ' : '*',
#endif
                        bf->fieldName,
                        bf->startBit,
                        bf->stopBit,
                        bf->defaultValue,
                        bf->currentValue
#ifdef EXTRA_READ_BEFORE_WRITE
                        , previousBf->currentValue
#endif
                        );
                }
            }
        }
#endif
    }
}

/*! If logging is enabled, write a log message describing an MacConfigSpace read operation. */
void HW_Logging_MacConfigSpaceRead
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The 32-bit offset being read from to.*/
uint32_t offset,
/*! The 32-bits of data read with the read operation.*/
uint32_t value
)
{
#ifdef REGLIBLOGGING
    PartNumber pn;
    uint32_t defaultRegVal, i;
    RegisterDefinition defaultRegDef, readValRegDef;
    BitField* bf;
#endif

    if( HW_log || HW_PyLog )
    {
		HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,CR,%08X,%08X", adapterNumber, offset, value);
#ifdef REGLIBLOGGING
        pn = HW_Logging_GetPartNumber(adapterNumber);
        if (pn != INVALID)
        {
            /* Check if this register is defined for the relevant part number. */
            getRegisterState(pn, offset, 0, &defaultRegDef);
            if (defaultRegDef.registerDefinitionState == Defined)
            {
                /* Get the default value, and the register definition for the default value. */
                defaultRegVal = getDefaultValue(&defaultRegDef);
                getRegisterState(pn, offset, defaultRegVal, &defaultRegDef);

                /* Get the register definition for the read value. */
                getRegisterState(pn, offset, value, &readValRegDef);

                /* Write a line for each bitfield to the log. */
                for (i = 0; i < readValRegDef.numberOfBitFields; i++)
                {
                    bf = &(readValRegDef.bitField[i]);
#ifdef EXTRA_READ_BEFORE_WRITE
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%08X,%08X,%c,,%s,%X,%X,0x%X,0x%X,",
#else
                    HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, ",,,%08X,%08X,%c,%s,%X,%X,0x%X,0x%X",
#endif
                        offset,
                        value,
                        (bf->defaultValue == bf->currentValue) ? ' ' : '*',
                        bf->fieldName,
                        bf->startBit,
                        bf->stopBit,
                        bf->defaultValue,
                        bf->currentValue);
                }
            }
        }
#endif
    }
}
/*! If logging is enabled, write a log message describing an asynchronous MacConfigSpace read request operation. */
void HW_Logging_BlockMacConfigSpaceRead
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 32-bit address being read from.*/
  uint32_t offset
)
{
    if( HW_log || HW_PyLog )
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,ACR,%08X,", adapterNumber, offset);
}

/*! If logging is enabled, write a log message describing an unflushed MacConfigSpace write operation. */
void HW_Logging_BlockMacConfigSpaceWrite
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 32-bit address being written to.*/
  uint32_t offset,
  /*! Mask of used bits of data being sent out with the write command.*/
  uint32_t mask,
  /*! The 32-bit value to write.*/
  uint32_t value
)
{
    if( HW_log || HW_PyLog )
        HW_Log(HWLOG_DEBUG, HWLOG_GENLOG, "%u,ACW,%08X,%08X,%08X", adapterNumber, offset, mask, value);
}
