# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetSetting
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $DisplayName = '',
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object]
    $DisplayValue = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $RegistryKeyword = '',
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object]
    $RegistryValue = $null,
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Setting = $null,
    [parameter(Mandatory = $false)]
    [switch]
    $Reset
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesSet = @()
        $script:WarningMessagesSet = @()
        $FinalObject = @()
        GetIntelEthernetDevices @('icei*', 'i40ei*')
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
        $script:MSNetAdvProperty = @(Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue)
    }
    Process
    {
        $Adapters = $Adapter

        if (IsSettingObject $Adapters)
        {
            $Setting = $Adapters
            $Adapters = $null
        }

        do
        {
            if (!(ValidateParams $DisplayName $DisplayValue $RegistryKeyword $RegistryValue $Setting $AdapterName $Adapters))
            {
                break
            }

            if ($null -ne $Setting)
            {
                $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Setting.Name $null ([ref]$script:ErrorMessagesSet)
                $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet) | Get-Unique)
            }
            else
            {
                $PreProcessedAdapterNames = ValidateSetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesSet)
                $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))
            }

            try
            {
                $PreErrorActionPreference = $global:ErrorActionPreference
                $global:ErrorActionPreference = 'SilentlyContinue'
                foreach ($a in $AdapterNamesArray)
                {
                    $TmpStatusMsg = CheckDeviceError $a
                    if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                    {
                        $script:ErrorMessagesSet += $TmpStatusMsg
                    }

                    $script:isProfileSetting = $false
                    $AdapterSettings = @()
                    if ($null -ne $Setting)
                    {
                        $AdapterSettings = GetSetSettings $null $Setting.RegistryKeyword $a
                    }
                    else
                    {
                        $AdapterSettings = GetSetSettings $DisplayName $RegistryKeyword $a
                    }

                    if ($AdapterSettings.RegistryKeyword -Contains 'NetworkAddress' -and -not $Reset)
                    {
                        if ($false -eq (ValidateLAASetting $DisplayValue $RegistryValue))
                        {
                            break
                        }
                    }

                    if ($script:isProfileSetting)
                    {
                        $FinalObject += SetProfileSettings $a $DisplayValue $RegistryValue
                    }

                    foreach ($adapterSetting in $AdapterSettings)
                    {
                        if ($null -ne $DisplayValue)
                        {
                            Set-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName -DisplayValue $DisplayValue
                            $FinalObject += ValidateSetting $adapterSetting $DisplayValue $null $a
                        }
                        elseif ($null -ne $RegistryValue)
                        {
                            Set-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName -RegistryValue $RegistryValue
                            $FinalObject += ValidateSetting $adapterSetting $null $RegistryValue $a
                        }
                        elseif ($Reset)
                        {
                            Reset-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName
                            $FinalObject += ValidateSetting $adapterSetting $adapterSetting.DefaultDisplayValue $null $a
                        }
                    }
                }
            }
            finally
            {
                $global:ErrorActionPreference = $PreErrorActionPreference
            }
        } while ($false)
    }
    End
    {
        $FinalObject

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function ValidateParams($DisplayName, $DisplayValue, $RegistryKeyword, $RegistryValue, $Setting, $AdapterName, $Adapters)
{
    $bContinue = $true

    do
    {
        # if DisplayName, RegistryKeyword, & Setting are not used, or both DisplayName & RegistryKeyword are used - stop execution
        if (([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword) -and $null -eq $Setting) -or
            (-not [string]::IsNullOrEmpty($DisplayName) -and -not [string]::IsNullOrEmpty($RegistryKeyword)))
        {
            $bContinue = $false
            break
        }

        # if both DisplayValue & RegistryValue are not used & Reset isn't used, or both DisplayValue & RegistryValue are used - stop exection
        if (([string]::IsNullOrEmpty($DisplayValue) -and [string]::IsNullOrEmpty($RegistryValue) -and -not $Reset) -or
            (-not [string]::IsNullOrEmpty($DisplayValue) -and -not [string]::IsNullOrEmpty($RegistryValue)))
        {
            $bContinue = $false
            break
        }

        # if either DisplayValue & RegistryValue are used and $Reset is used - stop execution
        if (-not [string]::IsNullOrEmpty($DisplayValue) -or -not [string]::IsNullOrEmpty($RegistryValue) -and $Reset)
        {
            $bContinue = $false
            break
        }

        # if Setting is used and either DisplayName, RegistryKeyword are used - stop execution
        if ($null -ne $Setting -and (-not [string]::IsNullOrEmpty($DisplayName) -or -not [string]::IsNullOrEmpty($RegistryKeyword)))
        {
            $bContinue = $false
            break
        }

        # if Setting is used and either $AdapterName or $Adapters is used - stop execution
        if ($null -ne $Setting -and (-not [string]::IsNullOrEmpty($AdapterName) -or $Adapters))
        {
            $bContinue = $false
            break
        }

    } while ($false)

    if ($false -eq $bContinue)
    {
        $script:ErrorMessagesSet += $Messages.InvalidParams
    }

    return $bContinue
}

function GetSetSettings($DisplayName, $RegistryKeyword, $AdapterName)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            if ($TmpDisplayName -eq 'Profile')
            {
                $script:isProfileSetting = $true
            }
            else
            {
                $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName})

                if (-not $TmpSetting)
                {
                    $script:ErrorMessagesSet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
                }
                else
                {
                    $SettingArray += $TmpSetting
                }
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            if ($TmpRegistryKeyword -eq 'PerformanceProfile')
            {
                $script:isProfileSetting = $true
            }
            else
            {
                $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword})

                if (-not $TmpSetting)
                {
                    $script:ErrorMessagesSet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
                }
                else
                {
                    $SettingArray += $TmpSetting
                }
            }
        }
    }

    return $SettingArray
}

function ValidateSetting($Setting, $DisplayValue = $null, $RegistryValue = $null, $AdapterName)
{
    $IntelEthernetSetting = Get-IntelEthernetSetting -Name $a -DisplayName $Setting.DisplayName -WarningAction:SilentlyContinue
    # setting type 'int'
    if ($null -ne $IntelEthernetSetting.PSTypeNames -and $IntelEthernetSetting.PSTypeNames[0] -eq "IntelEthernetSettingInt")
    {
        if ($null -ne $DisplayValue)
        {
            $SettingIntValue = $DisplayValue
        }
        else
        {
            $SettingIntValue = $RegistryValue
        }

        $bValidIntValue = $false
        # check if numeric value
        if ($SettingIntValue -match "^[\d]+$")
        {
            if ([int]$SettingIntValue -ge $IntelEthernetSetting.Min -and [int]$SettingIntValue -le $IntelEthernetSetting.Max)
            {
                if (($SettingIntValue % $IntelEthernetSetting.Step) -eq 0)
                {
                    $bValidIntValue = $true
                }
            }
        }

        if (-not $bValidIntValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $SettingIntValue, $Setting.RegistryKeyword
        }
        elseif ($SettingIntValue -ne $IntelEthernetSetting.DisplayValue)
        {
            $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.RegistryKeyword
        }
        else
        {
            return $IntelEthernetSetting
        }
    }
    else
    {
        $bValidEnumValue = $false
        if ($null -ne $DisplayValue)
        {
            if ($IntelEthernetSetting.DescriptionMap -notcontains $DisplayValue -and $IntelEthernetSetting.RegistryKeyword -ne "NetworkAddress")
            {
                $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $DisplayValue, $Setting.RegistryKeyword
            }
            elseif ($DisplayValue.ToString() -ne $IntelEthernetSetting.DisplayValue)
            {
                $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.DisplayName
            }
            else
            {
                $bValidEnumValue = $true
            }
        }
        elseif ($null -ne $RegistryValue)
        {
            if ($IntelEthernetSetting.PossibleValues -notcontains $RegistryValue -and $IntelEthernetSetting.RegistryKeyword -ne "NetworkAddress")
            {
                $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $RegistryValue, $Setting.RegistryKeyword
            }
            elseif (($RegistryValue.ToString() -ne $IntelEthernetSetting.RegistryValue))
            {
                $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.DisplayName
            }
            else
            {
                $bValidEnumValue = $true
            }
        }

        if ($bValidEnumValue)
        {
            return $IntelEthernetSetting
        }
    }
}

function ValidateLAASetting($DisplayValue, $RegistryValue)
{
    $Valid = $False
    do
    {
        if (-not [string]::IsNullOrEmpty($DisplayValue))
        {
            $Value = $DisplayValue
        }
        else
        {
            $Value = $RegistryValue
        }
        # first check if the string has correct length
        if ($Value.Length -eq $NETWORK_ADDRESS_LEN)
        {
            # filter out all non-hexadecimal numbers
            try
            {
                $ValueInt = [Convert]::ToUInt64($Value, 16)
            }
            catch
            {
                break
            }
            # check for reserved and multicast addresses
            if ($ValueInt -band 0x010000000000 -or $ValueInt -eq 0 -or $ValueInt -eq 0xFFFFFFFFFFFF)
            {
                break
            }
            $Valid = $True
        }
    } while($false)

    if ($false -eq $Valid)
    {
        $script:ErrorMessagesSet += $Messages.InvalidSettingValueGeneric
    }
    return $Valid
}

function IsSettingObject($PipedObject)
{
    $bSettingObject = $false
    if ($null -ne $PipedObject -and $null -ne $PipedObject[0].PSTypeNames -and
        ($PipedObject[0].PSTypeNames[0]).Contains("IntelEthernetSetting"))
    {
        $bSettingObject = $true
    }

    return $bSettingObject
}

function SetProfileSettings($AdapterName, $DisplayValue, $RegistryValue)
{
    $ProfileSetting = InitializeProfileSetting $AdapterName

    if ($null -ne $DisplayValue)
    {
        if ($ProfileSetting.DescriptionMap -notcontains $DisplayValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $DisplayValue, $ProfileSetting.RegistryKeyword
        }
        else
        {
            $ProfileRegValue = $PROFILE_VALUE_TO_NAME.GetEnumerator().Where({ $_.Value -eq $DisplayValue}).Name
        }
    }
    elseif ($null -ne $RegistryValue)
    {
        if ($ProfileSetting.PossibleValues -notcontains $RegistryValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $RegistryValue, $ProfileSetting.RegistryKeyword
        }
        else
        {
            $ProfileRegValue = $RegistryValue
        }
    }
    elseif ($Reset)
    {
        $ProfileRegValue = $ProfileSetting.DefaultValue
    }

    if ($null -ne $ProfileRegValue)
    {
        SetAdapterPropertyInRegistry $AdapterName 'PerformanceProfile' $ProfileRegValue

        $DriverFamily = $script:SupportedAdapters.Where({ $_.Name -eq $AdapterName }).Service
        $ProfileCmdletValName = $PROFILE_VALUE_TO_NAME[$ProfileRegValue.ToString()]
        $tmp = GetProfileSettingsFromXml $ProfileCmdletValName $DriverFamily
        $tmp.GetEnumerator().ForEach(
            {
                Set-NetAdapterAdvancedProperty -InterfaceDescription $AdapterName -RegistryKeyword $_.Key -RegistryValue $_.Value
            })

        $script:MSNetAdvProperty = Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue
        return InitializeProfileSetting $AdapterName
    }
}

# SIG # Begin signature block
# MIIt8gYJKoZIhvcNAQcCoIIt4zCCLd8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCLN5FwYb/sNfWG
# 5Obua/HWn5GcYyrg0JLLeysq+NyofKCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCG1UwghtRAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQg2IoH7zCnafVGzSBjbBHM
# 0+22IxDMf8/4nuY80XD+86EwDQYJKoZIhvcNAQEBBQAEggGANjMe+m5l0DbWBCbV
# AK0AOHGKLmPzRTKwXPpSXVGkwu4fWDLUuXV1YNuUGZrnEXHpKL80lTXnul2W+qdv
# VX4m5s35AslmF6VNboeTDIjJTvzdAfDAnRN5EjnFFk2owRixfv7uyc8IIJkyB9/G
# pBaSm5VOlsaFCb+uHLcIny59SUahnynd92Y5p2EARQftAk0mIuL8X0eu2M8dgMx3
# t2et9APBqaLdD4U+YieZ9hlOolGNiSInlthOUK0KattNVdw8oyDIwdOpV5n/kvNF
# UuCfjlioBQF+gVpFphHV4HiWhUCnuTUDTcZNVemBfJslyHgXDczOGJX/QdUORhNH
# hM6GzetZzNR7nVraXubU+/sQaoeedwlTdvzfIGZ3KaILmuN9VS7Sl/rYOeVHXA3I
# 375xHRGZg8ueAeUspLN9k6/b3hVyz3S3H56dtc0CbSyYHe5EGcSq5JEUBox9HK/i
# XPFUlzVs/B3DDhM2FYtw/0K0dR3zT7zAO5tHK5qTbn0N3VWUoYIYzjCCGMoGCisG
# AQQBgjcDAwExghi6MIIYtgYJKoZIhvcNAQcCoIIYpzCCGKMCAQMxDzANBglghkgB
# ZQMEAgIFADCB9AYLKoZIhvcNAQkQAQSggeQEgeEwgd4CAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQghcNf77rQl4DEB1NZsrguWh0J8r6FjKxFfZWycb0e
# qowCFQDSWFJHPNrBL/Iqomis8WT8SMLKWBgPMjAyNDA5MTcyMTMzNTVaoHKkcDBu
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMTAwLgYDVQQDEydTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1w
# aW5nIFNpZ25lciBSMzWgghL/MIIGXTCCBMWgAwIBAgIQOlJqLITOVeYdZfzMEtjp
# iTANBgkqhkiG9w0BAQwFADBVMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMSwwKgYDVQQDEyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5n
# IENBIFIzNjAeFw0yNDAxMTUwMDAwMDBaFw0zNTA0MTQyMzU5NTlaMG4xCzAJBgNV
# BAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxMDAuBgNVBAMTJ1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgU2ln
# bmVyIFIzNTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAI3RZ/TBSJu9
# /ThJOk1hgZvD2NxFpWEENo0GnuOYloD11BlbmKCGtcY0xiMrsN7LlEgcyoshtP3P
# 2J/vneZhuiMmspY7hk/Q3l0FPZPBllo9vwT6GpoNnxXLZz7HU2ITBsTNOs9fhbdA
# Wr/Mm8MNtYov32osvjYYlDNfefnBajrQqSV8Wf5ZvbaY5lZhKqQJUaXxpi4TXZKo
# hLgxU7g9RrFd477j7jxilCU2ptz+d1OCzNFAsXgyPEM+NEMPUz2q+ktNlxMZXPF9
# WLIhOhE3E8/oNSJkNTqhcBGsbDI/1qCU9fBhuSojZ0u5/1+IjMG6AINyI6XLxM8O
# AGQmaMB8gs2IZxUTOD7jTFR2HE1xoL7qvSO4+JHtvNceHu//dGeVm5Pdkay3Et+Y
# Tt9EwAXBsd0PPmC0cuqNJNcOI0XnwjE+2+Zk8bauVz5ir7YHz7mlj5Bmf7W8SJ8j
# QwO2IDoHHFC46ePg+eoNors0QrC0PWnOgDeMkW6gmLBtq3CEOSDU8iNicwNsNb7A
# Bz0W1E3qlSw7jTmNoGCKCgVkLD2FaMs2qAVVOjuUxvmtWMn1pIFVUvZ1yrPIVbYt
# 1aTld2nrmh544Auh3tgggy/WluoLXlHtAJgvFwrVsKXj8ekFt0TmaPL0lHvQEe5j
# Hbufhc05lvCtdwbfBl/2ARSTuy1s8CgFAgMBAAGjggGOMIIBijAfBgNVHSMEGDAW
# gBRfWO1MMXqiYUKNUoC6s2GXGaIymzAdBgNVHQ4EFgQUaO+kMklptlI4HepDOSz0
# FGqeDIUwDgYDVR0PAQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAww
# CgYIKwYBBQUHAwgwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEF
# BQcCARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEoGA1UdHwRD
# MEEwP6A9oDuGOWh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1Rp
# bWVTdGFtcGluZ0NBUjM2LmNybDB6BggrBgEFBQcBAQRuMGwwRQYIKwYBBQUHMAKG
# OWh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGlu
# Z0NBUjM2LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20w
# DQYJKoZIhvcNAQEMBQADggGBALDcLsn6TzZMii/2yU/V7xhPH58Oxr/+EnrZjpIy
# vYTz2u/zbL+fzB7lbrPml8ERajOVbudan6x08J1RMXD9hByq+yEfpv1G+z2pmnln
# 5XucfA9MfzLMrCArNNMbUjVcRcsAr18eeZeloN5V4jwrovDeLOdZl0tB7fOX5F6N
# 2rmXaNTuJR8yS2F+EWaL5VVg+RH8FelXtRvVDLJZ5uqSNIckdGa/eUFhtDKTTz9L
# tOUh46v2JD5Q3nt8mDhAjTKp2fo/KJ6FLWdKAvApGzjpPwDqFeJKf+kJdoBKd2zQ
# uwzk5Wgph9uA46VYK8p/BTJJahKCuGdyKFIFfEfakC4NXa+vwY4IRp49lzQPLo7W
# ticqMaaqb8hE2QmCFIyLOvWIg4837bd+60FcCGbHwmL/g1ObIf0rRS9ceK4DY9rf
# BnHFH2v1d4hRVvZXyCVlrL7ZQuVzjjkLMK9VJlXTVkHpuC8K5S4HHTv2AJx6mOdk
# MJwS4gLlJ7gXrIVpnxG+aIniGDCCBhQwggP8oAMCAQICEHojrtpTaZYPkcg+XPTH
# 4z8wDQYJKoZIhvcNAQEMBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGlu
# ZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5NTlaMFUxCzAJ
# BgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMTI1Nl
# Y3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0EgUjM2MIIBojANBgkqhkiG9w0B
# AQEFAAOCAY8AMIIBigKCAYEAzZjYQ0GrboIr7PYzfiY05ImM0+8iEoBUPu8mr4wO
# gYPjoiIz5vzf7d5wu8GFK1JWN5hciN9rdqOhbdxLcSVwnOTJmUGfAMQm4eXOls3i
# QwfapEFWuOsYmBKXPNSpwZAFoLGl5y1EaGGc5LByM8wjcbSF52/Z42YaJRsPXY54
# 5E3QAPN2mxDh0OLozhiGgYT1xtjXVfEzYBVmfQaI5QL35cTTAjsJAp85R+KAsOfu
# L9Z7LFnjdcuPkZWjssMETFIueH69rxbFOUD64G+rUo7xFIdRAuDNvWBsv0iGDPGa
# R2nZlY24tz5fISYk1sPY4gir99aXAGnoo0vX3Okew4MsiyBn5ZnUDMKzUcQrpVav
# GacrIkmDYu/bcOUR1mVBIZ0X7P4bKf38JF7Mp7tY3LFF/h7hvBS2tgTYXlD7TnIM
# PrxyXCfB5yQq3FFoXRXM3/DvqQ4shoVWF/mwwz9xoRku05iphp22fTfjKRIVpm4g
# FT24JKspEpM8mFa9eTgKWWCvAgMBAAGjggFcMIIBWDAfBgNVHSMEGDAWgBT2d2rd
# P/0BE/8WoWyCAi/QCj0UJTAdBgNVHQ4EFgQUX1jtTDF6omFCjVKAurNhlxmiMpsw
# DgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYI
# KwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMEwGA1UdHwRFMEMwQaA/oD2GO2h0
# dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY1RpbWVTdGFtcGluZ1Jv
# b3RSNDYuY3JsMHwGCCsGAQUFBwEBBHAwbjBHBggrBgEFBQcwAoY7aHR0cDovL2Ny
# dC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljVGltZVN0YW1waW5nUm9vdFI0Ni5w
# N2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3
# DQEBDAUAA4ICAQAS13sgrQ41WAyegR0lWP1MLWd0r8diJiH2VVRpxqFGhnZbaF+I
# Q7JATGceTWOS+kgnMAzGYRzpm8jIcjlSQ8JtcqymKhgx1s6cFZBSfvfeoyigF8iC
# GlH+SVSo3HHr98NepjSFJTU5KSRKK+3nVSWYkSVQgJlgGh3MPcz9IWN4I/n1qfDG
# zqHCPWZ+/Mb5vVyhgaeqxLPbBIqv6cM74Nvyo1xNsllECJJrOvsrJQkajVz4xJwZ
# 8blAdX5umzwFfk7K/0K3fpjgiXpqNOpXaJ+KSRW0HdE0FSDC7+ZKJJSJx78mn+rw
# EyT+A3z7Ss0gT5CpTrcmhUwIw9jbvnYuYRKxFVWjKklW3z83epDVzoWJttxFpujd
# rNmRwh1YZVIB2guAAjEQoF42H0BA7WBCueHVMDyV1e4nM9K4As7PVSNvQ8LI1WRa
# TuGSFUd9y8F8jw22BZC6mJoB40d7SlZIYfaildlgpgbgtu6SDsek2L8qomG57Yp5
# qTqof0DwJ4Q4HsShvRl/59T4IJBovRwmqWafH0cIPEX7cEttS5+tXrgRtMjjTOp6
# A9l0D6xcKZtxnLqiTH9KPCy6xZEi0UDcMTww5Fl4VvoGbMG2oonuX3f1tsoHLaO/
# Fwkj3xVr3lDkmeUqivebQTvGkx5hGuJaSVQ+x60xJ/Y29RBr8Tm9XJ59AjCCBoIw
# ggRqoAMCAQICEDbCsL18Gzrno7PdNsvJdWgwDQYJKoZIhvcNAQEMBQAwgYgxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkg
# Q2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVV
# U0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTIxMDMyMjAw
# MDAwMFoXDTM4MDExODIzNTk1OVowVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1Nl
# Y3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFt
# cGluZyBSb290IFI0NjCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAIid
# 2LlFZ50d3ei5JoGaVFTAfEkFm8xaFQ/ZlBBEtEFAgXcUmanU5HYsyAhTXiDQkiUv
# pVdYqZ1uYoZEMgtHES1l1Cc6HaqZzEbOOp6YiTx63ywTon434aXVydmhx7Dx4IBr
# Aou7hNGsKioIBPy5GMN7KmgYmuu4f92sKKjbxqohUSfjk1mJlAjthgF7Hjx4vvyV
# DQGsd5KarLW5d73E3ThobSkob2SL48LpUR/O627pDchxll+bTSv1gASn/hp6IuHJ
# orEu6EopoB1CNFp/+HpTXeNARXUmdRMKbnXWflq+/g36NJXB35ZvxQw6zid61qmr
# lD/IbKJA6COw/8lFSPQwBP1ityZdwuCysCKZ9ZjczMqbUcLFyq6KdOpuzVDR3ZUw
# xDKL1wCAxgL2Mpz7eZbrb/JWXiOcNzDpQsmwGQ6Stw8tTCqPumhLRPb7YkzM8/6N
# nWH3T9ClmcGSF22LEyJYNWCHrQqYubNeKolzqUbCqhSqmr/UdUeb49zYHr7ALL8b
# AJyPDmubNqMtuaobKASBqP84uhqcRY/pjnYd+V5/dcu9ieERjiRKKsxCG1t6tG9o
# j7liwPddXEcYGOUiWLm742st50jGwTzxbMpepmOP1mLnJskvZaN5e45NuzAHteOR
# lsSuDt5t4BBRCJL+5EZnnw0ezntk9R8QJyAkL6/bAgMBAAGjggEWMIIBEjAfBgNV
# HSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQU9ndq3T/9ARP/
# FqFsggIv0Ao9FCUwDgYDVR0PAQH/BAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wEwYD
# VR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcw
# RaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0
# aWZpY2F0aW9uQXV0aG9yaXR5LmNybDA1BggrBgEFBQcBAQQpMCcwJQYIKwYBBQUH
# MAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIB
# AA6+ZUHtaES45aHF1BGH5Lc7JYzrftrIF5Ht2PFDxKKFOct/awAEWgHQMVHol9ZL
# Syd/pYMbaC0IZ+XBW9xhdkkmUV/KbUOiL7g98M/yzRyqUOZ1/IY7Ay0YbMniIibJ
# rPcgFp73WDnRDKtVutShPSZQZAdtFwXnuiWl8eFARK3PmLqEm9UsVX+55DbVIz33
# Mbhba0HUTEYv3yJ1fwKGxPBsP/MgTECimh7eXomvMm0/GPxX2uhwCcs/YLxDnBdV
# VlxvDjHjO1cuwbOpkiJGHmLXXVNbsdXUC2xBrq9fLrfe8IBsA4hopwsCj8hTuwKX
# JlSTrZcPRVSccP5i9U28gZ7OMzoJGlxZ5384OKm0r568Mo9TYrqzKeKZgFo0fj2/
# 0iHbj55hc20jfxvK3mQi+H7xpbzxZOFGm/yVQkpo+ffv5gdhp+hv1GDsvJOtJinJ
# mgGbBFZIThbqI+MHvAmMmkfb3fTxmSkop2mSJL1Y2x/955S29Gu0gSJIkc3z30vU
# /iXrMpWx2tS7UVfVP+5tKuzGtgkP7d/doqDrLF1u6Ci3TpjAZdeLLlRQZm867eVe
# XED58LXd1Dk6UvaAhvmWYXoiLz4JA5gPBcz7J311uahxCweNxE+xxxR3kT0WKzAS
# o5G/PyDez6NHdIUKBeE3jDPs2ACc6CkJ1Sji4PKWVT0/MYIEkTCCBI0CAQEwaTBV
# MQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# EyNTZWN0aWdvIFB1YmxpYyBUaW1lIFN0YW1waW5nIENBIFIzNgIQOlJqLITOVeYd
# ZfzMEtjpiTANBglghkgBZQMEAgIFAKCCAfkwGgYJKoZIhvcNAQkDMQ0GCyqGSIb3
# DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yNDA5MTcyMTMzNTVaMD8GCSqGSIb3DQEJ
# BDEyBDApCL4YImNfWCPunW/bYp5w33vAaHpWhMqR03yXRYLCahi3GqGroCpE5+xA
# MqOOu+MwggF6BgsqhkiG9w0BCRACDDGCAWkwggFlMIIBYTAWBBT4YJgZpvuILPfo
# UpfyoRlSGhZ3XzCBhwQUxq5U5HiG8Xw9VRJIjGnDSnr5wt0wbzBbpFkwVzELMAkG
# A1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2Vj
# dGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBSb290IFI0NgIQeiOu2lNplg+RyD5c
# 9MfjPzCBvAQUhT1jLZOCgmF80JA1xJHeksFC2scwgaMwgY6kgYswgYgxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0
# eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VS
# VHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5AhA2wrC9fBs656Oz3TbL
# yXVoMA0GCSqGSIb3DQEBAQUABIICAAtdI5sY1cuKG/9QcZUsGtg7BwgS2pvGQf/t
# fB6OxkIbA0IPTo/kPmsD+NmklZnUi7R/50XXeplxF4ujD197kuke3eUrgI0V9ByK
# lIPI8dyka/iEzPkfL1dXCtnbLBIrIqXk8HWDFSWALyNnUmqjTsoX5drcVNWvoOwp
# bEvravUoUQYRhXiAi+483hI/FqUDWMAE9tYl3Ixulcx9wTDdBAT9EKJoMln4qut8
# oGFcH/Bdn3Up7fPd92A5pzlx5VeLexks+xIjbB1d23gm7OhViKHdYCMALAYoMeTY
# nytaMsZqqs8HrO95IN4msyygpoUNry/f4eTwvV52I4uLJdynID8F2Yb+gdjMi8Bk
# UWpbYKjqYS+xnIhZcwhGl6aAnKoAzvmRUwXJvxTTiGEXCZtw1YOfLT8LAECQW1qc
# aqpxEJIiJdRgH38Q0haPw+aJHRqcsbBH1ocS4ZqyDJCpHfPjFrbyt18gb+s6F53V
# 6huhS5TNmc2R7utoj4nA6lRQow4e/xP3SeiVZ5F0VFx7wsj438GaN10C0+aZ9cJR
# QK//YAwkW38AZLj1+ej6FLp+tZqpMxd5YP+yuIUkA01yGxnfifkqurrjrexTZrIE
# LKZq1AwypYx8TLHUV12J+BwFDmDQkxUuMWKjk8h6ExII7094dnE1EW8pShOLezhT
# AaI5goDS
# SIG # End signature block
