﻿<#
/// <copyright>
/// INTEL CONFIDENTIAL
///
/// Copyright 2014 Intel Corporation
///
/// This software and the related documents are Intel copyrighted materials, and your use of
/// them is governed by the express license under which they were provided to you ("License").
/// Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
/// disclose or transmit this software or the related documents without Intel's prior written
/// permission.
///
/// This software and the related documents are provided as is, with no express or implied
/// warranties, other than those that are expressly stated in the License.
///
/// </copyright>
#>

#User input arguments
param
(
    [Parameter(Mandatory=$true)]
    [string]$Action,

    [string]$ConfigPath,

    [switch]$BDF
)

#Global file variables
$Script:context
$Script:service
$Script:lockobj
$script:options
$Script:Adapters = $null
$Script:bSave = $false
$Script:bRestore = $false
$Script:bRemove = $false
$Script:bUpSave = $false
$Script:bUpRestore = $false
$Script:bANSInstalled = $true
$Script:colImportAdapters = $null
$Script:EnabledSystemAdapters = $null

#Save off input parameters to be used in functions
$Script:Action = $Action
$Script:BDF = $BDF
$Script:ConfigPath = $ConfigPath
$Script:ScriptPath = $null


#------------------------ Messages ------------------------
$Script:Messages = @{
    Usage = @"
Intel(R) SaveRestore.ps1
Copyright 2014 Intel Corporation. All rights reserved.

  Usage: SaveRestore.ps1 -Action -ConfigPath -BDF
  Example: SaveRestore.ps1 –Action save –ConfigPath C:\*File Path*\config.txt
  -Action is required. Valid values are 'save' and 'restore.'
  -ConfigPath is optional. It specifies the path and file name of the main configuration save file. If not specified, it is the script path and default file name (Saved_Config.txt).
  NOTE: The Saved_StaticIP.txt file name does not change and is always saved and restored from the script path.
  -BDF is optional. If specified during a restore, it will restore settings to adapters using their bus/device/function.
  The default configuration file names are Saved_Config.txt and Saved_StaticIP.txt.
"@;

    # Performing a save/restore of configuration file at $Script:ConfigPath
    RunningSaveRestore        = "Performing a save/restore of configuration file at";

    SettingOEMValue           = "Setting OEM custom value.";

    # Restoring primary and secondary adapters on $Team.TeamName
    RestoringTeamAdapters     = "Restoring primary and secondary adapters on";

    # Restoring $Team.TeamName / $Vlan.VlanName
    Restoring                 = "Restoring"

    RestoringAdapterIP        = "Restoring adapter IP settings."
    RestoringTeamIP           = "Restoring team IP settings."
    RestoringVLANIP           = "Restoring VLAN IP settings."
    RestoringAdapterSettings  = "Restoring adapter settings."
    RestoringTeamSettings     = "Restoring team settings."
    RestoringVLANSettings     = "Restoring VLAN settings."
    RestoringAdapterPartition = "Restoring adapter partition settings."

    NICPartitioningDetected   = "Detected a device configured for NIC partitioning. We recommend you use the -BDF option when restoring."

    PerformingSave            = "Saving the configuration."
    PerformingUpgradeSave     = "Upgrading and saving the configuration."
    PerformingRestore         = "Restoring the configuration."
    PerformingUpgradeRestore  = "Upgrading and restoring the configuration."
    PerformingRemove          = "Removing the team and VLAN device configuration."

    # $Script:ConfigPath file already exists. Do you want to overwrite it (y/n)?
    FileExistsPrompt          = "file already exists. Do you want to overwrite it (y/n)?"

    #------------------------ Errors ------------------------

    # No objects found for $class
    NoObjectsFound            = "No objects found for"

    AdminRequired             = "The save/restore script requires administrative rights. Please log in as an Administrator and try again."
    PROSetRequired            = "The save/restore script requires Intel® PROSet to be installed. Please install Intel PROSet and try again."

    # Could not find the restore configuration file at $Script:ConfigPath. Please provide a path to the configuration file.
    ConfigNotFoundPart1       = "Could not find the restore configuration file at"
    ConfigNotFoundPart2       = "Please provide a valid path to the configuration file."

    # Configuration path $FilePath does not exist. Please provide a valid path.
    ConfigPathDoesntExist1    = "A directory in"
    ConfigPathDoesntExist2    = "does not exist. Please provide a valid path."

    ConfigFileNotSpecified    = "No configuration file name specified. Please provide a valid file name."

    # Unable to set ... on ...
    UnableToSetPart1          = "Unable to set"
    UnableToSetPart2          = "on"

    #----------------------- Warnings -----------------------
    UnableToFindDevice        = "Unable to find Device. Please verify configuration file matches your system configuration."
    UnableToFindANS           = "Intel(R) ANS is not present on the system. Any Team and VLAN information contained in the configuration file will not be restored."
}

#-----------------------------------------------------[Functions]------------------------------------------------------


# Set WMI lock
Function BeginApply
{
    # retrieve instance
        $Script:service = Get-WMiobject -class IAnet_netservice -namespace root\intelncs2

    # create [in] parameters
        $inparams = $Script:service.GetMethodParameters("BeginApply")

    # spawn management object
        $Script:options = new-object System.Management.InvokeMethodOptions

    # add context for this script
        $Script:options.Context.Add('SaveRestoreApply', $true)
        $Script:options.Context.Add('GET_EXTENSIONS', $true)
        $Script:options.Context.Add('GET_EXT_KEYS_ONLY', $false)
        $Script:options.Context.Add('IANet_PartialData', 512)

    # obtain lock
        $Script:lockobj = $Script:service.InvokeMethod("BeginApply", $inparams, $Script:options)

    # this is necessary to set the lock into a new object
        $Script:context = new-object System.Management.ManagementNamedValueCollection

    # adding a new single named value to the collection with the value of the lock objects client handle
        $Script:context.Add('ClientSetId', [int] $Script:lockobj.ClientSetHandle)
        $Script:context.Add('SaveRestoreApply', $true)
}

# Remove WMI lock
Function ReleaseLock
{
    $inparams = $Script:service.GetMethodParameters("Apply")
    $inparams.ClientSetHandle=[int]$Script:lockobj.ClientSetHandle
    $null = $Script:Service.InvokeMethod("Apply", $inparams, $null)
}

Function NewObject ($class)
{
    $classdef = New-Object System.Management.ManagementClass $class
    # the new object options
    $classdef.Options.UseAmendedQualifiers = $true
    $classdef.Options.Context = $Script:context

    return $classdef
}

Function DeleteObject ($object)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log $object
    # the delete options
    $deleteoptions = New-Object System.Management.DeleteOptions
    $deleteoptions.Context = $Script:context

    $object.Delete($deleteoptions)
}

Function SetSetting ($setting)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$setting = $setting"
    # the put options
    $putoptions = New-Object System.Management.PutOptions($Script:context)
    $putoptions.UseAmendedQualifiers = $true

    #perform put
    $null = $setting.Put($putoptions)
}

# Get specified class object from WMI
Function GetObject($class)
{
    $querystring = [string] "SELECT * FROM $class"
    $query = New-Object System.Management.ObjectQuery($querystring)

    # the enumeration options
    $enumerate_option = New-Object System.Management.EnumerationOptions
    $enumerate_option.UseAmendedQualifiers = $true
    $enumerate_option.Context = $Script:context

    # setup scope
    $DMiXPath = New-Object System.Management.ManagementPath
    $DMiXPath.NamespacePath = "root\intelncs2"
    $scope = New-Object System.Management.ManagementScope($DMiXPath)
    $scope.Connect()

    # the searcher for the object
    $searcher = New-Object System.Management.ManagementObjectSearcher($scope, $query, $enumerate_option)
    $collection = $searcher.Get()

    # If the Get() above fails, it won't be caught until the return happens. So catch it and display an appropriate message.
    try
    {
        return $collection
    }
    catch
    {
        $ErrorMessage = [string]$Messages.NoObjectsFound
        Write-Error "$ErrorMessage $class"
        Write-Log "$ErrorMessage $class"
        exit
    }
}

# Get associated objects given the object path and where condition
Function GetAssociated($path, $where)
{
    $querystring = [string] "ASSOCIATORS OF {$path} WHERE $where"

    $query = New-Object System.Management.ObjectQuery($querystring)

    # the enumeration options
    $enumerate_option = New-Object System.Management.EnumerationOptions
    $enumerate_option.UseAmendedQualifiers = $true
    $enumerate_option.Context = $Script:context

    # setup scope
    $DMiXPath = New-Object System.Management.ManagementPath
    $DMiXPath.NamespacePath = "root\intelncs2"
    $scope = New-Object System.Management.ManagementScope($DMiXPath)
    $scope.Connect()

    # the searcher for the object
    $searcher = New-Object System.Management.ManagementObjectSearcher($scope, $query, $enumerate_option)
    $collection = $searcher.Get()

    return $collection
}

# Invoke a method given it's name, class and parameters
Function InvokeMethod($Class,$Method,$ColParameterName,$ColParameterValue)
{
    $Invoke_option = New-Object System.Management.InvokeMethodOptions
    $Invoke_option.Context = $Script:context

    $params = $null
    if ($null -ne $ColParameterName)
    {
        $params = $Class.psbase.GetMethodParameters($Method)
        if ($ColParameterName.count -gt 1)
        {
            for($i=0; $i -lt $ColParameterName.count; $i++)
            {
                $params.psbase.SetPropertyValue($ColParameterName[$i],$ColParameterValue[$i])
            }
        }
        else
        {
            $params.psbase.SetPropertyValue($ColParameterName,$ColParameterValue)
        }
    }

    return $Class.psbase.InvokeMethod($Method,$params,$Invoke_option)
}

# Function used to write objects to config file.
Function WriteObjectToConfigFile($NewObjects,$Properties)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewObjects = $NewObjects"
    Write-Log "`$Properties = $Properties"
    if ($null -ne $NewObjects)
    {
        $ExportObject = $NewObjects | Select-Object $Properties | ConvertTo-Csv -NoTypeInformation
        $ExportObject = $ExportObject[1..$ExportObject.Count]
        WriteTextToConfigFile $ExportObject
    }
}

# Function used to write text to config file.
Function WriteTextToConfigFile($NewText)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewText = $NewText"
    $NewText | Out-File $Script:ConfigPath -Encoding utf8 -Append
}

# Function used to write objects to the IP config file.
Function WriteObjectToIPFile($NewObjects,$Properties)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewObjects = $NewObjects"
    Write-Log "`$Properties = $Properties"
    if ($null -ne $NewObjects)
    {
        $ExportObject = $NewObjects | Select-Object $Properties | ConvertTo-Csv -NoTypeInformation
        $ExportObject = $ExportObject[1..$ExportObject.Count]
        WriteTextToIPFile $ExportObject
    }
}

# Function used to write text to IP config file.
Function WriteTextToIPFile($NewText)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$NewText = $NewText"
    $NewText | Out-File $Script:ScriptPath\Saved_StaticIP.txt -Encoding utf8 -Append
}

# Function used to read objects from config file given a file section.
Function ReadFromConfigFile($FileSection)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$FileSection = $FileSection"
    $FileObjects=@()
    $FileObjects = Get-Content $Script:ConfigPath

    switch($FileSection)
    {
        "Adapters"
        {
            #Find the section for adapters and add 1 to not include AdaptersStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"AdaptersStart") + 1
            #Find the end of the section and remove 1 to not include AdaptersEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"AdaptersEnd") - 1

            $colProperty = "Name","OriginalDisplayName","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "AdapterSettings"
        {
            #Find the section for adapters and add 1 to not include AdapterSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"AdapterSettingsStart") + 1
            #Find the end of the section and remove 1 to not include AdapterSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"AdapterSettingsEnd") - 1

            $colProperty = "Name","Caption","CurrentValue","CurrentValues","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "OEMSetting"
        {
            #Find the section for adapters and add 1 to not include OEMSettingStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"OEMSettingStart") + 1
            #Find the end of the section and remove 1 to not include OEMSettingEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"OEMSettingEnd") - 1

            $colProperty = "OEMCustomizeable"
            break
        }
        "PMSettings"
        {
            #Find the section for adapters and add 1 to not include PMSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"PMSettingsStart") + 1
            #Find the end of the section and remove 1 to not include PMSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"PMSettingsEnd") - 1

            $colProperty = "Name","Caption","CurrentValue","CurrentValues","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "Teams"
        {
            #Find the section for adapters and add 1 to not include TeamsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"TeamsStart") + 1
            #Find the end of the section and remove 1 to not include TeamsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"TeamsEnd") - 1

            $colProperty = "TeamName","TeamMembers","TeamMode","PrimaryAdapter","SecondaryAdapter"
            break
        }
        "TeamSettings"
        {
            #Find the section for adapters and add 1 to not include TeamSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"TeamSettingsStart") + 1
            #Find the end of the section and remove 1 to not include TeamSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"TeamSettingsEnd") - 1

            $colProperty = "TeamName","Caption","CurrentValue","CurrentValues"
            break
        }
        "Vlans"
        {
            #Find the section for adapters and add 1 to not include VlansStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"VlansStart") + 1
            #Find the end of the section and remove 1 to not include VlansEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"VlansEnd") - 1

            $colProperty = "ParentName","VLANID","VLANNAME","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
            break
        }
        "VlanSettings"
        {
            #Find the section for adapters and add 1 to not include VlanSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"VlanSettingsStart") + 1
            #Find the end of the section and remove 1 to not include VlanSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"VlanSettingsEnd") - 1

            $colProperty = "ParentName","Name","VLANID","Caption","CurrentValue","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
            break
        }
        "NICPARTSettings"
        {
            #Find the section for adapters and add 1 to not include NICPARTSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"NICPARTSettingsStart") + 1
            #Find the end of the section and remove 1 to not include NICPARTSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"NICPARTSettingsEnd") - 1

            $colProperty = "Name","PartitionNumber","Identifier","MinBWPercent","MaxBWPercent"
            break
        }
        Default
        {
            return
        }
    }

    #If no items were found in the provided section return nothing
    if ($CsvEnd -lt $CsvObjectStart)
    {
        return

    }
    #Else return only the provided section and convert the text into objects
    else
    {
        return $FileObjects[$CsvObjectStart..$CsvEnd] | ConvertFrom-Csv -Header $colProperty
    }
}

# Function used to read objects from config file.
Function ReadFromIPFile($FileSection)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$FileSection = $FileSection"
    $FileObjects=@()
    $FileObjects = Get-Content $Script:ScriptPath\Saved_StaticIP.txt

    switch($FileSection)
    {
        "AdapterIPSettings"
        {
            #Find the section for adapters and add 1 to not include AdapterIPSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"AdapterIPSettingsStart") + 1
            #Find the end of the section and remove 1 to not include AdapterIPSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"AdapterIPSettingsEnd") - 1

            $colProperty = "Name","Caption","CurrentValue","CurrentValues","PermanentAddress","PCIDeviceID","SlotID"
            break
        }
        "TeamIPSettings"
        {
            #Find the section for adapters and add 1 to not include TeamIPSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"TeamIPSettingsStart") + 1
            #Find the end of the section and remove 1 to not include TeamIPSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"TeamIPSettingsEnd") - 1

            $colProperty = "TeamName","Caption","CurrentValue","CurrentValues"
            break
        }
        "VlanIPSettings"
        {
            #Find the section for adapters and add 1 to not include VlanIPSettingsStart identifier
            $CsvObjectStart = [array]::indexof($FileObjects,"VlanIPSettingsStart") + 1
            #Find the end of the section and remove 1 to not include VlanIPSettingsEnd identifier
            $CsvEnd = [array]::indexof($FileObjects,"VlanIPSettingsEnd") - 1

            $colProperty = "ParentName","VLANID","Caption","CurrentValue","CurrentValues","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
            break
        }
        Default
        {
            return
        }
    }

    # If no items were found in the provided section return nothing
    if ($CsvEnd -lt $CsvObjectStart)
    {
        return
    }
    # Else return only the provided section and convert the text into objects
    else
    {
        return $FileObjects[$CsvObjectStart..$CsvEnd] | ConvertFrom-Csv -Header $colProperty
    }
}

# Add the parent adapter identifiers to the setting objects
Function AddParentIDs($Setting)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$Setting = $Setting"
    #Add the PermanentAddress, PCIDeviceID and SlotID to the settings object
    $SettingAdapter = $Script:Adapters | Where-Object {$_.DeviceID -eq $Setting.ParentId}
    if ($SettingAdapter)
    {
        $Setting | Add-Member -Name "PermanentAddress" -Value  $SettingAdapter.PermanentAddress -MemberType NoteProperty
        $Setting | Add-Member -Name "PCIDeviceID" -Value  $SettingAdapter.PCIDeviceID -MemberType NoteProperty
        $AdapterSlotID =  $SettingAdapter.SlotID.split(":")[0..2] #return only the first three elements in array
        $AdapterBDF = [string]::Join(':',$AdapterSlotID)
        $Setting | Add-Member -Name "SlotID" -Value  $AdapterBDF -MemberType NoteProperty
    }
}

Function PrintUsage
{
    Write-Host $Messages.Usage
    exit
}

Function CheckForAdminRights
{
    $winIdent = [Security.Principal.WindowsIdentity]::GetCurrent()
    $WinPrinc = [Security.Principal.WindowsPrincipal] $winIdent
    $AdminId  = [Security.Principal.WindowsBuiltInRole] "Administrator"
    if (-Not $WinPrinc.IsInRole($AdminId))
    {
        Write-Error $Messages.AdminRequired
        exit
    }
}

Function CheckForDMiXInstall
{
    Write-Log $MyInvocation.MyCommand
    $DMiXInstall = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\DMIX -Name InstalledDMIX -ErrorAction SilentlyContinue

    if ((!$DMiXInstall) -or ($DMiXInstall.InstalledDMIX -ne 1))
    {
        Write-Error $Messages.PROSetRequired
        exit
    }
}

Function CheckForANSInstall
{
    Write-Log $MyInvocation.MyCommand
    $ANSInstall = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\DMIX -Name InstalledDMIX_ANS -ErrorAction SilentlyContinue

    if ((!$ANSInstall) -or ($ANSInstall.InstalledDMIX_ANS -ne 1))
    {
        $Script:bANSInstalled = $false
    }
}

Function CheckIfConfigFileExists
{
    Write-Log $MyInvocation.MyCommand
    if (!(Test-Path $Script:ConfigPath -PathType Leaf))
    {
        $ErrorMessagePart1 = [string]$Messages.ConfigNotFoundPart1
        $ErrorMessagePart2 = [string]$Messages.ConfigNotFoundPart2

        Write-Error  "$ErrorMessagePart1 $Script:ConfigPath. $ErrorMessagePart2"
        exit
    }
}

Function CheckIfConfigShouldBeOverwritten
{
    Write-Log $MyInvocation.MyCommand
    if (!(Test-Path $Script:ConfigPath.ToString() -PathType Container))
    {
        $FileExistsString = [string]$Messages.FileExistsPrompt
        $confirmation = Read-Host -Prompt "$Script:ConfigPath $FileExistsString"
        if ($confirmation -notmatch "[yY]")
        {
            exit
        }
    }
}

Function Write-Log($string)
{
    $timestamp = "[$(Get-Date -Format G)] "
    $string = $timestamp + $string
    Write-Verbose -Message $string
}

# Sets a global variable for the configuration file paths to be saved/restored from.
Function SetupSaveRestoreLocation
{
    Write-Log $MyInvocation.MyCommand
    $Script:ScriptPath = Get-Location

    # If the user did not specify a path, use the scripts path
    if ($Script:bUpSave -eq $true -or $Script:bUpRestore -eq $true)
    {
        $Script:ConfigPath = Get-Location
        $Script:ConfigPath = $Script:ConfigPath + "\Upgrade_Saved_Config.txt"
    }
    elseif (!$Script:ConfigPath)
    {
        $Script:ConfigPath = Get-Location
        $Script:ConfigPath = $Script:ConfigPath + "\Saved_Config.txt"

        #Check if the file exist during a save
        if ($Script:bSave -eq $true -and (Test-Path $Script:ConfigPath -PathType Leaf))
        {
            CheckIfConfigShouldBeOverwritten
        }
    }
    else
    {
        #Only prompt to overwrite if we are doing save and file exists
        if (($Script:bSave -eq $true -or $Script:bUpSave -eq $true) -and (Test-Path $Script:ConfigPath -PathType Leaf))
        {
            CheckIfConfigShouldBeOverwritten
        }

        #Current dir is default
        $FilePath = "."
        #Find position of the last backslash before the filename
        $FileNamePos = $Script:ConfigPath.ToString().LastIndexOf("\")
        #pos = -1 means ConfigPath contains just the file name - use current dir in this case
        if ($FileNamePos -ne -1)
        {
            #Separate the filename from the path to verify path exists
            $FilePath = $Script:ConfigPath.ToString().Substring(0, $FileNamePos)
        }

        #Check that config path exists
        if (!(Test-Path $FilePath -PathType Container))
        {
            $ErrorMessagePart1 = [string]$Messages.ConfigPathDoesntExist1
            $ErrorMessagePart2 = [string]$Messages.ConfigPathDoesntExist2
            Write-Error "$ErrorMessagePart1 $FilePath $ErrorMessagePart2"
            exit
        }

        #Check the input of the config file when Saving
        if ($Script:bSave -eq $true -or $Script:bUpSave -eq $true)
        {
            Try
            {
                Out-File $Script:ConfigPath  -ErrorAction Stop
            }
            Catch
            {
                Write-Error $Messages.ConfigFileNotSpecified
                Write-Log $Messages.ConfigFileNotSpecified
                exit
            }
        }
        #Check the input of the config file when Restoring
        elseif (!(Test-Path $Script:ConfigPath))
        {
            Write-Error $Messages.ConfigFileNotSpecified
            exit
        }
    }

    Write-Host $Messages.RunningSaveRestore $Script:ConfigPath
}

# Get the present and enabled adapters on the system.
Function GetAdaptersOnSystem
{
    Write-Log $MyInvocation.MyCommand
    #Only store adapters that have a status of 3 (which means enabled)
    $Script:EnabledSystemAdapters = GetObject "IANet_PhysicalEthernetAdapter" | Where-Object {($_.StatusInfo -eq 3)}  | Sort-Object -Property Name
    #Only store adapters that support DMiX or Extended DMiX capability or if it is intel vendor capable and this is an upgrade.
    $Script:Adapters = $Script:EnabledSystemAdapters | Where-Object {($_.Capabilities -eq 73) -or ($_.Capabilities -eq 74) -or (($_.Capabilities -eq 47) -and (($bUpSave -eq $true) -or ($bUpRestore -eq $true)))}
    #Save Teamable adapters to be referenced for teams
}

# Remove any present configuration files
Function RemoveOldFiles
{
    Write-Log $MyInvocation.MyCommand
    # check if the file exists before trying to remove it
    if (Test-Path $Script:ConfigPath -PathType Leaf)
    {
        Remove-Item $Script:ConfigPath
    }
    if (Test-Path $Script:ScriptPath\Saved_StaticIP.txt)
    {
        Remove-Item $Script:ScriptPath\Saved_StaticIP.txt
    }
}

# Save adapters on system to the configuration file
Function SaveAdapters
{
    Write-Log $MyInvocation.MyCommand
    WriteTextToConfigFile "AdaptersStart"

    $colProperty = "Name","OriginalDisplayName","PermanentAddress","PCIDeviceID","SlotID"
    WriteObjectToConfigFile $Script:EnabledSystemAdapters $colProperty

    WriteTextToConfigFile "AdaptersEnd"
    WriteTextToConfigFile ""
}

# Save OEM customization value
Function SaveOEMCustomizeableSetting
{
    Write-Log $MyInvocation.MyCommand
    WriteTextToConfigFile "OEMSettingStart"
    $OEMSetting = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\NCS2 -Name OEMCustomizeable -ErrorAction SilentlyContinue

    # If a value is present save it to the config file
    if ($null -ne $OEMSetting)
    {
        WriteObjectToConfigFile $OEMSetting OEMCustomizeable
    }

    WriteTextToConfigFile "OEMSettingEnd"
    WriteTextToConfigFile ""
}

# Save any legacy power management settings
Function SavePowerManagementSettings
{
    Write-Log $MyInvocation.MyCommand
    $colProperty = "Name","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"PermanentAddress","PCIDeviceID","SlotID"

    WriteTextToConfigFile "PMSettingsStart"

    foreach ($Adapter in $Script:Adapters)
    {
        try
        {
            #Get the Power Management settings for adapter
            $PowerSettingValues = InvokeMethod $Adapter GetPowerUsageOptions $null $null
            $PowerSettingProperties = @{Name = $Adapter.Name;
                                        CurrentValue = $PowerSettingValues.AutoPowerSaveModeEnabled;
                                        CurrentValues = {};
                                        PermanentAddress = $Adapter.PermanentAddress;
                                        PCIDeviceID = $Adapter.PCIDeviceID;
                                        SlotID = $Adapter.SlotID}

            # Check each Power Management setting to see if it NULL before trying to save it to the config file
            # if there is a value, create a custom object and save it to the config file
            if ($null -ne $PowerSettingValues.AutoPowerSaveModeEnabled)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "AutoPowerSaveModeEnabled" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.ReduceSpeedOnPowerDown)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "ReduceSpeedOnPowerDown" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.SmartPowerDown)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "SmartPowerDown" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.SavePowerNowEnabled)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "SavePowerNowEnabled" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
            if ($null -ne $PowerSettingValues.EnhancedASPMPowerSaver)
            {
                $Setting = New-Object PSObject -Property $PowerSettingProperties
                $Setting | Add-Member -Name "Caption" -Value  "EnhancedASPMPowerSaver" -MemberType NoteProperty
                WriteObjectToConfigFile $Setting $colProperty
            }
        }
        catch
        {
            Write-Log "Unable to Save Power Management Settings"
        }
    }

    WriteTextToConfigFile "PMSettingsEnd"
    WriteTextToConfigFile ""
}

# Some settings need to be saved in a certain order, this function stops them from being saved now so they can be saved later by returning
# whether it should be saved immidiately or later (using true or false).
Function SaveAdapterSettingLater($Setting, $bIPSetting)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$Setting = $Setting"
    Write-Log "`$bIPSetting = $bIPSetting"
    $bRet = $true

    # Don't save the settings now if it is an IP, DCB, performance profile, SRIOV, VMQueues, or NUMVF setting
    if (($bIPSetting -eq $false) -and
        ($Setting.GroupId -ne 12 ) -and
        ($Setting.GroupId -ne 8 ) -and
        ($Setting.Caption -ne "PerformanceProfile") -and
        ($Setting.Caption -ne "*FlowControl") -and
        ($Setting.Caption -ne "*PriorityVLANTag") -and
        ($Setting.Caption -ne "*SRIOV") -and
        ($Setting.Caption -ne "VMQueues") -and
        ($Setting.Caption -ne "*NumVFs"))
    {
        $bRet = $false
    }

    return $bRet
}

# Check if the given setting is an IP setting and save it in the IP config file and return if it is an IP setting (true or false).
# Depending on the device different properties of the object need to be saved.
Function SaveIPSetting($Setting,$DeviceType,$EnabledDHCP,$bSaveIPv6Settings)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$DeviceType = $DeviceType"
    Write-Log "`$Setting = $Setting"
    Write-Log "`$bEnabledDHCP = $EnabledDHCP"
    Write-Log "`$bSaveIPv6Settings = $bSaveIPv6Settings"
    $bIPSetting = $false
    $bSaveIpSetting = $false

    # Check if the passed in setting is one of these IP settings.
    # Some IP settings need DHCP enabled to restore.
    switch($Setting.Caption)
    {
        "IPAddress"
        {
            $bIPSetting = $true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "IPv4Address"
        {
            $bIPSetting=$true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "IPv6Address"
        {
            $bIPSetting=$true
            if ($bSaveIPv6Settings -eq $true)
            {
                $bSaveIpSetting = $true
            }
        }
        "PrefixLength"
        {
            $bIPSetting=$true
            if ($bSaveIPv6Settings -eq $true)
            {
                $bSaveIpSetting = $true
            }
        }
        "SubnetMask"
        {
            $bIPSetting=$true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "DefaultGateway"
        {
            $bIPSetting=$true
            if ($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "DefaultGatewayIPv6"
        {
            $bIPSetting=$true
            if ($bSaveIPv6Settings -eq $true)
            {
                $bSaveIpSetting = $true
            }
        }
        "NameServer"
        {
            $bIPSetting=$true
            if($EnabledDHCP -eq 0)
            {
                $bSaveIpSetting = $true
            }
        }
        "NameServerList"
        {
            $bIPSetting=$true
            $bSaveIpSetting = $true
        }
        "NetbiosOptions"
        {
            $bIPSetting=$true
            $bSaveIpSetting = $true
        }
    }

    # Save IP settings with different properties depending on the device type
    if ($bSaveIpSetting -eq $true -and $DeviceType -eq "Adapter")
    {
        $colProperty = "Name","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"PermanentAddress","PCIDeviceID","SlotID"
        WriteObjectToIPFile $Setting $colProperty
    }
    elseif ($bSaveIpSetting -eq $true -and $DeviceType -eq "Team")
    {
        $colProperty = "TeamName","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"}
        WriteObjectToIPFile $Setting $colProperty
    }
    elseif ($bSaveIpSetting -eq $true -and $DeviceType -eq "Vlan")
    {
        $colProperty = "ParentName","VLANID","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
        WriteObjectToIPFile $Setting $colProperty
    }

    return $bIPSetting
}

# Save the adapter settings
Function SaveAdapterSettings()
{
    Write-Log $MyInvocation.MyCommand
    $colProperty = "Name","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"},"PermanentAddress","PCIDeviceID","SlotID"

    # Save power management settings
    SavePowerManagementSettings

    WriteTextToConfigFile "AdapterSettingsStart"
    WriteTextToIPFile "AdapterIPSettingsStart"

    foreach ($Adapter in $Script:Adapters)
    {
        # Get the settings associated with the adapter
        $colSettings = GetAssociated $Adapter.path.path "ResultClass = IANet_AdapterSetting"
        $colSettings = $colSettings  | Sort-Object -Property Name,Caption

        # Check status of EnablDHCP for IP settings later
        $EnableDHCPCol = $colSettings | Where-Object {($_.Caption -eq "EnableDHCP")} | Select-Object $colProperty
        $EnableDHCPv6Col = $colSettings | Where-Object {($_.Caption -eq "EnableDHCPv6")} | Select-Object $colProperty
        $ManualIPv6Col = $colSettings | Where-Object {($_.Caption -eq "ManualIPv6")} | Select-Object $colProperty

        #Get the DHCP enable value for the specific adapter
        $EnableDHCP = $EnableDHCPCol | Where-Object {($_.Name -eq $Adapter.Name)}

        # Get the DHCPv6 enable value, and the ManualIPv6 value
        # EnableDHCPv6 is 0 (disabled) only if the "Managed Address Configuration Flag" and the "Other Stateful Configuration Flag" are both 0
        # (see https://blogs.technet.microsoft.com/teamdhcp/2009/03/03/dhcpv6-understanding-of-address-configuration-in-automatic-mode-and-installation-of-dhcpv6-server/)
        # ManualIPv6 is 1 (manual) only if the PrefixOrigin and SuffixOrigin of any IPv6 address are both "Manual" - other values indicate autoconfiguration
        $EnableDHCPv6 = $EnableDHCPv6Col | Where-Object {($_.Name -eq $Adapter.Name)}
        $ManualIPv6 = $ManualIPv6Col | Where-Object {($_.Name -eq $Adapter.Name)}

        $bSaveIPv6Settings = $false
        #if EnableDHCPv6 = 0 and ManualIPv6 = 1, we want to save IPv6 addresses and settings. Otherwise, all IPv6 addresses were autoconfigured and we don't need to save/restore them
        if($EnableDHCPv6.CurrentValue -eq 0 -and $ManualIPv6.CurrentValue -eq 1)
        {
            $bSaveIPv6Settings = $true
        }

        foreach ($Setting in $colSettings)
        {
            AddParentIDs $Setting

            #check if setting is an IP setting save them in the IP config file instead of the Saved_Config file
            $bIPSetting = SaveIPSetting $Setting "Adapter" $EnableDHCP.CurrentValue $bSaveIPv6Settings

            #Check to see if the setting should be saved later
            $bRet = SaveAdapterSettingLater $Setting $bIPSetting
            if ($bRet -eq $false)
            {
                WriteObjectToConfigFile $Setting $colProperty
            }
        }

        # Check if DCB is being updated and if so, don't save the settings so the default values are restored
        $RestoreDCB = $true
        $FCoEUpdate = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Prounstl -Name DCB_Update_FCoE  -ErrorAction SilentlyContinue
        if ($null -ne $FCoEUpdate)
        {
            if ($FCoEUpdate.DCB_Update_FCoE -eq 1)
            {
                #FCoE is changing don't save settings
                $RestoreDCB = $false
            }
        }
        $iSCSIUpdate = Get-ItemProperty -Path HKLM:\SOFTWARE\Intel\Prounstl -Name DCB_Update_iSCSI  -ErrorAction SilentlyContinue
        if ($null -ne $iSCSIUpdate)
        {
            if ($iSCSIUpdate.DCB_Update_iSCSI -eq 1)
            {
                #iSCSI is changing don't save settings
                $RestoreDCB = $false
            }
        }

        #Save *SRIOV after *VMQ
        $SRIOVSetting = $colSettings | Where-Object {($_.Caption -eq "*SRIOV")}
        WriteObjectToConfigFile $SRIOVSetting $colProperty

        #Save DCB Settings if this is not an upgrade or if it is an upgrade, and we are modifying DCB
        if (($Script:bUpSave -eq $false) -or ($RestoreDCB -eq $true))
        {
            $colDCBSettings = $colSettings | Where-Object {($_.GroupId -eq 12) -or ($_.GroupId -eq 8)}
            WriteObjectToConfigFile $colDCBSettings $colProperty
        }

        #Save the performance profile
        $ProfileSetting = $colSettings | Where-Object {($_.Caption -eq "PerformanceProfile")}
        WriteObjectToConfigFile $ProfileSetting $colProperty

        #Save the FlowControl after performance profile and DCB
        $FlowControlSetting = $colSettings | Where-Object {($_.Caption -eq "*FlowControl")}
        WriteObjectToConfigFile $FlowControlSetting $colProperty

        #Save the PriorityVLANTag after performance profile and DCB
        $PriorityVLANTagSetting = $colSettings | Where-Object {($_.Caption -eq "*PriorityVLANTag")}
        WriteObjectToConfigFile $PriorityVLANTagSetting $colProperty

        #Save VMQueues and *NUMVFs last
        $VMQQueuesSetting = $colSettings | Where-Object {($_.Caption -eq "VMQueues")}
        WriteObjectToConfigFile $VMQQueuesSetting $colProperty
        $NumVFsSetting = $colSettings | Where-Object {($_.Caption -eq "*NumVFs")}
        WriteObjectToConfigFile $NumVFsSetting $colProperty
    }

    WriteTextToConfigFile "AdapterSettingsEnd"
    WriteTextToConfigFile ""

    WriteTextToIPFile "AdapterIPSettingsEnd"
    WriteTextToIPFile ""
}

#Save team information
Function SaveTeams
{
    Write-Log $MyInvocation.MyCommand
    #Get current teams on system
    $colProperty = "TeamName","TeamMembers","TeamMode","PrimaryAdapter","SecondaryAdapter"
    $colItems = GetObject "IANet_TeamOfAdapters" | Select-Object $colProperty | Sort-Object -Property TeamName


    WriteTextToConfigFile "TeamsStart"

    # if there are teams on system save the team information to the config file.
    if ($colItems)
    {
        # convert Member arrays into a "single" value to save
        foreach ($item in $colItems)
        {
            $TeamMembers = $item.TeamMembers
            $item.TeamMembers = @()
            for (($i = 0); $i -lt $TeamMembers.Count; $i++)
            {
                $adapter = GetObject IANet_PhysicalEthernetAdapter | Where-Object {($_.OriginalDisplayName -eq $TeamMembers[$i])}
                $PermanentAddress = $adapter | Select-Object -ExpandProperty "PermanentAddress"
                $PCIDeviceID = $adapter | Select-Object -ExpandProperty "PCIDeviceID"
                $SlotID = $adapter | Select-Object -ExpandProperty "SlotID"
                $item.TeamMembers +=  $TeamMembers[$i] + "," + $PermanentAddress + "," + $PCIDeviceID + "," + $SlotID
            }
            $item.TeamMembers = $item.TeamMembers -join '|'
        }

        WriteObjectToConfigFile $colItems *
    }

    WriteTextToConfigFile "TeamsEnd"
    WriteTextToConfigFile ""
}

# Save team settings
Function SaveTeamSettings
{
    Write-Log $MyInvocation.MyCommand
    # Get the current team settings
    $colProperty = "TeamName","Caption","CurrentValue",@{expression={$_.CurrentValues -join ","};label="CurrentValues"}
    $colSettings = GetObject "IANet_TeamSetting" | Sort-Object -Property Name,Caption

    WriteTextToConfigFile "TeamSettingsStart"
    WriteTextToIPFile "TeamIPSettingsStart"

    # Check status of EnablDHCP for IP settings later
    $EnableDHCPCol = $colSettings | Where-Object {($_.Caption -eq "EnableDHCP")} | Select-Object $colProperty
    $EnableDHCPv6Col = $colSettings | Where-Object {($_.Caption -eq "EnableDHCPv6")} | Select-Object $colProperty
    $ManualIPv6Col = $colSettings | Where-Object {($_.Caption -eq "ManualIPv6")} | Select-Object $colProperty

    foreach ($Setting in $colSettings)
    {
        # Get the DHCP enable value for the specific Team
        $EnableDHCP = $EnableDHCPCol | Where-Object {($_.TeamName -eq $Setting.Name)}

        # Get the DHCPv6 enable value, and the ManualIPv6 value
        # EnableDHCPv6 is 0 (disabled) only if the "Managed Address Configuration Flag" and the "Other Stateful Configuration Flag" are both 0
        # (see https://blogs.technet.microsoft.com/teamdhcp/2009/03/03/dhcpv6-understanding-of-address-configuration-in-automatic-mode-and-installation-of-dhcpv6-server/)
        # ManualIPv6 is 1 (manual) only if the PrefixOrigin and SuffixOrigin of any IPv6 address are both "Manual" - other values indicate autoconfiguration
        $EnableDHCPv6 = $EnableDHCPv6Col | Where-Object {($_.TeamName -eq $Setting.Name)}
        $ManualIPv6 = $ManualIPv6Col | Where-Object {($_.TeamName -eq $Setting.Name)}

        $bSaveIPv6Settings = $false
        # if EnableDHCPv6 = 0 and ManualIPv6 = 1, we want to save IPv6 addresses and settings. Otherwise, all IPv6 addresses were autoconfigured and we don't need to save/restore them
        if ($EnableDHCPv6.CurrentValue -eq 0 -and $ManualIPv6.CurrentValue -eq 1)
        {
            $bSaveIPv6Settings = $true
        }

        # Save the IP Settings in the IP config file
        $bIPSetting = SaveIPSetting $Setting "Team" $EnableDHCP.CurrentValue $bSaveIPv6Settings

        if ($bIPSetting -eq $false)
        {
            WriteObjectToConfigFile $Setting $colProperty
        }
    }

    WriteTextToConfigFile "TeamSettingsEnd"
    WriteTextToConfigFile ""

    WriteTextToIPFile "TeamIPSettingsEnd"
    WriteTextToIPFile ""
}

# Save vlan information
Function SaveVlans
{
    Write-Log $MyInvocation.MyCommand
    #Get the vlans on the system
    $colProperty = "ParentName", "VLANID", "VLANNAME", "ParentPermanentAddress", "ParentPCIDeviceID", "ParentSlotID"
    $colVlans = GetObject "IANet_VLAN" | Sort-Object -Property Parent,VLANID


    WriteTextToConfigFile "VlansStart"

    # Save untagged vlan last if there are vlans on the system
    if ($colVlans)
    {
        $colTaggedVlans = $colVlans | Where-Object {($_.VLANID -ne 0)}
        WriteObjectToConfigFile $colTaggedVlans $colProperty
        $colUnTaggedVlans = $colVlans | Where-Object {($_.VLANID -eq 0)}
        WriteObjectToConfigFile $colUnTaggedVlans $colProperty
    }

    WriteTextToConfigFile "VlansEnd"
    WriteTextToConfigFile ""
}

# Save vlan settings
Function SaveVlanSettings
{
    Write-Log $MyInvocation.MyCommand
    #Get vlan settings on system
    $colProperty = "ParentName","Name","VLANID","Caption","CurrentValue","ParentPermanentAddress","ParentPCIDeviceID","ParentSlotID"
    $colSettings = GetObject "IANet_VLANSetting" | Sort-Object -Property Name,Caption

    WriteTextToConfigFile "VlanSettingsStart"
    WriteTextToIPFile "VlanIPSettingsStart"

    # Check status of EnablDHCP for IP settings later
    $EnableDHCPCol = $colSettings | Where-Object {($_.Caption -eq "EnableDHCP")} | Select-Object $colProperty
    $EnableDHCPv6Col = $colSettings | Where-Object {($_.Caption -eq "EnableDHCPv6")} | Select-Object $colProperty
    $ManualIPv6Col = $colSettings | Where-Object {($_.Caption -eq "ManualIPv6")} | Select-Object $colProperty

    foreach ($Setting in $colSettings)
    {
        # Get the DHCP enable value for the specific adapter
        $EnableDHCP = $EnableDHCPCol | Where-Object {($_.Name -eq $Setting.Name)}

        # Get the DHCPv6 enable value, and the ManualIPv6 value
        # EnableDHCPv6 is 0 (disabled) only if the "Managed Address Configuration Flag" and the "Other Stateful Configuration Flag" are both 0
        # (see https://blogs.technet.microsoft.com/teamdhcp/2009/03/03/dhcpv6-understanding-of-address-configuration-in-automatic-mode-and-installation-of-dhcpv6-server/)
        # ManualIPv6 is 1 (manual) only if the PrefixOrigin and SuffixOrigin of any IPv6 address are both "Manual" - other values indicate autoconfiguration
        $EnableDHCPv6 = $EnableDHCPv6Col | Where-Object {($_.Name -eq $Setting.Name)}
        $ManualIPv6 = $ManualIPv6Col | Where-Object {($_.Name -eq $Setting.Name)}

        $bSaveIPv6Settings = $false
        # if EnableDHCPv6 = 0 and ManualIPv6 = 1, we want to save IPv6 addresses and settings. Otherwise, all IPv6 addresses were autoconfigured and we don't need to save/restore them
        if ($EnableDHCPv6.CurrentValue -eq 0 -and $ManualIPv6.CurrentValue -eq 1)
        {
            $bSaveIPv6Settings = $true
        }

        # Save the IP Settings in the IP config file
        $bIPSetting = SaveIPSetting $Setting "Vlan" $EnableDHCP.CurrentValue $bSaveIPv6Settings

        if ($bIPSetting -eq $false)
        {
            WriteObjectToConfigFile $Setting $colProperty
        }
    }

    WriteTextToConfigFile "VlanSettingsEnd"
    WriteTextToConfigFile ""

    WriteTextToIPFile "VlanIPSettingsEnd"
    WriteTextToIPFile ""
}

#Save NICPART settings
Function SaveNICPARTSettings
{
    Write-Log $MyInvocation.MyCommand
    $colProperty = "Name","PartitionNumber","Identifier","MinBWPercent","MaxBWPercent"

    WriteTextToConfigFile "NICPARTSettingsStart"

    #start lock
    BeginApply

    try
    {
        #Get the partition information for all partitions on system and save them
        $PartitionArray = InvokeMethod $Script:service GetPartitionsForPort "szDeviceID" $null
            WriteObjectToConfigFile $PartitionArray.Partitions $colProperty
    }
    catch
    {
        Write-log "Unable to Save NIC Partition Settings"
    }

    #release lock
    ReleaseLock

    WriteTextToConfigFile "NICPARTSettingsEnd"
    WriteTextToConfigFile ""
}

#Remove teams and vlans on system
Function RemoveTeamsAndVlans
{
    Write-Log $MyInvocation.MyCommand

    $IANetObjs = @('IANet_TeamOfAdapters','IANet_Vlan')

    foreach ($IANetObj in $IANetObjs)
    {
        $colDevices = @()
        $colDevices += GetObject $IANetObj

        BeginApply

        foreach ($Device in $colDevices)
        {
            try
            {
                DeleteObject $Device
            }
            catch
            {
                Write-Log "Unable to DeleteObject `$Device $Device"
            }
        }

        ReleaseLock
    }
}

# Check that adapters in configuration file are present and enabled on the system.
Function CheckAdaptersExist
{
    Write-Log $MyInvocation.MyCommand
    $Script:colImportAdapters  = ReadFromConfigFile "Adapters"
    if ($null -ne $Script:colImportAdapters )
    {
        foreach ($Adapter in $Script:colImportAdapters)
        {
            # Find the matching file adapter using the appropriate property to compare name,address or ID
            if ($Script:BDF -eq $false)
            {
                if ($bUpRestore -eq $true)
                {
                    $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $Adapter.PermanentAddress)}
                }
                else
                {
                    $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $Adapter.PCIDeviceID)}
                }

                if ($null -eq $ImportedAdapter)
                {
                    $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.Name -eq $Adapter.Name)}
                }
            }
            else
            {
                $ImportedAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $Adapter.SlotID)}
            }


            if ($null -eq $ImportedAdapter)
            {
                Write-Warning UnableToFindDevice
                Write-Log "Invalid `$Adapter = $Adapter"
            }
        }
    }
}

# Restore custome OEM value
Function RestoreOEMCustomizeableSetting
{
    Write-Log $MyInvocation.MyCommand
    $ImportOEMSetting = ReadFromConfigFile "OEMSetting"
    if ($null -ne $ImportOEMSetting)
    {
        Write-Host $Messages.SettingOEMValue
        Set-ItemProperty -Path HKLM:\SOFTWARE\Intel\Network_Services\NCS2 -Name OEMCustomizeable -Value $ImportOEMSetting.OEMCustomizeable -Type DWord
    }
}

Function RestoreSettings($Setting, $stringClass, $enumClass)
{
    if ($Setting)
    {
        # IPv6 Address and DefaultGateway must be set with netsh, currently no API to set setting in Middleware
        if ($Setting.Caption -eq "IPv6Address")
        {
            # netsh interface ipv6 add address $ConnectionName $Address
            $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
            $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex


            # delete current IPv6 addresses to overwrite them with saved config, if there are no addresses then we catch the exception and continue since there is nothing to delete.
            try
            {
                $AllAddresses = Get-WmiObject Win32_NetworkAdapterConfiguration -Namespace "root/cimv2" | Where-Object {($_.InterfaceIndex -eq $ConnectionInterfaceIndex)} | Select-Object -ExpandProperty IPAddress -ErrorAction SilentlyContinue
            }
            catch
            {
                Write-Log "No IPv6Address found."
            }

            $IPv6Array = @()
            foreach ($Address in $AllAddresses)
            {
                if (-not $Address.Contains("."))
                {
                    $IPv6Array += $Address
                }
            }

            #wrap interfaceindex in quotes for netsh
            $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

            foreach ($IP in $IPv6Array)
            {
                $DeleteCommand = "netsh interface ipv6 delete address $ConnectionInterfaceIndex $IP"
                Invoke-Expression $DeleteCommand
            }

            foreach ($Value in $Setting.CurrentValues) #CurrentValues is a list of IPv6Addresses split by comma, set each address
            {
                $Command = "netsh interface ipv6 add address $ConnectionInterfaceIndex $Value"
                Invoke-Expression $Command
            }
        }

        elseif ($Setting.Caption -eq "DefaultGatewayIPv6")
        {
            # netsh interface ipv6 add route ::/0 $ConnectionName $Address
            $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
            $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex

            # delete current IPv6 gateways to overwrite them with saved config, if there are no gateways then we catch the exception and continue since there is nothing to delete.
            try
            {
                $AllGateways = Get-WmiObject Win32_NetworkAdapterConfiguration -Namespace "root/cimv2" | Where-Object {($_.InterfaceIndex -eq $ConnectionInterfaceIndex)} | Select-Object -ExpandProperty DefaultIPGateway -ErrorAction SilentlyContinue
            }
            catch
            {
                Write-Log "No DefaultGatewayIPv6 found."
            }

            $IPv6Array = @()
            foreach ($Address in $AllGateways)
            {
                if (-not $Address.Contains("."))
                {
                    $IPv6Array += $Address
                }
            }

            #wrap interfaceindex in quotes for netsh
            $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

            foreach ($IP in $IPv6Array)
            {
                $DeleteCommand = "netsh interface ipv6 delete route ::/0 $ConnectionInterfaceIndex $IP"
                Invoke-Expression $DeleteCommand
            }

            foreach ($Value in $Setting.CurrentValues)
            {
                $Command = "netsh interface ipv6 add route ::/0 $ConnectionInterfaceIndex $Value"
                Invoke-Expression $Command
            }
        }

        elseif ($Setting.Caption -eq "EnableDHCPv6")
        {
            $EnableDHCPv6Setting = Get-WmiObject $enumClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "EnableDHCPv6")} | Select-Object -ExpandProperty CurrentValue

            if (($Setting.CurrentValue -eq "0") -and ($EnableDHCPv6Setting.CurrentValue -eq "1"))
            {
                $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
                $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex

                $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

                $dhcpv6routercmd = "netsh int ipv6 set int $ConnectionInterfaceIndex routerdiscovery=disabled managedaddress=disabled otherstateful=disabled"

                Invoke-Expression $dhcpv6routercmd
                Start-Sleep -s 1
            }

            elseif (($Setting.CurrentValue -eq "1") -and ($EnableDHCPv6Setting.CurrentValue -eq "0"))
            {
                $ConnectionName = Get-WmiObject $stringClass -Namespace "root/intelncs2" | Where-Object {($_.Name -eq $Setting.Name) -and ($_.Caption -eq "ConnectionName")} | Select-Object -ExpandProperty CurrentValue
                $ConnectionInterfaceIndex = Get-WmiObject Win32_NetworkAdapter -Namespace "root/cimv2" | Where-Object {($_.NetConnectionID -eq $ConnectionName)} | Select-Object -ExpandProperty InterfaceIndex

                $ConnectionInterfaceIndex = "`"$($ConnectionInterfaceIndex)`""

                $dhcpv6managedcmd = "netsh int ipv6 set int $ConnectionInterfaceIndex managedaddress=enabled"

                Invoke-Expression $dhcpv6managedcmd
                Start-Sleep -s 1
            }
        }
        else
        {
            $result = SetSetting $Setting
            Write-Log "SetSetting $result"
        }
    }
}

# Restore adapter settings
Function RestoreAdapterSettings($colImportedSettings)
{
    Write-Log $($MyInvocation.MyCommand)
    Write-Log "`$colImportedSettings = $colImportedSettings"
    # Get adapter settings on system
    $colSetting = GetObject "IANet_AdapterSetting" | Sort-Object -Property Name,Caption
    # Add parent identifiers to settings on system
    foreach ($cSetting in $colSetting)
    {
        AddParentIDs $cSetting
    }

    # start lock
    BeginApply

    # Loop through the passed in settings from the config file
    foreach ($ImportedSetting in $colImportedSettings) #for each setting in AdapterIPSettings
    {
        try
        {
            # Find the matching system setting using the appropriate property to compare name,address or ID
            if ($Script:BDF -eq $false)
            {
                # finds the setting with the same caption and device name
                if (($bUpRestore -eq $true))
                {
                    $Setting = $colSetting | Where-Object {($_.PermanentAddress -eq $ImportedSetting.PermanentAddress) -and ($_.Caption -eq $ImportedSetting.Caption)}
                }
                else
                {
                    $Setting = $colSetting | Where-Object {($_.PCIDeviceID -eq $ImportedSetting.PCIDeviceID) -and ($_.Caption -eq $ImportedSetting.Caption)}
                }

                if ($null -eq $Setting)
                {
                    $Setting = $colSetting | Where-Object {($_.Name -eq $ImportedSetting.Name) -and ($_.Caption -eq $ImportedSetting.Caption)}
                }
            }
            else
            {
                $Setting = $colSetting | Where-Object {($_.SlotID -eq $ImportedSetting.SlotID) -and ($_.Caption -eq $ImportedSetting.Caption)}
            }

            # If the setting in the file can't be found on the system, continue to the next setting
            if ($null -eq $Setting)
            {
                Write-Log "Not Restored: $ImportedSetting"
                continue
            }

            # See if 'CurrentValues' is a property over the system setting, if so set that value from the config file setting
            if (($Setting.Properties | Select-Object -ExpandProperty Name) -contains "CurrentValues")
            {
                # If no value was present from the file setting, then set it to empty
                if ($null -eq $ImportedSetting.CurrentValues)
                {
                    $Setting.CurrentValues = {}
                }
                else
                {
                    $Setting.CurrentValues = $ImportedSetting.CurrentValues.Split(",")
                }
            }
            else
            {
                $Setting.CurrentValue = $ImportedSetting.CurrentValue
            }

            RestoreSettings $Setting "IANet_AdapterSettingString" "IANet_AdapterSettingEnum"
        }
        catch
        {
            $ErrorMessagePart1 = [string]$Messages.UnableToSetPart1
            $ErrorMessagePart2 = [string]$Messages.UnableToSetPart2
            # Unable to set ... on ...
            Write-Log "$ErrorMessagePart1 $($Setting.Caption) $ErrorMessagePart2 $($Setting.Name)"
        }
    }

    ReleaseLock
}

# Set the primary and secondary adapters on the team
Function RestorePrimaryAndSecondary($Team)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$Team = $Team"
    Write-Host $Messages.RestoringTeamAdapters $Team.TeamName

    # Find the adapter objects from the configuration file that match the team primary and secondary adapter names
    $PrimaryImportAdapter = $Script:colImportAdapters | Where-Object {$_.OriginalDisplayName -eq $Team.PrimaryAdapter}
    $SecondaryImportAdapter = $Script:colImportAdapters | Where-Object {$_.OriginalDisplayName -eq $Team.SecondaryAdapter}

    # Find the matching system adapter using the appropriate property to compare name,address or ID
    if ($Script:BDF -eq $false)
    {
        if ($bUpRestore)
        {
            $PrimaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $PrimaryImportAdapter.PermanentAddress)}
        }
        else
        {
            $PrimaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $PrimaryImportAdapter.PCIDeviceID)}
        }
    }
    else
    {
        $PrimaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $PrimaryImportAdapter.SlotID)}
    }

    if ($Script:BDF -eq $false)
    {
        if ($bUpRestore -eq $true)
        {
            $SecondaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $SecondaryImportAdapter.PermanentAddress)}
        }
        else
        {
            $SecondaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $SecondaryImportAdapter.PCIDeviceID)}
        }
    }
    else
    {
        $SecondaryAdapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $SecondaryImportAdapter.SlotID)}
    }

    # Set the primary and secondary adapters on the team
    $Team = GetObject "IANet_TeamOfAdapters" | Where-Object {$_.TeamName -eq $Team.TeamName} | Sort-Object -Property TeamName
    $param = $Team.psbase.GetMethodParameters("ValidateAndSetAdapterPriority")
    $param.PrimaryAdapter = $PrimaryAdapter
    $param.SecondaryAdapter = $SecondaryAdapter
    $param.SetValuesOnNoError = 2
    $value = $Team.psbase.InvokeMethod("ValidateAndSetAdapterPriority", $param, $invokeoptions)
    Write-Log $Value
}

# Restore teams from configuration file
Function RestoreTeams
{
    Write-Log $MyInvocation.MyCommand
    # Get the team objects from the configuration file
    $colImportTeams = ReadFromConfigFile "Teams"

    if ($null -ne $colImportTeams)
    {
        BeginApply

        foreach ($Team in $colImportTeams)
        {
            Write-Host $Messages.Restoring $Team.TeamName

            try
            {
                # Create a new team object
                $classdef = NewObject root\intelncs2:IANet_TeamOfAdapters

                # fill in the CreateTeam's method parameters with the new object
                $param = $classdef.psbase.GetMethodParameters("CreateTeam2Ex")

                $TeamedAdapters = $Team.TeamMembers.split("|")
                $param.Adapters = @()

                $bCreateTeam = $true
                # Set the teammembers for creation
                foreach ($Adapter in $TeamedAdapters)
                {
                    $Adapter = $Adapter.Split(",")
                    $AdapterName = $Adapter[0]
                    $AdapterPermanentAddress = $Adapter[1]
                    $AdapterPCIDeviceID = $Adapter[2]
                    $AdapterSlotID = $Adapter[3]
                    $Adapter = $null

                    # Find the matching system adapter using the appropriate property to compare address or ID
                    if ($Script:BDF -eq $false)
                    {
                        if ($bUpRestore -eq $true)
                        {
                            $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.PermanentAddress -eq $AdapterPermanentAddress)}
                        }
                        else
                        {
                            $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.PCIDeviceID -eq $AdapterPCIDeviceID)}
                        }

                        # Older versions of SaveRestore.ps1 won't have the Adapter's PermanentAddress, PCIDeviceID, or SlotID saved, so use the name.
                        if ($null -eq $Adapter)
                        {
                            $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.Name -eq $AdapterName)}
                        }
                    }
                    else
                    {
                        $Adapter = $Script:EnabledSystemAdapters | Where-Object {($_.SlotID -eq $AdapterSlotID)}
                    }

                    # If a teammember can't be found output an error and breakout of team creation
                    if ($null -eq $Adapter)
                    {
                        Write-Warning UnableToFindDevice
                        Write-Log $AdapterName
                        $bCreateTeam = $false
                        break
                    }
                    $param.Adapters += $Adapter
                }

                # If an error was found, don't try and create the team
                if ($bCreateTeam -eq $false)
                {
                    Continue
                }

                $param.TeamMode = $Team.TeamMode
                $param.TeamName = $Team.TeamName

                # the invoke options
                $invokeoptions = New-Object System.Management.InvokeMethodOptions
                $invokeoptions.Context = $Script:context

                # call the CreateTeam method to create the new team
                $null = $classdef.psbase.InvokeMethod("CreateTeam2Ex", $param, $invokeoptions)

                # Set primary and secondary adapters
                RestorePrimaryAndSecondary $Team
            }
            catch
            {
                Write-Log "Unable to Restore Teams"
            }
        }

        ReleaseLock
    }
}

#Restore team settings
Function RestoreTeamSettings($colImportSettings)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$colImportSettings = $colImportSettings"
    # Get the current team settings
    $colSetting = GetObject "IANet_TeamSetting" | Sort-Object -Property Name,Caption

    BeginApply

    foreach ($Setting in $colSetting)
    {
        try
        {
            # If the current setting using 'CurrentValues' then set that value.
            if (($Setting.Properties | Select-Object -ExpandProperty Name) -contains "CurrentValues")
            {
                $ValueObject = $colImportSettings | Where-Object {($_.TeamName -eq $Setting.Name) -and ($_.Caption -eq $Setting.Caption)} | Select-Object CurrentValues
                if ($null -eq $ValueObject.CurrentValues)
                {
                    $Setting.CurrentValues = {}
                }
                else
                {
                    $Setting.CurrentValues = $ValueObject.CurrentValues.Split(",")
                }
            }
            else
            {
                $ValueObject = $colImportSettings | Where-Object {($_.TeamName -eq $Setting.Name) -and ($_.Caption -eq $Setting.Caption)} | Select-Object CurrentValue
                $Setting.CurrentValue = $ValueObject.CurrentValue
            }

            #If a matching system setting was found set it to the restore value
            if($null -ne $ValueObject)
            {
                RestoreSettings $Setting "IANet_TeamSettingString" "IANet_TeamSettingEnum"
            }

        }
        catch
        {
            $ErrorMessagePart1 = [string]$Messages.UnableToSetPart1
            $ErrorMessagePart2 = [string]$Messages.UnableToSetPart2
            # Unable to set ... on ...
            Write-Log "$ErrorMessagePart1 $($Setting.Caption) $ErrorMessagePart2 $($Setting.Name)"
        }
    }

    ReleaseLock
}

Function RestoreVlans
{
    Write-Log $MyInvocation.MyCommand
    # Get vlan objects from config file
    $colImportVlans = ReadFromConfigFile "Vlans"

    if ($null -ne $colImportVlans)
    {
        foreach ($Vlan in $colImportVlans)
        {
            # Start lock
            BeginApply
            try
            {
                # Find the matching system adapter using the appropriate property to compare name,address or ID
                if ($Script:BDF -eq $false)
                {
                    if ($bUpRestore -eq $true)
                    {
                        $Adapter = $Script:Adapters | Where-Object {($_.PermanentAddress -eq $Vlan.ParentPermanentAddress)}
                    }
                    else
                    {
                        $Adapter = $Script:Adapters | Where-Object {($_.PCIDeviceID -eq $Vlan.ParentPCIDeviceID)}
                    }

                    # Older versions of SaveRestore.ps1 won't have the Adapter's PermanentAddress, PCIDeviceID, or SlotID saved, so use the name.
                    if ($null -eq $Adapter)
                    {
                        $Adapter = $Script:Adapters | Where-Object {($_.Name -eq $Vlan.ParentName)}
                    }
                }
                else
                {
                    $Adapter = $Script:Adapters | Where-Object {($_.SlotID -eq $Vlan.ParentSlotID)}
                }

                if ($null -ne $Adapter)
                {
                    # Get the vlan creation class associated with adapter parent object
                    $VlanCreator = GetAssociated $Adapter.path.path "ResultClass = IANet_802dot1QVLANService" | Where-Object {1}
                }
                # Check if vlan needs to be created on a team
                else
                {
                    # Get the logical ethernet adapter object for the team parent
                    $Team = GetObject "IANet_LogicalEthernetAdapter" | Where-Object {( NScompare $_.Name $Vlan.ParentName )}

                    if ($null -ne $Team)
                    {
                        # Get the vlan creation class associated with team parent object
                        $VlanCreator = GetAssociated $Team.path.path "ResultClass = IANet_802dot1QVLANService" | Where-Object {1}
                    }
                }

                # If the vlan creation class was found continue on to create the vlan
                if ($null -ne $VlanCreator)
                {
                    Write-Host $Messages.Restoring $Vlan.VlanName

                    # fill in the CreateVlan's method parameters
                    $param = $VlanCreator.psbase.GetMethodParameters("CreateVlan")

                    $param.Name = $Vlan.VLANNAME
                    $param.VLANNumber = $Vlan.VLANID

                    # the invoke options
                    $invokeoptions = New-Object System.Management.InvokeMethodOptions
                    $invokeoptions.Context = $Script:context

                    # call the CreateVlan method to create the new vlan
                    $null = $VlanCreator.psbase.InvokeMethod("CreateVlan", $param, $invokeoptions)
                }
            }
            catch
            {
                Write-Log "Unable to Restore VLANs"
            }
            ReleaseLock
        }
    }
}

# Restore vlan settings
Function RestoreVlanSettings($colImportSettings)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "`$colImportSettings = $colImportSettings"
    # Get the current Vlan settings
    $vlanSettings = GetObject "IANet_VLANSetting" | Sort-Object -Property ParentName,VLANID,Caption

    # Start lock
    BeginApply

    foreach ($Setting in $vlanSettings)
    {
        try
        {
            if ($Script:BDF -eq $false)
            {
                #finds the setting with the same caption and device name
                if ($bUpRestore -eq $true)
                {
                    $ValueObject = $colImportSettings | Where-Object {($_.ParentPermanentAddress -eq $Setting.ParentPermanentAddress) -and ($_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
                }
                else
                {
                    $ValueObject = $colImportSettings | Where-Object {($_.ParentPCIDeviceID -eq $Setting.ParentPCIDeviceID) -and ($_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
                }

                # Older versions of SaveRestore.ps1 won't have the Adapter's PermanentAddress, PCIDeviceID, or SlotID saved, so use the name.
                if ($null -eq $ValueObject)
                {
                    $ValueObject = $colImportSettings | Where-Object {( NSCompare $_.ParentName $Setting.ParentName ) -and ( $_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
                }
            }
            else
            {
                $ValueObject = $colImportSettings | Where-Object {($_.ParentSlotID -eq $Setting.ParentSlotID) -and ($_.Caption -eq $Setting.Caption) -and ($_.VLANID -eq $Setting.VLANID)}
            }

            if (($Setting.Properties | Select-Object -ExpandProperty Name) -contains "CurrentValues")
            {
                if ($null -eq $ValueObject.CurrentValues)
                {
                    $Setting.CurrentValues = {}
                }
                else
                {
                    $Setting.CurrentValues = $ValueObject.CurrentValues.Split(",")
                }
            }
            else
            {
                $Setting.CurrentValue = $ValueObject.CurrentValue
            }

            # If a matching system setting was found set it to the restore value
            if($null -ne $ValueObject)
            {
                RestoreSettings $Setting "IANet_VLANSettingString" "IANet_VLANSettingEnum"
            }

        }
        catch
        {
            $ErrorMessagePart1 = [string]$Messages.UnableToSetPart1
            $ErrorMessagePart2 = [string]$Messages.UnableToSetPart2
            # Unable to set ... on ...
            Write-Log "$ErrorMessagePart1 $($Setting.Caption) $ErrorMessagePart2 $($Setting.Name)"
        }
    }

    ReleaseLock
}

# Restore the IP settings from the IP config file
Function RestoreIPSettings
{
    Write-Log $MyInvocation.MyCommand
    # Restore Adapter IP settings from the IP config file
    $colImportSettings = ReadFromIPFile "AdapterIPSettings"

    if ($null -ne $colImportSettings)
    {
        foreach ($setting in $colImportSettings)
        {
            # If the staticip is set on a disconnected adapter from cmdline tools (netsh, New-NetIPAddress, Set-NetIPInterface, etc)
            # the private 169.254.x.x IP that Microsoft assigns when it can't get an address from the DHCP server
            # comes along and saverestore ends up restoring that IP with a 255.255.0.0 subnetmask instead.
            # Only capture the first ip, which is what static ip and subnetmask the user intended to set
            # ignore the 169.254.x.x IP and 255.255.0.0 subnetmask
            if (($setting.Caption -eq 'IPv4Address') -or
                ($setting.Caption -eq 'SubnetMask'))
            {
                if (($setting.CurrentValues.ToCharArray()) -contains [char]',')
                {
                    $setting.CurrentValues = $setting.CurrentValues.Substring(0, $setting.CurrentValues.lastIndexOf(','))
                }
            }
        }

        Write-Host $Messages.RestoringAdapterIP
        RestoreAdapterSettings $colImportSettings
    }

    if ($Script:bANSInstalled)
    {
        # Restore Team IP settings from the IP config file
        $colImportSettings = ReadFromIPFile "TeamIPSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringTeamIP
            RestoreTeamSettings $colImportSettings $colSetting
        }

        # Restore Vlan IP settings from the IP config file
        $colImportSettings = ReadFromIPFile "VlanIPSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringVLANIP
            RestoreVlanSettings $colImportSettings
        }
    }
}

# Restore devices and settings from configuration files
Function RestoreAdvancedSettingsAndDevices
{
    Write-Log $MyInvocation.MyCommand
    #Restore Adapter avanced settings from config file
    $colImportSettings = ReadFromConfigFile "AdapterSettings"
    if ($null -ne $colImportSettings)
    {
        Write-Host $Messages.RestoringAdapterSettings
        RestoreAdapterSettings $colImportSettings
    }

    if (!$Script:bANSInstalled)
    {
        Write-Warning UnableToFindANS
    }
    else
    {
        #Restore Teams
        RestoreTeams

        #Restore Team advanced settings from config file
        $colImportSettings = ReadFromConfigFile "TeamSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringTeamSettings
            RestoreTeamSettings $colImportSettings
        }

        #Restore Vlans
        RestoreVlans

        #Restore Vlan avanced settings from config file
        $colImportSettings = ReadFromConfigFile "VlanSettings"
        if ($null -ne $colImportSettings)
        {
            Write-Host $Messages.RestoringVLANSettings
            RestoreVlanSettings $colImportSettings
        }
    }

    if (Test-Path $Script:ScriptPath\Saved_StaticIP.txt)
    {
        #Restore IP Settings from IP config file
        RestoreIPSettings
    }
}

#Restore NICPART settings
Function RestoreNICPARTSettings
{
    Write-Log $MyInvocation.MyCommand
    #Restore NICPART settings from config file
    $colImportPartitions = ReadFromConfigFile "NICPARTSettings"
    Write-Host $Messages.RestoringAdapterPartition

    BeginApply

    try
    {
        if ($null -ne $colImportPartitions)
        {
            #Get the partition information for all partitions on system and save them
            $PartitionObject = InvokeMethod $Script:service GetPartitionsForPort "szDeviceID" $null
            $PartitionArray = $PartitionObject.Partitions


            foreach ($Partition in $PartitionArray)
            {
                $ImportPartition = $colImportPartitions | Where-Object {($_.PartitionNumber -eq $Partition.PartitionNumber) -and ($_.Identifier -eq $Partition.Identifier)}

                if ($null -ne $ImportPartition)
                {
                    if ($Script:BDF -eq $true)
                    {
                        $Partition.Name = "IGNORE"
                        $Partition.DeviceGuid = "IGNORE"
                    }
                    else
                    {
                        $Partition.Identifier = "IGNORE"
                        $Partition.DeviceGuid = "IGNORE"
                    }

                    $Partition.MinBWPercent = $ImportPartition.MinBWPercent
                    $Partition.MaxBWPercent = $ImportPartition.MaxBWPercent
                }
            }

            $Output = InvokeMethod $Script:service ValidateAndSetBAndwidthsForPort "Partitions","SetValuesOnNoError" $PartitionArray,"1"
        }
    }
    catch
    {
        Write-Log "Unable to Restore NIC Partition Settings"
    }

    ReleaseLock
}


<#
.SYNOPSIS
    Make sure users know about BDF parameter when using NICPART enabled adapters
.DESCRIPTION
    By default, the save restore script uses the adapter's friendly name to match the name of the adapter
    in the configuration file to adapter in the system.  Because of the way Windows enumerates devices and
    assigns friendly names, the name of the adapter saved in the configuration file may not be the exact
    same adapter/port/partition on another system. The /bdf command line option should be used when
    restoring settings of NICPART enabled devices on multiple systems.
.NOTES
    This must be invoked after GetAdaptersOnSystem - as it relies on the contents of $Script:Adapters.
    The global variable is used to save time when restoring settings on user systems (to avoid polling
    the system again just to get the same data).
#>
function CheckNICPART_BDFWarning
{
    Write-Log $MyInvocation.MyCommand
    $NICPART = $Script:Adapters | Where-Object {($_.Capabilities -eq 76)}
    if ($NICPART -and $Script:BDF -eq $false)
    {
        Write-Host $Messages.NICPartitioningDetected
    }

}

<#
.SYNOPSIS
    "No Space Compare" Compares two strings without whitespace
.DESCRIPTION
    Intel(R) Network Connections Software recently introduced a change to ANS team prefixes. The old format
    was "TEAM : ". The new format is "TEAM: ". To preserve user configurations during an upgrade it is
    necessary to compare some devices without space differences.
#>
function NSCompare ($s1, $s2)
{
    Write-Log $MyInvocation.MyCommand
    Write-Log "Comparing `$s1=$s1 and `$s2=$s2..."
    $s1temp = $s1 -replace '\s', ''
    $s2temp = $s2 -replace '\s', ''
    return ($s1temp -eq $s2temp)
}

#-----------------------------------------------------[Execution]------------------------------------------------------

# Determine which action was specified by the user
switch($Script:Action)
{
    "Save" {$Script:bSave=$true
            Write-Host $Messages.PerformingSave
            break}
    "UpSave" {$Script:bUpSave=$true
            Write-Host $Messages.PerformingUpgradeSave
            break}
    "Restore" {$Script:bRestore=$true
            Write-Host $Messages.PerformingRestore
            break}
    "UpRestore" {$Script:bUpRestore=$true
            Write-Host $Messages.PerformingUpgradeRestore
            break}
    "Remove" {$Script:bRemove=$true
            Write-Host $Messages.PerformingRemove
            break}
    "?" {PrintUsage}
    "help"{PrintUsage}
    default {PrintUsage}
}

CheckForAdminRights
CheckForDMiXInstall
CheckForANSInstall

# Setup the save/restore path
SetupSaveRestoreLocation

# If a save is the primary action
if (($Script:bSave -eq $true) -or ($Script:bUpSave -eq $true))
{
    GetAdaptersOnSystem

    RemoveOldFiles

    SaveAdapters

    SaveOEMCustomizeableSetting

    SaveAdapterSettings

    if ($Script:bANSInstalled)
    {
        SaveTeams

        SaveTeamSettings

        SaveVlans

        SaveVlanSettings
    }

    SaveNICPARTSettings
}
# If a restore is the primary action
elseif (($Script:bRestore -eq $true) -or ($Script:bUpRestore -eq $true))
{
    CheckIfConfigFileExists

    if ($Script:bANSInstalled)
    {
        RemoveTeamsAndVlans
    }

    GetAdaptersOnSystem

    CheckNICPART_BDFWarning

    CheckAdaptersExist

    RestoreOEMCustomizeableSetting

    RestoreAdvancedSettingsAndDevices

    RestoreNICPARTSettings
}
# If remove is the primary action
elseif ($Script:bRemove -eq $true)
{
    RemoveTeamsAndVlans
}


# SIG # Begin signature block
# MIIocQYJKoZIhvcNAQcCoIIoYjCCKF4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAg5XcJXSwWFqph
# dupHD17uM3jd+S7JJX0NITYAtsMChqCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1TCCFdECAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEIMjIQRxcpfJhSpwoVocMxvnr
# z62LiXT7IUQ/DnDGqETTMA0GCSqGSIb3DQEBAQUABIIBgCOWlnmmzN+8pO8B/jzJ
# fQIYgC4uEVUNF4Jdgu+OaiKZRUL4DZQn2cWcygHmwZQODSkCCOETMf+tSHHBrgoD
# Oy/daH8kAzc6MxgsN7gZTPfUUOyaFH+SNhuqvxd5eGTi9ytrg9Y3Ah+V3wWP2CjP
# ghUkIA36d9apGICRRIzIWRoElxr0EFN4RMpr1vLGxT39IUQXeI+xuf72e9TvZ6ID
# +40UjpiW0Byp3Oud85pVUucwr+lbVmbwx1JwpwOHgjPQo3EScI/TZIdGNOv6vJsb
# 4SPjMJDOKWne10HPppo3yZpg6r5cPumQ7b/FTioDYTgGMCk/MWoNqEvzA5wWi5VT
# U2OT0wWurammJ1KRlwvLKA2R/J83odIDZxWz2qjcQZSBiqnJMC7+XEDw0czFogSa
# cDUtND1h6CvAotCWkm/40ilf3T6ISPTyyiZApVihSwf5sja1cXG7kUksxW7IM0bP
# 8+cmGqwzPj3WV1YWby7pYxNs0oWqh+C/uhQ/KXPKPnmeW6GCE08wghNLBgorBgEE
# AYI3AwMBMYITOzCCEzcGCSqGSIb3DQEHAqCCEygwghMkAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwgfAGCyqGSIb3DQEJEAEEoIHgBIHdMIHaAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIGqeQbcO//4yG+csJxKpJI/6Axt6LYtT/BsZ1mReauiH
# AhUAr69rs/uEX8PziHAN330DvGu1JDkYDzIwMjQwMTI0MTAxNDEzWqBupGwwajEL
# MAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBT
# aWduZXIgIzSggg3pMIIG9TCCBN2gAwIBAgIQOUwl4XygbSeoZeI72R0i1DANBgkq
# hkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5j
# aGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwHhcNMjMw
# NTAzMDAwMDAwWhcNMzQwODAyMjM1OTU5WjBqMQswCQYDVQQGEwJHQjETMBEGA1UE
# CBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# DCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjNDCCAiIwDQYJKoZI
# hvcNAQEBBQADggIPADCCAgoCggIBAKSTKFJLzyeHdqQpHJk4wOcO1NEc7GjLAWTk
# is13sHFlgryf/Iu7u5WY+yURjlqICWYRFFiyuiJb5vYy8V0twHqiDuDgVmTtoeWB
# IHIgZEFsx8MI+vN9Xe8hmsJ+1yzDuhGYHvzTIAhCs1+/f4hYMqsws9iMepZKGRNc
# rPznq+kcFi6wsDiVSs+FUKtnAyWhuzjpD2+pWpqRKBM1uR/zPeEkyGuxmegN77tN
# 5T2MVAOR0Pwtz1UzOHoJHAfRIuBjhqe+/dKDcxIUm5pMCUa9NLzhS1B7cuBb/Rm7
# HzxqGXtuuy1EKr48TMysigSTxleGoHM2K4GX+hubfoiH2FJ5if5udzfXu1Cf+hgl
# TxPyXnypsSBaKaujQod34PRMAkjdWKVTpqOg7RmWZRUpxe0zMCXmloOBmvZgZpBY
# B4DNQnWs+7SR0MXdAUBqtqgQ7vaNereeda/TpUsYoQyfV7BeJUeRdM11EtGcb+Re
# DZvsdSbu/tP1ki9ShejaRFEqoswAyodmQ6MbAO+itZadYq0nC/IbSsnDlEI3iCCE
# qIeuw7ojcnv4VO/4ayewhfWnQ4XYKzl021p3AtGk+vXNnD3MH65R0Hts2B0tEUJT
# cXTC5TWqLVIS2SXP8NPQkUMS1zJ9mGzjd0HI/x8kVO9urcY+VXvxXIc6ZPFgSwVP
# 77kv7AkTAgMBAAGjggGCMIIBfjAfBgNVHSMEGDAWgBQaofhhGSAPw0F3RSiO0TVf
# BhIEVTAdBgNVHQ4EFgQUAw8xyJEqk71j89FdTaQ0D9KVARgwDgYDVR0PAQH/BAQD
# AgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwSgYDVR0g
# BEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6Ly9zZWN0
# aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1UdHwQ9MDswOaA3oDWGM2h0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNybDB0Bggr
# BgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQuc2VjdGlnby5jb20v
# U2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggIBAEybZVj64HnP7xXD
# Mm3eM5Hrd1ji673LSjx13n6UbcMixwSV32VpYRMM9gye9YkgXsGHxwMkysel8Cbf
# +PgxZQ3g621RV6aMhFIIRhwqwt7y2opF87739i7Efu347Wi/elZI6WHlmjl3vL66
# kWSIdf9dhRY0J9Ipy//tLdr/vpMM7G2iDczD8W69IZEaIwBSrZfUYngqhHmo1z2s
# IY9wwyR5OpfxDaOjW1PYqwC6WPs1gE9fKHFsGV7Cg3KQruDG2PKZ++q0kmV8B3w1
# RB2tWBhrYvvebMQKqWzTIUZw3C+NdUwjwkHQepY7w0vdzZImdHZcN6CaJJ5OX07T
# jw/lE09ZRGVLQ2TPSPhnZ7lNv8wNsTow0KE9SK16ZeTs3+AB8LMqSjmswaT5qX01
# 0DJAoLEZKhghssh9BXEaSyc2quCYHIN158d+S4RDzUP7kJd2KhKsQMFwW5kKQPqA
# bZRhe8huuchnZyRcUI0BIN4H9wHU+C4RzZ2D5fjKJRxEPSflsIZHKgsbhHZ9e2hP
# jbf3E7TtoC3ucw/ZELqdmSx813UfjxDElOZ+JOWVSoiMJ9aFZh35rmR2kehI/shV
# Cu0pwx/eOKbAFPsyPfipg2I2yMO+AIccq/pKQhyJA9z1XHxw2V14Tu6fXiDmCWp8
# KwijSPUV/ARP380hHHrl9Y4a1LlAMIIG7DCCBNSgAwIBAgIQMA9vrN1mmHR8qUY2
# p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5l
# dyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNF
# UlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNh
# dGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/IJC9nIAhVJO5z6A+U++zWsB21hoE
# pc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnkuyWPCE5u2hj8BBZJmbyGr1XEQeYf0
# RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnTv/M5u7mkI0Ks0BXDf56iXNc48Ray
# cNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwPDU+zkQGObp0ndVXRFzs0IXuXAZSv
# f4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN9Q0/29DqhYyKVnHRLZRMyIw80xSi
# nL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gpGaQpL0bj3duRX58/Nj4OMGcrRrc1
# r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMSHv2qyghYh3ekFesZVf/QOVQtJu5F
# GjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdonigbSlofe6GsO8Ft96XZpkyAcSpcs
# dxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179g6JTZ8qx+o2hZMmIklnLqEbAyfKm
# /31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkUAx4p+hulIq6lw02C0I3aa7fb9xhA
# V3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLvbxhQiddPnTO9GrWdod6VQXqngwID
# AQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/WqorSs9UgOHYm8Cd8rIDZsswHQYD
# VR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRVMA4GA1UdDwEB/wQEAwIBhjASBgNV
# HRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAG1UgaUzXRbhtVOB
# kXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJWZApbGHLtVDb4n35nwDvQMOt0+Lk
# VvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc87QRD9XVw+K81Vh4v0h24URnbY+w
# QxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYIuM1cfwGWvnJSajtCN3wWeDmTk5Sb
# sdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8VmBvbz1kteraMrQoohkv6ob1olcGKB
# c2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncEvAQ2ted3y5wujSMYuaPCRx3wXdah
# c1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSmTqo9CfUFbLfSZFrYKiLCS53xOV5M
# 3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/W0NdZT1QgW6r8NDT/LKzH7aZlib0
# PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYybxzrF06Uw3hlIDsPQaof6aFBnf6x
# uKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/qEZ0eBsambTJdtDgJK0kHqv3sMNrx
# py/Pt/360KOE2See+wFmd7lWEOEgbsausfm2usg1XTN2jvF8IAwqd661ogKGuinu
# tFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELDCCBCgCAQEwgZEwfTELMAkGA1UEBhMC
# R0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9y
# ZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJT
# QSBUaW1lIFN0YW1waW5nIENBAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCWCGSAFlAwQC
# AgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkF
# MQ8XDTI0MDEyNDEwMTQxM1owPwYJKoZIhvcNAQkEMTIEMI4eNDSTdmLrda8+yLWv
# nrEo6zPiCfw1ZKjVOQxihx0uhrp3Q1ElewCXL0BZHjY4zzCB7QYLKoZIhvcNAQkQ
# Agwxgd0wgdowgdcwFgQUrmKvdQoMvUfWRh91aOK8jOfKT5QwgbwEFALWW5Xig3DB
# VwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQEF
# AASCAgCUb+HtdUFND6vYI1XTEbudFxoCKP5trqH9nCdshAhEt8uEoLjXINVxDeNK
# MF7unnfxAuO0Vv8dw6ntPKrQi1hqof63B6/4ZhFwuaQjj/CTLPkapeLW30fbUtIJ
# ntbbPbTat1RWeFRbVNUf5AdvE3UbQQuX7s3U1YsD1JGlmLoI0wkx07r8Lqi0Q3K7
# tkvNjzZqP3UlsnyaMHRjWmvAqCxUo9/43zk/77OCPPimsDJKU/mVipQD4qIE6bSs
# t/PeN02WUiVT+K3OzScC/2UUbzhjSVPvG0o/XMfSnPEDLLVQBYa8LWGtlH6353ue
# izs3/5Zp8Gj5H0ndf+3tdmTNF1OR/y7WU+mtXb2S5qYlpJfHp5saCYGkZ7oy7LEK
# /pqEb6VSCOqTQUEnFFJuGxeARD9gVH/+1SPAsid5SrfP5yUj7ZPa9/Thv5RR9Pyw
# 9xTabGdd6Rr5CUVqxke5ZX+AjVJK/hAMrpLEAmo1ogFiYhv0AFqayKAIWaz2Wh8J
# 3e0HxAHcjlUllS2nA7rOBUgRxfKOeoaKIwU2AWvx4Irll67ptYq+obLUnVu98rbe
# lterbhUNgfK6Y0AKleluDEKs8fxpIshZNXWbAR0adGdCS/8AxpW0DMe9S0IVFgIK
# TS4GPdGXS1us2gjzsUUEtVQNYsI/hLEvmhal/rjOBz9AUvNEEQ==
# SIG # End signature block
