# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope = "function")] param()
#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Register-IntelEthernetRDMACounterSet
{
    [CmdletBinding(DefaultParameterSetName = 'ByAdapter')]
    Param(
        [parameter(Mandatory = $false, ParameterSetName = 'ByName')]
        [SupportsWildcards()]
        [ValidateNotNullOrEmpty()]
        [String[]]
        $Name = '',
        [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true, ParameterSetName = 'ByAdapter')]
        [ValidateNotNullOrEmpty()]
        [object[]]
        $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:WarningMessagesRegister = @()
        $script:ErrorMessagesRegister = @()
        $bIceaRegistered = $false
        $bIavfRegistered = $false
        $BeforePreference = $global:ErrorActionPreference

        GetIntelEthernetDevices @('iavf')
        $script:SupportedAdapters = @(GetSupportedAdaptersByDriver @("icea", "iavf"))
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapter ([ref]$script:ErrorMessagesRegister)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesRegister))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesRegister += $TmpStatusMsg
            }

            $strDriverName = ($script:SupportedAdapters.Where({ $_.Name -eq $a })).Service
            if ([string]::IsNullOrEmpty($strDriverName))
            {
                $script:ErrorMessagesRegister += $Messages.RegisterCountersFailure -f $a
                continue
            }

            if (($bIceaRegistered -and $strDriverName -eq 'icea') -or ($bIavfRegistered -and $strDriverName -eq 'iavf'))
            {
                Write-Host ($Messages.RegisterCountersSuccess -f $a)
                continue
            }

            $strOSSpecificDriverName = GetOSSpecificDriverName $strDriverName

            $strDriverStoreFolderPath = GetDriverInfParentFolder $strOSSpecificDriverName "Register"

            if ([string]::IsNullOrEmpty($strOSSpecificDriverName) -or [string]::IsNullOrEmpty($strDriverStoreFolderPath))
            {
                $script:ErrorMessagesRegister += $Messages.InvalidSystemConfigFailure
                break
            }

            $strManifestName = "manifest_" + $strOSSpecificDriverName + ".man"
            $strManifestPath = $strDriverStoreFolderPath + "\" + $strManifestName

            try
            {
                $global:ErrorActionPreference = "Stop"
                [System.Environment]::SetEnvironmentVariable($strDriverName + "InstallDir", $strDriverStoreFolderPath, [System.EnvironmentVariableTarget]::Machine)

                $strLodctrPath = [System.Environment]::SystemDirectory + "\lodctr.exe"
                & "$strLodctrPath" "/m:`"$strManifestPath`"" | Out-Null
            }
            catch
            {
                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"

                try
                {
                    # If SetEnvironmentVariable caused error, this will error again.
                    [System.Environment]::SetEnvironmentVariable($strDriverName + "InstallDir", '', [System.EnvironmentVariableTarget]::Machine)
                }
                catch
                {
                    Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                }

                $script:ErrorMessagesRegister += $Messages.RegisterCountersFailure -f $a
                continue
            }

            $ERROR_SUCCESS = 0
            if ($ERROR_SUCCESS -ne $LASTEXITCODE)
            {
                $script:ErrorMessagesRegister += $Messages.RegisterCountersFailure -f $a
                [System.Environment]::SetEnvironmentVariable($strDriverName + "InstallDir", '', [System.EnvironmentVariableTarget]::Machine)

                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                continue
            }

            if ($strDriverName -eq 'icea')
            {
                $bIceaRegistered = $true
            }
            elseif ($strDriverName -eq 'iavf')
            {
                $bIavfRegistered = $true
            }

            Write-Host ($Messages.RegisterCountersSuccess -f $a)
        }
    }
    End
    {
        $global:ErrorActionPreference = $BeforePreference

        foreach ($WarningMessage in $script:WarningMessagesRegister)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesRegister)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Unregister-IntelEthernetRDMACounterSet
{
    [CmdletBinding(DefaultParameterSetName = 'ByAdapter')]
    Param(
        [parameter(Mandatory = $false, ParameterSetName = 'ByName')]
        [SupportsWildcards()]
        [ValidateNotNullOrEmpty()]
        [String[]]
        $Name = '',
        [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true, ParameterSetName = 'ByAdapter')]
        [ValidateNotNullOrEmpty()]
        [object[]]
        $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:WarningMessagesUnregister = @()
        $script:ErrorMessagesUnregister = @()
        $bIceaUnregistered = $false
        $bIavfUnregistered = $false
        $BeforePreference = $global:ErrorActionPreference

        GetIntelEthernetDevices @('iavf')
        $script:SupportedAdapters = @(GetSupportedAdaptersByDriver @("icea", "iavf"))
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapter ([ref]$script:ErrorMessagesUnregister)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesUnregister))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesUnregister += $TmpStatusMsg
            }

            $strDriverName = ($script:SupportedAdapters.Where({ $_.Name -eq $a })).Service
            if ([string]::IsNullOrEmpty($strDriverName))
            {
                $script:ErrorMessagesUnregister += $Messages.UnregisterCountersFailure -f $a
                continue
            }

            if (($bIceaUnregistered -and $strDriverName -eq 'icea') -or ($bIavfUnregistered -and $strDriverName -eq 'iavf'))
            {
                Write-Host ($Messages.UnregisterCountersSuccess -f $a)
                continue
            }

            try
            {
                $global:ErrorActionPreference = "Stop"
                [Environment]::SetEnvironmentVariable($strDriverName + 'InstallDir', '', [System.EnvironmentVariableTarget]::Machine)

                $strUnlodctrPath = [System.Environment]::SystemDirectory + "\unlodctr.exe"
                & "$strUnlodctrPath" "/p:Intel-$strDriverName" | Out-Null
            }
            catch
            {
                $script:ErrorMessagesUnregister += $Messages.UnregisterCountersFailure -f $a
                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                continue
            }

            $ERROR_SUCCESS = 0
            $ERROR_UNREGISTER_REPEAT = 259
            if ($ERROR_SUCCESS -ne $LASTEXITCODE -and $ERROR_UNREGISTER_REPEAT -ne $LASTEXITCODE)
            {
                $script:ErrorMessagesUnregister += $Messages.UnregisterCountersFailure -f $a
                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                continue
            }

            if ($strDriverName -eq 'icea')
            {
                $bIceaUnregistered = $true
            }
            elseif ($strDriverName -eq 'iavf')
            {
                $bIavfUnregistered = $true
            }

            Write-Host ($Messages.UnregisterCountersSuccess -f $a)
        }
    }
    End
    {
        $global:ErrorActionPreference = $BeforePreference

        foreach ($WarningMessage in $script:WarningMessagesUnregister)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesUnregister)
        {
            Write-Error $ErrorMessage
        }
    }
}

# SIG # Begin signature block
# MIIocgYJKoZIhvcNAQcCoIIoYzCCKF8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCB9kj6IaoM1SXKz
# Y86UmJPK0oJXaL6Cdop/IitU7BJH26CCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQCmOovGKSxCae+j/hQscnOWMA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjMwMTI1MDAwMDAwWhcNMjQwMTI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDGlXgQt1127m+5LYJmaFBzYTdi9cRozQjdZt/+
# +l4e/XT+sHZemhE0o/x+nydO/Ai7lhA6+Rs0Juw3vSXdGjKZLJ2G19nKCeICKfyK
# /rTX7GkdhwZnX8yQJEWU2VuZ/Ad9MtybHiZY3tIlpauWvNmfILMXWL+8DZOzVifw
# KbQgObHaYERAsfT2ZJ0HYp8lo5XepZy8SCeY/kRShw37vMeqNnuJtjnoB81/sZmI
# bZfLL4XxYn3ESyeeXgGhpplv3DkBUFM6YdD7lNXN7yp7h1gCJUA648U9EMOoQAHi
# ewLOPQSX/xgNqHfk3FUbcwZp07/zLrHtKvDNdPFrgkTBz81xGzVHOi48Lh6FBGW7
# UFEq/jI5TOk/AcM5eDvsizy4dA6J0AJ0o3lez+jAP2tDccEt7bn1aaamcjFpDOol
# MktQz0H07XqmRgVT3wHGHiDswDNUCxbk3ervsNG1afAPcFh5Hx4up8hpJoF7YTOf
# 6OHm8k3dRtjF1DC78oNYgqkpuNMCAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBRvq1HVd47vo0fOeEIdOUOyHE5Z
# ojAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBAJan0xi9l18PVDDZYZmI6kX64OJRsJvMbeAOd/iCwfoO1W2d
# +vin3je7xjz7TX1tERtLQGbRiraogMGVMco6+BSXJn0mGwpPcxsaRLpNbcbfXPHx
# niq3yO3gbSOrUU1Z9kScqlrqgbaAMKWnwlhy8MYbo9H2wIXCtrZ9kCuwoI9iilKS
# jjQqWs4085hIBzp3Kc32V4fLkuKGH1KYeTRMm8hP4+V0D+vBVFSu7PFh5t8p93y+
# Y/n7icZkyci6GUMUbVi8YK2z5KDr/ohPWS3a+DP+gAtjafTITgucRazljocMCIIg
# eWtL1Gwq0hvcpQOn+pdNGg8vUIXuVtziS9NmJKVfHEnOXtvte0ep9ziC5yaE8ui0
# 3e0ouY6xuhqPY5+mh9Wve5BX5ju1ewEUuImurs0xjtNBr3bscylS3AkssWz0LBko
# gR/aT9luMyUm/rC/2JLMrDlRTuiWergSd1ffFeLGnQBWtvd0snyFK3ZlVbJelDJP
# 47S4Vj1s45eyKo2DYTGCFdUwghXRAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRAKY6i8YpLEJp76P+FCxyc5YwDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgp0e0smAnMFthh5rNo3fK
# 9MoYRro8pCQVWpU5rp/oMZgwDQYJKoZIhvcNAQEBBQAEggGAYuWprjqS/iko8KQq
# DKBox3Hy7GCkX6qlO0aaBAcbsJYA0iN3Oc1bQl6wKqBvPfcFMVDLGjS653QcD64S
# 2eLKsnDlaoCgDNw5gQbR9JtGozXONjlTkjqt7mOSw79xQA0IuGwQ3+88PVHJLI31
# F8dYByK2xBRC90INoWo0QvwPXDKh7z8UdNCEYkLt4Ux1wNoWQFd0o0VlM15P+HY9
# m9swOhg6orWUG/v0T5whpJYpWDNbV6OEbw8Ju0Tnp41Ph2DXgDBpjIUTvkaIevuZ
# J11fhQHaTq2bZ8TW/bU3RYFrLKHT8FZ+ap2pgUQUkjmyr7gAWFPNo0GjgFgMxbW0
# 1L0OEr/EHzgw/UFOYgkQft9nHpks0SdTyYpmv5A1IJBvJBJHf4TxlygDQSqFtgBs
# 0OUOMv/GRdYWnuv3Cp0BZW9qU8+WHs4C6yoAm+lzYnlX2mbwubnkGPdlYP+tMNi0
# JNfIqdov9DJrmtc5S70o9V2J/pPfzQrKh4b1DqEjLEnTEjnPoYITTjCCE0oGCisG
# AQQBgjcDAwExghM6MIITNgYJKoZIhvcNAQcCoIITJzCCEyMCAQMxDzANBglghkgB
# ZQMEAgIFADCB7wYLKoZIhvcNAQkQAQSggd8EgdwwgdkCAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQgSCrt83zb5b8zjy4yoZ/hvWHrLwn7hcblU2qVatiM
# RcICFBGqp0vn3hB3de+aD2Sc/g1pDmHGGA8yMDIzMTEwNzE5MDExNFqgbqRsMGox
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# U2lnbmVyICM0oIIN6TCCBvUwggTdoAMCAQICEDlMJeF8oG0nqGXiO9kdItQwDQYJ
# KoZIhvcNAQEMBQAwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFu
# Y2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMB4XDTIz
# MDUwMzAwMDAwMFoXDTM0MDgwMjIzNTk1OVowajELMAkGA1UEBhMCR0IxEzARBgNV
# BAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UE
# AwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBTaWduZXIgIzQwggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQCkkyhSS88nh3akKRyZOMDnDtTRHOxoywFk
# 5IrNd7BxZYK8n/yLu7uVmPslEY5aiAlmERRYsroiW+b2MvFdLcB6og7g4FZk7aHl
# gSByIGRBbMfDCPrzfV3vIZrCftcsw7oRmB780yAIQrNfv3+IWDKrMLPYjHqWShkT
# XKz856vpHBYusLA4lUrPhVCrZwMlobs46Q9vqVqakSgTNbkf8z3hJMhrsZnoDe+7
# TeU9jFQDkdD8Lc9VMzh6CRwH0SLgY4anvv3Sg3MSFJuaTAlGvTS84UtQe3LgW/0Z
# ux88ahl7brstRCq+PEzMrIoEk8ZXhqBzNiuBl/obm36Ih9hSeYn+bnc317tQn/oY
# JU8T8l58qbEgWimro0KHd+D0TAJI3VilU6ajoO0ZlmUVKcXtMzAl5paDgZr2YGaQ
# WAeAzUJ1rPu0kdDF3QFAaraoEO72jXq3nnWv06VLGKEMn1ewXiVHkXTNdRLRnG/k
# Xg2b7HUm7v7T9ZIvUoXo2kRRKqLMAMqHZkOjGwDvorWWnWKtJwvyG0rJw5RCN4gg
# hKiHrsO6I3J7+FTv+GsnsIX1p0OF2Cs5dNtadwLRpPr1zZw9zB+uUdB7bNgdLRFC
# U3F0wuU1qi1SEtklz/DT0JFDEtcyfZhs43dByP8fJFTvbq3GPlV78VyHOmTxYEsF
# T++5L+wJEwIDAQABo4IBgjCCAX4wHwYDVR0jBBgwFoAUGqH4YRkgD8NBd0UojtE1
# XwYSBFUwHQYDVR0OBBYEFAMPMciRKpO9Y/PRXU2kNA/SlQEYMA4GA1UdDwEB/wQE
# AwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMEoGA1Ud
# IARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8vc2Vj
# dGlnby5jb20vQ1BTMAgGBmeBDAEEAjBEBgNVHR8EPTA7MDmgN6A1hjNodHRwOi8v
# Y3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcmwwdAYI
# KwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnNlY3RpZ28uY29t
# L1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6
# Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBMm2VY+uB5z+8V
# wzJt3jOR63dY4uu9y0o8dd5+lG3DIscEld9laWETDPYMnvWJIF7Bh8cDJMrHpfAm
# 3/j4MWUN4OttUVemjIRSCEYcKsLe8tqKRfO+9/YuxH7t+O1ov3pWSOlh5Zo5d7y+
# upFkiHX/XYUWNCfSKcv/7S3a/76TDOxtog3Mw/FuvSGRGiMAUq2X1GJ4KoR5qNc9
# rCGPcMMkeTqX8Q2jo1tT2KsAulj7NYBPXyhxbBlewoNykK7gxtjymfvqtJJlfAd8
# NUQdrVgYa2L73mzECqls0yFGcNwvjXVMI8JB0HqWO8NL3c2SJnR2XDegmiSeTl9O
# 048P5RNPWURlS0Nkz0j4Z2e5Tb/MDbE6MNChPUitemXk7N/gAfCzKko5rMGk+al9
# NdAyQKCxGSoYIbLIfQVxGksnNqrgmByDdefHfkuEQ81D+5CXdioSrEDBcFuZCkD6
# gG2UYXvIbrnIZ2ckXFCNASDeB/cB1PguEc2dg+X4yiUcRD0n5bCGRyoLG4R2fXto
# T4239xO07aAt7nMP2RC6nZksfNd1H48QxJTmfiTllUqIjCfWhWYd+a5kdpHoSP7I
# VQrtKcMf3jimwBT7Mj34qYNiNsjDvgCHHKv6SkIciQPc9Vx8cNldeE7un14g5glq
# fCsIo0j1FfwET9/NIRx65fWOGtS5QDCCBuwwggTUoAMCAQICEDAPb6zdZph0fKlG
# Nqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpO
# ZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVT
# RVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoXDTM4MDExODIzNTk1OVow
# fTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4G
# A1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQD
# ExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyAIVSTuc+gPlPvs1rAdtYa
# BKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhObtoY/AQWSZm8hq9VxEHmH
# 9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5pCNCrNAVw3+eolzXOPEW
# snDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EBjm6dJ3VV0Rc7NCF7lwGU
# r3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWMilZx0S2UTMiMPNMU
# opy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G493bkV+fPzY+DjBnK0a3
# Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3pBXrGVX/0DlULSbu
# RRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH3uhrDvBbfel2aZMgHEqX
# LHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqNoWTJiJJZy6hGwMny
# pv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfobpSKupcNNgtCN2mu32/cY
# QFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInXT50zvRq1naHelUF6p4MC
# AwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0G
# A1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNVHQ8BAf8EBAMCAYYwEgYD
# VR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAI
# MAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3Qu
# Y29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHkuY3JsMHYGCCsG
# AQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2NydC51c2VydHJ1c3QuY29t
# L1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUFBzABhhlodHRwOi8v
# b2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBtVIGlM10W4bVT
# gZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ2+J9+Z8A70DDrdPi
# 5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V1cPivNVYeL9IduFEZ22P
# sEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8Blr5yUmo7Qjd8Fng5k5OU
# m7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZLXq2jK0KKIZL+qG9aJXBi
# gXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXnd8ucLo0jGLmjwkcd8F3W
# oXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1BWy30mRa2Coiwkud8Tle
# TN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9UIFuq/DQ0/yysx+2mZYm
# 9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4ZSA7D0GqH+mhQZ3+
# sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbGpm0yXbQ4CStJB6r97DDa
# 8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7xfCAMKneutaIChrop
# 7rRaALGMq+P5CslUXdS5anSevUiumDGCBCwwggQoAgEBMIGRMH0xCzAJBgNVBAYT
# AkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZv
# cmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBS
# U0EgVGltZSBTdGFtcGluZyBDQQIQOUwl4XygbSeoZeI72R0i1DANBglghkgBZQME
# AgIFAKCCAWswGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJ
# BTEPFw0yMzExMDcxOTAxMTRaMD8GCSqGSIb3DQEJBDEyBDDm3iKujqr6/2ECFVdV
# ez366Pq6mziyl5mexbtuGzCmwyrIXKa/OSh5CP1QuDXQ9hgwge0GCyqGSIb3DQEJ
# EAIMMYHdMIHaMIHXMBYEFK5ir3UKDL1H1kYfdWjivIznyk+UMIG8BBQC1luV4oNw
# wVcAlfqI+SPdk3+tjzCBozCBjqSBizCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Ck5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUg
# VVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlm
# aWNhdGlvbiBBdXRob3JpdHkCEDAPb6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEB
# BQAEggIAZVZ1++jdNhBvnP4CPat0NKYN9MhxLl5qAUnlwvjGv6v4+o5DuyIiDP9o
# Uunm2YjRPlvQc0aAmhOBTBh0yWjbqBY2uhIl+ZOK3FSZuvlo0kAbaruCqkjPv283
# MSTxw2xMP1juxqg4BDmQvUaNT9pIoIP1wN0w7dAATsODD22G8w6E8FbUfFG1Q1kJ
# EF/SK+qq4hevOC+/+bR029Sw9meSbCOMoS1klhwdLG6gq4+wkdZs93DuOqsOoXT1
# oJs51hIpuNdlkD7OvPv6N+FrdFbPOnAdccGRIOxomj21SKuAIF4K10Zd8GRW+XCj
# nLcbwNqPglOkAuO4nMSVaj0qAHkTRv835oI+hBV6/QNBTetGzbng7t2o38zXZVdG
# VuQggHxR1qNqXkI5L2ntHK9UHWanYHbIzlrOJXmI9D4AoB7AJr/7zW6OmUgmZ9+O
# 0wVgVTMNlKnA7DNvDC2AeWfwlEu9m7r3vCwsySEem2B41PhyMlKKshNUc4o5Oiz6
# yODJimxe8MK2qtWg3LtL0O3KolDGoTFXZKRGnSY2gYlAFFhLiMXVyzsPQNkJvJ7z
# LKCiYkGXq4UAzYa79ZdeBbZUrE5eHOtVNKPSQ+h06w4y4QtZp8TzATrqCxb5cnkf
# fMrDoHhe3C14strxW5DaGhggQp1sbw5hOYYLNX9P1m8QdpW88hQ=
# SIG # End signature block
