# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernet
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $FinalObject = @()
        GetIntelEthernetDevices @('icei*', 'i40ei*')
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
        GetIntelDriverInfo
        $script:MSNetHwInfo = @(Get-NetAdapterHardwareInfo -ErrorAction SilentlyContinue)
        $script:MSNetAdvProperty = @(Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue)
    }
    Process
    {
        $Adapters = $Adapter
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNamesArray)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
            }

            $SupportedSpeeds = GetAdapterSupportedSpeeds $a
            $AdapterStatuses = GetAdapterStatus $a
            $BusTypes = GetAdapterBusType $a
            $Capabilities = GetAdapterCapabilities $a $SupportedSpeeds
            $ConnectionNames = GetAdapterConnectionName $a
            $DDPPackageNameAndVersions = GetAdapterDDPPackageNameAndVersion $a
            $DDPPackageTrackIds = GetAdapterDDPPackageTrackId $a
            $DeviceStatuses = GetAdapterDeviceStatus $a $SupportedSpeeds
            $DriverVersion = GetDriverVersion $a
            $EEEStatuses = GetAdapterEEEStatus $a
            $ETrackIDs = GetAdapterETrackId $a
            $FullDuplex = GetFullDuplex $a
            $LinkSpeedDuplex = GetLinkSpeedDuplex $a
            $MaxSpeeds = GetAdapterMaxSpeed $a $SupportedSpeeds
            $MediaTypes = GetAdapterMediaType $a
            $MiniPortNames = GetAdapterMiniPortName $a
            $NVMVersions = GetAdapterNVMVersion $a
            $NegotiatedLinkSpeed = GetAdapterNegotiatedLinkSpeed $a
            $NegotiatedLinkWidth = GetAdapterNegotiatedLinkWidth $a
            $NetlistVersions = GetAdapterNetlistVersion $a
            $OemFwVersions = GetOemFwVersion $a
            $OriginalDisplayNames = GetOriginalDisplayName $a
            $PCIDeviceIDs = GetAdapterPCIDeviceID $a
            $PartNumbers = GetAdapterPartNumber $a
            $PciLocations = GetAdapterPCILocation $a
            $AdapterPort = GetAdapterPort $a
            $SanMacAddresses = GetAdapterSanMacAddress $a

            # Assemble it all together in PSCustomObject
            $FinalObject += [PsCustomObject] @{
                PSTypeName          = 'IntelEthernetAdapter'
                AdapterStatus       = $AdapterStatuses
                BusType             = $BusTypes.BusType
                BusTypeString       = $BusTypes.BusTypeString
                Capabilities        = $Capabilities
                ConnectionName      = $ConnectionNames
                DDPPackageName      = $DDPPackageNameAndVersions.Name
                DDPPackageTrackId   = $DDPPackageTrackIds
                DDPPackageVersion   = $DDPPackageNameAndVersions.Version
                DeviceStatus        = $DeviceStatuses.DeviceStatus
                DeviceStatusString  = $DeviceStatuses.DeviceStatusString
                DriverVersion       = $DriverVersion
                EEE                 = $EEEStatuses.EEEStatus
                EEEString           = $EEEStatuses.EEEStatusString
                ETrackID            = $ETrackIDs
                FWVersion           = $OemFwVersions
                FullDuplex          = $FullDuplex
                Location            = $PciLocations
                MaxSpeed            = $MaxSpeeds
                MediaType           = $MediaTypes.MediaType
                MediaTypeString     = $MediaTypes.MediaTypeString
                MiniPortInstance    = $MiniPortNames.Instance
                MiniPortName        = $MiniPortNames.Name
                NVMVersion          = $NVMVersions
                Name                = $a
                NegotiatedLinkSpeed       = $NegotiatedLinkSpeed.NegotiatedLinkSpeed
                NegotiatedLinkSpeedString = $NegotiatedLinkSpeed.NegotiatedLinkSpeedString
                NegotiatedLinkWidth       = $NegotiatedLinkWidth.NegotiatedLinkWidth
                NegotiatedLinkWidthString = $NegotiatedLinkWidth.NegotiatedLinkWidthString
                NetlistVersion      = $NetlistVersions
                OriginalDisplayName = $OriginalDisplayNames
                PCIDeviceID         = $PCIDeviceIDs
                PartNumber          = $PartNumbers
                PortNumber          = $AdapterPort.Port
                PortNumberString    = $AdapterPort.PortString
                SANMacAddress       = $SanMacAddresses
                Speed               = $LinkSpeedDuplex.Speed
                SpeedString         = $LinkSpeedDuplex.SpeedString
                }
        }
    }
    End
    {
        $FinalObject | Sort-Object -Property Location

        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function GetAdapterBusType($AdapterName)
{
    $BusType       = 0
    $BusTypeString = $Messages.Unknown

    foreach ($Bus in $script:BusTypesArray)
    {
        if (($null -ne $Bus) -and ($Bus.InstanceName -eq $AdapterName))
        {
            $BusType       = [int]$Bus.BusType
            $BusTypeString = $BusTypeMap[[int]$Bus.BusType]
            break
        }
    }

    return [PsCustomObject] @{
        BusType       = $BusType
        BusTypeString = $BusTypeString }
}

function GetAdapterConnectionName($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceAlias
}

function GetAdapterDDPPackageNameAndVersion($AdapterName)
{
    $Name    = $Messages.NotSupported
    $Version = $Messages.NotSupported

    foreach ($DDPPkgName in $script:DDPPkgNamesArray)
    {
        if (($null -ne $DDPPkgName) -and ($DDPPkgName.InstanceName -eq $AdapterName))
        {
            if ($DDPPkgName.Name -and ($null -ne $DDPPkgName.Major) -and ($null -ne $DDPPkgName.Minor))
            {
                $Name = [System.Text.Encoding]::ASCII.GetString($DDPPkgName.Name).Trim(0)
                $Version = $DDPPkgName.Major.ToString() + "." + $DDPPkgName.Minor.ToString()
            }
            break
        }
    }
    return [PsCustomObject] @{
        Name    = $Name
        Version = $Version }
}


function GetAdapterDDPPackageTrackId($AdapterName)
{
    $TrackId = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetTrackId" $AdapterName "WmiGetTrackId" $Params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true) -and ($Result.Track_Id))
    {
        $TrackId = '0x{0:X}' -f $Result.Track_Id
    }
    return $TrackId
}


function GetAdapterDeviceStatus($AdapterName, $SupportedSpeeds)
{
    $DeviceStatus = 0
    $DeviceStatusString = $Messages.Unknown

    $AdapterNames = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})

    foreach($TmpAdapter in $AdapterNames)
    {
        if ("Up" -eq $TmpAdapter.Status)
        {
            $MaxSpeed = GetAdapterMaxSpeed $AdapterName $SupportedSpeeds
            $CurrentSpeed = (GetLinkSpeedDuplex $AdapterName).Speed
            if ($CurrentSpeed -lt $MaxSpeed)
            {
                $DeviceStatus = 4
                $DeviceStatusString = $Messages.LinkUpNotMax
            }
            else
            {
                $DeviceStatus = 1
                $DeviceStatusString = $Messages.LinkUp
            }
        }
        elseif ("Disconnected" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 2
            $DeviceStatusString = $Messages.LinkDown
        }
        elseif ("Disabled" -eq $TmpAdapter.Status)
        {
            $DeviceStatus = 0
            $DeviceStatusString = $Messages.Disabled
        }
    }

    return [PsCustomObject] @{
        DeviceStatus       = $DeviceStatus
        DeviceStatusString = $DeviceStatusString}

}

function GetAdapterETrackId($AdapterName)
{
    $strETrackId = $Messages.NotSupported
    foreach ($ETrackId in $script:ETrackIdsArray)
    {
        if (($null -ne $ETrackId) -and ($ETrackId.InstanceName -eq $AdapterName))
        {
            if ($ETrackId.Id)
            {
                $strETrackId = '0x{0:X}' -f $ETrackId.Id
            }
            break
        }
    }
    return $strETrackId
}


function GetAdapterNVMVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($NVMVersion in $script:NVMVersionsArray)
    {
        if (($null -ne $NVMVersion) -and ($NVMVersion.InstanceName -eq $AdapterName))
        {
            if ($NVMVersion.Version)
            {
                $Version = (($NVMVersion.Version -band 0xffff) -shr 8).ToString() + "." + (($NVMVersion.Version -band 0xff)).ToString("X2")
            }
            break
        }
    }
    return $Version
}

function GetAdapterNetlistVersion($AdapterName)
{
    $NetlistVersion = $Messages.NotSupported
    $Params = @{Version = [uint32]1; Size = [uint32]12; Type = [uint32]1;}
    $Result = InvokeCimMethod "IntlLan_GetNVMNetlistInfo" $AdapterName "WmiGetNVMNetlistInfo" $params
    if (($null -ne $Result) -and ($Result.ReturnValue -eq $true) -and ($Result.VersionStr))
    {
        $NetlistVersion = [System.Text.Encoding]::ASCII.GetString($Result.VersionStr).Trim(0)
    }
    return $NetlistVersion
}

function GetAdapterPartNumber($AdapterName)
{
    $PartNumberString = $Messages.NotSupported
    foreach ($PartNumber in $script:PartNumbersArray)
    {
        if (($null -ne $PartNumber) -and ($PartNumber.InstanceName -eq $AdapterName))
        {
            if ($PartNumber.PartNumberString)
            {
                $PartNumberString = [System.Text.Encoding]::ASCII.GetString($PartNumber.PartNumberString).Trim(0)
            }
            break
        }
    }
    return $PartNumberString
}

function GetAdapterSanMacAddress($AdapterName)
{
    $MacAddress = $Messages.NotSupported

    foreach ($SanMacAddress in $script:SanMacAddressesArray)
    {
        if (($null -ne $SanMacAddress) -and ($SanMacAddress.InstanceName -eq $AdapterName))
        {
            if ($SanMacAddress.SanMacAddr)
            {
                $MacAddress = ""
                for ($i = 0; $i -lt 6; $i++)
                {
                    # convert to string hex representation; X - hex, 2 - add leading zeroes if needed
                    $MacAddress += $SanMacAddress.SanMacAddr[$i].ToString("X2")
                }
            }

            break
        }
    }
    return $MacAddress
}

function GetAdapterMediaType($AdapterName)
{
    $MediaType = [PsCustomObject] @{
        MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
        MediaTypeString = $Messages.Unknown }

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    switch ($Service)
    {
        {($_ -eq 'icea') -or ($_ -eq 'scea')}   { $MediaType = GetAdapterMediaTypeIce $AdapterName; break }
        {($_ -eq 'i40ea') -or ($_ -eq 'i40eb')} { $MediaType = GetAdapterMediaTypeI40e $AdapterName; break }
    }
    return $MediaType
}

function GetAdapterMediaTypeIce($AdapterName)
{
    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
    $MediaTypeString = $Messages.Unknown

    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 8))
        {
            # Interpreting PhyInfo array values:
            # 0 == PHY type
            # 1 == link info <-- Bit 0 (value of 1 means has link)
            # 2 == an_info
            # 3 == ext_info
            # 4 == module type [0]
            # 5 == module type [1]
            # 6 == module type [2]
            # 7 == media interface  <-- 1=Backplane, 2=QSFP, 3=SFP
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkInfo = $PhyInfo.PhyInfo[1];
            $MediaInterface = $PhyInfo.PhyInfo[7];

            if ($LinkInfo -band 0x01)
            {
                switch ($PhyType)
                {
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL25GAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUIAOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4AOC_ACC,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2AOC_ACC}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_AOC_ACC
                        $MediaTypeString = $Messages.AOCACC
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL5GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL10GBaseKR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseKR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR
                        $MediaTypeString = $Messages.BackplaneKR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR1
                        $MediaTypeString = $Messages.BackplaneKR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2
                        $MediaTypeString = $Messages.BackplaneKR2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR2PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR2_PAM4
                        $MediaTypeString = $Messages.BackplaneKR2PAM4
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseKR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseKR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4
                        $MediaTypeString = $Messages.BackplaneKR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseKR4PAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR4_PAM4
                        $MediaTypeString = $Messages.BackplaneKR4PAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseKRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_PAM4
                        $MediaTypeString = $Messages.BackplaneKRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseKRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KR_S
                        $MediaTypeString = $Messages.BackplaneKRS
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseKX,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseKX}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Backplane_KX
                        $MediaTypeString = $Messages.BackplaneKX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GSFIC2C,
                    [int][CVL_PHY_TYPE]::CVL25GAUIC2C,
                    [int][CVL_PHY_TYPE]::CVL40GXLAUI,
                    [int][CVL_PHY_TYPE]::CVL50GLAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI2,
                    [int][CVL_PHY_TYPE]::CVL50GAUI1,
                    [int][CVL_PHY_TYPE]::CVL100GCAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI4,
                    [int][CVL_PHY_TYPE]::CVL100GAUI2}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Chip_to_Chip
                        $MediaTypeString = $Messages.ChiptoChip
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL1000BaseT,
                    [int][CVL_PHY_TYPE]::CVL2Point5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL5GBaseT,
                    [int][CVL_PHY_TYPE]::CVL10GBaseT,
                    [int][CVL_PHY_TYPE]::CVL25GBaseT}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100BaseTX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Copper_TX
                        $MediaTypeString = $Messages.CopperTX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL10GSFIDA)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach
                        $MediaTypeString = $Messages.DirectAttach
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseCP2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CP2
                        $MediaTypeString = $Messages.DirectAttachCP2
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR
                        $MediaTypeString = $Messages.DirectAttachCR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCR1)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR1
                        $MediaTypeString = $Messages.DirectAttachCR1
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR2
                        $MediaTypeString = $Messages.DirectAttachCR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseCR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseCR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR4
                        $MediaTypeString = $Messages.DirectAttachCR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseCRPAM4)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_PAM4
                        $MediaTypeString = $Messages.DirectAttachCRPAM4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL25GBaseCRS)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Direct_Attach_CR_S
                        $MediaTypeString = $Messages.DirectAttachCRS
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseDR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_DR
                        $MediaTypeString = $Messages.FiberDR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL50GBaseFR)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_FR
                        $MediaTypeString = $Messages.FiberFR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseLR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseLR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR
                        $MediaTypeString = $Messages.FiberLR
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseLR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseLR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LR4
                        $MediaTypeString = $Messages.FiberLR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseLX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_LX
                        $MediaTypeString = $Messages.FiberLX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL10GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL25GBaseSR,
                    [int][CVL_PHY_TYPE]::CVL50GBaseSR}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR
                        $MediaTypeString = $Messages.FiberSR
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL100GBaseSR2)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR2
                        $MediaTypeString = $Messages.FiberSR2
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL40GBaseSR4,
                    [int][CVL_PHY_TYPE]::CVL100GBaseSR4}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SR4
                        $MediaTypeString = $Messages.FiberSR4
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL1000BaseSX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_SX
                        $MediaTypeString = $Messages.FiberSX
                        break
                    }
                    ([int][CVL_PHY_TYPE]::CVL2Point5gBaseX)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Fiber_X
                        $MediaTypeString = $Messages.FiberX
                        break
                    }
                    {$_ -in [int][CVL_PHY_TYPE]::CVL100MSGMII,
                    [int][CVL_PHY_TYPE]::CVL1GSGMII}
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
            else
            {
                switch ($MediaInterface)
                {
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BACKPLANE)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BACKPLANE
                        $MediaTypeString = $Messages.Backplane
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_QSFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_QSFP
                        $MediaTypeString = $Messages.QSFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SFP)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SFP
                        $MediaTypeString = $Messages.SFP
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_BASE_T)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_BASE_T
                        $MediaTypeString = $Messages.CopperT
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_SGMII)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_SGMII
                        $MediaTypeString = $Messages.SGMII
                        break
                    }
                    ([int][CPK_PHY_INFO]::CPK_PHYINFO_MEDIA_FIBER)
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_INTERFACE_CVL_FIBER
                        $MediaTypeString = $Messages.Fiber
                        break
                    }
                    default
                    {
                        $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_CVL_Unknown
                        $MediaTypeString = $Messages.Unknown
                        break
                    }
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterMediaTypeI40e($AdapterName)
{
    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
    $MediaTypeString = $Messages.Unknown

    foreach ($PhyInfo in $script:PhyInfoArray)
    {
        if (($null -ne $PhyInfo) -and ($PhyInfo.InstanceName -eq $AdapterName) -and ($PhyInfo.PhyInfo.Length -ge 13))
        {
            $PhyType = $PhyInfo.PhyInfo[0];
            $LinkType = ([uint32]$PhyInfo.PhyInfo[11] -shl 24) + ([uint32]$PhyInfo.PhyInfo[10] -shl 16) + ([uint32]$PhyInfo.PhyInfo[9] -shl 8) + [uint32]$PhyInfo.PhyInfo[8];
            $LinkTypeExt = $PhyInfo.PhyInfo[12];

            if ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_EMPTY)
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_UNKNOWN
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_KR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_KR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                $MediaTypeString = $Messages.Backplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SFP_Cu) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                $MediaTypeString = $Messages.SFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_LR4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_LR) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_LR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                $MediaTypeString = $Messages.LRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_SR) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_SR4) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_SR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                $MediaTypeString = $Messages.SRFiber
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_T) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_100BASE_TX) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_2_5GBASE_T) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_5GBASE_T))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                $MediaTypeString = $Messages.Copper
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_CR1_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4_CU) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_40GBASE_CR4))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                $MediaTypeString = $Messages.QSFPDirectAttach
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_10GBASE_KX4) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_1000BASE_KX))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                $MediaTypeString = $Messages.KXBackplane
            }
            elseif (($LinkType -band [int][LinkType]::LINK_TYPE_XAUI) -or
                    ($LinkType -band [int][LinkType]::LINK_TYPE_XLAUI))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                $MediaTypeString = $Messages.XAUI
            }
            elseif (($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_AOC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25G_ACC) -or
                    ($LinkTypeExt -band [int][LinkTypeExt]::LINK_TYPE_25GBase_CR))
            {
                $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                $MediaTypeString = $Messages.SFP28DirectAttach
            }
            # old FW or HW different than XL710
            elseif ($LinkTypeExt -eq 0 -and $LinkType -eq 0)
            {
                if (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_KR4) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KR) -or
                    ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_KR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_BACKPLANE
                    $MediaTypeString = $Messages.Backplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SFPP_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFPDIRECTATTACH
                    $MediaTypeString = $Messages.SFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_LR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_LR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_LR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_LR_FIBER
                    $MediaTypeString = $Messages.LRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_SR) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_SR4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_SR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SR_FIBER
                    $MediaTypeString = $Messages.SRFiber
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_T) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_100BASE_TX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_COPPER
                    $MediaTypeString = $Messages.Copper
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_CR1_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4_CU) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_40GBASE_CR4))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_QSFPDIRECTATTACH
                    $MediaTypeString = $Messages.QSFPDirectAttach
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_10GBASE_KX4) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_1000BASE_KX))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_KX4BACKPLANE
                    $MediaTypeString = $Messages.KXBackplane
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XAUI) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_XLAUI))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_XAUI
                    $MediaTypeString = $Messages.XAUI
                }
                elseif (($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_AOC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_ACC) -or
                        ($PhyType -eq [int][I40E_PHY_TYPE]::I40E_PHY_TYPE_25GBASE_CR))
                {
                    $MediaType = [int][NCS_ADAPTER_MEDIA_TYPE]::NCS_MEDIA_SFP28DIRECTATTACH
                    $MediaTypeString = $Messages.SFP28DirectAttach
                }
            }
        }
    }

    return [PsCustomObject] @{
        MediaType = $MediaType
        MediaTypeString = $MediaTypeString }
}

function GetAdapterPCILocation($AdapterName)
{
    $PCILocation = $Messages.NotSupported
    foreach ($HwInfo in $script:MSNetHwInfo)
    {
        if ($AdapterName -eq $HwInfo.ifDesc)
        {
            $PCILocation = $HwInfo.Bus.ToString() + ":" + $HwInfo.Device.ToString()  + ":" + $HwInfo.Function.ToString() + ":" + $HwInfo.Segment.ToString()
            break
        }
    }
    return $PCILocation
}

function GetOriginalDisplayName($AdapterName)
{
    return ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).ifDesc
}

function GetOemFwVersion($AdapterName)
{
    $Version = $Messages.NotSupported
    foreach ($FwVersion in $script:FwVersionsArray)
    {
        if ($FwVersion.InstanceName -eq $AdapterName)
        {
            # driver can return an array of zeroes - don't attempt to construct a string using it
            if (($null -ne $FwVersion.SingleNvmVersion) -and $FwVersion.SingleNvmVersion[0] -ne 0)
            {
                $Version = [System.Text.Encoding]::ASCII.GetString($FwVersion.SingleNvmVersion).Trim(0)
            }
            break
        }
    }
    return $Version
}

function GetAdapterPCIDeviceID($AdapterName)
{
    return ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).PnPDeviceID
}

function GetAdapterNegotiatedLinkWidth($AdapterName)
{
    $NegotiatedLinkWidth = 0
    $NegotiatedLinkWidthString = $Messages.NotAvailable

    $AdapterHWInfo = $script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})

    if ($AdapterHWInfo)
    {
        $NegotiatedLinkWidth = $AdapterHWInfo.PciExpressCurrentLinkWidth
        $NegotiatedLinkWidthString = "x" + $NegotiatedLinkWidth.ToString()
    }

    return [PsCustomObject] @{
        NegotiatedLinkWidth = $NegotiatedLinkWidth
        NegotiatedLinkWidthString = $NegotiatedLinkWidthString }
}

function GetAdapterNegotiatedLinkSpeed($AdapterName)
{
    $NegotiatedLinkSpeed = ($script:MSNetHwInfo.Where({$_.ifDesc -eq $AdapterName})).PciExpressCurrentLinkSpeedEncoded
    switch ($NegotiatedLinkSpeed)
    {
        0 {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
        1 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed25; break}
        2 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed50; break}
        3 {$NegotiatedLinkSpeedString = $Messages.NegLaneSpeed80; break}
        default {$NegotiatedLinkSpeedString = $Messages.Unknown; break}
    }
    return [PsCustomObject] @{
        NegotiatedLinkSpeed = $NegotiatedLinkSpeed
        NegotiatedLinkSpeedString = $NegotiatedLinkSpeedString }
}

function GetLinkSpeedDuplex($AdapterName)
{
    $AdapterObj = $script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})
    $Speed = $AdapterObj.Speed
    $SpeedString = $AdapterObj.LinkSpeed
    $FullDuplex = $AdapterObj.FullDuplex

    if ("Up" -ne $AdapterObj.Status)
    {
        $Speed = 0
        $SpeedString = $Messages.NotAvailable
    }
    elseif ($true -eq $FullDuplex)
    {
        $SpeedString += " " + $Messages.FullDuplex
    }

    return [PsCustomObject] @{
        Speed = $Speed
        SpeedString = $SpeedString }
}

function GetFullDuplex($AdapterName)
{
    $FullDuplex = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).FullDuplex
    if (-not $FullDuplex)
    {
        $FullDuplex = ""
    }
    return $FullDuplex
}

function GetAdapterPort($AdapterName)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName

    switch ($AdapterRegKey.Port)
    {
        0 {$PortNumberString = $Messages.PortA; break}
        1 {$PortNumberString = $Messages.PortB; break}
        2 {$PortNumberString = $Messages.PortC; break}
        3 {$PortNumberString = $Messages.PortD; break}
        default {$PortNumberString = $Messages.NotSupported; break}
    }

    return [PsCustomObject] @{
        Port       = $AdapterRegKey.Port
        PortString = $PortNumberString }
}

function GetDriverVersion($AdapterName)
{
    $DriverVersion = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).DriverVersion
    return $DriverVersion
}

function GetAdapterMiniPortName($AdapterName)
{
    $Name = ($script:PnpDevice.Where({$_.Name -eq $AdapterName})).Service
    $Instance = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    return [PsCustomObject] @{
        Name     = $Name
        Instance = $Instance }
}

function GetAdapterMaxSpeed($AdapterName, $SupportedSpeeds)
{
    if ($SupportedSpeeds.Length -gt 0)
    {
        # array is sorted, so we just return the last element
        return $SupportedSpeeds[-1]
    }
    return 0
}

function GetAdapterSupportedSpeeds($AdapterName)
{
    $SpeedDuplex = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "*SpeedDuplex"})
    if ($SpeedDuplex)
    {
        $RegistryValues = ($SpeedDuplex).ValidRegistryValues
    }

    $SupportedSpeeds = @()

    foreach ($i in $RegistryValues)
    {
        $SupportedSpeeds += $SupportedSpeedsMap[$i]
    }

    return $SupportedSpeeds | Sort-Object
}


function GetAdapterEEEStatus($AdapterName)
{
    $EEEStatus = 0
    $EEEStatusString = $Messages.NotSupported

    $Service = ($script:SupportedAdapters.Where({$_.Name -eq $AdapterName})).Service

    foreach ($EEE in $script:EEELinkStatusArray)
    {
        if (($null -ne $EEE) -and ($EEE.InstanceName -eq $AdapterName))
        {
            if ($EEE.EEELinkStatus -eq $false)
            {
                $EEELinkAdvertisement = 0
                if ($Service -in @("i40ea", "i40eb"))
                {
                    $EEELinkAdvertisement = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq "EEELinkAdvertisement"})
                    if ($EEELinkAdvertisement -and $EEELinkAdvertisement.RegistryValue)
                    {
                        $EEELinkAdvertisement = $EEELinkAdvertisement.RegistryValue
                    }
                    else
                    {
                        break
                    }
                }

                if ($EEELinkAdvertisement -gt 0)
                {
                    $EEEStatus = 3 #Not Negotiated
                    $EEEStatusString = $Messages.NotNegotiated
                }
                else
                {
                    $EEEStatus = 1 #Disabled
                    $EEEStatusString = $Messages.Disabled
                }
            }
            else
            {
                $EEEStatus = 2 #Active
                $EEEStatusString = $Messages.Active
            }
        }
    }

    return [PsCustomObject] @{
        EEEStatus       = $EEEStatus
        EEEStatusString = $EEEStatusString }
}

function GetAdapterStatus($AdapterName)
{
    $AdapterStatus = [ADAPTER_STATUS]::Installed -bor [ADAPTER_STATUS]::DriverLoaded -bor [ADAPTER_STATUS]::HasDiag

    $LinkStatus = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).Status
    if ($LinkStatus -eq 'Up')
    {
        $AdapterStatus = $AdapterStatus -bor [ADAPTER_STATUS]::HasLink
    }
    return $AdapterStatus
}

function GetAdapterCapabilities($AdapterName, $SupportedSpeeds)
{
    $Capabilities = @([int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_VENDOR_INTEL)

    foreach ($SupportedSpeed in $SupportedSpeeds)
    {
        switch ($SupportedSpeed)
        {
            10000000     {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10_MBPS; break}
            100000000    {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100_MBPS; break}
            1000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_1000_MBPS; break}
            2500000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_2500_MBPS; break}
            5000000000   {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_5000_MBPS; break}
            10000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_10000_MBPS; break}
            40000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_40000_MBPS; break}
            25000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_25000_MBPS; break}
            50000000000  {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_50000_MBPS; break}
            100000000000 {$Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_SPEED_100000_MBPS; break}
        }
    }
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_PERFORMANCE_PROFILE
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DIAGNOSTIC_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_FLASH_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_CYPRESS
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_IDENTIFY_ADAPTER_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_NDIS_IOAT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_EXTENDED_DMIX_SUPPORT
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_DCB
    $Capabilities += [int][ADAPTER_CAPABILITY]::NCS_ADAPTER_CAP_JUMBO_FRAMES

    return , ($Capabilities | Sort-Object)
}

function GetIntelDriverInfo()
{
    $script:BusTypesArray        = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_BusType -Property BusType -ErrorAction SilentlyContinue
    $script:DDPPkgNamesArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPackageInfo -Property Name, Major, Minor -ErrorAction SilentlyContinue
    $script:ETrackIdsArray       = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EetrackId -Property Id -ErrorAction SilentlyContinue
    $script:NVMVersionsArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EepromVersion -Property Version -ErrorAction SilentlyContinue
    $script:SanMacAddressesArray = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetSanMacAddress -Property SanMacAddr -ErrorAction SilentlyContinue
    $script:PartNumbersArray     = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_PartNumberString -Property PartNumberString -ErrorAction SilentlyContinue
    $script:PhyInfoArray         = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetPhyInfo -Property PhyInfo -ErrorAction SilentlyContinue
    $script:FwVersionsArray      = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetOemProductVer -Property SingleNvmVersion -ErrorAction SilentlyContinue
    $script:EEELinkStatusArray   = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EEELinkStatus -Property EEELinkStatus -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIocgYJKoZIhvcNAQcCoIIoYzCCKF8CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCACKIU7lkdvjS75
# O4GGqXKUGpY27jd8+uVCpwn4LBgV1qCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQCmOovGKSxCae+j/hQscnOWMA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjMwMTI1MDAwMDAwWhcNMjQwMTI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDGlXgQt1127m+5LYJmaFBzYTdi9cRozQjdZt/+
# +l4e/XT+sHZemhE0o/x+nydO/Ai7lhA6+Rs0Juw3vSXdGjKZLJ2G19nKCeICKfyK
# /rTX7GkdhwZnX8yQJEWU2VuZ/Ad9MtybHiZY3tIlpauWvNmfILMXWL+8DZOzVifw
# KbQgObHaYERAsfT2ZJ0HYp8lo5XepZy8SCeY/kRShw37vMeqNnuJtjnoB81/sZmI
# bZfLL4XxYn3ESyeeXgGhpplv3DkBUFM6YdD7lNXN7yp7h1gCJUA648U9EMOoQAHi
# ewLOPQSX/xgNqHfk3FUbcwZp07/zLrHtKvDNdPFrgkTBz81xGzVHOi48Lh6FBGW7
# UFEq/jI5TOk/AcM5eDvsizy4dA6J0AJ0o3lez+jAP2tDccEt7bn1aaamcjFpDOol
# MktQz0H07XqmRgVT3wHGHiDswDNUCxbk3ervsNG1afAPcFh5Hx4up8hpJoF7YTOf
# 6OHm8k3dRtjF1DC78oNYgqkpuNMCAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBRvq1HVd47vo0fOeEIdOUOyHE5Z
# ojAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBAJan0xi9l18PVDDZYZmI6kX64OJRsJvMbeAOd/iCwfoO1W2d
# +vin3je7xjz7TX1tERtLQGbRiraogMGVMco6+BSXJn0mGwpPcxsaRLpNbcbfXPHx
# niq3yO3gbSOrUU1Z9kScqlrqgbaAMKWnwlhy8MYbo9H2wIXCtrZ9kCuwoI9iilKS
# jjQqWs4085hIBzp3Kc32V4fLkuKGH1KYeTRMm8hP4+V0D+vBVFSu7PFh5t8p93y+
# Y/n7icZkyci6GUMUbVi8YK2z5KDr/ohPWS3a+DP+gAtjafTITgucRazljocMCIIg
# eWtL1Gwq0hvcpQOn+pdNGg8vUIXuVtziS9NmJKVfHEnOXtvte0ep9ziC5yaE8ui0
# 3e0ouY6xuhqPY5+mh9Wve5BX5ju1ewEUuImurs0xjtNBr3bscylS3AkssWz0LBko
# gR/aT9luMyUm/rC/2JLMrDlRTuiWergSd1ffFeLGnQBWtvd0snyFK3ZlVbJelDJP
# 47S4Vj1s45eyKo2DYTGCFdUwghXRAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRAKY6i8YpLEJp76P+FCxyc5YwDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQgNLPKlUxBcDESrGl12T2m
# dJfeVOdOH5k4wI+8aZSWHywwDQYJKoZIhvcNAQEBBQAEggGAlfbod8ueIVJkEpJS
# MT8Ty0jqlILn58ACuxAZGJ7OnFzi68dunYNkdQzvvkRysOjK8BpCbFGzQu3qpTR+
# iN6tDJsbc1vgXvl9KP1Hrzui0MJxzp5KXz/gB44g0O/aZMbZkhu6fmEnp3mLG6hn
# lTkxPTmoKXxYlib4dPgsLksjS9KEc8CerwF5DTZM99EKzkESgCxqT0zg4Ux2j0/S
# WjFdGuziA4hVjb1O7FDyH2whvSHSPB0E7EsoBvu/y7z11vXO44YEOs3Y6TJjBpLs
# 4lKUkNyvOpR3hctx5e7G+lBLPCLa/L7MmEM9wjZ3MZ9oc4xTb2QSB4CbBXIGzpSy
# 8ta7xsN4wTSGfe0QFw+Xc8GXh1Gk727r/56JeV2Sg30CVYRU7zdNE35KAtwIaglE
# HEdx6Jp+n1XQFSl8LxIRqkOiwuXg8Apesmi1PAFsoq1RhYHjwCeh0cXMgm9RO2dm
# 6aIsyfbCHBsj4+54rMI9en2QYdG1QGaqPp1wzfVWrpRg5uEroYITTjCCE0oGCisG
# AQQBgjcDAwExghM6MIITNgYJKoZIhvcNAQcCoIITJzCCEyMCAQMxDzANBglghkgB
# ZQMEAgIFADCB7wYLKoZIhvcNAQkQAQSggd8EgdwwgdkCAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQg1dtMGTe5DAYbfM1TmBGz9shg228GI3B3PCiNXm3H
# PmECFCP6Truw1vp7LrE3Tx9c5DhvmQB5GA8yMDIzMTEwNzE5MDA1N1qgbqRsMGox
# CzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# U2lnbmVyICM0oIIN6TCCBvUwggTdoAMCAQICEDlMJeF8oG0nqGXiO9kdItQwDQYJ
# KoZIhvcNAQEMBQAwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFu
# Y2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMB4XDTIz
# MDUwMzAwMDAwMFoXDTM0MDgwMjIzNTk1OVowajELMAkGA1UEBhMCR0IxEzARBgNV
# BAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UE
# AwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBTaWduZXIgIzQwggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQCkkyhSS88nh3akKRyZOMDnDtTRHOxoywFk
# 5IrNd7BxZYK8n/yLu7uVmPslEY5aiAlmERRYsroiW+b2MvFdLcB6og7g4FZk7aHl
# gSByIGRBbMfDCPrzfV3vIZrCftcsw7oRmB780yAIQrNfv3+IWDKrMLPYjHqWShkT
# XKz856vpHBYusLA4lUrPhVCrZwMlobs46Q9vqVqakSgTNbkf8z3hJMhrsZnoDe+7
# TeU9jFQDkdD8Lc9VMzh6CRwH0SLgY4anvv3Sg3MSFJuaTAlGvTS84UtQe3LgW/0Z
# ux88ahl7brstRCq+PEzMrIoEk8ZXhqBzNiuBl/obm36Ih9hSeYn+bnc317tQn/oY
# JU8T8l58qbEgWimro0KHd+D0TAJI3VilU6ajoO0ZlmUVKcXtMzAl5paDgZr2YGaQ
# WAeAzUJ1rPu0kdDF3QFAaraoEO72jXq3nnWv06VLGKEMn1ewXiVHkXTNdRLRnG/k
# Xg2b7HUm7v7T9ZIvUoXo2kRRKqLMAMqHZkOjGwDvorWWnWKtJwvyG0rJw5RCN4gg
# hKiHrsO6I3J7+FTv+GsnsIX1p0OF2Cs5dNtadwLRpPr1zZw9zB+uUdB7bNgdLRFC
# U3F0wuU1qi1SEtklz/DT0JFDEtcyfZhs43dByP8fJFTvbq3GPlV78VyHOmTxYEsF
# T++5L+wJEwIDAQABo4IBgjCCAX4wHwYDVR0jBBgwFoAUGqH4YRkgD8NBd0UojtE1
# XwYSBFUwHQYDVR0OBBYEFAMPMciRKpO9Y/PRXU2kNA/SlQEYMA4GA1UdDwEB/wQE
# AwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMEoGA1Ud
# IARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEWF2h0dHBzOi8vc2Vj
# dGlnby5jb20vQ1BTMAgGBmeBDAEEAjBEBgNVHR8EPTA7MDmgN6A1hjNodHRwOi8v
# Y3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcmwwdAYI
# KwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnNlY3RpZ28uY29t
# L1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYIKwYBBQUHMAGGF2h0dHA6
# Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBMm2VY+uB5z+8V
# wzJt3jOR63dY4uu9y0o8dd5+lG3DIscEld9laWETDPYMnvWJIF7Bh8cDJMrHpfAm
# 3/j4MWUN4OttUVemjIRSCEYcKsLe8tqKRfO+9/YuxH7t+O1ov3pWSOlh5Zo5d7y+
# upFkiHX/XYUWNCfSKcv/7S3a/76TDOxtog3Mw/FuvSGRGiMAUq2X1GJ4KoR5qNc9
# rCGPcMMkeTqX8Q2jo1tT2KsAulj7NYBPXyhxbBlewoNykK7gxtjymfvqtJJlfAd8
# NUQdrVgYa2L73mzECqls0yFGcNwvjXVMI8JB0HqWO8NL3c2SJnR2XDegmiSeTl9O
# 048P5RNPWURlS0Nkz0j4Z2e5Tb/MDbE6MNChPUitemXk7N/gAfCzKko5rMGk+al9
# NdAyQKCxGSoYIbLIfQVxGksnNqrgmByDdefHfkuEQ81D+5CXdioSrEDBcFuZCkD6
# gG2UYXvIbrnIZ2ckXFCNASDeB/cB1PguEc2dg+X4yiUcRD0n5bCGRyoLG4R2fXto
# T4239xO07aAt7nMP2RC6nZksfNd1H48QxJTmfiTllUqIjCfWhWYd+a5kdpHoSP7I
# VQrtKcMf3jimwBT7Mj34qYNiNsjDvgCHHKv6SkIciQPc9Vx8cNldeE7un14g5glq
# fCsIo0j1FfwET9/NIRx65fWOGtS5QDCCBuwwggTUoAMCAQICEDAPb6zdZph0fKlG
# Nqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpO
# ZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVT
# RVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoXDTM4MDExODIzNTk1OVow
# fTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4G
# A1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQD
# ExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyAIVSTuc+gPlPvs1rAdtYa
# BKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhObtoY/AQWSZm8hq9VxEHmH
# 9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5pCNCrNAVw3+eolzXOPEW
# snDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EBjm6dJ3VV0Rc7NCF7lwGU
# r3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWMilZx0S2UTMiMPNMU
# opy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G493bkV+fPzY+DjBnK0a3
# Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3pBXrGVX/0DlULSbu
# RRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH3uhrDvBbfel2aZMgHEqX
# LHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqNoWTJiJJZy6hGwMny
# pv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfobpSKupcNNgtCN2mu32/cY
# QFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInXT50zvRq1naHelUF6p4MC
# AwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0G
# A1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNVHQ8BAf8EBAMCAYYwEgYD
# VR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAI
# MAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3Qu
# Y29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHkuY3JsMHYGCCsG
# AQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2NydC51c2VydHJ1c3QuY29t
# L1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUFBzABhhlodHRwOi8v
# b2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBtVIGlM10W4bVT
# gZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ2+J9+Z8A70DDrdPi
# 5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V1cPivNVYeL9IduFEZ22P
# sEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8Blr5yUmo7Qjd8Fng5k5OU
# m7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZLXq2jK0KKIZL+qG9aJXBi
# gXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXnd8ucLo0jGLmjwkcd8F3W
# oXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1BWy30mRa2Coiwkud8Tle
# TN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9UIFuq/DQ0/yysx+2mZYm
# 9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4ZSA7D0GqH+mhQZ3+
# sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbGpm0yXbQ4CStJB6r97DDa
# 8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7xfCAMKneutaIChrop
# 7rRaALGMq+P5CslUXdS5anSevUiumDGCBCwwggQoAgEBMIGRMH0xCzAJBgNVBAYT
# AkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZv
# cmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBS
# U0EgVGltZSBTdGFtcGluZyBDQQIQOUwl4XygbSeoZeI72R0i1DANBglghkgBZQME
# AgIFAKCCAWswGgYJKoZIhvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJ
# BTEPFw0yMzExMDcxOTAwNTdaMD8GCSqGSIb3DQEJBDEyBDC5YtkqRCH9twZ/hjl2
# 2Qz4e1eSOKM+H6Ui3SWuOtUM805xsYFgI41Ar9SvJ4P5bNMwge0GCyqGSIb3DQEJ
# EAIMMYHdMIHaMIHXMBYEFK5ir3UKDL1H1kYfdWjivIznyk+UMIG8BBQC1luV4oNw
# wVcAlfqI+SPdk3+tjzCBozCBjqSBizCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Ck5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUg
# VVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlm
# aWNhdGlvbiBBdXRob3JpdHkCEDAPb6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEB
# BQAEggIAYsvqZHD9O4a7aSP5lrlXsHhH7M3kIQICst3HYgVqr+rEwIUtDQPigj7E
# PI64g4BGvW/o010unKYqCoVGEaWV+dv4xvMf4BIDl5dD0d8MocA7Cy+2XUgleI5l
# 6UN2+lqK6GG5iY37ZfXbbxbsr/VXB/RjgvEljgK+G9XLZPcoQg+IwMj4RQjmEdOC
# JNv8aKZwllVhygPXV+zzfGtea/xs1hV2uFd2FMJmUjne8VaGrlVqIlXVb9WLjVha
# /0ZJDa9V1PEbq7RxeG6DOWJWh0J8jHbNlwbv0Ih1FPfXlf1sTko5owL7rCOnt+Dt
# rWl4VWQXSjkLOM/aIX6ch4jsVwG0M17zmYcdx8S549cLVyClTsGO0aiel3HFk602
# g1VUf6iR0KELN3X30RARkWLhgBroKHN4NYQcSDDKH2hEmES4fhHX5Fc0bUTFN5Ih
# 4PnTZDYOPxjSHIIs58FZA7FAglZdDRNgGPOO/mXk+5UjMtGistqYYJDBitxHtbaP
# Ngx2J2hKdp3O1YGlWLqWLNrvbtXnHKDURruHuKXVCUZ/JjHGiLGAjRbrNR9uk1++
# 1SPPtRPKNcn4C+z5moaHOsM1AQSUodJ+MxdNVA7LgI8yYzpmPVI0qIxSSxecjUNI
# 6lIJuasZFA923kAtzP8qTfWpRIcl8O/wRF1Ps3auCB/NhF/K+MA=
# SIG # End signature block
