# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'Global:FormatEnumerationLimit')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesGet))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
                continue
            }

            $Local:IntelLogConfigurations = @{}
            $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
            $Local:IntelLogConfigModules = @{}
            $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"
            $Local:IntelLogConfigLevels = @{}
            $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

            if ($IntelLogConfigLevels.Count -eq 0)
            {
                $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
            }

            $DisplayConfigNames = @()
            if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Get" $a))
            {
                if ($script:ErrorMessagesGet.Count -gt 0)
                {
                    break
                }
                continue
            }

            $DisplayModules = @()
            $DisplayLevels = @()
            $DriverPassed = $null

            if ($DisplayConfigNames)
            {
                $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"
                if (!(ValidateMethodOutput $MethodOutput "Get"))
                {
                    continue
                }

                GetConfigLogs $DisplayConfigNames ([ref]$DisplayModules) ([ref]$DisplayLevels) $MethodOutput
            }
            else
            {
                $DriverPassed = $ModuleNames.Where({$DRIVER_MODULE_NAME -like $_})

                if (-Not $DriverPassed -and $SubModule)
                {
                    $script:ErrorMessagesGet += $Messages.InvalidParams
                    break
                }
                elseif ($SubModule)
                {
                    $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesGet)
                    if ($SubModuleSum -eq -1)
                    {
                        break
                    }
                }

                if ($DriverPassed -or -Not $ModuleNames)
                {
                    $LogConfigs += GetDriverLogConfig $DriverPassed $SubModuleSum $a ([ref]$script:ErrorMessagesGet)
                }

                $MethodOutput = $null
                if ($IntelLogConfigModules.Count -gt 0)
                {
                    $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"

                    if (!(ValidateMethodOutput $MethodOutput "Get"))
                    {
                        continue
                    }
                }

                # ValidateGetModuleParam will error if Driver is passed and not present here.
                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                ValidateGetModuleParam ([ref]$DisplayModules) ([ref]$DisplayLevels) $ModuleNames $MethodOutput $a
            }

            if ($DisplayModules -and $DisplayLevels)
            {
                $LogConfigs += GatherOutput $a $DisplayModules $DisplayLevels $DriverPassed
            }
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }
        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false,Position=2)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String]
    $Level = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = '',
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $SubModule = ''
    )
    Begin
    {
        $ModuleNames = $Module
        $LevelName = $Level
        $ConfigurationNames = $Configuration
        $SubModuleSum = $SUBMODULE_DEFAULT_SUM
        $script:WarningMessagesSet = @()
        $script:ErrorMessagesSet = @()
        $LogConfigs = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        do
        {
            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesSet)
            $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))

            if ([string]::IsNullOrEmpty($ConfigurationNames) -and ([string]::IsNullOrEmpty($ModuleNames)))
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigOrModule
                break
            }

            foreach($a in $AdapterNames)
            {
                $TmpStatusMsg = CheckDeviceError $a
                if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                {
                    $script:ErrorMessagesSet += $TmpStatusMsg
                    continue
                }

                $DisplayModules = @()
                $DisplayLevels = @()
                $InputModulesId = [uint32[]]::new(64)
                $InputLevelId = [uint32[]]::new(64)

                $Local:IntelLogConfigurations = @{}
                $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS"
                $Local:IntelLogConfigModules = @{}
                $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES"

                $IntelLogConfigModules[$DRIVER_MODULE_NAME] = $DRIVER_MODULE_ID

                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                $ModuleCount = 0
                $DisplayConfigNames = @()
                if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) "Set" $a))
                {
                    break
                }

                $bDriverPassed = $false
                $InputConfigurationIDs = $null
                if ($DisplayConfigNames)
                {
                    $InputConfigurationIDs = $IntelLogConfigurations[$DisplayConfigNames]
                }
                else
                {
                    ValidateSetModuleParam ([ref]$InputModulesId) ([ref]$DisplayModules) $ModuleNames $a

                    if ($DisplayModules -contains $DRIVER_MODULE_NAME)
                    {
                        $bDriverPassed = $true
                    }
                    elseif ($SubModule)
                    {
                        $script:ErrorMessagesSet += $Messages.InvalidParams
                        break
                    }

                    $ModuleCount = $DisplayModules.Count
                }

                if ($ModuleCount -ne 0)
                {
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $ModuleCount $a $bDriverPassed))
                    {
                        break
                    }

                    if ($bDriverPassed)
                    {
                        # Driver module is not included in Wmi method params
                        $ModuleCount -= 1

                        if ($InputLevelId[0] -lt 3)
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidDriverLevel
                            break
                        }

                        if ($SubModule)
                        {
                            $SubModuleSum = GetSubModuleBitmask $SubModule ([ref]$script:ErrorMessagesSet)
                            if ($SubModuleSum -eq -1)
                            {
                                break
                            }
                        }

                        if (-not (SetSubModuleRegistryValue $a $SubModuleSum))
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                            break
                        }

                        $LevelName = GetLevelName($InputLevelId[0])

                        if (-Not (SetDriverLogLevel $a $LevelName))
                        {
                            $script:ErrorMessagesSet += $Messages.InvalidSystemConfigFailure
                            break
                        }
                    }

                    if ($ModuleCount -gt 0)
                    {
                        $params = @{Count    = [uint32]$ModuleCount;
                                ModuleID = $InputModulesId;
                                Loglevel = $InputLevelId}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogLevels" $params
                    }

                    if ($bDriverPassed)
                    {
                        # Driver is set, display it. Avoid call to Validate if no other modules passed.
                        if ($ModuleCount -eq 0 -or (ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules -SubModule $SubModuleSum
                        }
                        else
                        {
                            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule $SubModuleSum
                        }
                    }
                    elseif (ValidateMethodOutput $MethodOutput "Set")
                    {
                        $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules
                    }
                }
                elseif ($null -ne $InputConfigurationIDs)
                {
                    $count = 1
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $count $a))
                    {
                        break
                    }

                    $bContinue = $true
                    foreach ($ConfigID in $InputConfigurationIDs)
                    {
                        $params = @{ConfigurationID = [uint32]$ConfigID;
                                    Level = $InputLevelId[0]}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogConfiguration" $params

                        if (!(ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $bContinue = $false
                            break
                        }
                    }

                    if (!$bContinue)
                    {
                        break
                    }
                    $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Configuration $DisplayConfigNames
                }
            }
        } while ($false)
    }
    End
    {
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Start-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [ValidateNotNullOrEmpty()]
    [string]
    $Path = '',
    [parameter(Mandatory=$false)]
    [switch]
    $Force,
    [parameter(Mandatory=$false)]
    [switch]
    $Append
    )
    Begin
    {
        $SubModuleSum = 0xFFFF
        # Set LogmanLevel to Normal by default
        $LogmanLevel = 4
        $script:ErrorMessagesStart = @()
        $script:WarningMessagesStart = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesStart += $Messages.AdapterNotFound -f $Name
                break
            }

            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStart)
            $AdapterName = GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesStart)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $TmpStatusMsg = CheckDeviceError $AdapterName
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesStart += $TmpStatusMsg
                break
            }

            $UseDefaultPath = $false
            if (([string]::IsNullOrEmpty($Path)))
            {
                $UseDefaultPath = $true
            }

            $LogmanGUID = GetLogmanGUID $AdapterName ([ref]$script:ErrorMessagesStart)
            $LogName = GetLogName $AdapterName ([ref]$script:ErrorMessagesStart)

            if ($null -eq $LogmanGUID -or $null -eq $LogName)
            {
                break
            }

            if (LogmanProcessRunning $AdapterName $LogName $LogmanGUID)
            {
                break
            }

            $LogmanPath = ''
            if (-not (ValidatePathParams ([ref]$LogmanPath) $UseDefaultPath $LogName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $DriverLogLevel = GetDriverLogLevel $AdapterName
            if ($DriverLogLevel)
            {
                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $AdapterName "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS"

                if ($IntelLogConfigLevels.Count -eq 0)
                {
                    $IntelLogConfigLevels = $DRIVER_LOG_LEVEL_MAP
                }

                # Driver log levels are normal levels value + 1
                $LogmanLevel = $IntelLogConfigLevels[$DriverLogLevel] + 1

                $LogMetadataName = "$LogName.txt"

                $LogMetadataPath = ''
                if (-Not $UseDefaultPath)
                {
                    if ([IO.Path]::GetExtension($Path) -eq ".txt")
                    {
                        $Path += ".txt"
                    }
                    else
                    {
                        $Path = [IO.Path]::ChangeExtension($Path, ".txt")
                    }
                }

                $iBeforeErrorCount = $script:ErrorMessagesStart.Count
                if (-not (ValidatePathParams ([ref]$LogMetadataPath) $UseDefaultPath $LogMetadataName ([ref]$script:ErrorMessagesStart)))
                {
                    if ($iBeforeErrorCount -lt $script:ErrorMessagesStart.Count -and $script:ErrorMessagesStart[-1] -eq $Messages.LogmanFileExists -f $AdapterName)
                    {
                        # Driver Metadata File is always replaced if it exists.
                        $script:ErrorMessagesStart = $script:ErrorMessagesStart | Select-Object -First ($script:ErrorMessagesStart.Count - 1)
                    }
                    else
                    {
                        break
                    }
                }

                $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
                if ($null -eq $AdapterRegistryPath)
                {
                    $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
                    break
                }

                $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
                if ($SubModuleRegistryValue)
                {
                    $SubModuleSum = $SubModuleRegistryValue.SubModule

                    if (-Not (CreateDriverLogMetadataFile $AdapterName))
                    {
                        break
                    }
                }
            }

            LogmanStart $LogmanPath $LogName $LogmanGUID $SubModuleSum $LogmanLevel
            if (ValidateLogmanStart $AdapterName)
            {
                $LogmanPath = Resolve-Path -Path $LogmanPath -ErrorAction SilentlyContinue
                Write-Output $LogmanPath
            }

            break
        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesStart)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStart)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Stop-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null
    )
    Begin
    {
        $script:ErrorMessagesStop = @()
        $script:WarningMessagesStop = @()
        $bErrorObtained = $false
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStop)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesStop))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesStop += $TmpStatusMsg
                continue
            }

            if (-Not $bErrorObtained)
            {
                $LogName = GetLogName $a ([ref]$script:ErrorMessagesStop)
            }
            else
            {
                $LogName = GetLogName $a ([ref]@())
            }

            if ($null -eq $LogName)
            {
                $bErrorObtained = $true
                continue
            }

            LogmanStop $LogName
            if (-Not $bErrorObtained)
            {
                if (-Not (ValidateLogmanStop $a ([ref]$script:ErrorMessagesStop)))
                {
                    $bErrorObtained = $true
                }
            }
        }
    }
    End
    {
        foreach ($WarningMessage in $script:WarningMessagesStop)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStop)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Disable-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesDisable = @()
        $script:WarningMessagesDisable = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesDisable)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesDisable))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesDisable += $TmpStatusMsg
                continue
            }

            $DriverConfig = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME -SubModule @($SUB_MODULE_MAP.Keys)[0] -Level "Normal"
            if (-Not $DriverConfig)
            {
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiDisableFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Disable"))
                {
                    continue
                }
            }

            # Will pass Driver as module, so returns IntelDriverLogConfiguration
            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*" -SubModule @($SUB_MODULE_MAP.Keys)[0]
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesDisable)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesDisable)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Reset-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesReset = @()
        $script:WarningMessagesReset = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = @(Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue)
        }
    }
    Process
    {
        $Adapters = $Adapter

        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesReset)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesReset))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesReset += $TmpStatusMsg
                continue
            }

            # Set driver level back to Normal
            $null = Set-IntelEthernetLogConfig -Name $a -Module $DRIVER_MODULE_NAME

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiResetFwLogLevels"

            if ($null -ne $MethodOutput)
            {
                if (!(ValidateMethodOutput $MethodOutput "Reset"))
                {
                    continue
                }
            }

            $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module "*"
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesReset)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesReset)
        {
            Write-Error $ErrorMessage
        }
    }
}

Function GatherOutput($AdapterName, $DisplayModules, $DisplayLevels, $DriverPassed)
{
    $LogConfigs = @()
    $ValidLevels = $IntelLogConfigLevels.Keys
    $LogConfigCount = $DisplayModules.Count
    for ($i = 0; $i -lt $LogConfigCount; $i++)
    {
        if ($DriverPassed)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                SubModule  = "";
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
        else
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                Name        = $AdapterName;
                Module      = $DisplayModules[$i];
                Level       = $DisplayLevels[$i];
                ValidSubModules = @();
                ValidLevels = $ValidLevels}
        }
    }

    return $LogConfigs
}

Function GetModuleName($ModuleId)
{
    return $IntelLogConfigModules.GetEnumerator().Where({$_.Value -eq $ModuleId}).Name
}

Function GetLevelName($LevelId)
{
    return $IntelLogConfigLevels.GetEnumerator().Where({$_.Value -eq $LevelId}).Name
}

Function GetDriverLogLevel($AdapterName)
{
    $DriverRegistryValue = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $DriverRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    }

    if ($null -ne $DriverRegistryValue)
    {
        $DriverRegistryValue = $DriverRegistryValue.DriverLogLevel
    }

    return $DriverRegistryValue
}

Function GetDriverLogConfig($DriverPassed, $SubModuleSum, $AdapterName, [ref]$ErrorMessages)
{
    $LogConfigs = @()
    $InactiveSubModuleArray = @()
    $ActiveSubModuleArray = @()

    $DriverLevel = GetDriverLogLevel $AdapterName
    if (-Not $DriverLevel)
    {
        # Set to Normal by default
        $DriverLevel = @($DRIVER_LOG_LEVEL_MAP.Keys)[1]
        if (-Not (SetDriverLogLevel $AdapterName $DriverLevel))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }
    }

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -eq $AdapterRegistryPath)
    {
        $ErrorMessages.Value += $Messages.NoCmdletSupport -f $AdapterName
        break
    }

    $SubModuleRegistryValue = Get-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -ErrorAction SilentlyContinue
    if (-Not $SubModuleRegistryValue -or [string]::IsNullOrEmpty($SubModuleRegistryValue.SubModule))
    {
        # If driver passed but set has not been used, enable default driver logging
        if (-not (SetSubModuleRegistryValue $a $SUBMODULE_DEFAULT_SUM))
        {
            $ErrorMessages.Value += $Messages.InvalidSystemConfigFailure
            break
        }

        $RegistrySubModuleSum = $SUBMODULE_DEFAULT_SUM
    }
    else
    {
        $RegistrySubModuleSum = $SubModuleRegistryValue.SubModule
    }

    foreach ($DisplayName in $SUB_MODULE_MAP.Keys)
    {
        if ($SUB_MODULE_MAP[$DisplayName] -band $SubModuleSum)
        {
            if ($SUB_MODULE_MAP[$DisplayName] -band $RegistrySubModuleSum)
            {
                $ActiveSubModuleArray += $DisplayName
            }
            else
            {
                $InactiveSubModuleArray += $DisplayName
            }
        }
    }

    if ($DriverPassed)
    {
        foreach ($ActiveSubModule in $ActiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $ActiveSubModule;
                                            Level       = $DriverLevel;
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
        foreach ($InactiveSubModule in $InactiveSubModuleArray)
        {
            $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelDriverLogConfiguration';
                                            Name        = $AdapterName;
                                            Module      = $DRIVER_MODULE_NAME;
                                            SubModule   = $InactiveSubModule;
                                            Level       = @($DRIVER_LOG_LEVEL_MAP.Keys)[-1]; # Disabled
                                            ValidSubModules = $SUB_MODULE_MAP.Keys;
                                            ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
        }
    }
    else
    {
        $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                                        Name            = $AdapterName;
                                        Module          = $DRIVER_MODULE_NAME;
                                        SubModule       = @($ActiveSubModuleArray);
                                        Level           = $DriverLevel;
                                        ValidSubModules = $SUB_MODULE_MAP.Keys;
                                        ValidLevels = @($DRIVER_LOG_LEVEL_MAP.Keys)[0,1] }
    }

    return $LogConfigs
}

Function GetSubModuleBitmask($SubModuleArray, [ref]$ErrorMessages)
{
    $iSubModuleSum = 0

    foreach ($SubModule in $SubModuleArray)
    {
        if ($SubModule -match "^0[xX][A-Fa-f0-9]+$" -or $SubModule -match "^[1-9]\d*$")
        {
            if ([int]$SubModule -le $SUM_SUBMODULES)
            {
                $iSubModuleSum = $iSubModuleSum -bor $SubModule
            }
            else
            {
                $iSubModuleSum = $SUM_SUBMODULES
                continue
            }
        }
        elseif ($SUB_MODULE_MAP.Keys -Contains $SubModule)
        {
            $iSubModuleSum = $iSubModuleSum -bor $SUB_MODULE_MAP[$SubModule]
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidSubModule -f $a, $SubModule
            $iSubModuleSum = -1
            break
        }
    }

    return $iSubModuleSum
}

Function GetLogmanGUID($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_WPP_GUID; break }
            'i40eb' { return $I40EB_WPP_GUID; break }
            'icea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $ICEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            'scea' {
                $WppGuid = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID"

                if (-Not $WppGuid)
                {
                    $WppGuid = $SCEA_WPP_GUID
                }

                return $WppGuid
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function GetLogName($AdapterName, [ref]$ErrorMessages)
{
    $CurrentAdapter = $script:PnpDevice.Where({ $_.FriendlyName -eq $AdapterName })
    if ($CurrentAdapter)
    {
        switch ($CurrentAdapter.Service)
        {
            'i40ea' { return $I40EA_LOGNAME; break }
            'i40eb' { return $I40EB_LOGNAME; break }
            'icea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $ICEA_LOGNAME
                }

                return $LogName
                break
            }
            'scea' {
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME"
                if (-Not $LogName)
                {
                    $LogName = $SCEA_LOGNAME
                }

                return $LogName
                break
            }
            default {
                $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
                if (-not ($ErrorMessages.Value -contains $TmpErrorMsg))
                {
                    $ErrorMessages.Value += $TmpErrorMsg
                }

                break
            }
        }
    }

    return $null
}

Function SetDriverLogLevel($AdapterName, $DriverLevel)
{
    $bSuccess = $false
    $SetDriverLogLevel = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetDriverLogLevel = Set-ItemProperty -Path $AdapterRegistryPath -Name $DRIVERLOG_REGISTRYVAR_NAME -Type "String" -Value $DriverLevel -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetDriverLogLevel -and -Not [string]::IsNullOrEmpty($SetDriverLogLevel.DriverLogLevel))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function SetSubModuleRegistryValue($AdapterName, $SubModuleValue)
{
    $bSuccess = $false
    $SetSubModule = $null

    $AdapterRegistryPath = GetAdapterRegistryConfigPath $AdapterName ""
    if ($null -ne $AdapterRegistryPath)
    {
        $SetSubModule = Set-ItemProperty -Path $AdapterRegistryPath -Name $SUBMODULE_REGISTRYVAR_NAME -Value $SubModuleValue -Type "DWORD" -PassThru -ErrorAction SilentlyContinue
    }

    if ($null -ne $SetSubModule -and -Not [string]::IsNullOrEmpty($SetSubModule.SubModule))
    {
        $bSuccess = $true
    }

    return $bSuccess
}

Function CreateDriverLogMetadataFile($AdapterName)
{
    $strDriverName = ($script:SupportedAdapters.Where({ $_.Name -eq $AdapterName })).Service

    $strOSSpecificDriverName = (GetOSSpecificDriverName $strDriverName)
    if (-Not $strOSSpecificDriverName)
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    $DriverFolder = GetDriverInfParentFolder $strOSSpecificDriverName "Start"
    $DriverMetaData = Get-Item "$DriverFolder\$strOSSpecificDriverName.sys" -ErrorAction SilentlyContinue

    if ($DriverMetaData)
    {
        $DriverVersionInfo = $DriverMetaData.VersionInfo
        $DriverLength = "$strOSSpecificDriverName.sys Length: " + $DriverMetaData.Length
        $DriverChecksum = Get-FileHash $DriverMetaData

        try
        {
            Set-Content $LogMetadataPath -Value $DriverVersionInfo -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverLength -ErrorAction Stop
            Add-Content $LogMetadataPath -Value $DriverChecksum -ErrorAction Stop
        }
        catch
        {
            $script:ErrorMessagesStart += $Messages.PathIncorrect
        }
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName
        return $false
    }

    return $true
}

Function ValidateMethodOutput($MethodOutput, $Verb)
{
    Write-Verbose -Message "MethodOutput: $MethodOutput verb: $Verb"

    do
    {
        $Result = $true
        if ($null -eq $MethodOutput)
        {
            switch ($Verb)
            {
                Get {$script:WarningMessagesGet += $Messages.NoCmdletSupport -f $a; break}
                Set {$script:WarningMessagesSet += $Messages.NoCmdletSupport -f $a; break}
                Start {$script:WarningMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
                Stop {$script:WarningMessagesStop += $Messages.NoCmdletSupport -f $a; break}
                Disable {$script:WarningMessagesDisable += $Messages.NoCmdletSupport -f $a; break}
                Reset {$script:WarningMessagesReset += $Messages.NoCmdletSupport -f $a; break}
            }

            $Result = $false
            break
        }

        if ($MethodOutput.OutStatus -ne 0)
        {
            switch ($Verb)
            {
                Get {$script:ErrorMessagesGet += $Messages.OperationFailed -f $a, $Verb; break}
                Set {$script:ErrorMessagesSet += $Messages.OperationFailed -f $a, $Verb; break}
                Start {$script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, $Verb; break}
                Stop {$script:ErrorMessagesStop += $Messages.OperationFailed -f $a, $Verb; break}
                Disable {$script:ErrorMessagesDisable += $Messages.OperationFailed -f $a, $Verb; break}
                Reset {$script:ErrorMessagesReset += $Messages.OperationFailed -f $a, $Verb; break}
            }

            $Result = $false
        }
    } while ($false)

    Return $Result
}

Function ValidateGetModuleParam([ref]$DisplayModules, [ref]$DisplayLevels, $ModuleNames, $MethodOutput, $AdapterName)
{
    if ($ModuleNames)
    {
        foreach ($ModuleName in $ModuleNames)
        {
            $ModuleIds = $null
            $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
            $ModuleIDCount = $ModuleIDNames.Count

            if ($ModuleIDCount -gt 0)
            {
                $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })

                if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
                {
                    $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                    foreach ($ModuleId in $ModuleIds)
                    {
                        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
                        {
                            if ($ModuleId -eq $MethodOutput.ModuleID[$i])
                            {
                                $DisplayModules.Value += GetModuleName($ModuleId)
                                $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
                                break
                            }
                        }
                    }
                }
            }
            else
            {
                #if user passes in an array of modules and one (or more) is spelled incorrectly,
                #show error msg for that 1 and continue on getting the modules for the rest
                $script:ErrorMessagesGet += $Messages.InvalidModule -f $AdapterName, $ModuleName
            }
        }
    }
    else
    {
        GetAllModulesLevel $MethodOutput $DisplayModules $DisplayLevels
    }
}

Function GetAllModulesLevel($MethodOutput, [ref]$DisplayModules, [ref]$DisplayLevels)
{
    if ($MethodOutput)
    {
        for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$i])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
            # driver returns an array of 64 but only 31 modules populated
            if ($i -eq 31)
            {
                break
            }
        }
    }
}

Function ValidateLevelParam([ref]$InputLevelId, [ref]$DisplayLevels, $LevelName, $Count, $AdapterName, $DriverPassed)
{
    do
    {
        $NewLevelId = $null
        $Result = $true

        $LevelMatchArray = $IntelLogConfigLevels.GetEnumerator().Where({$_.Name -like $LevelName})
        if ($LevelMatchArray)
        {
            $NewLevelId = @($LevelMatchArray.Value)
        }
        # Handle default case of "Normal" if Level is not passed
        elseif ($DriverPassed -and -not $LevelName)
        {
            $NewLevelId = @($IntelLogConfigLevels["Normal"])
        }

        if ($null -eq $NewLevelId -or 0 -eq $NewLevelId.Count)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevel -f $AdapterName, $LevelName, $ModuleNames[0]
            $Result = $false
            break
        }

        if ($NewLevelId.count -gt 1)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevelWildcard
            $Result = $false
            break
        }

        for ($i = 0; $i -lt $Count; $i++)
        {
            $InputLevelId.Value[$i] = $NewLevelId[0]
            $DisplayLevels.Value += GetLevelName($NewLevelId[0])
        }

    } while ($false)

    Return $Result
}

Function ValidateSetModuleParam([ref]$InputModulesId, [ref]$DisplayModules, $ModuleNames, $AdapterName)
{
    $count = 0
    foreach ($ModuleName in $ModuleNames)
    {
        $ModuleIds = $null
        $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName
        $ModuleIDCount = $ModuleIDNames.Count

        if ($ModuleIDCount -gt 0)
        {
            $ModuleIDNames = $ModuleIDNames.Where({ $_ -NotLike $DRIVER_MODULE_NAME })
            if (-Not $ModuleIDNames -or $ModuleIDCount -gt $ModuleIDNames.Count)
            {
                $DisplayModules.Value += $DRIVER_MODULE_NAME
            }

            if ($ModuleIDNames -and $ModuleIDNames.Count -gt 0)
            {
                $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                foreach ($ModuleId in $ModuleIds)
                {
                    $InputModulesId.Value[$count] = $ModuleId
                    $DisplayModules.Value += GetModuleName($ModuleId)
                    $count++
                }
            }
        }
        else
        {
            #if user passes in an array of modules and one (or more) is spelled incorrectly,
            #show error msg for that 1 and continue on getting the modules for the rest
            $script:ErrorMessagesSet += $Messages.InvalidModule -f $AdapterName, $ModuleName
        }
    }
}

Function LogmanProcessRunning($AdapterName, $LogName, $GUID)
{
    $Result = $false

    $LogmanQueryString = logman query $LogName -ets
    if ($LogmanQueryString -match ($GUID))
    {
        $script:ErrorMessagesStart += $Messages.LogmanRunning -f $AdapterName
        $Result = $true
    }

    return $Result
}

Function LogmanStart($LogmanPath, $LogName, $GUID, $SubModuleSum, $Level)
{
    if ($Append)
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level -a
    }
    else
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID $SubModuleSum $Level
    }
}

Function LogmanStop($LogName)
{
    $null = logman stop $LogName -ets
}


Function ValidateLogmanStart($AdapterName)
{
    $Result = $true
    # -2147024629, -2147024773, -2147024893, -2147024891 error codes mean directory name is invalid
    # or filename, directory name, or volume label syntax is incorrect or permission is denied.
    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2147024629 -or $LASTEXITCODE -eq -2147024773 -or $LASTEXITCODE -eq -2147024893 -or $LASTEXITCODE -eq -2147024891)
    {
        $script:ErrorMessagesStart += $Messages.PathIncorrect
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, "Start"
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStop($AdapterName, [ref]$ErrorMessages)
{
    $Result = $true

    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2144337918)
    {
        $ErrorMessages.Value += $Messages.NoLogsStarted -f $AdapterName
        $Result = $false
    }
    else
    {
        $ErrorMessages.Value += $Messages.OperationFailed -f $AdapterName, "Stop"
        $Result = $false
    }
    Return $Result
}

Function ValidateConfigurationParam($ConfigurationNames, $ModuleNames, [ref]$DisplayConfigNames, $Verb, $a)
{
    $Result = $true

    do
    {
        if ($ConfigurationNames)
        {
            if ($ModuleNames)
            {
                switch ($Verb)
                {
                    Set { $script:ErrorMessagesSet += $Messages.InvalidParamsConfigAndModule; break }
                    Get { $script:ErrorMessagesGet += $Messages.InvalidParamsConfigAndModule; break }
                }
                $Result = $false
                break
            }

            if ($IntelLogConfigurations.Count -eq 0)
            {
                if ($Verb -eq "Set")
                {
                    $script:ErrorMessagesSet += $Messages.NoConfigurationSupport -f $a
                }
                elseif ($Verb -eq "Get")
                {
                    $script:WarningMessagesGet += $Messages.NoConfigurationSupport -f $a
                }

                $Result = $false
                break
            }

            foreach ($ConfigName in $ConfigurationNames)
            {
                $TmpDisplayConfigNames = $IntelLogConfigurations.Keys -Like $ConfigName

                if ($TmpDisplayConfigNames.Count -gt 0)
                {
                    $DisplayConfigNames.Value += $TmpDisplayConfigNames
                }
                else
                {
                    #if user passes in an array of configs and one (or more) is spelled incorrectly,
                    #show error msg for that 1 and continue on getting the config for the rest
                    switch ($Verb)
                    {
                        Set { $script:ErrorMessagesSet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                        Get { $script:ErrorMessagesGet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName; break }
                    }

                    $Result = $false
                }
            }
        }
    } while ($false)

    return $Result
}

Function GetConfigLogs($DisplayConfigNames, [ref]$DisplayModules, [ref]$DisplayLevels, $MethodOutput)
{
    $ConfigurationID = $IntelLogConfigurations[$DisplayConfigNames]
    foreach ($ConfigID in $ConfigurationID)
    {
        $params = @{ConfigurationID = [uint32]$ConfigID}
        $GetConfigMethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevelsForConfiguration" $params

        for ($i = 0; $i -lt $GetConfigMethodOutput.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$GetConfigMethodOutput.ModuleID[$i]])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$GetConfigMethodOutput.ModuleID[$i]])
        }
    }
}

function GetConfigHashTable($AdapterName, $ConfigPath)
{
    $ConfigHashTable = @{}

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $ConfigPath
    if ($null -ne $FWLogConfigPath)
    {
        $LogConfigs = Get-Item $FWLogConfigPath -ErrorAction SilentlyContinue
        if ($null -ne $LogConfigs)
        {
            $LogConfigs = $LogConfigs.Property
            foreach ($Config in $LogConfigs)
            {
                $ConfigHashTable.add($Config, (Get-ItemProperty -Path $FWLogConfigPath -Name $Config -ErrorAction SilentlyContinue).$Config)
            }
        }
    }

    return $ConfigHashTable
}

function GetAdapterRegistryConfigPath($AdapterName, $Path)
{
    $ConfigPath = $null
    $AdapterObject = $script:MSNetAdapters.Where({$_.ifDesc -eq $AdapterName})
    $972Key = @(Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -Name "NetCfgInstanceId" -ErrorAction SilentlyContinue)
    $Adapter972Key = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterObject.InterfaceGuid})
    if ($Adapter972Key)
    {
        $ConfigPath = $Adapter972Key.PSPath + $Path
    }

    return $ConfigPath
}

function GetItemPropertyValue($AdapterName, $Path, $PropertyName)
{
    $PropertyItem = $null

    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $Path
    if ($null -ne $FWLogConfigPath)
    {
        $PropertyItem = Get-ItemProperty -Path $FWLogConfigPath -Name $PropertyName -ErrorAction SilentlyContinue
    }

    $PropertyValue = $null
    if ($null -ne $PropertyItem)
    {
        $PropertyValue = $PropertyItem.$PropertyName
    }

    return $PropertyValue
}

# SIG # Begin signature block
# MIIocQYJKoZIhvcNAQcCoIIoYjCCKF4CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCWHYOF1gGMOvcg
# 7t6MGE6DwANnu0CF3XTMI6S6JZ5sMKCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1TCCFdECAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEIDVbLIPJVnbwpmd3i1xXD/IX
# 4imuZUWFHcUqE5rvJgFtMA0GCSqGSIb3DQEBAQUABIIBgJA6gslq/D5OJzM0beGZ
# 3iurs5BWLDIpcuk5ggeiSR7s1bDawFDN+fwhBiD53aaVZiPNGIp9zA3NtzyBDIsP
# Fr43yQsbaeTEzQkms2VyQRPd/8T6F7Jk68T1oW67HRede7OUAgRcHmDmB3+0WXZo
# YhUUwgR6RQ4he/b+RRu3ZBcx2UY0jO/D6zrmhjf1Rud3abQOV6G6qKYHucoRcQt5
# bijR2apWT4SEB88jkoJ+gtFoMa/EdUtHODmzE8pNouCBIfvzyzMPBCEWsNjqFNtX
# zYCu+v7X85lfkXPvVgdMqqKVj+3BH0JAkZvrZGYnx/74PMeWWDmNQhNtMXT8vFVS
# ryx9vOSy2hD5g38FmnAO+pp8K3AI40WWAP76NRrf+/x5QfaOQ4of/nNefPTLsCwr
# /XSPapc8iq6ortqgLddvOUvmvCEF0avl0m1Ud6qA+lhHYcQJXsxYFYR8WEKHUqyA
# DjOivw5rbQEgItKmmUvuDJjhee6YA63qNslG4KFLTyFPjKGCE08wghNLBgorBgEE
# AYI3AwMBMYITOzCCEzcGCSqGSIb3DQEHAqCCEygwghMkAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwgfAGCyqGSIb3DQEJEAEEoIHgBIHdMIHaAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIGumWbxTtj/vVEPrzmhpI3/H0yk2gNaPlTXiurSncjWB
# AhUAkwaDi+U4+CREfXL8f3XSV8RGO0EYDzIwMjQwMzE4MTgyMzQ3WqBupGwwajEL
# MAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3Rp
# Z28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBT
# aWduZXIgIzSggg3pMIIG9TCCBN2gAwIBAgIQOUwl4XygbSeoZeI72R0i1DANBgkq
# hkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5j
# aGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwHhcNMjMw
# NTAzMDAwMDAwWhcNMzQwODAyMjM1OTU5WjBqMQswCQYDVQQGEwJHQjETMBEGA1UE
# CBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQD
# DCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjNDCCAiIwDQYJKoZI
# hvcNAQEBBQADggIPADCCAgoCggIBAKSTKFJLzyeHdqQpHJk4wOcO1NEc7GjLAWTk
# is13sHFlgryf/Iu7u5WY+yURjlqICWYRFFiyuiJb5vYy8V0twHqiDuDgVmTtoeWB
# IHIgZEFsx8MI+vN9Xe8hmsJ+1yzDuhGYHvzTIAhCs1+/f4hYMqsws9iMepZKGRNc
# rPznq+kcFi6wsDiVSs+FUKtnAyWhuzjpD2+pWpqRKBM1uR/zPeEkyGuxmegN77tN
# 5T2MVAOR0Pwtz1UzOHoJHAfRIuBjhqe+/dKDcxIUm5pMCUa9NLzhS1B7cuBb/Rm7
# HzxqGXtuuy1EKr48TMysigSTxleGoHM2K4GX+hubfoiH2FJ5if5udzfXu1Cf+hgl
# TxPyXnypsSBaKaujQod34PRMAkjdWKVTpqOg7RmWZRUpxe0zMCXmloOBmvZgZpBY
# B4DNQnWs+7SR0MXdAUBqtqgQ7vaNereeda/TpUsYoQyfV7BeJUeRdM11EtGcb+Re
# DZvsdSbu/tP1ki9ShejaRFEqoswAyodmQ6MbAO+itZadYq0nC/IbSsnDlEI3iCCE
# qIeuw7ojcnv4VO/4ayewhfWnQ4XYKzl021p3AtGk+vXNnD3MH65R0Hts2B0tEUJT
# cXTC5TWqLVIS2SXP8NPQkUMS1zJ9mGzjd0HI/x8kVO9urcY+VXvxXIc6ZPFgSwVP
# 77kv7AkTAgMBAAGjggGCMIIBfjAfBgNVHSMEGDAWgBQaofhhGSAPw0F3RSiO0TVf
# BhIEVTAdBgNVHQ4EFgQUAw8xyJEqk71j89FdTaQ0D9KVARgwDgYDVR0PAQH/BAQD
# AgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgwSgYDVR0g
# BEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggrBgEFBQcCARYXaHR0cHM6Ly9zZWN0
# aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1UdHwQ9MDswOaA3oDWGM2h0dHA6Ly9j
# cmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNybDB0Bggr
# BgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQuc2VjdGlnby5jb20v
# U2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggIBAEybZVj64HnP7xXD
# Mm3eM5Hrd1ji673LSjx13n6UbcMixwSV32VpYRMM9gye9YkgXsGHxwMkysel8Cbf
# +PgxZQ3g621RV6aMhFIIRhwqwt7y2opF87739i7Efu347Wi/elZI6WHlmjl3vL66
# kWSIdf9dhRY0J9Ipy//tLdr/vpMM7G2iDczD8W69IZEaIwBSrZfUYngqhHmo1z2s
# IY9wwyR5OpfxDaOjW1PYqwC6WPs1gE9fKHFsGV7Cg3KQruDG2PKZ++q0kmV8B3w1
# RB2tWBhrYvvebMQKqWzTIUZw3C+NdUwjwkHQepY7w0vdzZImdHZcN6CaJJ5OX07T
# jw/lE09ZRGVLQ2TPSPhnZ7lNv8wNsTow0KE9SK16ZeTs3+AB8LMqSjmswaT5qX01
# 0DJAoLEZKhghssh9BXEaSyc2quCYHIN158d+S4RDzUP7kJd2KhKsQMFwW5kKQPqA
# bZRhe8huuchnZyRcUI0BIN4H9wHU+C4RzZ2D5fjKJRxEPSflsIZHKgsbhHZ9e2hP
# jbf3E7TtoC3ucw/ZELqdmSx813UfjxDElOZ+JOWVSoiMJ9aFZh35rmR2kehI/shV
# Cu0pwx/eOKbAFPsyPfipg2I2yMO+AIccq/pKQhyJA9z1XHxw2V14Tu6fXiDmCWp8
# KwijSPUV/ARP380hHHrl9Y4a1LlAMIIG7DCCBNSgAwIBAgIQMA9vrN1mmHR8qUY2
# p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5l
# dyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNF
# UlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNh
# dGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/IJC9nIAhVJO5z6A+U++zWsB21hoE
# pc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnkuyWPCE5u2hj8BBZJmbyGr1XEQeYf0
# RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnTv/M5u7mkI0Ks0BXDf56iXNc48Ray
# cNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwPDU+zkQGObp0ndVXRFzs0IXuXAZSv
# f4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN9Q0/29DqhYyKVnHRLZRMyIw80xSi
# nL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gpGaQpL0bj3duRX58/Nj4OMGcrRrc1
# r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMSHv2qyghYh3ekFesZVf/QOVQtJu5F
# GjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdonigbSlofe6GsO8Ft96XZpkyAcSpcs
# dxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179g6JTZ8qx+o2hZMmIklnLqEbAyfKm
# /31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkUAx4p+hulIq6lw02C0I3aa7fb9xhA
# V3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLvbxhQiddPnTO9GrWdod6VQXqngwID
# AQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/WqorSs9UgOHYm8Cd8rIDZsswHQYD
# VR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRVMA4GA1UdDwEB/wQEAwIBhjASBgNV
# HRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAG1UgaUzXRbhtVOB
# kXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJWZApbGHLtVDb4n35nwDvQMOt0+Lk
# VvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc87QRD9XVw+K81Vh4v0h24URnbY+w
# QxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYIuM1cfwGWvnJSajtCN3wWeDmTk5Sb
# sdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8VmBvbz1kteraMrQoohkv6ob1olcGKB
# c2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncEvAQ2ted3y5wujSMYuaPCRx3wXdah
# c1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSmTqo9CfUFbLfSZFrYKiLCS53xOV5M
# 3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/W0NdZT1QgW6r8NDT/LKzH7aZlib0
# PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYybxzrF06Uw3hlIDsPQaof6aFBnf6x
# uKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/qEZ0eBsambTJdtDgJK0kHqv3sMNrx
# py/Pt/360KOE2See+wFmd7lWEOEgbsausfm2usg1XTN2jvF8IAwqd661ogKGuinu
# tFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELDCCBCgCAQEwgZEwfTELMAkGA1UEBhMC
# R0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9y
# ZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJT
# QSBUaW1lIFN0YW1waW5nIENBAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCWCGSAFlAwQC
# AgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkF
# MQ8XDTI0MDMxODE4MjM0N1owPwYJKoZIhvcNAQkEMTIEMPtlxcI9XRFZKkerw7fQ
# Ye+j/CjMfi88NdeGI0DrFeVZWqsQ3eR/g9UpTIIzEGHw1TCB7QYLKoZIhvcNAQkQ
# Agwxgd0wgdowgdcwFgQUrmKvdQoMvUfWRh91aOK8jOfKT5QwgbwEFALWW5Xig3DB
# VwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQEF
# AASCAgArLMgqx1xPwzBA8M4QM0r0D70Euv2s5bE5cMuTaIGZRN8VsL1S++3Im/G1
# /hzE27tOXjSiLBeOq50hjf812s9HNWBz9WAEvvuwABexSBSyA6gdZ7S4xccWCRo5
# xuFIYR3bse41AmH69MU2u81v9sFGy1aLLueARnv5OBa2c8glqImnhPUx2+5FvHMR
# ye0O3XRuH1njL5v13C0CyPOmg1U5kSHYX5cTUfD78QoWUjJRgoZSOGyGmwDNiwTq
# 4LXtgWj7oCQ+apFR4wWnWB9qCJF7Mtc6S90ZhxNaE+9lMAJsANpfhQdNn3h2irAK
# rD81Q7Lak1DT0Zb4WlXOgOwU4PIzV0v//TsIJstgEqQ7pDTGrIOB/MLzMRRSkGWp
# KSUzvPVILBNYLxxnpKtin5CwdksxBk8EmW9mq/Oys7fTbNNxsFpaeBQUM6VzvOdY
# Rgp242a5MVBIhc8BlWrCuDK2FDL2vNMrinfL+Unhs3O9/b/OObMvXuCby7yVUqoV
# DQlF3X0nRSA2XwKtbw//LhiVgcojre/UcbD2Xxnd7jpLhLxHK1pkykrHvd4oYmJD
# JZEP17Ecg78twQJc9DN2DcWQoyt2HntrVpS5tApthB/lyOsZ5XPQNw348LRPrmOA
# xQ7HErIBtyuF5frM3IgkK1xzVtCrJGsmGpVYNGsQnskJNd4q9A==
# SIG # End signature block
