# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>


# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'Global:FormatEnumerationLimit')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#-----------------------------------------------[Initializations]------------------------------------------------------

$script:INTEL_LOG_CONFIG_MODULES = [ordered]@{
    "General" = 0x0;
    "Control" = 0x1;
    "Link Management" = 0x2;
    "Link Topology Detection" = 0x3;
    "Link Control Technology" = 0x4;
    "I2C" = 0x5;
    "SDP" = 0x6;
    "MDIO" = 0x7;
    "Admin Queue" = 0x8;
    "HDMA" = 0x9;
    "LLDP" = 0xA;
    "DCBx" = 0xB;
    "DCB" = 0xC;
    "XLR" = 0xD;
    "NVM" = 0xE;
    "Authentication" = 0xF;
    "VPD" = 0x10;
    "IOSF" = 0x11;
    "Parser" = 0x12;
    "Switch" = 0x13;
    "Scheduler" = 0x14;
    "TX Queue Management" = 0x15;
    "ACL" = 0x16;
    "Post" = 0x17;
    "Watchdog" = 0x18;
    "Task Dispatcher" = 0x19;
    "Manageability" = 0x1A;
    "SyncE" = 0x1B;
    "Health" = 0x1C
    "Time Sync" = 0x1D;
    "PF Registration" = 0x1E;
    "Module Version" = 0x1F
}

$script:INTEL_LOG_CONFIG_LEVELS = [ordered]@{
    "Disabled" = 0x0;
    "Error"    = 0x1;
    "Warning"  = 0x2;
    "Normal"   = 0x3
    "Verbose"  = 0x4;
}

$script:INTEL_LOG_CONFIGURATIONS =[ordered]@{
    "Initialization" = 0x0;
    "NVM" = 0x1;
    "IO" = 0x2;
    "Link Management" = 0x3;
    "RX" = 0x4;
    "TX" = 0x5;
    "AQ Interface" = 0x6;
    "Manageability" = 0x7;
    "Protocols" = 0x8;
    "Infrastructure" = 0x9;
    "XLR" = 0xA;
    "Qos" = 0xB;
    "Diagnostics" = 0xC;
    "TimeSync" = 0xD;
}

$script:CVLGUID = "{BDD04ED8-F4BB-4B36-BB76-D2FFC123EC67}"
$script:GUID = "{2104EB78-B564-47BF-AB55-DBFD9338627D}"

#-----------------------------------------------------[Messages]-------------------------------------------------------

Import-LocalizedData -BindingVariable "Messages" -UICulture "en"

#-----------------------------------------------------[Functions]------------------------------------------------------
# Constants / enums
. $PSScriptRoot\Constants.ps1
# Import Get-IntelNetEthernet functions
. $PSScriptRoot\Get-IntelNetEthernet.ps1


#.ExternalHelp IntelNetDriverCmdlets.dll-Help.xml
function Get-IntelNetAdapterLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [CimInstance[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $ConfigurationNames = $Configuration
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $Global:FormatEnumerationLimit
        $Global:FormatEnumerationLimit = $INTEL_LOG_CONFIG_LEVELS.Count
    }
    Process
    {
        $Adapters = $Adapter

        $AdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters

        foreach ($a in $AdapterNames)
        {
            $DisplayConfigNames = @()
            if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) $a))
            {
                break
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"
            if (!(ValidateMethodOutput $MethodOutput "Get"))
            {
                continue
            }

            $DisplayModules = @()
            $DisplayLevels = @()

            if ($DisplayConfigNames)
            {
                GetConfigLogs $DisplayConfigNames ([ref]$DisplayModules) ([ref]$DisplayLevels) $MethodOutput
            }
            else
            {
                ValidateGetModuleParam ([ref]$DisplayModules) ([ref]$DisplayLevels) $ModuleNames $MethodOutput $a
            }

            if ($DisplayModules -and $DisplayLevels)
            {
                $LogConfigs += GatherOutput $a $DisplayModules $DisplayLevels
            }
        }
    }
    End
    {
        $Global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($warnMsg in $WarningMessagesGet)
        {
            Write-Warning $warnMsg
        }

        foreach ($msg in $ErrorMessagesGet)
        {
            Write-Error $msg
        }
    }
}


#.ExternalHelp IntelNetDriverCmdlets.dll-Help.xml
function Set-IntelNetAdapterLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [CimInstance[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$true,Position=2)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String]
    $Level = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $LevelName = $Level
        $ConfigurationNames = $Configuration
        $script:WarningMessagesSet = @()
        $script:ErrorMessagesSet = @()
        $LogConfigs = @()
    }
    Process
    {
        do
        {
            $Adapters = $Adapter

            $AdapterNames = ValidateSetAdapterNameParams $AdapterName $Adapters

            if ([string]::IsNullOrEmpty($ConfigurationNames) -and ([string]::IsNullOrEmpty($ModuleNames)))
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigOrModule
                break
            }

            foreach($a in $AdapterNames)
            {
                $DisplayModules = @()
                $DisplayLevels = @()
                $InputModulesId = [uint32[]]::new(64)
                $InputLevelId = [uint32[]]::new(64)

                $DisplayConfigNames = @()
                if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) $a))
                {
                    break
                }

                if ($DisplayConfigNames)
                {
                    $InputConfigurationIDs = $INTEL_LOG_CONFIGURATIONS[$DisplayConfigNames]
                }
                else
                {
                    ValidateSetModuleParam ([ref]$InputModulesId) ([ref]$DisplayModules) $ModuleNames $a
                    $ModuleCount = $DisplayModules.Count
                }

                if ($null -ne $ModuleCount -and $ModuleCount -ne 0)
                {
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $ModuleCount $a))
                    {
                        break
                    }

                    $params = @{Count    = [uint32]$ModuleCount;
                                ModuleID = $InputModulesId;
                                Loglevel = $InputLevelId}

                    $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogLevels" $params

                    if (ValidateMethodOutput $MethodOutput "Set")
                    {
                        $LogConfigs += Get-IntelNetAdapterLogConfig -Name $a -Module $DisplayModules
                    }
                }
                else
                {
                    $count = 1
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $count $a))
                    {
                        break
                    }

                    $bContinue = $true
                    foreach ($ConfigID in $InputConfigurationIDs)
                    {
                        $params = @{ConfigurationID = [uint32]$ConfigID;
                                    Level = $InputLevelId[0]}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogConfiguration" $params

                        if (!(ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $bContinue = $false
                            break
                        }
                    }

                    if (!$bContinue)
                    {
                        break
                    }
                    $LogConfigs += Get-IntelNetAdapterLogConfig -Name $a -Configuration $DisplayConfigNames
                }
            }
        } while ($false)
    }
    End
    {
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($warnMsg in $WarningMessagesSet)
        {
            Write-Warning $warnMsg
        }

        foreach ($msg in $ErrorMessagesSet)
        {
            Write-Error $msg
        }
    }
}

#.ExternalHelp IntelNetDriverCmdlets.dll-Help.xml
function Start-IntelNetAdapterLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [CimInstance]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [ValidateNotNullOrEmpty()]
    [string]
    $Path = '',
    [parameter(Mandatory=$false)]
    [switch]
    $Force,
    [parameter(Mandatory=$false)]
    [switch]
    $Append
    )
    Begin
    {
        $script:ErrorMessagesStart = @()
        $script:WarningMessagesStart = @()
        $script:ErrorMessagesSet = @()
    }
    Process
    {
        do
        {
            if ($Name -match '\*')
            {
                $ErrorMessagesStart += $Messages.AdapterNotFound + $Name
                break
            }

            $AdapterName = ValidateSetAdapterNameParams $Name $Adapter
            $script:ErrorMessagesStart += $script:ErrorMessagesSet

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (([string]::IsNullOrEmpty($Path)))
            {
                $UseDefaultPath = $true
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $AdapterName "WmiGetFwLogLevels"

            if (ValidateMethodOutput $MethodOutput "Start")
            {
                $DeviceFamilyName = (Get-CimInstance win32_networkadapter | Where-Object {$_.Name -Like ($AdapterName)}).ServiceName
                if ($DeviceFamilyName -eq "icea")
                {
                    $CurrentGUID = $script:CVLGUID
                }
                else
                {
                    $CurrentGUID = $script:GUID
                }

                if (LogmanProcessRunning $AdapterName $DeviceFamilyName $CurrentGUID)
                {
                    break
                }

                $LogmanPath = ''
                if (!(ValidatePathParam ([ref]$LogmanPath) $UseDefaultPath $DeviceFamilyName))
                {
                    break
                }

                LogmanStart $LogmanPath $DeviceFamilyName $CurrentGUID
                if (ValidateLogmanStart)
                {
                    $LogmanPath = Resolve-Path -Path $LogmanPath -ErrorAction SilentlyContinue
                    Write-Output $LogmanPath
                }
                break
            }
        } while ($false)
    }
    End
    {
        foreach ($warnMsg in $WarningMessagesStart)
        {
            Write-Warning $warnMsg
        }

        foreach ($msg in $ErrorMessagesStart)
        {
            Write-Error $msg
        }
    }
}

#.ExternalHelp IntelNetDriverCmdlets.dll-Help.xml
function Stop-IntelNetAdapterLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [CimInstance]
    $Adapter = $null
    )
    Begin
    {
        $script:ErrorMessagesStop = @()
        $script:WarningMessagesStop = @()
        $script:ErrorMessagesGet = @()
    }
    Process
    {
        $AdapterNames = ValidateGetAdapterNameParams $Name $Adapter
        $script:ErrorMessagesStop += $script:ErrorMessagesGet

        foreach ($a in $AdapterNames)
        {
            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"

            if (ValidateMethodOutput $MethodOutput "Stop")
            {
                $DeviceFamilyName = (Get-CimInstance win32_networkadapter | Where-Object {$_.Name -Like ($a)}).ServiceName

                LogmanStop $DeviceFamilyName
                if (!(ValidateLogmanStop $a))
                {
                    break
                }
                break
            }
        }
    }
    End
    {
        foreach ($warnMsg in $WarningMessagesStop)
        {
            Write-Warning $warnMsg
        }

        foreach ($msg in $ErrorMessagesStop)
        {
            Write-Error $msg
        }
    }
}

#.ExternalHelp IntelNetDriverCmdlets.dll-Help.xml
function Disable-IntelNetAdapterLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [CimInstance[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesDisable = @()
        $script:WarningMessagesDisable = @()
        $script:ErrorMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $Global:FormatEnumerationLimit
        $Global:FormatEnumerationLimit = $INTEL_LOG_CONFIG_LEVELS.Count
    }
    Process
    {
        $Adapters = $Adapter

        $AdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters
        $script:ErrorMessagesDisable += $script:ErrorMessagesGet

        foreach ($a in $AdapterNames)
        {
            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiDisableFwLogLevels"

            if (!(ValidateMethodOutput $MethodOutput "Disable"))
            {
                continue
            }

            $LogConfigs += Get-IntelNetAdapterLogConfig -Name $a
        }
    }
    End
    {
        $Global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($warnMsg in $WarningMessagesDisable)
        {
            Write-Warning $warnMsg
        }

        foreach ($msg in $ErrorMessagesDisable)
        {
            Write-Error $msg
        }
    }
}

#.ExternalHelp IntelNetDriverCmdlets.dll-Help.xml
function Reset-IntelNetAdapterLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [CimInstance[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesReset = @()
        $script:WarningMessagesReset = @()
        $script:ErrorMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $Global:FormatEnumerationLimit
        $Global:FormatEnumerationLimit = $INTEL_LOG_CONFIG_LEVELS.Count
    }
    Process
    {
        $Adapters = $Adapter

        $AdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters
        $script:ErrorMessagesReset += $script:ErrorMessagesGet

        foreach ($a in $AdapterNames)
        {
            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiResetFwLogLevels"

            if (!(ValidateMethodOutput $MethodOutput "Reset"))
            {
                continue
            }

            $LogConfigs += Get-IntelNetAdapterLogConfig -Name $a
        }
    }
    End
    {
        $Global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($warnMsg in $WarningMessagesReset)
        {
            Write-Warning $warnMsg
        }

        foreach ($msg in $ErrorMessagesReset)
        {
            Write-Error $msg
        }
    }
}

Function GatherOutput($AdapterName, $DisplayModules, $DisplayLevels)
{
    $LogConfigs = @()
    $ValidLevels = $INTEL_LOG_CONFIG_LEVELS.Keys
    $LogConfigCount = $DisplayModules.Count
    for ($i = 0; $i -lt $LogConfigCount; $i++)
    {
        $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                                           Name        = $AdapterName;
                                           Module      = $DisplayModules[$i];
                                           Level       = $DisplayLevels[$i];
                                           ValidLevels = $ValidLevels}
    }

    return $LogConfigs
}

Function GetModuleName($ModuleId)
{
    return $INTEL_LOG_CONFIG_MODULES.GetEnumerator().Where({$_.Value -eq $ModuleId}).Name
}

Function GetLevelName($LevelId)
{
    return $INTEL_LOG_CONFIG_LEVELS.GetEnumerator().Where({$_.Value -eq $LevelId}).Name
}

Function ValidateGetAdapterNameParams($AdapterName, $Adapters)
{
    $AdapterNames = @()

    if ((-Not $Adapters) -and $AdapterName)
    {
        foreach($n in $AdapterName)
        {
            $TmpAdapterName = (Get-CimInstance win32_networkadapter | Where-Object {$_.Name -Like ($n)}).Name
            if (-Not $TmpAdapterName)
            {
                $script:ErrorMessagesGet += $Messages.AdapterNotFound + $n
                continue
            }
            $AdapterNames += $TmpAdapterName
        }
    }
    elseif ((-Not $AdapterName) -and $Adapters)
    {
        foreach ($a in $Adapters)
        {
            if ($a.CreationClassName -eq "MSFT_NetAdapter")
            {
                $AdapterNames += $a.ifDesc
            }
            elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                    $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter")
            {
                $AdapterNames += $a.Name
            }
            else
            {
                $script:ErrorMessagesGet += $Messages.AdapterNotFound + $a
            }
        }
    }
    elseif (-Not ($AdapterName -and $Adapters))
    {
        $AdapterNames = (Get-CimInstance win32_networkadapter | Where-Object {$_.Manufacturer -eq "Intel"}).Name
    }
    elseif ($AdapterName -and $Adapters)
    {
        $script:ErrorMessagesGet += $Messages.InvalidParamsAdapterAndName
    }

    return $AdapterNames
}

function InvokeCimMethod($ClassName, $InstanceName = "", $MethodName, $params = @{}, $Namespace = "root\wmi")
{
    $query = "Select * from $ClassName"
    if ($InstanceName)
    {
        $query+= " where instancename like '$InstanceName'"
    }

    Invoke-CimMethod -Query $query -MethodName $MethodName -Arguments $params -Namespace $Namespace -ErrorAction SilentlyContinue
}

Function ValidateMethodOutput($MethodOutput, $Verb)
{
    do
    {
        $Result = $true
        if ($null -eq $MethodOutput)
        {
            switch ($Verb)
            {
                Get {$script:WarningMessagesGet += $Messages.NoCmdletSupport -f $a; break}
                Set {$script:WarningMessagesSet += $Messages.NoCmdletSupport -f $a; break}
                Start {$script:WarningMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
                Stop {$script:WarningMessagesStop += $Messages.NoCmdletSupport -f $a; break}
                Disable {$script:WarningMessagesDisable += $Messages.NoCmdletSupport -f $a; break}
                Reset {$script:WarningMessagesReset += $Messages.NoCmdletSupport -f $a; break}
            }

            $Result = $false
            break
        }

        if ($MethodOutput.OutStatus -ne 0)
        {
            switch ($Verb)
            {
                Get {$script:ErrorMessagesGet += $Messages.OperationFailed -f $a, $Verb; break}
                Set {$script:ErrorMessagesSet += $Messages.OperationFailed -f $a, $Verb; break}
                Start {$script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, $Verb; break}
                Stop {$script:ErrorMessagesStart += $Messages.OperationFailed -f $a, $Verb; break}
                Disable {$script:ErrorMessagesDisable += $Messages.OperationFailed -f $a, $Verb; break}
                Reset {$script:ErrorMessagesReset += $Messages.OperationFailed -f $a, $Verb; break}
            }

            $Result = $false
        }
    } while ($false)

    Return $Result
}

Function ValidateGetModuleParam([ref]$DisplayModules, [ref]$DisplayLevels, $ModuleNames, $MethodOutput, $AdapterName)
{
    if ($ModuleNames)
    {
        foreach ($ModuleName in $ModuleNames)
        {
            $ModuleIds = $null
            $ModuleIDNames = ($INTEL_LOG_CONFIG_MODULES.GetEnumerator() | Where-Object {$_.Name -like $ModuleName}).Name

            if ($null -ne $ModuleIDNames)
            {
                $ModuleIds = $INTEL_LOG_CONFIG_MODULES[$ModuleIDNames]

                foreach ($ModuleId in $ModuleIds)
                {
                    for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
                    {
                        if ($ModuleId -eq $MethodOutput.ModuleID[$i])
                        {
                            $DisplayModules.Value += GetModuleName($ModuleId)
                            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
                            break
                        }
                    }
                }
            }
            else
            {
                #if user passes in an array of modules and one (or more) is spelled incorrectly,
                #show error msg for that 1 and continue on getting the modules for the rest
                $script:ErrorMessagesGet += $Messages.InvalidModule -f $AdapterName, $ModuleName
            }
        }
    }
    else
    {
        GetAllModulesLevel $MethodOutput $DisplayModules $DisplayLevels
    }
}

Function GetAllModulesLevel($MethodOutput, [ref]$DisplayModules, [ref]$DisplayLevels)
{
    for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
    {
        $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$i])
        $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
        # driver returns an array of 64 but only 31 modules populated
        if ($i -eq 31)
        {
            break
        }
    }
}

Function ValidateSetAdapterNameParams($AdapterName, $Adapters)
{
    $AdapterNames = @()

    do
    {
        if ($AdapterName -and $Adapters)
        {
            $script:ErrorMessagesSet += $Messages.InvalidParamsAdapterAndName
            break
        }
        elseif ($AdapterName)
        {
            foreach ($n in $AdapterName)
            {
                $TmpAdapters = (Get-CimInstance win32_networkadapter | Where-Object {$_.Name -Like ($n)}).Name
                if (-Not $TmpAdapters)
                {
                    $script:ErrorMessagesSet += $Messages.AdapterNotFound + $n
                    continue
                }
                $AdapterNames += $TmpAdapters
            }
        }
        elseif ($Adapters)
        {
            foreach ($a in $Adapters)
            {
                if ($a.CreationClassName -eq "MSFT_NetAdapter")
                {
                    $AdapterNames += $a.ifDesc
                }
                elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                        $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $script:ErrorMessagesSet += $Messages.AdapterNotFound + $a
                }
            }
        }
        else
        {
            $script:ErrorMessagesSet += $Messages.InvalidParamsAdapterOrName
            break
        }
    } while ($false)

    Return $AdapterNames
}

Function ValidateLevelParam([ref]$InputLevelId, [ref]$DisplayLevels, $LevelName, $Count, $AdapterName)
{
    do
    {
        $NewLevelId = $null
        $Result = $true
        $NewLevelId = ($INTEL_LOG_CONFIG_LEVELS.GetEnumerator() | Where-Object {$_.Name -like $LevelName}).Value
        if ($NewLevelId.count -gt 1)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevelWildcard
            $Result = $false
            break
        }

        if ($null -eq $NewLevelId)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevel -f $AdapterName, $LevelName, $ModuleNames[0]
            $Result = $false
            break
        }

        for ($i = 0; $i -lt $Count; $i++)
        {
            $InputLevelId.Value[$i] = $NewLevelId
            $DisplayLevels.Value += GetLevelName($NewLevelId)
        }

    } while ($false)

    Return $Result
}

Function ValidateSetModuleParam([ref]$InputModulesId, [ref]$DisplayModules, $ModuleNames, $AdapterName)
{
    $count = 0
    foreach ($ModuleName in $ModuleNames)
    {
        $ModuleIds = $null
        $ModuleIDNames = ($INTEL_LOG_CONFIG_MODULES.GetEnumerator() | Where-Object {$_.Name -like $ModuleName}).Name

        if ($null -ne $ModuleIDNames)
        {
            $ModuleIds = $INTEL_LOG_CONFIG_MODULES[$ModuleIDNames]

            foreach ($ModuleId in $ModuleIds)
            {
                $InputModulesId.Value[$count] = $ModuleId
                $DisplayModules.Value += GetModuleName($ModuleId)
                $count++
            }
        }
        else
        {
            #if user passes in an array of modules and one (or more) is spelled incorrectly,
            #show error msg for that 1 and continue on getting the modules for the rest
            $script:ErrorMessagesSet += $Messages.InvalidModule -f $AdapterName, $ModuleName
        }
    }
}

Function LogmanProcessRunning($AdapterName, $DeviceFamilyName, $GUID)
{
    $Result = $false
    $LogName = GetLogName($DeviceFamilyName)

    $LogmanQueryString = logman query $LogName -ets
    if ($LogmanQueryString -match ($GUID))
    {
        $script:ErrorMessagesStart += $Messages.LogmanRunning -f $AdapterName
        $Result = $true
    }

    return $Result
}

Function LogmanStart($LogmanPath, $DeviceFamilyName, $GUID)
{
    $LogName = GetLogName($DeviceFamilyName)

    if ($Append)
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID 0xFFFF 4 -a
    }
    else
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID 0xFFFF 4
    }
}

Function LogmanStop($DeviceFamilyName)
{
    $LogName = GetLogName($DeviceFamilyName)

    $null = logman stop $LogName -ets
}

Function ValidatePathParam([ref]$LogmanPath, $UseDefaultPath, $DeviceFamilyName)
{
    $Result = $true

    if ($UseDefaultPath)
    {
        if ($DeviceFamilyName -eq "icea")
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\E810_Log"
        }
        else
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\" + $DeviceFamilyName + "_Log"
        }

        if (-not (Test-Path -Path $DefaultPath))
        {
            New-Item -Path $ENV:LOCALAPPDATA -Name "\Intel\Wired Networking" -ItemType "directory" -ErrorAction SilentlyContinue
        }
        $LogmanPath.Value = $DefaultPath
    }
    else
    {
        $LogmanPath.Value = $Path
    }

    $isPathFile = Test-Path -Path $LogmanPath.Value -PathType Leaf

    if (($isPathFile) -and (-not $Append) -and (-not $Force))
    {
        $script:ErrorMessagesStart += $Messages.LogmanFileExists -f $AdapterName
        return $false
    }
    if ((Test-Path -Path $LogmanPath.Value) -and (-not $isPathFile))
    {
        $script:ErrorMessagesStart += $Messages.FolderFileNameExits
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStart()
{
    $Result = $true
    # -2147024629, -2147024773, -2147024893, -2147024891 error codes mean directory name is invalid
    # or filename, directory name, or volume label syntax is incorrect or permission is denied.
    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2147024629 -or $LASTEXITCODE -eq -2147024773 -or $LASTEXITCODE -eq -2147024893 -or $LASTEXITCODE -eq -2147024891)
    {
        $script:ErrorMessagesStart += $Messages.PathIncorrect
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.StartCallFailed
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStop($AdapterName)
{
    $Result = $true

    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2144337918)
    {
        $script:ErrorMessagesStop += $Messages.NoLogsStarted -f $AdapterName
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStop += $Messages.StopCallFailed
        $Result = $false
    }
    Return $Result
}

Function GetLogName($DeviceFamilyName)
{
    if ($DeviceFamilyName -eq "icea")
    {
        $LogName = "E810_Log"
    }
    else
    {
        $LogName = $DeviceFamilyName + "_Log"
    }

    return $LogName
}

Function ValidateConfigurationParam($ConfigurationNames, $ModuleNames, [ref]$DisplayConfigNames, $a)
{
    $Result = $true

    do
    {
        if ($ConfigurationNames)
        {
            if ($ModuleNames)
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigAndModule
                $script:ErrorMessagesGet += $Messages.InvalidParamsConfigAndModule
                $Result = $false
                break
            }

            foreach ($ConfigName in $ConfigurationNames)
            {
                $TmpDisplayConfigNames = ($INTEL_LOG_CONFIGURATIONS.GetEnumerator() | Where-Object {$_.Name -like $ConfigName}).Name
                if ($null -ne $TmpDisplayConfigNames)
                {
                    $DisplayConfigNames.Value += $TmpDisplayConfigNames
                }
                else
                {
                    #if user passes in an array of configs and one (or more) is spelled incorrectly,
                    #show error msg for that 1 and continue on getting the config for the rest
                    $script:ErrorMessagesSet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName
                    $script:ErrorMessagesGet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName
                    $Result = $false
                }
            }
        }
    } while ($false)

    return $Result
}

Function GetConfigLogs($DisplayConfigNames, [ref]$DisplayModules, [ref]$DisplayLevels, $MethodOutput)
{
    $ConfigurationID = $INTEL_LOG_CONFIGURATIONS[$DisplayConfigNames]
    foreach ($ConfigID in $ConfigurationID)
    {
        $params = @{ConfigurationID = [uint32]$ConfigID}
        $GetConfigMethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevelsForConfiguration" $params

        for ($i = 0; $i -lt $GetConfigMethodOutput.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$GetConfigMethodOutput.ModuleID[$i]])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$GetConfigMethodOutput.ModuleID[$i]])
        }
    }
}

# SIG # Begin signature block
# MIIo6AYJKoZIhvcNAQcCoIIo2TCCKNUCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDx5r+RY62O4DS5
# B0Nr8bdZOb8nNH2gEIkEYygLIxzUSKCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFjkwghY1AgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCA0WEn/zdFx
# 9cNOgRsjxpygfPFYTvlj/r2l6bqNqts1wDANBgkqhkiG9w0BAQEFAASCAYDb03R5
# dGrk8i7YSo2jGzfNhC3SYtaiNHpIC5D/RPa7Z6eTJIlLaY+83h9VJUjjXDps4kn9
# /jLOLg1Qyz3R5+ti8SYMI0qr3hizY7Qw7nwwjvM0VChm0LXvMxLmnn1eXcwXpeRG
# O+ZD37SHjjFPikLOnnwco/mmUbER/j1Z5O8RSkhGkMd6o2STDi8WTg0SpAB+dEpW
# NjHeGJyplJ0qzKAX3Iszz7YF6ep4KjE1QlmQHgsNtlAkqQyvDevctV4jXx4xSj4s
# UP2Zd2FQTN5v5yUtX5lctJZXUzMwrYQKyDlV+AHTrWE7OJXR2pnHqxrRu1AeHpU9
# S+jiFMkgPlkXJtCoX+44xOUxFM1Ka79o5kUcN/rsQ+EndR6o4cRk+KT2hHtkdb/R
# QO2v3Uhh7/4WwV3IGWFtm0Rx8VAS84aOhO3CeKkajrtB24b9AR08O8cYom5Bsb0O
# QkiiRVR+HtL/PeVPZUyUKeYEURnZZa0BMxKHJ0pLaTbbf+lwyfRZN3xe1SShghOM
# MIITiAYKKwYBBAGCNwMDATGCE3gwghN0BgkqhkiG9w0BBwKgghNlMIITYQIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIIBGQYLKoZIhvcNAQkQAQSgggEIBIIBBDCCAQACAQEG
# CisGAQQBsjECAQEwMTANBglghkgBZQMEAgEFAAQgwtaaAr1ZaAZw15L7mD0Rn9NN
# QPgfQat48wftB6v53lUCFCLPOQNkQPHFJmkb5IjtTdr4PgWBGA8yMDIyMDUxNzA3
# MjgyOFoCCE1RMY2Kokb/oIGKpIGHMIGEMQswCQYDVQQGEwJHQjEbMBkGA1UECBMS
# R3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9T
# ZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBp
# bmcgU2lnbmVyICMyoIIN+zCCBwcwggTvoAMCAQICEQCMd6AAj/TRsMY9nzpIg41r
# MA0GCSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVy
# IE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28g
# TGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAe
# Fw0yMDEwMjMwMDAwMDBaFw0zMjAxMjIyMzU5NTlaMIGEMQswCQYDVQQGEwJHQjEb
# MBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgw
# FgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRp
# bWUgU3RhbXBpbmcgU2lnbmVyICMyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEAkYdLLIvB8R6gntMHxgHKUrC+eXldCWYGLS81fbvA+yfaQmpZGyVM6u9A
# 1pp+MshqgX20XD5WEIE1OiI2jPv4ICmHrHTQG2K8P2SHAl/vxYDvBhzcXk6Th7ia
# 3kwHToXMcMUNe+zD2eOX6csZ21ZFbO5LIGzJPmz98JvxKPiRmar8WsGagiA6t+/n
# 1rglScI5G4eBOcvDtzrNn1AEHxqZpIACTR0FqFXTbVKAg+ZuSKVfwYlYYIrv8azN
# h2MYjnTLhIdBaWOBvPYfqnzXwUHOrat2iyCA1C2VB43H9QsXHprl1plpUcdOpp0p
# b+d5kw0yY1OuzMYpiiDBYMbyAizE+cgi3/kngqGDUcK8yYIaIYSyl7zUr0QcloIi
# lSqFVK7x/T5JdHT8jq4/pXL0w1oBqlCli3aVG2br79rflC7ZGutMJ31MBff4I13E
# V8gmBXr8gSNfVAk4KmLVqsrf7c9Tqx/2RJzVmVnFVmRb945SD2b8mD9EBhNkbunh
# FWBQpbHsz7joyQu+xYT33Qqd2rwpbD1W7b94Z7ZbyF4UHLmvhC13ovc5lTdvTn8c
# xjwE1jHFfu896FF+ca0kdBss3Pl8qu/CdkloYtWL9QPfvn2ODzZ1RluTdsSD7oK+
# LK43EvG8VsPkrUPDt2aWXpQy+qD2q4lQ+s6g8wiBGtFEp8z3uDECAwEAAaOCAXgw
# ggF0MB8GA1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQWBBRp
# dTd7u501Qk6/V9Oa258B0a7e0DAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIw
# ADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBABgNVHSAEOTA3MDUGDCsGAQQBsjEB
# AgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzBEBgNV
# HR8EPTA7MDmgN6A1hjNodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FU
# aW1lU3RhbXBpbmdDQS5jcmwwdAYIKwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNo
# dHRwOi8vY3J0LnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5j
# cnQwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3
# DQEBDAUAA4ICAQBKA3iQQjPsexqDCTYzmFW7nUAGMGtFavGUDhlQ/1slXjvhOcRb
# uumVkDc3vd/7ZOzlgreVzFdVcEtO9KiH3SKFple7uCEn1KAqMZSKByGeir2nGvUC
# FctEUJmM7D66A3emggKQwi6Tqb4hNHVjueAtD88BN8uNovq4WpquoXqeE5MZVY8J
# kC7f6ogXFutp1uElvUUIl4DXVCAoT8p7s7Ol0gCwYDRlxOPFw6XkuoWqemnbdaQ+
# eWiaNotDrjbUYXI8DoViDaBecNtkLwHHwaHHJJSjsjxusl6i0Pqo0bglHBbmwNV/
# aBrEZSk1Ki2IvOqudNaC58CIuOFPePBcysBAXMKf1TIcLNo8rDb3BlKao0AwF7Ap
# FpnJqreISffoCyUztT9tr59fClbfErHD7s6Rd+ggE+lcJMfqRAtK5hOEHE3rDbW4
# hqAwp4uhn7QszMAWI8mR5UIDS4DO5E3mKgE+wF6FoCShF0DV29vnmBCk8eoZG4BU
# +keJ6JiBqXXADt/QaJR5oaCejra3QmbL2dlrL03Y3j4yHiDk7JxNQo2dxzOZgjdE
# 1CYpJkCOeC+57vov8fGP/lC4eN0Ult4cDnCwKoVqsWxo6SrkECtuIf3TfJ035CoG
# 1sPx12jjTwd5gQgT/rJkXumxPObQeCOyCSziJmK/O6mXUczHRDKBsq/P3zCCBuww
# ggTUoAMCAQICEDAPb6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkg
# Q2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVV
# U0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAw
# MDAwMFoXDTM4MDExODIzNTk1OVowfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdy
# ZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IENBMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGY
# PyCQvZyAIVSTuc+gPlPvs1rAdtYaBKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5
# LsljwhObtoY/AQWSZm8hq9VxEHmH9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ
# 07/zObu5pCNCrNAVw3+eolzXOPEWsnDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8
# Dw1Ps5EBjm6dJ3VV0Rc7NCF7lwGUr3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64
# TfUNP9vQ6oWMilZx0S2UTMiMPNMUopy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4
# KRmkKS9G493bkV+fPzY+DjBnK0a3Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFz
# Eh79qsoIWId3pBXrGVX/0DlULSbuRRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3a
# J4oG0paH3uhrDvBbfel2aZMgHEqXLHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe
# /YOiU2fKsfqNoWTJiJJZy6hGwMnypv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5
# FAMeKfobpSKupcNNgtCN2mu32/cYQFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy
# 728YUInXT50zvRq1naHelUF6p4MCAwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5
# v1qqK0rPVIDh2JvAnfKyA2bLMB0GA1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIE
# VTAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAK
# BggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/
# aHR0cDovL2NybC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRp
# b25BdXRob3JpdHkuY3JsMHYGCCsGAQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0
# cDovL2NydC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0
# MCUGCCsGAQUFBzABhhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3
# DQEBDAUAA4ICAQBtVIGlM10W4bVTgZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5
# SVmQKWxhy7VQ2+J9+Z8A70DDrdPi5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZY
# XPO0EQ/V1cPivNVYeL9IduFEZ22PsEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZW
# CLjNXH8Blr5yUmo7Qjd8Fng5k5OUm7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPF
# Zgb289ZLXq2jK0KKIZL+qG9aJXBigXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53
# BLwENrXnd8ucLo0jGLmjwkcd8F3WoXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGU
# pk6qPQn1BWy30mRa2Coiwkud8TleTN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9x
# v1tDXWU9UIFuq/DQ0/yysx+2mZYm9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETm
# Mm8c6xdOlMN4ZSA7D0GqH+mhQZ3+sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/
# 6hGdHgbGpm0yXbQ4CStJB6r97DDa8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5
# trrINV0zdo7xfCAMKneutaIChrop7rRaALGMq+P5CslUXdS5anSevUiumDGCBC0w
# ggQpAgEBMIGSMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNo
# ZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQQIRAIx3oACP
# 9NGwxj2fOkiDjWswDQYJYIZIAWUDBAICBQCgggFrMBoGCSqGSIb3DQEJAzENBgsq
# hkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjIwNTE3MDcyODI4WjA/BgkqhkiG
# 9w0BCQQxMgQwLhLsNMPl8OZd1ncwhMAnbheoLjw5zQf+FiQewcodflW7+s0ct1bD
# 15V5iyuUK9iCMIHtBgsqhkiG9w0BCRACDDGB3TCB2jCB1zAWBBSVETcQHYgvMb1R
# P5Sa2kxorYwI9TCBvAQUAtZbleKDcMFXAJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgx
# CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJz
# ZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQD
# EyVVU0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5AhAwD2+s3WaY
# dHypRjaneC25MA0GCSqGSIb3DQEBAQUABIICAFKwIGfqOWrR4N5iBvuyOIeb+R4t
# 1nAqZ8knX72lbt9nPrgt4F7Pa2HBxjS0Is0DquGFvyJrwEav0u7+j/6dQhgz8oc3
# nge185ZoYvPNl2bCuwcg7r2OzGwHucU0Jtt0LHp5D7IN7dhoaTsTKrONwcfprh0i
# DXbPp5/OnR551BxtWUJ0/Mdhh/qAQ/WW6adv2LXQmd+5FwcmAn/3oxZVY97+dDBc
# G7Yr0YjbnzDcAaROL772S86LFzXAfdPbDC6fjjjAkJ2deTKXNOIfefiv+tjjBAli
# NinFQOq3wA/Y+165gQw6QI2U0frvu24kJscVy331s/nKk5LHH4O5WbX4YiZNMNUa
# Gs6KeTlTlVdDjus+36daHNOWWt7zaKG5gjocT+Ka3IB9NgCBT69T6Ffka15NmG2K
# etiyH/17JsUP/zshC/Wv8gt5RBzp6ZFMn1NSsnkfJ4iI4VSUFalHyP0Dl/OuiMNf
# OMaZIzN5LclK8zmO0ud4fMOnCR3BTs9mO0BADihMpqlEIpO8uvK6M/UK/HngL9tq
# n1aZtKW/h+VTy91CdPNNULYOd4szsQJxDJ8HTJzgVlYv+NgtulPCHK0lazzaC9vb
# RHn2qbLeenkJBBsXbQemz5LBFKHfXqNqzrWRrbyCbRou+41bpKO5OQt6+GtD0G3b
# LHxKNGfpxlBLFHVi
# SIG # End signature block
