/*

  Copyright(c) 1999 - 2020 Intel Corporation. All rights reserved.
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in
      the documentation and/or other materials provided with the
      distribution.
    * Neither the name of Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived
      from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*
 *
 * Module Name:
 *   hwbus_t.h
 *
 * Abstract:
 *   This file includes all the types required for the NAL hardware
 *   bus interface.
 *
 */

#ifndef _NAL_HWBUS_TYPES_H_
#define _NAL_HWBUS_TYPES_H_

#include "pci_t.h"
#include "cardbus_t.h"

#define NAL_MAX_BARS 6
#define NAL_DEVICE_SIGNATURE 0xA55A5AA5

#if defined(NAL_EFI) || defined(NAL_EFI2)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;
    UINT32 Signature;
    VOID* Handle;
} NAL_OS_SLOT_ID;

#elif defined(NAL_LINUX) || defined(NAL_ESXI)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;
    UINT32 Signature;
} NAL_OS_SLOT_ID;
#elif defined(NAL_ESXIN)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID         Pci;
    UINT32              Signature;
    NAL_BASE_DRIVER_ID  BaseDriverId;
    BOOLEAN             IsNativeDevice;
} NAL_OS_SLOT_ID;
#elif defined(NAL_SOLARIS)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;
    UINT32 Signature;
    INT32 DevInfo;
} NAL_OS_SLOT_ID;

#elif defined(NAL_DOS)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;
    UINT32 Signature;
} NAL_OS_SLOT_ID;

#elif defined(NAL_WINNT_WDM)

/*********************************************************************************************************//**
**
** \details         NAL_OS_SLOT_ID 32-bit value  data type is a type that generically identifies a device. Its
**                  contents vary per operating system and are somewhat opaque for the application. While its
**                  members differ per operating system, the common data is the signature value. This UINT32
**                  value allows the caller to determine if the NAL_DEVICE_LOCATION actually contains a
**                  NAL_OS_SLOT_ID rather than a PCI_SLOT_ID.  If the NAL_DEVICE_LOCATION.OsDeviceLocation.
**                  Signature == NAL_DEVICE_SIGNATURE, then the device is a NAL_OS_SLOT_ID.
**
*************************************************************************************************************/

typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;                   /**< PCI_SLOT_ID Structure */
    CHAR        RegistryKey[256];      /**< Registry key name for device */
    UINT32      Signature;             /**< Signature */
} NAL_OS_SLOT_ID;

#elif defined(NAL_WINNT4)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;
    UINT32 Signature;
} NAL_OS_SLOT_ID;

#elif defined(NAL_FREEBSD)
typedef struct _NAL_OS_SLOT_ID
{
    PCI_SLOT_ID Pci;
    UINT32 Signature;
} NAL_OS_SLOT_ID;
#else

#error "Undefined NAL Operating System in hwbus_t.h."
#endif
/*********************************************************************************************************//**
**
** \ingroup         NAL_enums
**
** \details         VF_SLOT_ID union.
**
*************************************************************************************************************/

typedef union _VF_SLOT_ID
{
    struct
    {
        UINT32      SlotId;                           /**< Slot ID */
        UINT32      ReservedForSignature;             /**< Memory reserved for signature */
        UINT32      ParentSlotId;                     /**< Present slot ID */
        UINT16      SupportedModule;                  /**< which of our modulex files supports this device */
        BOOLEAN     IsVirtualDevice;                  /**< Determines if devise is virtual */
        UINT8       Pad;                              /**< PAD */
    } slot_id;
    struct
    {
        struct
        {
#if defined (NAL_BIG_ENDIAN)
            UINT32 DwordLo : 2;
            UINT32 _Byte : 2;
            UINT32 Reserved : 4;
            UINT32 DwordHi : 8;
            UINT32 Device : 7;
            UINT32 Function : 1;
            UINT32 Bus : 8;
#else
            UINT32  Bus                   : 8;       /**< Bus */
            UINT32  Device                : 7;       /**< Device */
            UINT32  Function              : 1;       /**< Function */
            UINT32  Dword                 : 10;      /**< Dword */
            UINT32  _Byte                 : 2;       /**< _Byte */
            UINT32  Reserved              : 4;       /**< Reserved */
#endif
        } vf_slot_id;
        UINT32              ReservedForSignature;    /**< Reserved */
        PCI_EXPRESS_SLOT_ID ParentDevice;            /**< Parent device */
    };
} VF_SLOT_ID;
/*********************************************************************************************************//**
**
** \ingroup         NAL_enums
**
** \details         NAL_DEVICE_LOCATION union, which is returned from NalScanForDevices
**                  (\ref NalScanForDevices) function, and stores informations about device location.
**                  It is used to uniquely identify an adapter on a bus. This type is implemented as a union
**                  of many bus location identifiers. Note, that the NAL_DEVICE_LOCATION type may be any of
**                  below members. This is passed to NalInitializeAdapter in order to identify the device to
**                  initialize.
**
*************************************************************************************************************/

typedef union _NAL_DEVICE_LOCATION
{
    PCI_SLOT_ID         Pci;                   /**< PCI */
    PCI_EXPRESS_SLOT_ID PciExpress;            /**< PCI Express */
    VF_SLOT_ID          Vf;                    /**< VF */
    CARDBUS_SLOT_ID     Cardbus;               /**< Cardbus */
    NAL_OS_SLOT_ID      OsDeviceLocation;      /**< OS Slot ID, unique to each OS implementation */
    UINT64              Reserved;              /**< Reserved */
} NAL_DEVICE_LOCATION;

#pragma pack(1)
/*********************************************************************************************************//**
**
** \details         NAL_SUBST_DEVICE_TABLE structure contains substitute device table.
**
*************************************************************************************************************/

typedef struct _NAL_SUBST_DEVICE_TABLE
{
    UINT16 DeviceId;                        /**< Device Id value */
    NAL_DEVICE_LOCATION DeviceLocation;     /**< NAL_DEVICE_LOCATION structure, which contains informations
                                             about device*/
} NAL_SUBST_DEVICE_TABLE;
/*********************************************************************************************************//**
**
** \details         PCIEXP_CONFIG_STORE struture describes PCI express configuration store.
**
*************************************************************************************************************/

typedef struct _PCIEXP_CONFIG_STORE
{
    NAL_DEVICE_LOCATION DeviceLocation;             /**< NAL_DEVICE_LOCATION structure which contains
                                                    informations about device location */
    PCI_EXPRESS_CONFIG_SPACE ConfigSpace;           /**< PCI_EXPRESS_CONFIG_SPACE structure */
}PCIEXP_CONFIG_STORE;

/*********************************************************************************************************//**
**
** \ingroup         NAL_Enums
**
** \details         NAL_IO_RESOURCE_TYPE enumeration describes possible resource types.
**
*************************************************************************************************************/
typedef enum _NAL_IO_RESOURCE_TYPE
{
    NAL_IO_TYPE_UNUSED = 0,     /**< 0 - Resource unused */
    NAL_IO_TYPE_IO,             /**< 1 - IO memory */
    NAL_IO_TYPE_MEM             /**< 2 - Memory */
} NAL_IO_RESOURCE_TYPE;

/*********************************************************************************************************//**
**
** \details         NAL_IO_RESOURCE structure stores information about BAR of a device
**
*************************************************************************************************************/
typedef struct _NAL_IO_RESOURCE
{
    NAL_IO_RESOURCE_TYPE Type;                  /**< Type of the resource - memory or IO */
    NAL_PHYSICAL_ADDRESS MemoryAddress;         /**< Physical Address of the device memory */
} NAL_IO_RESOURCE;

/*********************************************************************************************************//**
**
** \details         NAL_INTERRUPT_INFO structure contains informations about interrupts.
**
*************************************************************************************************************/
typedef struct _NAL_INTERRUPT_INFO
{
    BOOLEAN Valid;              /**< Flag which determines if interrupt is valid */
    UINT32 Irq;                 /**< IRQ */
    UINT32 Vector;              /**< Interrupction vector */
    UINT32 Trigger;             /**< Interrupction trigger */
    UINT32 Affinity;            /**< Interrupction affinity */
    UINT32 Mode;                /**< Interrupction mode*/
    UINT64 Reserved;            /**< Reserved field */
} NAL_INTERRUPT_INFO;

/*********************************************************************************************************//**
**
** \ingroup         NAL_Enums
**
** \details         NAL_DEVICE_CLASS enumeration contains device classes.
**
*************************************************************************************************************/
typedef enum _NAL_DEVICE_CLASS
{
    NAL_CLASS_LEGACY = 0,               /**< 0 - Legacy device */
    NAL_CLASS_STORAGE,                  /**< 1 - Storage device */
    NAL_CLASS_NETWORK,                  /**< 2 - Network device */
    NAL_CLASS_DISPLAY,                  /**< 3 - Display device */
    NAL_CLASS_MULTIMEDIA,               /**< 4 - Multimedia device */
    NAL_CLASS_MEMORY_CONTROLLER,        /**< 5 - Memory controller */
    NAL_CLASS_BRIDGE_DEVICES,           /**< 6 - Bridge device */
    NAL_CLASS_SIMPLE_COMMUNICATION,     /**< 7 - Simple communication device */
    NAL_CLASS_SYSTEM,                   /**< 8 - System device */
    NAL_CLASS_USB,                      /**< 9 - USB device */
    NAL_CLASS_UNKNOWN = 0xFF            /**< 0xFF - Unknown device */
} NAL_DEVICE_CLASS;

typedef UINT8 NAL_DEVICE_SUBCLASS;

/*********************************************************************************************************//**
**
** \details         NAL_DEVICE structure with basic PCI information about the adapter. Allows to identify the
**                  device in the NAL before adapter initialization. This structure contains memory resources
**                  and vendor/device information in order to identify the device.
**
*************************************************************************************************************/
typedef struct _NAL_DEVICE
{
    UINT32 Signature;                               /**< The signature identifies the device as a NAL_DEVICE.
                                                    This is important because this structure may be unioned
                                                    with a PCI_DEVICE and this allows identifying the device
                                                    as a NAL_DEVICE. */
    UINT16 VendorId;                                /**< PCI vendor ID - allows to identify company of the
                                                    device. For Intel it is 0x8086 */
    UINT16 DeviceId;                                /**< Device ID of the PCI device. Allows to identify type
                                                     of the PCI device */
    UINT16 SubSystemId;                             /**< SubSystem identification number */
    UINT16 SubVendorId;                             /**< SubVendor identification number */
    UINT8 RevisionId;                               /**< Revision of the device */
    NAL_DEVICE_CLASS Class;                         /**< Class of the PCI device. The basic device classes:
                                                     0x00 - Devices before class codes
                                                     0x01 - Mass storage controller
                                                     0x02 - Network controller
                                                     For more you can see the NAL_DEVICE_CLASS type */
    NAL_DEVICE_SUBCLASS SubClass;                   /**< NAL_DEVICE_SUBCLASS enumeration value */
    UINT32 ResourceCount;                           /**< This is the number of memory resources for the device. */
    NAL_IO_RESOURCE IoResource[NAL_MAX_BARS];       /**< NAL_IO_RESOURCE structure This is an array of
                                                    resources for the device. Each resource contains a type
                                                    and a memory address. Where possible, these will be stored
                                                    in order of the PCI BAR that they�d be found*/
    NAL_INTERRUPT_INFO InterruptInfo;               /**< NAL_INTERRUPT_INFO structure. This is the information
                                                    required to register an interrupt subroutine for the device */
    NAL_DEVICE_LOCATION DeviceLocation;             /**< NAL_DEVICE_LOCATION structure */
    BOOLEAN DeviceIsOsConfigured;                   /**< Flag which determines if operating system is configured */
} NAL_DEVICE;
#pragma pack()

/*********************************************************************************************************//**
**
** \details         NAL_ADAPTER_VENDOR_INFO structure contains vendor information stored within the device.
**                  This allows the device to be identified as one type over another. This structure is
**                  usually populated from the PCI or bus scan results.
**
*************************************************************************************************************/
typedef struct _NAL_ADAPTER_VENDOR_INFO
{
    UINT16 Vendor;           /**< Vendor value */
    UINT16 Device;           /**< Device value */
    UINT16 SubVendor;        /**< Subvendor value */
    UINT16 SubDevice;        /**< Subdevice value */
    UINT8 Revision;          /**< Revision value */
} NAL_ADAPTER_VENDOR_INFO;

#endif
