// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2015 - 2020 Intel Corporation */
#include "main.h"

#define DRV_VER_MAJOR 1
#define DRV_VER_MINOR 3
#define DRV_VER_BUILD 19
#define DRV_VER	__stringify(DRV_VER_MAJOR) "."		\
	__stringify(DRV_VER_MINOR) "." __stringify(DRV_VER_BUILD)

static int resource_profile;
module_param(resource_profile, int, 0644);
MODULE_PARM_DESC(resource_profile, "Resource Profile: 0=PF only(default), 1=Weighted VF, 2=Even Distribution");

static int max_rdma_vfs = 32;
module_param(max_rdma_vfs, int, 0644);
MODULE_PARM_DESC(max_rdma_vfs, "Maximum VF count: 0-32, default=32");

bool irdma_upload_context;
module_param(irdma_upload_context, bool, 0644);
MODULE_PARM_DESC(irdma_upload_context, "Upload QP context, default=false");

static int limits_sel;
module_param(limits_sel, int, 0644);
MODULE_PARM_DESC(limits_sel, "Resource limits selector, Range: 0-7, default=0");

static int gen1_limits_sel = 2;
module_param(gen1_limits_sel, int, 0644);
MODULE_PARM_DESC(gen1_limits_sel, "x722 resource limits selector, Range: 0-5, default=2");

static int roce_ena;
module_param(roce_ena, int, 0644);
MODULE_PARM_DESC(roce_ena, "RoCE enable: 1=enable RoCEv2 on all ports (not supported on x722), 0=iWARP(default)");

static int en_rem_endpoint_trk;
module_param(en_rem_endpoint_trk, int, 0644);
MODULE_PARM_DESC(en_rem_endpoint_trk, "Remote Endpoint Tracking: 1=enabled (not supported on x722), 0=disabled(default)");

MODULE_ALIAS("i40iw");
MODULE_AUTHOR("Intel Corporation, <e1000-rdma@lists.sourceforge.net>");
MODULE_DESCRIPTION("Intel(R) Ethernet Protocol Driver for RDMA");
MODULE_LICENSE("Dual BSD/GPL");
MODULE_VERSION(DRV_VER);

LIST_HEAD(irdma_handlers);
DEFINE_SPINLOCK(irdma_handler_lock);

static struct notifier_block irdma_inetaddr_notifier = {
	.notifier_call = irdma_inetaddr_event
};

static struct notifier_block irdma_inetaddr6_notifier = {
	.notifier_call = irdma_inet6addr_event
};

static struct notifier_block irdma_net_notifier = {
	.notifier_call = irdma_net_event
};

static struct notifier_block irdma_netdevice_notifier = {
	.notifier_call = irdma_netdevice_event
};

/**
 * set_protocol_used - set protocol_used against HW generation and roce_ena flag
 * @rf: RDMA PCI function
 * @roce_ena: RoCE enabled flag
 */
static void set_protocol_used(struct irdma_pci_f *rf, bool roce_ena)
{
	switch (rf->rdma_ver) {
	case IRDMA_GEN_2:
		rf->protocol_used = roce_ena ? IRDMA_ROCE_PROTOCOL_ONLY :
					       IRDMA_IWARP_PROTOCOL_ONLY;
		break;
	case IRDMA_GEN_1:
		rf->protocol_used = IRDMA_IWARP_PROTOCOL_ONLY;
		break;
	}
}

void irdma_init_rf_config_params(struct irdma_pci_f *rf)
{
	/* Setup RF configurations from module parameters */
#ifdef DEVLINK_SUPPORTED
	struct irdma_dl_priv *dl_priv;

	dl_priv = platform_get_drvdata(rf->hdl->pdev);
	if (!dl_priv->reload) {
		rf->limits_sel = (rf->rdma_ver == IRDMA_GEN_1) ? gen1_limits_sel :
								 limits_sel;
		set_protocol_used(rf, roce_ena);
	} else {
		rf->limits_sel = dl_priv->limits_sel;
		set_protocol_used(rf, dl_priv->roce_ena);

	}
	rf->dl_priv = dl_priv;
#else
	rf->limits_sel = (rf->rdma_ver == IRDMA_GEN_1) ? gen1_limits_sel :
							 limits_sel;
	set_protocol_used(rf, roce_ena);
#endif /* DEVLINK_SUPPORTED */
	rf->rsrc_profile = (resource_profile < IRDMA_HMC_PROFILE_EQUAL) ?
			    (u8)resource_profile + IRDMA_HMC_PROFILE_DEFAULT :
			    IRDMA_HMC_PROFILE_DEFAULT;
	rf->max_rdma_vfs = (rf->rsrc_profile != IRDMA_HMC_PROFILE_DEFAULT) ?
			    max_rdma_vfs : 0;
	rf->max_ena_vfs = rf->max_rdma_vfs;
	rf->en_rem_endpoint_trk = en_rem_endpoint_trk;
	rf->rst_to = IRDMA_RST_TIMEOUT_HZ;
}

/*
 * irdma_deinit_rf - Clean up resources allocated for RF
 * @rf: RDMA PCI function
 */
void irdma_deinit_rf(struct irdma_pci_f *rf)
{
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_pf_exit(rf->hdl);
#endif
	irdma_ctrl_deinit_hw(rf);
	irdma_del_handler(rf->hdl);
	kfree(rf->hdl);
}

/**
 * irdma_find_ice_handler - find a handler given a client info
 * @pcidev: pointer to pci dev info
 */
struct irdma_handler *irdma_find_handler(struct pci_dev *pcidev)
{
	struct irdma_handler *hdl;
	unsigned long flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_for_each_entry (hdl, &irdma_handlers, list) {
		if (hdl->rf.pcidev == pcidev) {
			spin_unlock_irqrestore(&irdma_handler_lock, flags);
			return hdl;
		}
	}
	spin_unlock_irqrestore(&irdma_handler_lock, flags);

	return NULL;
}

/**
 * irdma_add_handler - add a handler to the list
 * @hdl: handler to be added to the handler list
 */
void irdma_add_handler(struct irdma_handler *hdl)
{
	unsigned long flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_add(&hdl->list, &irdma_handlers);
	spin_unlock_irqrestore(&irdma_handler_lock, flags);
}

/**
 * irdma_del_handler - delete a handler from the list
 * @hdl: handler to be deleted from the handler list
 */
void irdma_del_handler(struct irdma_handler *hdl)
{
	unsigned long flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_del(&hdl->list);
	spin_unlock_irqrestore(&irdma_handler_lock, flags);
}

/**
 * irdma_register_notifiers - register tcp ip notifiers
 */
void irdma_register_notifiers(void)
{
	register_inetaddr_notifier(&irdma_inetaddr_notifier);
	register_inet6addr_notifier(&irdma_inetaddr6_notifier);
	register_netevent_notifier(&irdma_net_notifier);
	register_netdevice_notifier(&irdma_netdevice_notifier);
}

void irdma_unregister_notifiers(void)
{
	unregister_netevent_notifier(&irdma_net_notifier);
	unregister_inetaddr_notifier(&irdma_inetaddr_notifier);
	unregister_inet6addr_notifier(&irdma_inetaddr6_notifier);
	unregister_netdevice_notifier(&irdma_netdevice_notifier);
}

/**
 * irdma_add_ipv6_addr - add ipv6 address to the hw arp table
 * @iwdev: irdma device
 */
static void irdma_add_ipv6_addr(struct irdma_device *iwdev)
{
	struct net_device *ip_dev;
	struct inet6_dev *idev;
	struct inet6_ifaddr *ifp, *tmp;
	u32 local_ipaddr6[4];

	rcu_read_lock();
	for_each_netdev_rcu (&init_net, ip_dev) {
		if (((rdma_vlan_dev_vlan_id(ip_dev) < 0xFFFF &&
		      rdma_vlan_dev_real_dev(ip_dev) == iwdev->netdev) ||
		      ip_dev == iwdev->netdev) &&
		      (READ_ONCE(ip_dev->flags) & IFF_UP)) {
			idev = __in6_dev_get(ip_dev);
			if (!idev) {
				dev_err(idev_to_dev(&iwdev->rf->sc_dev),
					"ipv6 inet device not found\n");
				break;
			}
			list_for_each_entry_safe (ifp, tmp, &idev->addr_list,
						  if_list) {
				irdma_dbg(&iwdev->rf->sc_dev,
					  "INIT: IP=%pI6, vlan_id=%d, MAC=%pM\n",
					  &ifp->addr,
					  rdma_vlan_dev_vlan_id(ip_dev),
					  ip_dev->dev_addr);

				irdma_copy_ip_ntohl(local_ipaddr6,
						    ifp->addr.in6_u.u6_addr32);
				irdma_manage_arp_cache(iwdev->rf,
						       ip_dev->dev_addr,
						       local_ipaddr6, false,
						       IRDMA_ARP_ADD);
			}
		}
	}
	rcu_read_unlock();
}

/**
 * irdma_add_ipv4_addr - add ipv4 address to the hw arp table
 * @iwdev: irdma device
 */
static void irdma_add_ipv4_addr(struct irdma_device *iwdev)
{
	struct net_device *dev;
	struct in_device *idev;
	u32 ip_addr;

	rcu_read_lock();
	for_each_netdev_rcu (&init_net, dev) {
		if (((rdma_vlan_dev_vlan_id(dev) < 0xFFFF &&
		      rdma_vlan_dev_real_dev(dev) == iwdev->netdev) ||
		      dev == iwdev->netdev) && (READ_ONCE(dev->flags) & IFF_UP)) {
#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(5, 3, 0)) || defined(RHEL_8_2) || defined(RHEL_8_3))
			const struct in_ifaddr *ifa;

#endif
			idev = __in_dev_get_rcu(dev);
			if (!idev)
				continue;

#if ((LINUX_VERSION_CODE >= KERNEL_VERSION(5, 3, 0)) || defined(RHEL_8_2) || defined(RHEL_8_3))
			in_dev_for_each_ifa_rcu(ifa, idev) {
#else
			for_ifa(idev) {
#endif
				irdma_dbg(&iwdev->rf->sc_dev,
					  "CM: IP=%pI4, vlan_id=%d, MAC=%pM\n",
					  &ifa->ifa_address,
					  rdma_vlan_dev_vlan_id(dev),
					  dev->dev_addr);

				ip_addr = ntohl(ifa->ifa_address);
				irdma_manage_arp_cache(iwdev->rf, dev->dev_addr,
						       &ip_addr, true,
						       IRDMA_ARP_ADD);
			}
#if ((LINUX_VERSION_CODE < KERNEL_VERSION(5, 3, 0)) && !defined(RHEL_8_2) && !defined(RHEL_8_3))
			endfor_ifa(idev);
#endif
		}
	}
	rcu_read_unlock();
}

/**
 * irdma_add_ip - add ip addresses
 * @iwdev: irdma device
 *
 * Add ipv4/ipv6 addresses to the arp cache
 */
void irdma_add_ip(struct irdma_device *iwdev)
{
	irdma_add_ipv4_addr(iwdev);
	irdma_add_ipv6_addr(iwdev);
}

#ifdef DEVLINK_SUPPORTED
#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
static int irdma_devlink_rsrc_limits_validate(struct devlink *dl, u32 id,
					      union devlink_param_value val)
#else
static int irdma_devlink_rsrc_limits_validate(struct devlink *dl, u32 id,
					      union devlink_param_value val,
					      struct netlink_ext_ack *extack)
#endif
{
	u8 value = val.vu8;

	if (value > 7) {
		NL_SET_ERR_MSG_MOD(extack, "resource limits selector range is (0-7)");
		return -ERANGE;
	}

	return 0;
}

static int irdma_devlink_enable_roce_validate(struct devlink *dl, u32 id,
#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
					      union devlink_param_value val)
#else
					      union devlink_param_value val,
					      struct netlink_ext_ack *extack)
#endif
{
	struct irdma_dl_priv *priv = devlink_priv(dl);
	bool value = val.vbool;

	if (value && priv->hw_ver == IRDMA_GEN_1) {
		NL_SET_ERR_MSG_MOD(extack, "RoCE not supported on device");
		return -EOPNOTSUPP;
	}

	return 0;
}

#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
static int irdma_devlink_fragcnt_limit_validate(struct devlink *dl, u32 id,
					      union devlink_param_value val)
#else
static int irdma_devlink_fragcnt_limit_validate(struct devlink *dl, u32 id,
						union devlink_param_value val,
						struct netlink_ext_ack *extack)
#endif
{
	struct irdma_dl_priv *priv = devlink_priv(dl);
	u8 value = val.vu8;

	if (priv->hw_ver == IRDMA_GEN_1) {
		NL_SET_ERR_MSG_MOD(extack, "feature not supproted on this device");
		return -EOPNOTSUPP;
	} else if (value > 13 || value < 2) {
		NL_SET_ERR_MSG_MOD(extack, "fragment count limit range (2-13)");
		return -ERANGE;
	}

	return 0;
}

static int irdma_devlink_upload_ctx_get(struct devlink *devlink, u32 id,
					struct devlink_param_gset_ctx *ctx)
{
	ctx->val.vbool = irdma_upload_context;
	return 0;
}

static int irdma_devlink_upload_ctx_set(struct devlink *devlink, u32 id,
					struct devlink_param_gset_ctx *ctx)
{
	irdma_upload_context = ctx->val.vbool;
	return 0;
}
enum irdma_dl_param_id {
	IRDMA_DEVLINK_PARAM_ID_BASE = DEVLINK_PARAM_GENERIC_ID_MAX,
	IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
	IRDMA_DEVLINK_PARAM_ID_UPLOAD_CONTEXT,
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
#endif
	/* DCQCN congestion control */
	IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
	IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
	IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
	IRDMA_DEVLINK_PARAM_ID_MIN_RATE,
	IRDMA_DEVLINK_PARAM_ID_DCQCN_F,
	IRDMA_DEVLINK_PARAM_ID_DCQCN_T,
	IRDMA_DEVLINK_PARAM_ID_DCQCN_B,
	IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
	IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
	IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
	IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
};

static const struct devlink_param irdma_devlink_params[] = {
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
			     "resource_limits_selector", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_rsrc_limits_validate),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_UPLOAD_CONTEXT,
			     "upload_context", DEVLINK_PARAM_TYPE_BOOL,
			     BIT(DEVLINK_PARAM_CMODE_RUNTIME),
			     irdma_devlink_upload_ctx_get,
			     irdma_devlink_upload_ctx_set, NULL),
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
			     "roce_enable", DEVLINK_PARAM_TYPE_BOOL,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_enable_roce_validate),
#else
	DEVLINK_PARAM_GENERIC(ENABLE_ROCE, BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_enable_roce_validate),
#endif
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
			     "dcqcn_enable", DEVLINK_PARAM_TYPE_BOOL,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
			     "dcqcn_cc_cfg_valid", DEVLINK_PARAM_TYPE_BOOL,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
			     "dcqcn_min_dec_factor", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_MIN_RATE,
			     "dcqcn_min_rate", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_F,
			     "dcqcn_F", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_T,
			     "dcqcn_T", DEVLINK_PARAM_TYPE_U16,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_DCQCN_B,
			     "dcqcn_B", DEVLINK_PARAM_TYPE_U32,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
			     "dcqcn_rai_factor", DEVLINK_PARAM_TYPE_U16,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
			     "dcqcn_hai_factor", DEVLINK_PARAM_TYPE_U16,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
			     "dcqcn_rreduce_mperiod", DEVLINK_PARAM_TYPE_U32,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, NULL),
	DEVLINK_PARAM_DRIVER(IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
			     "fragment_count_limit", DEVLINK_PARAM_TYPE_U8,
			      BIT(DEVLINK_PARAM_CMODE_DRIVERINIT),
			      NULL, NULL, irdma_devlink_fragcnt_limit_validate),
};

#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
#ifdef DEVLINK_RELOAD_DOWN_NETNS_CHANGE
static int  irdma_devlink_reload_down(struct devlink *devlink, bool netns_change,
				      struct netlink_ext_ack *extack)
#else
static int irdma_devlink_reload_down(struct devlink *devlink,
				     struct netlink_ext_ack *extack)
#endif
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);

#ifdef DEVLINK_RELOAD_DOWN_NETNS_CHANGE
	if (netns_change) {
		NL_SET_ERR_MSG_MOD(extack, "Namespace change is not supported");
		return -EOPNOTSUPP;
	}

#endif
	switch (priv->hw_ver) {
	case IRDMA_GEN_2:
		irdma_remove_dev(priv->pdev);
		break;
	case IRDMA_GEN_1:
		i40iw_remove_dev(priv->pdev);
		break;
	default:
		return -ENODEV;
	}

	return 0;
}

#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS*/
#if defined(RHEL_7_7) || defined(RHEL_7_8) || defined(RHEL_7_9)
static int irdma_devlink_reload_up(struct devlink *devlink)
#else
static int irdma_devlink_reload_up(struct devlink *devlink,
				   struct netlink_ext_ack *extack)
#endif
{
	struct irdma_dl_priv *priv = devlink_priv(devlink);
	union devlink_param_value saved_value;
	int ret;

	devlink_param_driverinit_value_get(devlink,
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
				IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE,
#else
				DEVLINK_PARAM_GENERIC_ID_ENABLE_ROCE,
#endif
				&saved_value);
	priv->roce_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR,
				&saved_value);
	priv->limits_sel = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE,
				&saved_value);
	priv->dcqcn_ena = saved_value.vbool;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID,
				&saved_value);
	priv->dcqcn_params.cc_cfg_valid = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR,
				&saved_value);
	priv->dcqcn_params.min_dec_factor = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_MIN_RATE, &saved_value);
	priv->dcqcn_params.min_rate = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_F, &saved_value);
	priv->dcqcn_params.dcqcn_f = saved_value.vu8;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_T, &saved_value);
	priv->dcqcn_params.dcqcn_t = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_B, &saved_value);
	priv->dcqcn_params.dcqcn_b = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR,
				&saved_value);
	priv->dcqcn_params.rai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR,
				&saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu16;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD,
				&saved_value);
	priv->dcqcn_params.hai_factor = saved_value.vu32;
	devlink_param_driverinit_value_get(devlink,
				IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT,
				&saved_value);
	priv->fragcnt_limit = saved_value.vu8;
	switch (priv->hw_ver) {
	case IRDMA_GEN_2:
#ifndef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
		irdma_remove_dev(priv->pdev);
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
		priv->reload = true;
		ret = irdma_probe_dev(priv->pdev);
		break;
	case IRDMA_GEN_1:
#ifndef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
		i40iw_remove_dev(priv->pdev);
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
		priv->reload = true;
		ret = i40iw_probe_dev(priv->pdev);
		break;
	default:
		ret = -ENODEV;
		break;
	}

	priv->reload = false;
	return ret;
}

static const struct devlink_ops irdma_devlink_ops = {
#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	.reload_up = irdma_devlink_reload_up,
	.reload_down = irdma_devlink_reload_down,
#else
    	.reload = irdma_devlink_reload_up,
#endif /* IRDMA_DEVLINK_RELOAD_SPLIT_OPS */
};

static void irdma_devlink_unregister(struct platform_device *pdev,
				     enum irdma_vers hw_ver)
{
	struct irdma_dl_priv *priv = platform_get_drvdata(pdev);
	struct devlink *devlink = priv_to_devlink(priv);

#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	devlink_reload_disable(devlink);
#endif
	devlink_params_unregister(devlink, irdma_devlink_params,
				  ARRAY_SIZE(irdma_devlink_params));
	devlink_unregister(devlink);
	devlink_free(devlink);
}

static int irdma_devlink_register(struct platform_device *pdev,
				  enum irdma_vers hw_ver)
{
	struct devlink *devlink;
	struct irdma_dl_priv *priv;
	union devlink_param_value value;
	int ret;

	devlink = devlink_alloc(&irdma_devlink_ops, sizeof(struct irdma_dl_priv));
	if (!devlink)
		return -ENOMEM;

	priv = devlink_priv(devlink);
	priv->pdev = pdev;
	priv->hw_ver = hw_ver;
	priv->limits_sel = (hw_ver >= IRDMA_GEN_2) ? 0 : 2;
	priv->fragcnt_limit = (hw_ver >= IRDMA_GEN_2) ? 6 : 2;
	platform_set_drvdata(pdev, priv);

	ret = devlink_register(devlink, &pdev->dev);
	if (ret)
		goto err_dl_free;

	ret = devlink_params_register(devlink, irdma_devlink_params,
				      ARRAY_SIZE(irdma_devlink_params));
	if (ret)
		goto err_dl_unreg;

	value.vu8 = priv->limits_sel;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_LIMITS_SELECTOR, value);
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_UPLOAD_CONTEXT, value);
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
#ifndef DEVLINK_PARAM_GENERIC_ENABLE_ROCE_TYPE
				IRDMA_DEVLINK_PARAM_ID_ROCE_ENABLE, value);
#else
				DEVLINK_PARAM_GENERIC_ID_ENABLE_ROCE, value);
#endif
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_ENABLE, value);
	value.vbool = false;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_CC_CFG_VALID, value);
	value.vu8 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_MIN_DEC_FACTOR, value);
	value.vu8 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_MIN_RATE, value);
	value.vu8 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_F, value);
	value.vu16 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_T, value);
	value.vu32 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_DCQCN_B, value);
	value.vu16 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_RAI_FACTOR, value);
	value.vu16 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_HAI_FACTOR, value);
	value.vu32 = 0;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_RREDUCE_MPERIOD, value);
	value.vu8 = priv->fragcnt_limit;
	devlink_param_driverinit_value_set(devlink,
				IRDMA_DEVLINK_PARAM_ID_FRAGCNT_LIMIT, value);
	devlink_params_publish(devlink);
#ifdef IRDMA_DEVLINK_RELOAD_SPLIT_OPS
	devlink_reload_enable(devlink);
#endif

	return 0;

err_dl_unreg:
	devlink_unregister(devlink);
err_dl_free:
	devlink_free(devlink);

	return ret;
}

#endif /* DEVLINK_SUPPORTED */

static int irdma_init_dev(struct platform_device *pdev, enum irdma_vers hw_ver)
{
	int ret = -ENODEV;

	switch (hw_ver) {
	case IRDMA_GEN_2:
		ret = irdma_probe_dev(pdev);
		break;
	case IRDMA_GEN_1:
		ret = i40iw_probe_dev(pdev);
		break;
	default:
		break;
	}

	return ret;
}

static void irdma_deinit_dev(struct platform_device *pdev, enum irdma_vers hw_ver)
{
	switch (hw_ver) {
	case IRDMA_GEN_2:
		irdma_remove_dev(pdev);
		break;
	case IRDMA_GEN_1:
		i40iw_remove_dev(pdev);
		break;
	default:
		break;
	}
}

static enum irdma_vers irdma_get_hw_version(struct platform_device *pdev)
{
	enum irdma_vers hw_ver = IRDMA_GEN_RSVD;

	if (!strcmp(pdev->name, IRDMA_I40E_PDEV_NAME))
		hw_ver = IRDMA_GEN_1;
	else if (!strcmp(pdev->name, IRDMA_ICE_PDEV_NAME))
		hw_ver = IRDMA_GEN_2;

	return hw_ver;
}

static int irdma_probe(struct platform_device *pdev)
{
	int ret;
	enum irdma_vers hw_ver = irdma_get_hw_version(pdev);

	if (!hw_ver)
		return -ENODEV;

#ifdef DEVLINK_SUPPORTED
	ret = irdma_devlink_register(pdev, hw_ver);
	if (ret)
		return ret;

#endif

	ret = irdma_init_dev(pdev, hw_ver);

#ifdef DEVLINK_SUPPORTED
	if (ret)
		irdma_devlink_unregister(pdev, hw_ver);

#endif
	return ret;
}

static int irdma_remove(struct platform_device *pdev)
{
	enum irdma_vers hw_ver = irdma_get_hw_version(pdev);

	if (!hw_ver)
		return -ENODEV;

	irdma_deinit_dev(pdev, hw_ver);
#ifdef DEVLINK_SUPPORTED
	irdma_devlink_unregister(pdev, hw_ver);

#endif
	return 0;
}

static int irdma_suspend(struct platform_device *pdev, pm_message_t state)
{
	enum irdma_vers hw_ver = irdma_get_hw_version(pdev);

	if (!hw_ver)
		return -ENODEV;

	irdma_deinit_dev(pdev, hw_ver);

	return 0;
}

static int irdma_resume(struct platform_device *pdev)
{
	enum irdma_vers hw_ver = irdma_get_hw_version(pdev);

	if (!hw_ver)
		return -ENODEV;

	return irdma_init_dev(pdev, hw_ver);
}

static const struct platform_device_id irdma_platform_id_table[] = {
	{.name = IRDMA_ICE_PDEV_NAME},
	{.name = IRDMA_I40E_PDEV_NAME},
	{},
};

MODULE_DEVICE_TABLE(platform, irdma_platform_id_table);

static struct platform_driver irdma_pdriver = {
	.driver = {
		   .name = "irdma",
		   .owner = THIS_MODULE,
		  },
	.id_table = irdma_platform_id_table,
	.probe = irdma_probe,
	.remove = irdma_remove,
	.resume = irdma_resume,
	.suspend = irdma_suspend,
};

/**
 * irdma_init_module - driver initialization function
 *
 * First function to call when the driver is loaded
 * Register the driver as ice client and port mapper client
 */
static int __init irdma_init_module(void)
{
	int ret;

	pr_info("irdma driver version: %d.%d.%d\n", DRV_VER_MAJOR,
		DRV_VER_MINOR, DRV_VER_BUILD);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_init();
#endif
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
	ret = irdma_configfs_init();
	if (ret) {
		pr_err("Failed to register irdma to configfs subsystem\n");
#ifdef CONFIG_DEBUG_FS
		irdma_dbg_exit();
#endif
		return ret;
	}
#endif
	ret = platform_driver_register(&irdma_pdriver);
	if (ret) {
		pr_err("Failed irdma platform_driver_register()\n");
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
		irdma_configfs_exit();
#endif
#ifdef CONFIG_DEBUG_FS
		irdma_dbg_exit();
#endif
		return ret;
	}
	irdma_register_notifiers();

	return 0;
}

/**
 * irdma_exit_module - driver exit clean up function
 *
 * The function is called just before the driver is unloaded
 * Unregister the driver as ice client and port mapper client
 */
static void __exit irdma_exit_module(void)
{
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
	irdma_configfs_exit();
#endif
	irdma_unregister_notifiers();
	platform_driver_unregister(&irdma_pdriver);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_exit();
#endif
}

module_init(irdma_init_module);
module_exit(irdma_exit_module);
