// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2019 - 2020 Intel Corporation */
#include "ice_idc.h"
#include "main.h"
#include "ws.h"
#include "icrdma_hw.h"

/**
 * irdma_lan_register_qset - Register qset with LAN driver
 * @vsi: vsi structure
 * @tc_node: Traffic class node
 */
static enum irdma_status_code irdma_lan_register_qset(struct irdma_sc_vsi *vsi,
						      struct irdma_ws_node *tc_node)
{
	struct irdma_device *iwdev = vsi->back_vsi;
	struct ice_peer_dev *ldev = iwdev->ldev->if_ldev;
	struct ice_res rdma_qset_res = {};
	int ret;

	rdma_qset_res.cnt_req = 1;
	rdma_qset_res.res_type = ICE_RDMA_QSETS_TXSCHED;
	rdma_qset_res.res[0].res.qsets.qs_handle = tc_node->qs_handle;
	rdma_qset_res.res[0].res.qsets.tc = tc_node->traffic_class;
	rdma_qset_res.res[0].res.qsets.vsi_id = vsi->vsi_idx;
	ret = ldev->ops->alloc_res(ldev, &rdma_qset_res, 0);
	if (ret) {
		irdma_dbg(vsi->dev,
			  "WS: LAN alloc_res for rdma qset failed.\n");
		return IRDMA_ERR_NO_MEMORY;
	}

	tc_node->l2_sched_node_id = rdma_qset_res.res[0].res.qsets.teid;
	vsi->qos[tc_node->user_pri].l2_sched_node_id =
		rdma_qset_res.res[0].res.qsets.teid;

	return 0;
}

/**
 * irdma_lan_unregister_qset - Unregister qset with LAN driver
 * @vsi: vsi structure
 * @tc_node: Traffic class node
 */
static void irdma_lan_unregister_qset(struct irdma_sc_vsi *vsi,
				      struct irdma_ws_node *tc_node)
{
	struct irdma_device *iwdev = vsi->back_vsi;
	struct ice_peer_dev *ldev = iwdev->ldev->if_ldev;
	struct ice_res rdma_qset_res = {};

	rdma_qset_res.res_allocated = 1;
	rdma_qset_res.res_type = ICE_RDMA_QSETS_TXSCHED;
	rdma_qset_res.res[0].res.qsets.vsi_id = vsi->vsi_idx;
	rdma_qset_res.res[0].res.qsets.teid = tc_node->l2_sched_node_id;
	rdma_qset_res.res[0].res.qsets.qs_handle = tc_node->qs_handle;

	if (ldev->ops->free_res(ldev, &rdma_qset_res))
		irdma_dbg(vsi->dev,
		          "WS: LAN free_res for rdma qset failed.\n");
}

/**
 * irdma_prep_tc_change - Prepare for TC changes
 * @ldev: Peer device structure
 */
static void irdma_prep_tc_change(struct ice_peer_dev *ldev)
{
	struct irdma_device *iwdev;

	iwdev = irdma_get_device(ldev->netdev);
	if (!iwdev)
		return;

	if (iwdev->vsi.tc_change_pending)
		goto done;

	iwdev->vsi.tc_change_pending = true;
	irdma_sc_suspend_resume_qps(&iwdev->vsi, IRDMA_OP_SUSPEND);

	/* Wait for all qp's to suspend */
	wait_event_timeout(iwdev->suspend_wq,
			   !atomic_read(&iwdev->vsi.qp_suspend_reqs),
			   IRDMA_EVENT_TIMEOUT);
	irdma_ws_reset(&iwdev->vsi);
done:
	irdma_put_device(iwdev);
}

static void irdma_log_invalid_mtu(u16 mtu, struct irdma_sc_dev *dev)
{
	if (mtu < IRDMA_MIN_MTU_IPV4)
		dev_warn(idev_to_dev(dev),
		         "MTU setting [%d] too low for RDMA traffic. Minimum MTU is 576 for IPv4\n",
		         mtu);
	else if (mtu < IRDMA_MIN_MTU_IPV6)
		dev_warn(idev_to_dev(dev),
		         "MTU setting [%d] too low for RDMA traffic. Minimum MTU is 1280 for IPv6\\n",
		         mtu);
}

/**
 * irdma_event_handler - Called by LAN driver to notify events
 * @ldev: Peer device structure
 * @event: event from LAN driver
 */
static void irdma_event_handler(struct ice_peer_dev *ldev,
				struct ice_event *event)
{
	struct irdma_l2params l2params = {};
	struct irdma_device *iwdev;
	int i;

	iwdev = irdma_get_device(ldev->netdev);
	if (!iwdev)
		return;

	if (*event->type & BIT(ICE_EVENT_LINK_CHANGE)) {
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: LINK_CHANGE event\n");
	} else if (*event->type & BIT(ICE_EVENT_MTU_CHANGE)) {
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: new MTU = %d\n",
			  event->info.mtu);
		if (iwdev->vsi.mtu != event->info.mtu) {
			l2params.mtu = event->info.mtu;
			l2params.mtu_changed = true;
			irdma_log_invalid_mtu(l2params.mtu, &iwdev->rf->sc_dev);
			irdma_change_l2params(&iwdev->vsi, &l2params);
		}
	} else if (*event->type & BIT(ICE_EVENT_TC_CHANGE)) {
		if (!iwdev->vsi.tc_change_pending)
			goto done;

		l2params.tc_changed = true;
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: TC Change\n");
		iwdev->dcb = event->info.port_qos.num_tc > 1;

		for (i = 0; i < ICE_IDC_MAX_USER_PRIORITY; ++i)
			l2params.up2tc[i] = event->info.port_qos.up2tc[i];
		irdma_check_fc_for_tc_update(&iwdev->vsi, &l2params);
		irdma_change_l2params(&iwdev->vsi, &l2params);
	} else if (*event->type & BIT(ICE_EVENT_API_CHANGE)) {
		irdma_dbg(&iwdev->rf->sc_dev, "CLNT: API_CHANGE\n");
	} else if (*event->type & BIT(ICE_EVENT_CRIT_ERR)) {
		ibdev_warn(&iwdev->ibdev, "ICE OICR event notification: oicr = 0x%08x\n",
			   event->info.reg);
		if (event->info.reg & IRDMAPFINT_OICR_PE_CRITERR_M) {
			u32 pe_criterr;

			pe_criterr = readl(iwdev->rf->sc_dev.hw_regs[IRDMA_GLPE_CRITERR]);
#define IRDMA_Q1_RESOURCE_ERR 0x0001024d
			if (pe_criterr != IRDMA_Q1_RESOURCE_ERR) {
				ibdev_err(&iwdev->ibdev, "critical PE Error, GLPE_CRITERR=0x%08x\n",
					  pe_criterr);
				iwdev->rf->reset = true;
			} else {
				ibdev_warn(&iwdev->ibdev, "Q1 Resource Check\n");
			}
		}
		if (event->info.reg & IRDMAPFINT_OICR_HMC_ERR_M) {
			ibdev_err(&iwdev->ibdev, "HMC Error\n");
			iwdev->rf->reset = true;
		}
		if (event->info.reg & IRDMAPFINT_OICR_PE_PUSH_M) {
			ibdev_err(&iwdev->ibdev, "PE Push Error\n");
			iwdev->rf->reset = true;
		}
		if (iwdev->rf->reset)
			iwdev->rf->gen_ops.request_reset(iwdev->rf);
	}
done:
	irdma_put_device(iwdev);
}

/**
 * irdma_open - client interface operation open for RDMA device
 * @ldev: lan device information
 *
 * Called by the lan driver during the processing of client
 * register.
 */
static int irdma_open(struct ice_peer_dev *ldev)
{
	struct irdma_handler *hdl;
	struct irdma_device *iwdev;
	struct irdma_sc_dev *dev;
	struct ice_event events = {};
	struct irdma_pci_f *rf;
	struct irdma_priv_ldev *pldev;
	struct irdma_l2params l2params = {};
	int i, ret;

	hdl = irdma_find_handler(ldev->pdev);
	if (!hdl)
		return -ENODEV;

	rf = &hdl->rf;
	if (rf->init_state != CEQ0_CREATED)
		return -EINVAL;

	iwdev = ib_alloc_device(irdma_device, ibdev);
	if (!iwdev)
		return -ENOMEM;

	pldev = &rf->ldev;
	pldev->pf_vsi_num = ldev->pf_vsi_num;
	dev = &hdl->rf.sc_dev;

	iwdev->hdl = hdl;
	iwdev->rf = rf;
	iwdev->ldev = &rf->ldev;
	iwdev->roce_cwnd = IRDMA_ROCE_CWND_DEFAULT;
	iwdev->roce_ackcreds = IRDMA_ROCE_ACKCREDS_DEFAULT;
	iwdev->rcv_wnd = IRDMA_CM_DEFAULT_RCV_WND_SCALED;
	iwdev->rcv_wscale = IRDMA_CM_DEFAULT_RCV_WND_SCALE;
	iwdev->netdev = ldev->netdev;
	INIT_LIST_HEAD(&iwdev->ah_list);
	mutex_init(&iwdev->ah_list_lock);
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
	iwdev->iwarp_ecn_en = true;
	iwdev->iwarp_rtomin = 5;
	iwdev->up_up_map = IRDMA_DEFAULT_UP_UP_MAP;
#endif
	if (rf->protocol_used == IRDMA_ROCE_PROTOCOL_ONLY) {
#if IS_ENABLED(CONFIG_CONFIGFS_FS)
		iwdev->roce_rtomin = IRDMA_ROCE_RTOMIN_DEFAULT;
#endif
#ifdef DEVLINK_SUPPORTED
		iwdev->roce_dcqcn_en = rf->dl_priv->dcqcn_ena;
#endif
		iwdev->roce_mode = true;
	}
	l2params.mtu = ldev->netdev->mtu;
	l2params.num_tc = ldev->initial_qos_info.num_tc;
	l2params.num_apps = ldev->initial_qos_info.num_apps;
	l2params.vsi_prio_type = ldev->initial_qos_info.vsi_priority_type;
	l2params.vsi_rel_bw = ldev->initial_qos_info.vsi_relative_bw;
	for (i = 0; i < l2params.num_tc; i++) {
		l2params.tc_info[i].egress_virt_up =
			ldev->initial_qos_info.tc_info[i].egress_virt_up;
		l2params.tc_info[i].ingress_virt_up =
			ldev->initial_qos_info.tc_info[i].ingress_virt_up;
		l2params.tc_info[i].prio_type =
			ldev->initial_qos_info.tc_info[i].prio_type;
		l2params.tc_info[i].rel_bw =
			ldev->initial_qos_info.tc_info[i].rel_bw;
		l2params.tc_info[i].tc_ctx =
			ldev->initial_qos_info.tc_info[i].tc_ctx;
	}
	for (i = 0; i < ICE_IDC_MAX_USER_PRIORITY; i++)
		l2params.up2tc[i] = ldev->initial_qos_info.up2tc[i];

	iwdev->vsi_num = ldev->pf_vsi_num;
	ldev->ops->update_vsi_filter(ldev, ICE_RDMA_FILTER_BOTH, true);

	if (irdma_rt_init_hw(rf, iwdev, &l2params)) {
		ib_dealloc_device(&iwdev->ibdev);
		return -EIO;
	}

	ret = irdma_ib_register_device(iwdev);
	if (ret) {
		irdma_rt_deinit_hw(iwdev);
		ib_dealloc_device(&iwdev->ibdev);
		return ret;
	}

	events.reporter = ldev;
	set_bit(ICE_EVENT_LINK_CHANGE, events.type);
	set_bit(ICE_EVENT_MTU_CHANGE, events.type);
	set_bit(ICE_EVENT_TC_CHANGE, events.type);
	set_bit(ICE_EVENT_API_CHANGE, events.type);
	set_bit(ICE_EVENT_CRIT_ERR, events.type);

	ldev->ops->reg_for_notification(ldev, &events);
	irdma_dbg(dev, "INIT: Gen2 VSI[%d] open success ldev=%p\n",
		  ldev->pf_vsi_num, ldev);

	return 0;
}

static void irdma_cleanup_dead_qps(struct irdma_sc_vsi *vsi)
{
	struct irdma_sc_qp *qp = NULL;
	struct irdma_qp *iwqp;
	struct irdma_pci_f *rf;
	u8 i;

	for (i = 0; i < IRDMA_MAX_USER_PRIORITY; i++) {
		qp = irdma_get_qp_from_list(&vsi->qos[i].qplist, qp);
		while (qp) {
			iwqp = qp->qp_uk.back_qp;
			rf = iwqp->iwdev->rf;
			dma_free_coherent(ihw_to_dev(rf->sc_dev.hw), iwqp->q2_ctx_mem.size,
					  iwqp->q2_ctx_mem.va, iwqp->q2_ctx_mem.pa);
			dma_free_coherent(ihw_to_dev(rf->sc_dev.hw), iwqp->kqp.dma_mem.size,
					  iwqp->kqp.dma_mem.va, iwqp->kqp.dma_mem.pa);
			kfree(iwqp->kqp.sq_wrid_mem);
			kfree(iwqp->kqp.rq_wrid_mem);
			kfree(iwqp);
			qp = irdma_get_qp_from_list(&vsi->qos[i].qplist, qp);
		}
	}
}

/**
 * irdma_close - client interface operation close for iwarp/uda device
 * @ldev: lan device information
 * @reason: reason for closing
 *
 * Called by the lan driver during the processing of client unregister
 * Destroy and clean up the driver resources
 */
static void irdma_close(struct ice_peer_dev *ldev, enum ice_close_reason reason)
{
	struct irdma_handler *hdl;
	struct irdma_device *iwdev;
	struct irdma_pci_f *rf;

	hdl = irdma_find_handler(ldev->pdev);
	if (!hdl)
		return;

	rf = &hdl->rf;
	iwdev = list_first_entry_or_null(&rf->vsi_dev_list, struct irdma_device,
					 list);
	if (!iwdev)
		return;
	if (iwdev->up_map_en) {
		struct irdma_up_info up_map_info = {};

		*((u64 *)up_map_info.map) = IRDMA_DEFAULT_UP_UP_MAP;
		up_map_info.use_cnp_up_override = false;
		up_map_info.cnp_up_override = 0;
		up_map_info.hmc_fcn_idx = iwdev->rf->sc_dev.hmc_fn_id;
		irdma_cqp_up_map_cmd(&iwdev->rf->sc_dev, IRDMA_OP_SET_UP_MAP,
				     &up_map_info);
	}
	if (reason == ICE_REASON_GLOBR_REQ || reason == ICE_REASON_CORER_REQ ||
	    reason == ICE_REASON_PFR_REQ || rf->reset) {
		iwdev->reset = true;
		rf->reset = true;
	}

	irdma_ib_unregister_device(iwdev);
#ifndef IB_DEALLOC_DRIVER_SUPPORT
	irdma_rt_deinit_hw(iwdev);
	ib_dealloc_device(&iwdev->ibdev);
#endif
	 irdma_cleanup_dead_qps(&iwdev->vsi);
	if (!rf->reset)
		ldev->ops->update_vsi_filter(ldev, ICE_RDMA_FILTER_BOTH, false);
	if (rf->reset && try_module_get(THIS_MODULE)) {
		struct reset_work *work;

		work = kzalloc(sizeof(*work), GFP_ATOMIC);
		if (!work) {
			module_put(THIS_MODULE);
			dev_warn(idev_to_dev(&rf->sc_dev),
				 "IRDMA reset recovery failed. Please rmmod/modprobe irdma.\n");
			return;
		}
		work->rst_to = rf->rst_to;
		work->pdev = rf->hdl->pdev;
		INIT_DELAYED_WORK(&work->work, irdma_reset_task);
		irdma_remove_dev(rf->hdl->pdev);
		schedule_delayed_work(&work->work, work->rst_to * HZ);
	}

	pr_debug("INIT: Gen2 VSI[%d] close complete ldev=%p\n",
		 ldev->pf_vsi_num, ldev);
}

/**
 * irdma_remove_dev - GEN_2 device remove()
 * @pdev: platform device
 *
 * Called on module unload.
 */
int irdma_remove_dev(struct platform_device *pdev)
{
	struct ice_peer_dev_platform_data *pdata =
		dev_get_platdata(&pdev->dev);
	struct ice_peer_dev *ldev = pdata->peer_dev;
	struct irdma_handler *hdl;

	hdl = irdma_find_handler(ldev->pdev);
	if (!hdl)
		return 0;

	ldev->ops->peer_unregister(ldev);

	irdma_deinit_rf(&hdl->rf);

	/* TODO: Notify of cleanup completion (see: ig3rdma_if.c).
	 *
	 * ldev->ops->peer_unregister_finished(ldev);
	 */
	pr_debug("INIT: Gen2 device remove success ldev=%p\n", ldev);

	return 0;
}

static const struct ice_peer_ops irdma_peer_ops = {
	.close = irdma_close,
	.event_handler = irdma_event_handler,
	.open = irdma_open,
	.prep_tc_change = irdma_prep_tc_change,
};

static struct ice_peer_drv irdma_peer_drv = {
	.driver_id = ICE_PEER_RDMA_DRIVER,
	.name = KBUILD_MODNAME,
	.ver.major = ICE_PEER_MAJOR_VER,
	.ver.minor = ICE_PEER_MINOR_VER,
};

/**
 * icrdma_request_reset - Request a reset
 * @rf: RDMA PCI function
 */
static void icrdma_request_reset(struct irdma_pci_f *rf)
{
	struct ice_peer_dev *ldev = rf->ldev.if_ldev;

	dev_warn(idev_to_dev(&rf->sc_dev), "Requesting a reset\n");
	ldev->ops->request_reset(ldev, ICE_PEER_PFR);
}

/**
 * icrdma_cfg_dev_attr - update GEN_2 device attributes
 * @dev: device pointer
 *
 * update device attributes from user-configurable settings such
 * as fragment count limit
 */
static void icrdma_cfg_dev_attr(struct irdma_sc_dev *dev)
{
#ifdef DEVLINK_SUPPORTED
	struct irdma_pci_f *rf = container_of(dev, struct irdma_pci_f, sc_dev);
	struct irdma_dl_priv *dl_priv =
				platform_get_drvdata(rf->hdl->pdev);

	if (irdma_set_attr_from_fragcnt(dev, dl_priv->fragcnt_limit))
		dev_warn(idev_to_dev(dev),
		         "device limit update failed for fragment count %d\n",
		         dl_priv->fragcnt_limit);
#endif
}

/**
 * irdma_probe_dev - GEN_2 device probe()
 * @pdev: platform device
 *
 * Create device resources, set up queues, pble and hmc objects.
 * Return 0 if successful, otherwise return error
 */
int irdma_probe_dev(struct platform_device *pdev)
{
	struct ice_peer_dev_platform_data *pdata =
		dev_get_platdata(&pdev->dev);
	struct ice_peer_dev *ldev = pdata->peer_dev;
	struct irdma_handler *hdl;
	struct irdma_pci_f *rf;
	struct irdma_sc_dev *dev;
	struct irdma_priv_ldev *pldev;
	int err;

	pr_info("probe: ldev=%p, ldev->dev.pdev.bus->number=%d, ldev->netdev=%p\n",
		ldev, ldev->pdev->bus->number, ldev->netdev);
	if (ldev->ver.major != ICE_PEER_MAJOR_VER ||
	    ldev->ver.minor != ICE_PEER_MINOR_VER) {
		pr_err("version mismatch:\n");
		pr_err("expected major ver %d, caller specified major ver %d\n",
		       ICE_PEER_MAJOR_VER, ldev->ver.major);
		pr_err("expected minor ver %d, caller specified minor ver %d\n",
		       ICE_PEER_MINOR_VER, ldev->ver.minor);
		return -EINVAL;
	}

	hdl = irdma_find_handler(ldev->pdev);
	if (hdl)
		return -EBUSY;

	hdl = kzalloc(sizeof(*hdl), GFP_KERNEL);
	if (!hdl)
		return -ENOMEM;

	rf = &hdl->rf;
	pldev = &rf->ldev;
	hdl->ldev = pldev;
	hdl->pdev = pdev;
	rf->hdl = hdl;
	dev = &rf->sc_dev;
	dev->back_dev = rf;
	rf->gen_ops.init_hw = icrdma_init_hw;
	rf->gen_ops.request_reset = icrdma_request_reset;
	rf->gen_ops.register_qset = irdma_lan_register_qset;
	rf->gen_ops.unregister_qset = irdma_lan_unregister_qset;
	rf->check_fc = irdma_check_fc_for_qp;
	pldev->if_ldev = ldev;
	rf->rdma_ver = IRDMA_GEN_2;
	irdma_init_rf_config_params(rf);
	dev->pci_rev = ldev->pdev->revision;
	rf->default_vsi.vsi_idx = ldev->pf_vsi_num;
	/* save information from ldev to priv_ldev*/
	pldev->fn_num = PCI_FUNC(ldev->pdev->devfn);
	rf->hw.hw_addr = ldev->hw_addr;
	rf->pcidev = ldev->pdev;
	rf->netdev = ldev->netdev;
	pldev->ftype = ldev->ftype;
	pldev->msix_count = ldev->msix_count;
	pldev->msix_entries = ldev->msix_entries;
	irdma_add_handler(hdl);
	if (irdma_ctrl_init_hw(rf)) {
		err = -EIO;
		goto err_ctrl_init;
	}
	icrdma_cfg_dev_attr(&rf->sc_dev);
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_pf_init(hdl);
#endif
	ldev->peer_ops = &irdma_peer_ops;
	ldev->peer_drv = &irdma_peer_drv;
	err = ldev->ops->peer_register(ldev);
	if (err)
		goto err_peer_reg;

	irdma_dbg(dev, "INIT: Gen2 device probe success ldev=%p\n", ldev);

	return 0;

err_peer_reg:
#ifdef CONFIG_DEBUG_FS
	irdma_dbg_pf_exit(rf->hdl);
#endif
	irdma_ctrl_deinit_hw(rf);
err_ctrl_init:
	irdma_del_handler(rf->hdl);
	kfree(rf->hdl);

	return err;
}

/*
 * irdma_lan_vsi_ready - check to see if lan reset done
 * @pdev: platform device
 */
bool irdma_lan_vsi_ready(struct platform_device *pdev)
{
	struct ice_peer_dev_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct ice_peer_dev *ldev = pdata->peer_dev;

	return ldev->ops->is_vsi_ready(ldev) ? true : false;
}

/**
 * irdma_reset_task: worker for reset recovery
 * @work: work_struct pointer
 */
void irdma_reset_task(struct work_struct *work)
{
	struct reset_work *rst_work = container_of(to_delayed_work(work),
						   struct reset_work, work);
	struct platform_device *pdev = rst_work->pdev;

	/* Reset Recovery */
	if (!irdma_lan_vsi_ready(pdev))
		goto reschd;

	irdma_probe_dev(pdev);
	kfree(rst_work);
	module_put(THIS_MODULE);
	return;

reschd:
	if (!rst_work->rst_to) {
		pr_err("RF rebuild after reset timed out\n");
		kfree(rst_work);
		module_put(THIS_MODULE);
	} else {
		schedule_delayed_work(&rst_work->work, --rst_work->rst_to * HZ);
	}
}
