/**************************************************************************

Copyright (c) 2016 - 2020, Intel Corporation. All Rights Reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/

#ifndef EEPROM_CONFIG_H_
#define EEPROM_CONFIG_H_

#include "Ice.h"

/* Typedefs */
enum ValidMinSrev {NoneValid, NvmValid, OromValid, BothValid};

/* Defines */

#define SWAP_DWORD_WORDS(a) ((UINT32)(((a) & 0x0000FFFF) << 16) | (((a) & 0xFFFF0000) >> 16))

// EEPROM capabilities word
#define EEPROM_CAPABILITIES_WORD 0x33
#define EEPROM_CAPABILITIES_SIG  0x4000

// EEPROM size
#define EEPROM_SIZE_32K_WORDS    32768

// EMP Settings in Shared RAM mapped area
#define NVM_EMP_SR_SETTINGS_MODULE_PTR            0x48

#define ICE_VF_PER_DEVICE_MAX                 256
#define MAC_WORDS_LEN                         4
#define PF_MAC_ADDRESS_MODULE 0x10F
#define TLV_ID_CURRENT_LLDP 0x129
#define TLV_ID_MIN_SREV     0x130
#define NVM_MIN_SREV_VALID  BIT(0)
#define OROM_MIN_SREV_VALID BIT(1)

#define RDMA_ALTRAM_CFG               19
#define RDMA_ALTRAM_CFG_DIS           0
#define RDMA_ALTRAM_CFG_DIS_OVERRIDE  1
#define RDMA_ALTRAM_CFG_ENA           2
#define RDMA_ALTRAM_CFG_ENA_OVERRIDE  3

#define RDMA_CTRL_TLV                   0x135
#define RDMA_CTRL_TLV_TOPO              BIT (0)
#define RDMA_CTRL_TLV_LOW_OVERRIDE_ENA  BIT (1)
#define RDMA_CTRL_TLV_LOW_OVERRIDE_VAL  BIT (2)
#define RDMA_CTRL_TLV_HI_OVERRIDE_ENA   BIT (3)
#define RDMA_CTRL_TLV_HI_OVERRIDE_VAL   BIT (4)

#define ALTRAM_BIOS_MODE 0
#define ALTRAM_UEFI_MODE 1

#define DEFAULT_OVERRIDE_MASK_TLV           0x134


/* Structs */
#pragma pack(1)
typedef struct {
  UINT8 MediaDetectionMode              : 1;
  UINT8 EpctMediaDetectionChangeability : 1;
  UINT8 PortDisable                     : 1;
  UINT8 OverrideEnable                  : 1;
  UINT8 DisableAutomaticLinkOnStartup   : 1;
  UINT8 EeeEnable                       : 1;
  UINT8 Reserved1                       : 2;
  UINT8 PauseAbility                    : 2;
  UINT8 Reserved2                       : 4;
  UINT8 LesmEnable                      : 1;
  UINT8 AutoFecEnable                   : 1;
  UINT8 SetPhyConfigByte22;
  UINT8 Reserved3;
  UINT64 PhyTypesLow;
  UINT64 PhyTypesHigh;
} DEFAULT_OVERRIDE_MASK_S;
#pragma pack()

/* Interface function declarations */

/** Write SR buffer using shared code implementation.

   @param[in]   AdapterInfo    Points to the driver information
   @param[in]   ModulePointer  Pointer to module in words with respect to NVM beginning
   @param[in]   Offset         offset in words from module start
   @param[in]   Words          Number of words to write
   @param[in]   Data           Pointer to location with data to be written

   @retval    EFI_SUCCESS        Buffer successfully written
   @retval    EFI_ACCESS_DENIED  Access to desired NVM memory range is denied
   @retval    EFI_DEVICE_ERROR   Failed to write buffer
   @retval    EFI_DEVICE_ERROR   Waiting for ARQ response timeout
**/
EFI_STATUS
IceWriteNvmBuffer (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT8        ModulePointer,
  IN UINT32       Offset,
  IN UINT16       Words,
  IN VOID        *Data
  );

/** Writes data buffer to nvm using __i40e_write_nvm_buffer shared code function.

   Function works around the situation when the buffer spreads over two sectors.
   The entire buffer must be located inside the Shared RAM.

   @param[in]   AdapterInfo   Points to the driver information
   @param[in]   Offset        Buffer offset from the start of NVM
   @param[in]   Words         Number of words to write
   @param[in]   Data          Pointer to location with data to be written

   @retval   EFI_SUCCESS       NVM buffer written successfully
   @retval   EFI_DEVICE_ERROR  Failed to write buffer (or either of the sectors)
**/
EFI_STATUS
IceWriteNvmBufferExt (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT32       Offset,
  IN UINT16       Words,
  IN VOID        *Data
  );

/** Gets lan speed setting for adapter

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval     LINK_SPEED_AUTO_NEG  value allowing operation with the highest
                                    possibe speed
**/
UINTN
EepromGetLanSpeedStatus (
  UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Sets lan speed for adapter (unsupported)

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[in]   LanSpeed         Lan speed setting

   @retval      EFI_SUCCESS   Always returned
**/
EFI_STATUS
EepromSetLanSpeed (
  UNDI_PRIVATE_DATA *UndiPrivateData,
  UINT8              LanSpeed
  );

/** Restores the factory default MAC address.

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval      EFI_SUCCESS       New MAC address set successfully
   @retval      EFI_SUCCESS       Setting unsupported by SW
   @retval      EFI_DEVICE_ERROR  Failed to write factory default
                                  MAC value to alt. RAM
**/
EFI_STATUS
EepromMacAddressDefault (
  IN UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Does nothing. Necessary for UndiCommon to compile

   @param[in]   UndiPrivateData  Pointer to driver private data structure

   @retval      EFI_SUCCESS      Always returned
**/
EFI_STATUS
FixFwsm31Bit (
  IN UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Gets alternate and factory MAC addresses for PF0

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  AlternateMacAddress  Pointer to buffer for resulting alternate
                                     MAC address
   @param[out]  FactoryMacAddress    Pointer to buffer for resulting factory
                                     MAC address

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Pointer to EMP SR module is not initialized
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
   @retval      EFI_DEVICE_ERROR  Failed to read PF MAC addresses pointer
   @retval      EFI_DEVICE_ERROR  Failed to read factory MAC address
**/
EFI_STATUS
EepromPf0MacAddressGet (
  IN UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16           *AlternateMacAddress,
  OUT UINT16           *FactoryMacAddress
  );

/** Programs the port with an alternate MAC address, and backs up the factory default
   MAC address.

   @param[in]   UndiPrivateData   Pointer to driver private data structure
   @param[in]   NewMacAddress   Value to set the MAC address to

   @retval      EFI_SUCCESS       New MAC address set successfully
   @retval      EFI_SUCCESS       Setting unsupported by SW
   @retval      EFI_DEVICE_ERROR  Failed to write new MAC value to alt. RAM
**/
EFI_STATUS
EepromMacAddressSet (
  IN UNDI_PRIVATE_DATA *UndiPrivateData,
  IN UINT16            *MacAddress
  );

/** Reads the currently assigned MAC address and factory default MAC address for specified PF.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[in]   PhysicalFunction     Number of PF to read the MAC Addresses of
   @param[out]  FactoryMacAddress    Factory default MAC address of the adapter
   @param[out]  AlternateMacAddress  CLP Assigned MAC address of the adapter,
                                     or the factory MAC address if an alternate
                                     MAC address has not been assigned.

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Pointer to EMP SR module is not initialized
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
   @retval      EFI_DEVICE_ERROR  Failed to read PF MAC addresses pointer
   @retval      EFI_DEVICE_ERROR  Failed to read factory MAC address
**/
EFI_STATUS
EepromMacAddressGetForPf (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  UINT8             PhysicalFunction,
  OUT UINT16            *FactoryMacAddress,
  OUT UINT16            *AlternateMacAddress
  );

/** Reads the currently assigned MAC address and factory default MAC address.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  FactoryMacAddress    Factory default MAC address of the adapter
   @param[out]  AlternateMacAddress  CLP Assigned MAC address of the adapter,
                                     or the factory MAC address if an alternate
                                     MAC address has not been assigned.

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Pointer to EMP SR module is not initialized
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
   @retval      EFI_DEVICE_ERROR  Failed to read PF MAC addresses pointer
   @retval      EFI_DEVICE_ERROR  Failed to read factory MAC address
**/
EFI_STATUS
EepromMacAddressGet (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16            *FactoryMacAddress,
  OUT UINT16            *AlternateMacAddress
  );

/** Returns EEPROM capabilities word (0x33) for current adapter

   @param[in]   UndiPrivateData    Points to the driver instance private data
   @param[out]  CapabilitiesWord   EEPROM capabilities word (0x33) for current adapter

   @retval   EFI_SUCCESS       Capabilities word successfully read
   @retval   EFI_DEVICE_ERROR  Failed to read capabilities word
**/
EFI_STATUS
EepromGetCapabilitiesWord (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16            *CapabilitiesWord
  );

/** Updates NVM checksum.

   @param[in]   UndiPrivateData   Pointer to driver private data structure
   @param[in]   EmprRequested     Indicates whether the next PCIR will  be
                                  elevated to EMPR

   @retval      EFI_SUCCESS       Checksum successfully updated
   @retval      EFI_DEVICE_ERROR  Failed to update NVM checksum
**/
EFI_STATUS
EepromUpdateChecksumIce (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  BOOLEAN           EmprRequested
  );

/** Reads PBA string from NVM

   @param[in]   AdapterInfo    Points to the driver information
   @param[out]  PbaNumber      Pointer to buffer for PBA string
   @param[in]   PbaNumberSize  Size of PBA string

   @retval   EFI_SUCCESS            PBA string successfully read
   @retval   EFI_INVALID_PARAMETER  PbaNumber is NULL
   @retval   EFI_DEVICE_ERROR       Failed to read PBA flags word
   @retval   EFI_DEVICE_ERROR       Failed to read PBA module pointer
   @retval   EFI_INVALID_PARAMETER  PbaNumberSize is lower than 11
   @retval   EFI_DEVICE_ERROR       Failed to read PBA number word pointer
   @retval   EFI_DEVICE_ERROR       Returned PBA length is 0xFFFF or 0x0
   @retval   EFI_INVALID_PARAMETER  PbaNumberSize is not big enough
   @retval   EFI_DEVICE_ERROR       Failed to read PBA number word
**/
EFI_STATUS
ReadPbaString (
  IN  DRIVER_DATA *AdapterInfo,
  OUT UINT8       *PbaNumber,
  IN  UINT32       PbaNumberSize
  );


/** Checks if LLDP Admin Status is supported for Hii.

   @param[in]   UndiPrivateData  Pointer to driver private data structure.

   @retval      TRUE   Firmware supports LLDP Admin.
   @retval      FALSE  Firmware does not support LLDP Admin.
**/
BOOLEAN
IsLLDPSupported (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Read current LLDP Admin Status.

   @param[in]       UndiPrivateData  Pointer to driver private data structure.
   @param[in,out]   AdminStatus      Pointer to variable which will store read LLDP Admin status.

   @retval      EFI_SUCCESS            LLDP Admin read successfully.
   @retval      EFI_DEVICE_ERROR       Failed to read LLDP Admin.
   @retval      EFI_INVALID_PARAMETER  Wrong value read from NVM.
**/
EFI_STATUS
ReadLLDPAdminStatus (
  IN      UNDI_PRIVATE_DATA *UndiPrivateData,
  IN OUT  UINT8             *AdminStatus
  );

/** Write LLDP Admin Status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[in]   Start  If True only LLDP start is executed in other case LLDP stop.

   @retval      EFI_SUCCESS         LLDP Admin Status written successfully
   @retval      EFI_DEVICE_ERROR    Failed to write LLDP Admin Status
**/
EFI_STATUS
WriteLLDPAdminStatus (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  BOOLEAN Start
  );

/** Get Default/Restore LLDP Admin Status.

   @param[in]       UndiPrivateData  Pointer to driver private data structure.
   @param[in,out]   DefaultValue  Pointer to variable which should store default value for LLDP Agent.
                                  If NULL restore Factory Setting.
   @retval      EFI_SUCCESS  LLDP Admin get default/restore successful.
   @retval      EFI_DEVICE_ERROR  Failed to get default/restore of LLDP Admin.
   @retval      EFI_INVALID_PARAMETER  Value out of range
**/
EFI_STATUS
DefaultLLDPAdminStatus (
  IN      UNDI_PRIVATE_DATA *UndiPrivateData,
  IN OUT  UINT8             *DefaultValue
  );

#endif /* EEPROM_CONFIG_H_ */
