/**************************************************************************

Copyright (c) 2016 - 2020, Intel Corporation. All Rights Reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/
#include "EepromConfig.h"
#include "Ice.h"
#include "ice_dcb.h"

/* Function definitions */

/** Write SR buffer using shared code implementation.

   @param[in]   AdapterInfo    Points to the driver information
   @param[in]   ModulePointer  Pointer to module in words with respect to NVM beginning
   @param[in]   Offset         offset in words from module start
   @param[in]   Words          Number of words to write
   @param[in]   Data           Pointer to location with data to be written

   @retval    EFI_SUCCESS        Buffer successfully written
   @retval    EFI_ACCESS_DENIED  Access to desired NVM memory range is denied
   @retval    EFI_DEVICE_ERROR   Failed to write buffer
   @retval    EFI_DEVICE_ERROR   Waiting for ARQ response timeout
**/
EFI_STATUS
IceWriteNvmBuffer (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT8        ModulePointer,
  IN UINT32       Offset,
  IN UINT16       Words,
  IN VOID        *Data
  )
{
  enum ice_status            IceStatus = ICE_SUCCESS;
  EFI_STATUS                 Status = EFI_SUCCESS;

  Status = ClearAdminReceiveQueue (AdapterInfo);
  if (EFI_ERROR (Status)) {
    return Status;
  }

  IceStatus = ice_acquire_nvm (&AdapterInfo->Hw, ICE_RES_WRITE);
  if (IceStatus != ICE_SUCCESS) {
    DEBUGPRINT (CRITICAL, ("ice_acquire_nvm returned %d\n", IceStatus));
    return EFI_DEVICE_ERROR;
  }

  IceStatus = __ice_write_sr_buf (
                &AdapterInfo->Hw,
                ModulePointer + Offset,
                Words,
                (UINT16 *) Data
              );

  ice_release_nvm (&AdapterInfo->Hw);

  if (IceStatus !=  ICE_SUCCESS) {
    DEBUGPRINT (
      CRITICAL, ("__ice_write_sr_buf (%d, %d, %x) returned: status %d, asq stat %d\n",
      ModulePointer + Offset, Words, *((UINT16 *) Data), IceStatus, AdapterInfo->Hw.adminq.sq_last_status)
    );

    if (AdapterInfo->Hw.adminq.sq_last_status == ICE_AQ_RC_EPERM) {

      // Need to detect attempts to write RO area
      DEBUGPRINT (IMAGE, ("__ice_write_sr_buf returned EPERM\n"));
      return EFI_ACCESS_DENIED;
    } else {
      return EFI_DEVICE_ERROR;
    }
  }

  DEBUGPRINT (WOL, ("__ice_write_sr_buf returned ICE_SUCCESS\n"));

  Status = AwaitReceiveQueueEvent (
             AdapterInfo,
             ice_aqc_opc_nvm_write,
             1000
           );
  if (EFI_ERROR (Status)) {
    return Status;
  }

  return EFI_SUCCESS;
}

/** Writes data buffer to nvm using IceWriteNvmBuffer shared code function.

   Function works around the situation when the buffer spreads over two sectors.
   The entire buffer must be located inside the Shared RAM.

   @param[in]   AdapterInfo   Points to the driver information
   @param[in]   Offset        Buffer offset from the start of NVM
   @param[in]   Words         Number of words to write
   @param[in]   Data          Pointer to location with data to be written

   @retval   EFI_SUCCESS       NVM buffer written successfully
   @retval   EFI_DEVICE_ERROR  Failed to write buffer (or either of the sectors)
**/
EFI_STATUS
IceWriteNvmBufferExt (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT32       Offset,
  IN UINT16       Words,
  IN VOID        *Data
  )
{
  UINT16     SectorStart;
  UINT16     Margin;
  UINT16     Words1;
  UINT16     Words2;
  EFI_STATUS Status;

  DEBUGFUNC ("__ice_write_nvm_buffer");

  // Check if the buffer spreads over two sectors. Then we need to split
  // the buffer into two adjacent buffers, one for each sector and write them separatelly.
  SectorStart = (Offset / ICE_SR_SECTOR_SIZE_IN_WORDS) * ICE_SR_SECTOR_SIZE_IN_WORDS;
  Margin = (SectorStart + ICE_SR_SECTOR_SIZE_IN_WORDS) - Offset;
  if (Words > Margin) {
    Words1 = Margin;
    Words2 = Words - Margin;
  } else {
    Words1 = Words;
    Words2 = 0;
  }

  Status = IceWriteNvmBuffer (AdapterInfo, 0, Offset, Words1, Data);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("IceWriteNvmBuffer(%x) returned %r\n", Offset, Status));
    return EFI_DEVICE_ERROR;
  }
  if (Words2 > 0) {

    // Write the remaining part of the input data to the second sector
    Status = IceWriteNvmBuffer (
               AdapterInfo,
               0,
               SectorStart + ICE_SR_SECTOR_SIZE_IN_WORDS,
               Words2,
               (UINT16 *) Data + Words1
             );
    if (EFI_ERROR (Status)) {
      DEBUGPRINT (CRITICAL, ("IceWriteNvmBuffer returned %r\n", Status));
      return EFI_DEVICE_ERROR;
    }
  }
  return EFI_SUCCESS;
}

/** Gets lan speed setting for adapter

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval     LINK_SPEED_AUTO_NEG  value allowing operation with the highest
                                    possibe speed
**/
UINTN
EepromGetLanSpeedStatus (
  IN UNDI_PRIVATE_DATA *UndiPrivateData
  )
{
  //  Speed settings are currently not supported for 10 Gig driver. It's always set to autoneg to
  //  allow operation with the highest possible speed
  return LINK_SPEED_AUTO_NEG;
}

/** Sets lan speed for adapter (unsupported)

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[in]   LanSpeed         Lan speed setting

   @retval      EFI_SUCCESS   Always returned
**/
EFI_STATUS
EepromSetLanSpeed (
  UNDI_PRIVATE_DATA *UndiPrivateData,
  UINT8              LanSpeed
  )
{
  return EFI_SUCCESS;
}

/** Gets alternate and factory MAC addresses for PF0

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  AlternateMacAddress  Pointer to buffer for resulting alternate
                                     MAC address
   @param[out]  FactoryMacAddress    Pointer to buffer for resulting factory
                                     MAC address

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Pointer to EMP SR module is not initialized
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
   @retval      EFI_DEVICE_ERROR  Failed to read PF MAC addresses pointer
   @retval      EFI_DEVICE_ERROR  Failed to read factory MAC address
**/
EFI_STATUS
EepromPf0MacAddressGet (
  IN UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16           *AlternateMacAddress,
  OUT UINT16           *FactoryMacAddress
  )
{
  return EepromMacAddressGetForPf (UndiPrivateData, 0, FactoryMacAddress, AlternateMacAddress);
}

/** Reads the currently assigned MAC address and factory default MAC address for specified PF.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[in]   PhysicalFunction     Number of PF to read the MAC Addresses of
   @param[out]  FactoryMacAddress    Factory default MAC address of the adapter
   @param[out]  AlternateMacAddress  CLP Assigned MAC address of the adapter,
                                     or the factory MAC address if an alternate
                                     MAC address has not been assigned.

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Pointer to EMP SR module is not initialized
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
   @retval      EFI_DEVICE_ERROR  Failed to read PF MAC addresses pointer
   @retval      EFI_DEVICE_ERROR  Failed to read factory MAC address
**/
EFI_STATUS
EepromMacAddressGetForPf (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  UINT8             PhysicalFunction,
  OUT UINT16            *FactoryMacAddress,
  OUT UINT16            *AlternateMacAddress
  )
{
  struct ice_hw   *Hw = &UndiPrivateData->NicInfo.Hw;
  enum ice_status IceStatus;
  EFI_STATUS      Status = EFI_SUCCESS;
  UINT32          AltRamBuffer[2];
  UINT16          PermMacBuff[MAC_WORDS_LEN];
  UINT16          PfOffset = SKIP_TLV_LENGTH + (MAC_WORDS_LEN * PhysicalFunction);

  Status = ReadTlv (UndiPrivateData, PF_MAC_ADDRESS_MODULE, PfOffset, sizeof (PermMacBuff), &PermMacBuff);
  if (EFI_ERROR (Status)) {
    DEBUGPRINT (CRITICAL, ("Permanent MAC Address read failed with status %d\n", Status));
    return Status;
  }

  if (FactoryMacAddress != NULL) {
    CopyMem (FactoryMacAddress, &PermMacBuff, ETH_ALEN);
  }

  IceStatus = ice_aq_alternate_read (
                Hw,
                ICE_ALT_RAM_LAN_MAC_ADDRESS_LOW (PhysicalFunction),
                &AltRamBuffer[0],
                ICE_ALT_RAM_LAN_MAC_ADDRESS_HIGH (PhysicalFunction),
                &AltRamBuffer[1]
              );

  if (IceStatus != ICE_SUCCESS) {
    DEBUGPRINT (CRITICAL, ("ice_aq_alternate_read error\n"));
    return EFI_DEVICE_ERROR;
  }

  // Check if this Alternate RAM entry is valid and then use it,
  // otherwise return zeros
  if ((AltRamBuffer[1] & ALT_RAM_VALID_PARAM_BIT_MASK) != 0) {
    AlternateMacAddress[0] = SwapBytes16 ((UINT16) (AltRamBuffer[1] & 0x0000FFFF));
    AlternateMacAddress[1] = SwapBytes16 ((UINT16) ((AltRamBuffer[0] & 0xFFFF0000) >> 16));
    AlternateMacAddress[2] = SwapBytes16 ((UINT16) (AltRamBuffer[0] & 0x0000FFFF));
  } else {
    AlternateMacAddress[0] = 0;
    AlternateMacAddress[1] = 0;
    AlternateMacAddress[2] = 0;
  }

  return Status;
}

/** Reads the currently assigned MAC address and factory default MAC address.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  FactoryMacAddress    Factory default MAC address of the adapter
   @param[out]  AlternateMacAddress  CLP Assigned MAC address of the adapter,
                                     or the factory MAC address if an alternate
                                     MAC address has not been assigned.

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Pointer to EMP SR module is not initialized
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
   @retval      EFI_DEVICE_ERROR  Failed to read PF MAC addresses pointer
   @retval      EFI_DEVICE_ERROR  Failed to read factory MAC address
**/
EFI_STATUS
EepromMacAddressGet (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16            *FactoryMacAddress,
  OUT UINT16            *AlternateMacAddress
  )
{
  return EepromMacAddressGetForPf (
           UndiPrivateData,
           UndiPrivateData->NicInfo.Hw.pf_id,
           FactoryMacAddress,
           AlternateMacAddress
           );
}

/** Does nothing. Necessary for UndiCommon to compile

   @param[in]   UndiPrivateData  Pointer to driver private data structure

   @retval      EFI_SUCCESS      Always returned
**/
EFI_STATUS
FixFwsm31Bit (
  IN UNDI_PRIVATE_DATA *UndiPrivateData
  )
{
  return EFI_SUCCESS;
}

/** Programs the port with an alternate MAC address.

   @param[in]   UndiPrivateData   Pointer to driver private data structure
   @param[in]   NewMacAddress   Value to set the MAC address to

   @retval      EFI_SUCCESS       New MAC address set successfully
   @retval      EFI_SUCCESS       Setting unsupported by SW
   @retval      EFI_DEVICE_ERROR  Failed to write new MAC value to alt. RAM
**/
EFI_STATUS
EepromMacAddressSet (
  IN UNDI_PRIVATE_DATA *UndiPrivateData,
  IN UINT16            *NewMacAddress
  )
{
  enum  ice_status IceStatus;
  UINT32           AltRamBuffer[2];

  AltRamBuffer[0] = SwapBytes16 (NewMacAddress[2]) + ((UINT32) SwapBytes16 (NewMacAddress[1]) << 16);
  AltRamBuffer[1] = SwapBytes16 (NewMacAddress[0]);

  AltRamBuffer[1] |= ALT_RAM_VALID_PARAM_BIT_MASK;

  IceStatus = ice_aq_alternate_write (
                &UndiPrivateData->NicInfo.Hw,
                ICE_ALT_RAM_LAN_MAC_ADDRESS_LOW (UndiPrivateData->NicInfo.Hw.pf_id),
                AltRamBuffer[0],
                ICE_ALT_RAM_LAN_MAC_ADDRESS_HIGH (UndiPrivateData->NicInfo.Hw.pf_id),
                AltRamBuffer[1]
              );
  if (IceStatus != ICE_SUCCESS) {
    return EFI_DEVICE_ERROR;
  }

  return EFI_SUCCESS;
}

/** Restores the factory default MAC address.

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval      EFI_SUCCESS       New MAC address set successfully
   @retval      EFI_DEVICE_ERROR  Failed to restore factory default MAC
**/
EFI_STATUS
EepromMacAddressDefault (
  IN UNDI_PRIVATE_DATA *UndiPrivateData
  )
{
  enum  ice_status IceStatus;
  UINT32           AltRamBuffer[2];

  // Invalidate Alternate RAM entry by writing zeros
  AltRamBuffer[0] = 0;
  AltRamBuffer[1] = 0;

  IceStatus = ice_aq_alternate_write (
                &UndiPrivateData->NicInfo.Hw,
                ICE_ALT_RAM_LAN_MAC_ADDRESS_LOW (UndiPrivateData->NicInfo.Hw.pf_id),
                AltRamBuffer[0],
                ICE_ALT_RAM_LAN_MAC_ADDRESS_HIGH (UndiPrivateData->NicInfo.Hw.pf_id),
                AltRamBuffer[1]
              );
  if (IceStatus != ICE_SUCCESS) {
    return EFI_DEVICE_ERROR;
  }

  return EFI_SUCCESS;
}

/** Returns EEPROM capabilities word (0x33) for current adapter

   @param[in]   UndiPrivateData    Points to the driver instance private data
   @param[out]  CapabilitiesWord   EEPROM capabilities word (0x33) for current adapter

   @retval   EFI_SUCCESS       Capabilities word successfully read
   @retval   EFI_DEVICE_ERROR  Failed to read capabilities word
**/
EFI_STATUS
EepromGetCapabilitiesWord (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16            *CapabilitiesWord
  )
{
  enum ice_status IceStatus;
  UINT16          Word;

  IceStatus = ice_read_sr_word (
                &UndiPrivateData->NicInfo.Hw,
                EEPROM_CAPABILITIES_WORD,
                &Word
              );
  if (IceStatus != ICE_SUCCESS) {
    return EFI_DEVICE_ERROR;
  }

  Word &= ~EEPROM_CAPABILITIES_SIG;
  *CapabilitiesWord = Word;

  return EFI_SUCCESS;
}

/** Updates NVM checksum.

   @param[in]   UndiPrivateData   Pointer to driver private data structure
   @param[in]   EmprRequested     Indicates whether the next PCIR will  be
                                  elevated to EMPR

   @retval      EFI_SUCCESS       Checksum successfully updated
   @retval      EFI_DEVICE_ERROR  Failed to update NVM checksum
**/
EFI_STATUS
EepromUpdateChecksumIce (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  BOOLEAN           EmprRequested
  )
{
  enum ice_status IceStatus;
  EFI_STATUS      Status;
  struct ice_hw   *Hw;
  UINT8           CmdFlags2;

  Hw = &UndiPrivateData->NicInfo.Hw;

  Status = ClearAdminReceiveQueue (&UndiPrivateData->NicInfo);
  if (EFI_ERROR (Status)) {
    return Status;
  }

  // Software takes ownership over the NVM resource for activate
  IceStatus = ice_acquire_nvm (Hw, ICE_RES_WRITE);
  if (IceStatus != ICE_SUCCESS) {
    return EFI_DEVICE_ERROR;
  }

  // Determine whether EMPR is requested instead of the next PCIR
  // and set cmd_flags2 of AQ command accordingly
  CmdFlags2 = EmprRequested ? ICE_AQC_NVM_ACTIV_REQ_EMPR : 0;

  // Run NVM Activate command with no banks specified to perform SR dump
  IceStatus = ice_nvm_write_activate (Hw, 0, CmdFlags2);

  if (IceStatus != ICE_SUCCESS) {
    DEBUGPRINT (CRITICAL, ("ice_nvm_write_activate returned %d\n", IceStatus));
    Status = EFI_DEVICE_ERROR;
    goto ExitError;
  }

  Status = AwaitReceiveQueueEvent (
             &UndiPrivateData->NicInfo,
             ice_aqc_opc_nvm_write_activate,
             NVM_OPERATION_TIMEOUT_IN_1MS_UNITS
           );

ExitError:
  // Release NVM ownership
  ice_release_nvm (Hw);
  return Status;
}

/** Reads PBA string from NVM

   @param[in]   AdapterInfo    Points to the driver information
   @param[out]  PbaNumber      Pointer to buffer for PBA string
   @param[in]   PbaNumberSize  Size of PBA string

   @retval   EFI_SUCCESS            PBA string successfully read
   @retval   EFI_INVALID_PARAMETER  PbaNumber is NULL
   @retval   EFI_DEVICE_ERROR       Failed to read PBA flags word
   @retval   EFI_DEVICE_ERROR       Failed to read PBA module pointer
   @retval   EFI_INVALID_PARAMETER  PbaNumberSize is lower than 11
   @retval   EFI_DEVICE_ERROR       Failed to read PBA number word pointer
   @retval   EFI_DEVICE_ERROR       Returned PBA length is 0xFFFF or 0x0
   @retval   EFI_INVALID_PARAMETER  PbaNumberSize is not big enough
   @retval   EFI_DEVICE_ERROR       Failed to read PBA number word
**/
EFI_STATUS
ReadPbaString (
  IN  DRIVER_DATA *AdapterInfo,
  OUT UINT8       *PbaNumber,
  IN  UINT32       PbaNumberSize
  )
{
  enum ice_status Status;
  Status = ice_read_pba_string (&AdapterInfo->Hw, PbaNumber, PbaNumberSize);
  if (Status != ICE_SUCCESS) {
    DEBUGPRINT (HII, ("Error reading PBA string = %d\n", Status));
    return EFI_DEVICE_ERROR;
  } else {
    return EFI_SUCCESS;
  }
}


/** Checks if LLDP Admin Status is supported for Hii.

   @param[in]   UndiPrivateData  Pointer to driver private data structure.

   @retval      TRUE   Firmware supports LLDP Admin.
   @retval      FALSE  Firmware does not support LLDP Admin.
**/
BOOLEAN
IsLLDPSupported (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  )
{
  return TRUE;
}

/** Get current LLDP Admin Status per Lan Port.

   @param[in]   PortNumber  LAN port number for which conversion is done.
   @param[in]   RawToRead   Variable which we use to convert from.

   @retval      LLDP Admin Status for selected port.
**/
UINT8
GetLLDPAdminForPort (
  IN   UINT8    PortNumber,
  IN   UINT32   RawToRead
  )
{
  return (UINT8) ((RawToRead >> PortNumber * 4) & 0xF);
}

/** Convert raw LLDP Admin Status to expected value.

   @param[in]   PortNumber  LAN port number for which conversion is done.
   @param[in]   RawValue  Variable which needs to be converted.
   @param[out]  Value  Pointer to the location where the result of conversion should be stored.
**/
VOID
ConvertLLDPAdmin (
  IN      UINT8    PortNumber,
  IN      UINT32   RawValue,
  OUT     UINT8   *Value
  )
{
  UINT8            Temp;

  Temp = GetLLDPAdminForPort (PortNumber, RawValue);

  //Convert value to Enabled, Disabled or pass through.
  switch (Temp) {
    case 0x3:
    case 0x2:
    case 0x1:
      *Value = LLDP_ENABLE;
      break;
    case 0x0:
      *Value = LLDP_DISABLE;
      break;
    default:
      *Value = Temp;
      break;
  }
}

/** Read current LLDP Admin Status.

   @param[in]       UndiPrivateData  Pointer to driver private data structure.
   @param[in,out]   AdminStatus      Pointer to variable which will store read LLDP Admin status.

   @retval      EFI_SUCCESS            LLDP Admin read successfully.
   @retval      EFI_DEVICE_ERROR       Failed to read LLDP Admin.
   @retval      EFI_INVALID_PARAMETER  Wrong value read from NVM.
**/
EFI_STATUS
ReadLLDPAdminStatus (
  IN      UNDI_PRIVATE_DATA *UndiPrivateData,
  IN OUT  UINT8             *AdminStatus
  )
{

  EFI_STATUS           EfiStatus;
  UINT32               CurrentLLDP;
  UINT8                PhysicalPortNr;

  PhysicalPortNr = UndiPrivateData->NicInfo.PhysicalPortNumber;

  EfiStatus = ReadTlv (
                UndiPrivateData,
                TLV_ID_CURRENT_LLDP,
                SKIP_TLV_LENGTH,
                sizeof (CurrentLLDP),
                &CurrentLLDP
              );
  if (EFI_ERROR (EfiStatus)) {
    DEBUGPRINT (CRITICAL, ("ReadTlv returned %d\n", EfiStatus));
    return EFI_DEVICE_ERROR;
  }

  //Extract value for current LAN port
  ConvertLLDPAdmin (PhysicalPortNr, CurrentLLDP, AdminStatus);

  //If LLDP Admin Status is invalid = 0xF then read default value
  if (*AdminStatus == 0xF) {
    return DefaultLLDPAdminStatus (UndiPrivateData, AdminStatus);
  } else if (*AdminStatus > 3) {
    DEBUGPRINT (CRITICAL, ("Wrong LLDP value read from NVM\n"));
    return EFI_INVALID_PARAMETER;
  }

  return EFI_SUCCESS;
}

/** Write LLDP Admin Status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[in]   Start  If True only LLDP start is executed in other case LLDP stop.

   @retval      EFI_SUCCESS         LLDP Admin Status written successfully
   @retval      EFI_DEVICE_ERROR    Failed to write LLDP Admin Status
**/
EFI_STATUS
WriteLLDPAdminStatus (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  BOOLEAN            Start
  )
{
  enum ice_status_code IceStatus;

  if (Start) {
    IceStatus = ice_aq_start_lldp (&UndiPrivateData->NicInfo.Hw, TRUE, NULL);
    if (IceStatus != ICE_SUCCESS) {
      DEBUGPRINT (CRITICAL, ("ice_aq_start_lldp returned %d\n", IceStatus));
      return EFI_DEVICE_ERROR;
    }
  } else {
    IceStatus = ice_aq_stop_lldp (&UndiPrivateData->NicInfo.Hw, TRUE, TRUE, NULL);
    if (IceStatus != ICE_SUCCESS) {
      DEBUGPRINT (CRITICAL, ("ice_aq_stop_lldp returned %d\n", IceStatus));
      return EFI_DEVICE_ERROR;
    }
    IceStatus = ice_aq_set_dcb_parameters (&UndiPrivateData->NicInfo.Hw, TRUE, NULL);
    if (IceStatus != ICE_SUCCESS) {
      DEBUGPRINT (CRITICAL, ("ice_aq_set_dcb_parameters returned %d\n", IceStatus));
      return EFI_DEVICE_ERROR;
    }
  }

  return EFI_SUCCESS;
}

/** Read default LLDP Admin Status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure.
   @param[in]   OffsetInModule  Offset in words from module beginning to default LLDP Admin Status
   @param[out]  DefaultValue  Pointer to variable which should store default value for LLDP Agent.

   @retval      EFI_SUCCESS   LLDP Admin get default/restore successful.
   @retval      EFI_DEVICE_ERROR  Failed to get default/restore of LLDP Admin.
**/
EFI_STATUS
ReadDefaultLLDPAdminStatus (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  UINT16             OffsetInModule,
  OUT UINT16            *DefaultValue
  )
{
#define DEFAULT_LLDP_MODULE_TYPE_ID  0x0000
#define LLDP_CONF_POINTER_MOD_OFFSET 0x0023

  EFI_STATUS      EfiStatus;
  UINT16          DataPtr1;
  UINT16          DataPtr2;

  //First we need to get EMP SR Settings Pointer
  EfiStatus = ReadTlv (
                UndiPrivateData,
                DEFAULT_LLDP_MODULE_TYPE_ID,
                NVM_EMP_SR_SETTINGS_MODULE_PTR,
                sizeof (DataPtr1),
                &DataPtr1
              );
  if (EFI_ERROR (EfiStatus)) {
    goto ExitError;
  }

  //If MSB = 1 then offset is in 4KB sector units
  if ((DataPtr1 & 0x8000) != 0) {
    //To get offset in words we need to multiply it
    DataPtr1 = DataPtr1 * 0x800;
  }

  DataPtr1 += LLDP_CONF_POINTER_MOD_OFFSET;

  //Now we can read LLDP configuration pointer
  EfiStatus = ReadTlv (
                UndiPrivateData,
                DEFAULT_LLDP_MODULE_TYPE_ID,
                DataPtr1,
                sizeof (DataPtr2),
                &DataPtr2
              );
  if (EFI_ERROR (EfiStatus)) {
    goto ExitError;
  }

  //Finally we can read default LLDP value
  EfiStatus = ReadTlv (
                UndiPrivateData,
                DEFAULT_LLDP_MODULE_TYPE_ID,
                DataPtr1 + DataPtr2 + OffsetInModule,
                sizeof (*DefaultValue),
                DefaultValue
              );
  if (EFI_ERROR (EfiStatus)) {
    goto ExitError;
  }

  return EFI_SUCCESS;

  ExitError:
  DEBUGPRINT (CRITICAL, ("ReadTlv returned %d\n", EfiStatus));

  return EFI_DEVICE_ERROR;
}

/** Reset LLDP Admin Status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure.

   @retval      EFI_SUCCESS      LLDP Admin reset successful.
   @retval      EFI_DEVICE_ERROR     Failed to reset LLDP Admin.
**/
EFI_STATUS
ResetLLDPAdminStatus (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  )
{
  EFI_STATUS       EfiStatus;
  UINT8            PhysicalPortNr;
  UINT32           Mask;
  UINT32           Temp;

  PhysicalPortNr = UndiPrivateData->NicInfo.PhysicalPortNumber;

  //First we need to read current value
  EfiStatus = ReadTlv (
                UndiPrivateData,
                TLV_ID_CURRENT_LLDP,
                SKIP_TLV_LENGTH,
                sizeof (Temp),
                &Temp
              );
  if (EFI_ERROR (EfiStatus)) {
    DEBUGPRINT (CRITICAL, ("Failed to read current LLDP AdminStatus"));
    return EFI_DEVICE_ERROR;
  }

  //Then set 0xF for a corresponding port in variable, we want to change value only for a port
  Mask = 0xF << PhysicalPortNr * 4 ;
  Temp |= Mask;

  //Finally we can write entire double word to NVM
  EfiStatus = WriteTlv (
                UndiPrivateData,
                TLV_ID_CURRENT_LLDP,
                SKIP_TLV_LENGTH,
                sizeof (Temp),
                &Temp
              );
  if (EFI_ERROR (EfiStatus)) {
    DEBUGPRINT (CRITICAL, ("Failed to reset current LLDP AdminStatus"));
    return EFI_DEVICE_ERROR;
  }

  return EFI_SUCCESS;
}

/** Get Default/Restore LLDP Admin Status.

   @param[in]       UndiPrivateData  Pointer to driver private data structure.
   @param[in,out]   DefaultValue  Pointer to variable which should store default value for LLDP Agent.
                                  If NULL restore Factory Setting.
   @retval      EFI_SUCCESS  LLDP Admin get default/restore successful.
   @retval      EFI_DEVICE_ERROR  Failed to get default/restore of LLDP Admin.
   @retval      EFI_INVALID_PARAMETER  Value out of range
**/
EFI_STATUS
DefaultLLDPAdminStatus (
  IN      UNDI_PRIVATE_DATA *UndiPrivateData,
  IN OUT  UINT8             *DefaultValue
  )
{
  EFI_STATUS      EfiStatus;
  UINT32          Offset = 0x0001;
  UINT8           PhysicalPortNr;
  UINT16          Buffer;

  PhysicalPortNr = UndiPrivateData->NicInfo.PhysicalPortNumber;

  //When physical port greater than 3 we need to read second word from module
  if (PhysicalPortNr > 3) {
    PhysicalPortNr -= 4;
    Offset = 0x0002;
  }

  if (DefaultValue != NULL) {
    EfiStatus = ReadDefaultLLDPAdminStatus (UndiPrivateData, Offset, &Buffer);
    if (EFI_ERROR (EfiStatus)) {
      DEBUGPRINT (CRITICAL, ("ReadDefaultLLDPAdminStatus returned %d\n", EfiStatus));
      return EFI_DEVICE_ERROR;
    }
    ConvertLLDPAdmin (PhysicalPortNr, Buffer, DefaultValue);
    if (*DefaultValue > 3) {
      DEBUGPRINT (CRITICAL, ("Default value out of range\n"));
      return EFI_INVALID_PARAMETER;
    }
  } else {
    EfiStatus = ResetLLDPAdminStatus (UndiPrivateData);
    if (EFI_ERROR (EfiStatus)) {
      DEBUGPRINT (CRITICAL, ("ResetLLDPAdminStatus returned %d\n", EfiStatus));
      return EFI_DEVICE_ERROR;
    }
  }
  return EFI_SUCCESS;
}


