/* hwInterfaceExtended.h */

/***********************************************************************
*                    Copyright Aquantia Corporation
*                                Freeware
*
* $File: //depot/icm/proj/Atlantic/rev1.0/software/Software/Test/MBU-dependencies/hwInterface/include/hwInterfaceExtended.h $
*
* $Revision: #3 $
*
* $DateTime: 2016/09/21 08:49:55 $
*
* $Author: epomozov $
*
* $Label:  $
*
***********************************************************************/


/*! \file
    This file contains the extended functionality function prototypes for interfacing with system hardware boards.
    
*/




#ifndef HW_INTERFACE_EXTENDED_TOKEN
#define HW_INTERFACE_EXTENDED_TOKEN

#ifndef DLLEXPORT
#ifdef WIN32
#define DLLEXPORT __declspec(dllexport)
#else
#define DLLEXPORT
#endif
#endif

#ifdef __cplusplus
  extern "C" {
#endif

#include <stdio.h>
#include <stdint.h>

/*! \defgroup HW_Interface_Extended HW Interface Extended
This module contains the extended functionality function prototypes for interfacing with system hardware boards.
 @{
*/



#ifndef HWI_RUNTIME_DYNLINKING
/*! Returns the current default adapter number. */
DLLEXPORT uint32_t HW_GetDefaultAdapter(void);
#else
typedef uint32_t (__cdecl *HW_GetDefaultAdapter)();
#endif


#ifndef HWI_RUNTIME_DYNLINKING

/*! Explicitly designates adapterNumber as the current default adapter.  adapterNumber must be 
 * a valid index into the list of adapters.  The adapter must be open.   A return value of 1  
 * signifies success, 0 failure.  In case of failure, the default adapter will not be changed. */
DLLEXPORT uint32_t HW_SetDefaultAdapter
(
    const uint32_t adapterNumber   /*!< The number of the adapter */
);


#endif
/*#endif HWI_RUNTIME_DYNLINKING*/


#ifndef HWI_RUNTIME_DYNLINKING
/*! Returns the current default PHY_ID. */
DLLEXPORT uint32_t HW_GetDefault_PHY_ID (void);
#else
typedef uint32_t (__cdecl *HW_GetDefault_PHY_ID)();
#endif


#ifndef HWI_RUNTIME_DYNLINKING

/*! Sets the current default PHY_ID. */
DLLEXPORT void HW_SetDefault_PHY_ID 
(
  const uint32_t PHY_ID   /*!< The default PHY_ID */
);


void HW_CallException( const char *message );

/*! This function returns the HW_ErrorFlag.  If set, an error has occurred, and the error message will be stored in
    HW_ErrorMessage */
DLLEXPORT uint32_t HW_Fail(void);



/*! This function returns the HW_ErrorMessage.  If HW_ErrorFlag is set, the error message will be stored in
    HW_ErrorMessage */
DLLEXPORT char* HW_Error (void);



/*! This function returns the adapterNumber associated with a board name, where the adapter number references
    the adapter found in HW_FilteredAdapterList */
DLLEXPORT uint32_t HW_DeviceNumber
(
  const char *boardName   /*!< The name of the board in the HW_FilteredAdapterList */
);



/*! This function opens an adapter, where the adapter number references the adapter found in HW_FilteredAdapterList */
DLLEXPORT uint32_t HW_OpenAdapter
(
    uint32_t adapterNumber   /*!< The number of the adapter in the HW_FilteredAdapterList */
);



/*! This function closes a specific adapter, but does not exit the Nal OS */
DLLEXPORT uint32_t HW_CloseAdapter
(
    uint32_t adapterNumber   /*!< The number of the adapter in the HW_FilteredAdapterList */
);



/*! This function performs a write on the specified adapter */
DLLEXPORT void HW_Write
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);



/*! This function performs a read on the specified adapter */
DLLEXPORT uint32_t HW_Read
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block read to the default adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockRead
(
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block read to the default adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockWrite
(
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bit data to write.*/
  uint32_t data
);

/*! This function performs a write on the specified adapter */
DLLEXPORT void HW_Write8
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 8-bits of data being sent out with the write command. This may be either address or data.*/
  uint8_t data
);



/*! This function performs a write on the specified adapter */
DLLEXPORT void HW_Write16
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
	uint16_t data
);


/*! This function performs a write on the specified adapter */
DLLEXPORT void HW_Write32
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
	uint32_t data
);



/*! This function performs a read on the specified adapter */
DLLEXPORT uint8_t HW_Read8
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address
);


/*! This function performs a read on the specified adapter */
DLLEXPORT uint16_t HW_Read16
(
	/*! The number of the adapter in the HW_FilteredAdapterList. */
	uint32_t adapterNumber,
	/*! The address of the PHY being addressed. */
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address
);

/*! This function performs a read on the specified adapter */
DLLEXPORT uint32_t HW_Read32
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block read to the default adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockRead32
(
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block read to the default adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockWrite32
(
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 32-bit data to write.*/
  uint32_t data
);
/*! This function performs a write on the specified adapter */
DLLEXPORT void HW_Write64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
uint64_t data
);



/*! This function performs a read on the specified adapter */
DLLEXPORT uint64_t HW_Read64
(
/*! The number of the adapter in the HW_FilteredAdapterList. */
uint32_t adapterNumber,
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address
);



/*! This function sends a block read to the default adapter.  These block reads are accumulated until a
HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockRead64
(
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address
);



/*! This function sends a block read to the default adapter.  These block reads are accumulated until a
HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockWrite64
(
/*! The address of the PHY being addressed. */
uint32_t DEV_ID,
/*! The 16-bit address being written to.*/
uint32_t address,
/*! The 64-bit data to write.*/
uint64_t data
);

DLLEXPORT void HW_ConfigSpaceWrite
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
    uint32_t adapterNumber,
    /*! The 32-bit offset being written to.*/
    uint32_t offset,
    /*! Mask of used bits of data being sent out with the write command.*/
    uint32_t mask,
    /*! The 32-bits of data being sent out with the write command.*/
    uint32_t value
);

DLLEXPORT uint32_t HW_ConfigSpaceRead
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
    uint32_t adapterNumber,
    /*! The 32-bit offset being written to.*/
    uint32_t offset
);

#endif
/*#endif HWI_RUNTIME_DYNLINKING*/

#ifndef HWI_RUNTIME_DYNLINKING
/*! This function executes the block read on the default adapter, and returns a pointer to the read results. */
DLLEXPORT uint32_t * HW_BlockOperationExecute
(
    /*! The number of block reads processed. */
  uint32_t PHY_ID
);
#else
typedef uint32_t * (__cdecl *HW_BlockOperationExecute)(uint32_t);
#endif


#ifndef HWI_RUNTIME_DYNLINKING

/*! This function sends a block read to the specified adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockReadAdapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block write to the specified adapter.  These block writes are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of writes to execute is implementation specific. */
DLLEXPORT void HW_BlockWriteAdapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bit data to write.*/
  uint32_t data
);

/*! This function sends a block read to the specified adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockRead32Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block write to the specified adapter.  These block writes are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of writes to execute is implementation specific. */
DLLEXPORT void HW_BlockWrite32Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 32-bit data to write.*/
  uint32_t data
);

/*! This function sends a block read to the specified adapter.  These block reads are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void HW_BlockRead64Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block write to the specified adapter.  These block writes are accumulated until a 
    HW_BlockOperationExecute() call is issued. The maximum number of writes to execute is implementation specific. */
DLLEXPORT void HW_BlockWrite64Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 64-bit data to write.*/
  uint64_t data
);

DLLEXPORT void HW_BlockConfigSpaceRead(uint32_t offset);


DLLEXPORT void HW_BlockConfigSpaceWrite(uint32_t offset, uint32_t mask, uint32_t value);


DLLEXPORT void HW_BlockConfigSpaceReadAdapter(uint32_t adapterNumber, uint32_t offset);


DLLEXPORT void HW_BlockConfigSpaceWriteAdapter(uint32_t adapterNumber, uint32_t offset, uint32_t mask, uint32_t value);


/*! This function executes the block read on the specified adapter, and returns a pointer to the read results. */
DLLEXPORT uint32_t * HW_BlockOperationExecuteAdapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The number of block reads processed. */
  uint32_t DEV_ID
);



/*! This function returns the size of the results queue. */
DLLEXPORT uint32_t HW_BlockReadQueueSize (void);



/*! This function performs a write on the default adapter and default PHY_ID. */
DLLEXPORT void AQ_API_MDIO_Write_Default_PHY_ID
(
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);



/*! This function performs a read on the default adapter and default PHY_ID. */
DLLEXPORT uint32_t AQ_API_MDIO_Read_Default_PHY_ID
(
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function sends a block write to the default adapter and default PHY_ID.
 *  These block writes are accumulated until a  HW_BlockOperationExecute() call 
 *  is issued. The maximum number of writes to execute is implementation specific. */
DLLEXPORT void AQ_API_MDIO_BlockWrite_Default_PHY_ID
(
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);



/*! This function sends a block read to the default adapter and default PHY_ID.
 *  These block reads are accumulated until a HW_BlockOperationExecute() call 
 *  is issued. The maximum number of reads to execute is implementation specific. */
DLLEXPORT void AQ_API_MDIO_BlockRead_Default_PHY_ID
(
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);

#endif
/*#endif HWI_RUNTIME_DYNLINKING*/



#ifndef HWI_RUNTIME_DYNLINKING
/*! This function performs a read on the default adapter and default PHY_ID. */
DLLEXPORT void HW_BlockRead_Default_PHY_ID (
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);
#else
typedef void (__cdecl *HW_BlockRead_Default_PHY_ID)(uint32_t, uint32_t);
#endif



#ifndef HWI_RUNTIME_DYNLINKING
/*! This function sends a block write to the default adapter and default PHY_ID.
 *  These block writes are accumulated until a  HW_BlockOperationExecute() call 
 *  is issued. The maximum number of writes to execute is implementation specific. */
DLLEXPORT void HW_BlockWrite_Default_PHY_ID (
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);
#else
typedef void (__cdecl *HW_BlockWrite_Default_PHY_ID)(uint32_t, uint32_t, uint32_t);
#endif



#ifndef HWI_RUNTIME_DYNLINKING
/*! This function performs a read on the default adapter and default PHY_ID. */
DLLEXPORT uint32_t HW_Read_Default_PHY_ID (
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);
#else
typedef uint32_t (__cdecl *HW_Read_Default_PHY_ID)(uint32_t, uint32_t);
#endif



#ifndef HWI_RUNTIME_DYNLINKING
/*! This function performs a write on the default adapter and default PHY_ID. */
DLLEXPORT void HW_Write_Default_PHY_ID (
    /*! The address of the MMD within the PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);
#else
typedef void (__cdecl *HW_Write_Default_PHY_ID)(uint32_t, uint32_t, uint32_t);
#endif


#ifndef HWI_RUNTIME_DYNLINKING

/*! This function provides board level write access. */
DLLEXPORT void Board_Write
(
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command.*/
  uint32_t data
);



/*! This function provides board level read access. */
DLLEXPORT uint32_t Board_Read
(
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function provides board level write access. */
DLLEXPORT void Board_WriteAdapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command.*/
  uint32_t data
);



/*! This function provides board level read access. */
DLLEXPORT uint32_t Board_ReadAdapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 16-bit address being written to.*/
  uint32_t address
);


/*! This function provides generic MDIO write functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this write operation.*/
DLLEXPORT void AQ_API_MDIO_Write
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);

/*! This function provides generic MDIO read functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this read operation.*/
DLLEXPORT uint32_t AQ_API_MDIO_Read
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);


/*! This function provides generic buffered MDIO write functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this write operation. <BR><BR>
    Callers should first query for buffered I/O capability by calling AQ_API_MDIO_SupportsBlockOperations.<BR>
    Note that buffered MDIO writes will not be issued to the PHY until AQ_API_MDIO_BlockOperationExecute is
    called. */
DLLEXPORT void AQ_API_MDIO_BlockWrite
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);

/*! This function provides generic buffered MDIO read functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this read operation. <BR><BR>
    Callers should first query for buffered I/O capability by calling AQ_API_MDIO_SupportsBlockOperations.<BR>
    The buffered MDIO read results can be fetched by calling AQ_API_MDIO_BlockOperationExecute.*/
DLLEXPORT void AQ_API_MDIO_BlockRead
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t PHY_ID,
    /*! The address of the MMD within the Aquantia PHY being addressed.*/
  uint32_t MMD,
    /*! The 16-bit address being written to.*/
  uint32_t address
);

/*! This function provides board level write access. */
DLLEXPORT void Board_Write32
(
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command.*/
  uint32_t data
);



/*! This function provides board level read access. */
DLLEXPORT uint32_t Board_Read32
(
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function provides board level write access. */
DLLEXPORT void Board_Write32Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command.*/
  uint32_t data
);



/*! This function provides board level read access. */
DLLEXPORT uint32_t Board_Read32Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 16-bit address being written to.*/
  uint32_t address
);


/*! This function provides generic MDIO write functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this write operation.*/
DLLEXPORT void HW_API_Write8
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 8-bits of data being sent out with the write command. This may be either address or data.*/
  uint8_t data
);

/*! This function provides generic MDIO write functionality. It is the responsibility <BR>
of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
in order to accomplish this write operation.*/
DLLEXPORT void HW_API_Write16
(
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 16-bits of data being sent out with the write command. This may be either address or data.*/
	uint16_t data
);

/*! This function provides generic MDIO write functionality. It is the responsibility <BR>
of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
in order to accomplish this write operation.*/
DLLEXPORT void HW_API_Write32
(
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address,
	/*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
	uint32_t data
);

/*! This function provides generic MDIO read functionality. It is the responsibility <BR>
of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
in order to accomplish this read operation.*/
DLLEXPORT uint8_t HW_API_Read8
(
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address
);

/*! This function provides generic MDIO read functionality. It is the responsibility <BR>
of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
in order to accomplish this read operation.*/
DLLEXPORT uint16_t HW_API_Read16
(
	/*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
	system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
	uint32_t DEV_ID,
	/*! The 16-bit address being written to.*/
	uint32_t address
);


/*! This function provides generic MDIO read functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this read operation.*/
DLLEXPORT uint32_t HW_API_Read32
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);


/*! This function provides generic buffered MDIO write functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this write operation. <BR><BR>
    Callers should first query for buffered I/O capability by calling AQ_API_MDIO_SupportsBlockOperations.<BR>
    Note that buffered MDIO writes will not be issued to the PHY until AQ_API_MDIO_BlockOperationExecute is
    called. */
DLLEXPORT void AQ_API_BlockWrite32
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 32-bits of data being sent out with the write command. This may be either address or data.*/
  uint32_t data
);

/*! This function provides generic buffered MDIO read functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this read operation. <BR><BR>
    Callers should first query for buffered I/O capability by calling AQ_API_MDIO_SupportsBlockOperations.<BR>
    The buffered MDIO read results can be fetched by calling AQ_API_MDIO_BlockOperationExecute.*/
DLLEXPORT void AQ_API_BlockRead32
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);

/*! This function provides board level write access. */
DLLEXPORT void Board_Write64
(
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command.*/
  uint64_t data
);



/*! This function provides board level read access. */
DLLEXPORT uint64_t Board_Read64
(
    /*! The 16-bit address being written to.*/
  uint32_t address
);



/*! This function provides board level write access. */
DLLEXPORT void Board_Write64Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 16-bits of data being sent out with the write command.*/
  uint64_t data
);



/*! This function provides board level read access. */
DLLEXPORT uint64_t Board_Read64Adapter
(
    /*! The number of the adapter in the HW_FilteredAdapterList. */
  uint32_t adapterNumber,
    /*! The 16-bit address being written to.*/
  uint32_t address
);


/*! This function provides generic MDIO write functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this write operation.*/
DLLEXPORT void HW_API_Write64
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
  uint64_t data
);

/*! This function provides generic MDIO read functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this read operation.*/
DLLEXPORT uint64_t HW_API_Read64
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);


/*! This function provides generic buffered MDIO write functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this write operation. <BR><BR>
    Callers should first query for buffered I/O capability by calling AQ_API_MDIO_SupportsBlockOperations.<BR>
    Note that buffered MDIO writes will not be issued to the PHY until AQ_API_MDIO_BlockOperationExecute is
    called. */
DLLEXPORT void AQ_API_BlockWrite64
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address,
    /*! The 64-bits of data being sent out with the write command. This may be either address or data.*/
  uint64_t data
);

/*! This function provides generic buffered MDIO read functionality. It is the responsibility <BR>
    of the system designer to provide the specific MDIO address pointer updates, etc. <BR>
    in order to accomplish this read operation. <BR><BR>
    Callers should first query for buffered I/O capability by calling AQ_API_MDIO_SupportsBlockOperations.<BR>
    The buffered MDIO read results can be fetched by calling AQ_API_MDIO_BlockOperationExecute.*/
DLLEXPORT void AQ_API_BlockRead64
(
    /*! The address of the PHY being addressed. This is ultimately some combination of the <BR>
        system address and the target PHY's MDIO Address, set by the pins Addr [4:0].*/
  uint32_t DEV_ID,
    /*! The 16-bit address being written to.*/
  uint32_t address
);
/* Executes any buffered MDIO read/write operations pending from prior calls to AQ_API_MDIO_BlockWrite and/or <BR>
 * AQ_API_MDIO_BlockRead.  The return value is an array representing the fetched results of all pending calls to <BR>
 * AQ_API_MDIO_BlockRead.  Callers should track the number of pending block reads to determine the size of the <BR>
 * returned array. <BR> */
DLLEXPORT uint32_t * AQ_API_MDIO_BlockOperationExecute
(
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID
);

/* Returns the maximum number of buffered MDIO read/write operations supported by the MDIO implementation.  Callers <BR>
 * should call AQ_API_MDIO_BlockOperationExecute before issuing additional calls to AQ_API_MDIO_BlockWrite or <BR>
 * AQ_API_MDIO_BlockRead to avoid a buffer overflow. */
DLLEXPORT uint32_t AQ_API_MDIO_MaxBlockOperations
(
void
);

/*! Enable "single-wire" mode for MDIO operations with the specified DEV_ID (BBW32 only). */
DLLEXPORT void HW_BroadcastEnable
(
    /*! The address of the PHY being addressed. */
  uint32_t DEV_ID, 
    /*! 0: disable single-wire mode. non-0: enable. */
  uint32_t enable
);

#endif
/*#endif HWI_RUNTIME_DYNLINKING*/
/*@}*/

#ifdef __cplusplus
  }
#endif

#endif
