# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")] Param()

# Functions common to all cmdlets
Function ValidateGetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    if ((-Not $Adapters) -and $AdapterName)
    {
        foreach ($n in $AdapterName)
        {
            $TmpAdapterNames = @()
            try
            {
                $PhysicalAdapterArray = $script:PnpDevice.Where({ $_.Name -Like $n })
                $IntelAdapterArray = $PhysicalAdapterArray.Where({ $_.Manufacturer -eq "Intel" })

                if ($IntelAdapterArray)
                {
                    $TmpAdapterNames = $IntelAdapterArray.Name
                }
                elseif ($PhysicalAdapterArray)
                {
                    # Add non-Intel devices for warning display
                    $TmpAdapterNames = $PhysicalAdapterArray.Name
                }
            }
            catch
            {
                # Failed due to Adapters passed to Name parameter
                $AdapterNames = @()
                $ErrorMessages.Value += $Messages.InvalidParams
                break
            }

            if (-Not $TmpAdapterNames)
            {
                $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                continue
            }
            $AdapterNames += $TmpAdapterNames
        }
    }
    elseif ((-Not $AdapterName) -and $Adapters)
    {
        foreach ($a in $Adapters)
        {
            if (CheckPropertyExists $a "CreationClassName")
            {
                if ($a.CreationClassName -eq "MSFT_NetAdapter")
                {
                    $AdapterNames += $a.InterfaceDescription
                }
                elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                    $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                    $a.CreationClassName -eq "Win32_PnpEntity")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
            elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
            {
                $AdapterNames += $a.Name
            }
            else
            {
                $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
            }
        }
    }
    elseif (-Not ($AdapterName -and $Adapters))
    {
        $AdapterNames = ($script:PnpDevice.Where({ $_.Manufacturer -eq "Intel" })).Name
    }
    elseif ($AdapterName -and $Adapters)
    {
        $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
    }

    return $AdapterNames
}

Function ValidateSetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    do
    {
        if ($AdapterName -and $Adapters)
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
            break
        }
        elseif ($AdapterName)
        {
            foreach ($n in $AdapterName)
            {
                $TmpAdapterNames = @()
                try
                {
                    $PhysicalAdapterArray = $script:PnpDevice.Where({ $_.Name -Like $n })
                    $IntelAdapterArray = $PhysicalAdapterArray.Where({ $_.Manufacturer -eq "Intel" })

                    if ($IntelAdapterArray)
                    {
                        $TmpAdapterNames = $IntelAdapterArray.Name
                    }
                    elseif ($PhysicalAdapterArray)
                    {
                        # Add non-Intel devices for warning display
                        $TmpAdapterNames = $PhysicalAdapterArray.Name
                    }
                }
                catch
                {
                    # Failed due to Adapters passed to Name parameter
                    $AdapterNames = @()
                    $ErrorMessages.Value += $Messages.InvalidParams
                    break
                }

                if (-Not $TmpAdapterNames)
                {
                    $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                    continue
                }
                $AdapterNames += $TmpAdapterNames
            }
        }
        elseif ($Adapters)
        {
            foreach ($a in $Adapters)
            {
                if (CheckPropertyExists $a "CreationClassName")
                {
                    if ($a.CreationClassName -eq "MSFT_NetAdapter")
                    {
                        $AdapterNames += $a.InterfaceDescription
                    }
                    elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                        $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                        $a.CreationClassName -eq "Win32_PnpEntity")
                    {
                        $AdapterNames += $a.Name
                    }
                    else
                    {
                        $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                    }
                }
                elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterOrName
            break
        }
    } while ($false)

    return $AdapterNames
}

Function ValidatePathParams([ref]$LogPath, $UseDefaultPath, $LogName, [ref]$ErrorMessages)
{
    $Result = $true

    try
    {
        if ($UseDefaultPath)
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\" + $LogName

            if (-not (Test-Path -Path $DefaultPath -ErrorAction Stop))
            {
                New-Item -Path $ENV:LOCALAPPDATA -Name "\Intel\Wired Networking" -ItemType "directory" -ErrorAction SilentlyContinue
            }
            $LogPath.Value = $DefaultPath
        }
        else
        {
            $LogPath.Value = $Path
        }

        $PathRoot = GetPSDriveRoot $LogPath.Value
        if (-Not [string]::IsNullOrEmpty($PathRoot))
        {
            $PathRoot = $PathRoot.TrimEnd("\")
            $strPathNoQualifier = Split-Path $LogPath.Value -NoQualifier
            $LogPath.Value = $PathRoot + $strPathNoQualifier
        }

        $isPathFile = Test-Path -Path $LogPath.Value -PathType Leaf -ErrorAction Stop

        if (($isPathFile) -and (-not $Append) -and (-not $Force))
        {
            $ErrorMessages.Value += $Messages.LogmanFileExists -f $AdapterName
            $Result = $false
        }
        elseif (-not $isPathFile)
        {
            if (Test-Path -Path $LogPath.Value -ErrorAction Stop)
            {
                $ErrorMessages.Value += $Messages.FolderFileNameExits
                $Result = $false
            }
            else
            {
                $strAbsolutePath = [IO.Path]::GetFullPath($LogPath.Value)
                $strParentFolder = Split-Path -Path $strAbsolutePath

                if (-Not (Test-Path -Path $strParentFolder -ErrorAction Stop))
                {
                    $ErrorMessages.Value += $Messages.PathIncorrect
                    $Result = $false
                }
            }
        }
    }
    catch
    {
        $ErrorMessages.Value += $Messages.PathIncorrect
        $Result = $false
    }
    return $Result
}

function GetPSDriveRoot($Path)
{
    $strQualifier = Split-Path -Path $Path -Qualifier -ErrorAction SilentlyContinue
    if (-Not [string]::IsNullOrEmpty($strQualifier))
    {
        $strPSDriveName = $strQualifier.TrimEnd(":")
        $CurrentPSDrive = Get-PSDrive -Name $strPSDriveName -ErrorAction SilentlyContinue
        if ($null -ne $CurrentPSDrive)
        {
            return $CurrentPSDrive.Root
        }
    }

    return $null
}

function InvokeCimMethod($ClassName, $InstanceName = "", $MethodName, $params = @{}, $Namespace = "root\wmi")
{
    $query = "Select * from $ClassName"
    if ($InstanceName)
    {
        $query += " where instancename like '$InstanceName'"
    }

    Invoke-CimMethod -Query $query -MethodName $MethodName -Arguments $params -Namespace $Namespace -ErrorAction SilentlyContinue
}

function GetIntelEthernetDevices($AdditionalDriverArray = $null)
{
    $script:PnpDevice = @(Get-PnpDevice | Where-Object { $_.Class -eq "Net" } -ErrorAction SilentlyContinue)
    if ($script:PnpDevice)
    {
        $script:SupportedAdapters = @($script:PnpDevice.Where({ $_.Service -in @('icea', 'scea', 'i40ea', 'i40eb') }))

        foreach ($AdditionalDriver in $AdditionalDriverArray)
        {
            $script:SupportedAdapters += $script:PnpDevice.Where({ $_.Service -like $AdditionalDriver })
        }
    }
}

function GetSupportedAdaptersByDriver([array]$strSupportedDriverArray)
{
    return $script:SupportedAdapters.Where({ $_.Service -in $strSupportedDriverArray })
}

function GetSupportedAdapters($AdapterNames, [ref]$WarningMessages)
{
    $SupportedAdapterNames = @()
    $AdapterNames = $AdapterNames | Sort-Object
    foreach ($a in $AdapterNames)
    {
        if ($script:SupportedAdapters.Name -Contains $a)
        {
            $SupportedAdapterNames += $a
        }
        else
        {
            $WarningMessages.Value += $Messages.NoCmdletSupport -f $a
        }
    }

    return $SupportedAdapterNames
}

Function GetOSSpecificDriverName($DriverName)
{
    $WIN10_RS5_BUILD = 17682
    $VER_NT_SERVER = 3

    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iMinorVersion = $OSVersion.Minor
    $iBuildNumber = $OSVersion.Build

    $strOSSpecificDriverName = ''

    if (10 -eq $iMajorVersion -and 0 -eq $iMinorVersion)
    {
        if ($VER_NT_SERVER -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
        {
            $bIsWin2016ServerAndHigher = $true
        }
    }

    if ($bIsWin2016ServerAndHigher)
    {
        if (IsWin2022ServerOrLater)
        {
            $strOSSpecificDriverName = $DriverName
        }
        elseif ($iBuildNumber -ge $WIN10_RS5_BUILD)
        {
            $strOSSpecificDriverName = $DriverName + "68"
        }
        else
        {
            $strOSSpecificDriverName = $DriverName + "65"
        }
    }

    return $strOSSpecificDriverName
}

function GetDriverInfParentFolder($strDriverName, $Verb)
{
    $strDriverStorePathArray = Get-WindowsDriver -Online -ErrorAction SilentlyContinue | Select-Object OriginalFileName

    if ($null -eq $strDriverStorePathArray)
    {
        switch ($Verb)
        {
            Start {$script:ErrorMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
            Register
            {
                $script:ErrorMessagesRegister += $Messages.RegisterCountersFailure -f $a
                Write-Verbose "LASTEXITCODE: $LASTEXITCODE"
                break
            }
        }
    }

    # The system keeps a history of installed drivers, Get the latest installed driver inf path
    $strPathToInf = Get-ChildItem -Path $strDriverStorePathArray.OriginalFileName.Where({ $_ -like "*$strDriverName*" }) | Sort-Object -Descending -Property "CreationTime" | Select-Object -First 1

    [System.IO.Path]::GetDirectoryName($strPathToInf)
}

function CheckDeviceError($AdapterName)
{
    $SupportedAdapter = $script:SupportedAdapters.Where({ $_.FriendlyName -eq $AdapterName })

    if ($SupportedAdapter)
    {
        # if the device is not 'working properly'
        if ([Int32]$SupportedAdapter.ConfigManagerErrorCode -ne 0)
        {
            $PreErrorActionPreference = $global:ErrorActionPreference
            $global:ErrorActionPreference = 'SilentlyContinue'
            # Workaround due to ProblemDescription being empty by default - change current path
            Push-Location -Path (Get-Module -Name PnPDevice).ModuleBase
            $StatusMsg = $AdapterName + ": " + $SupportedAdapter.ProblemDescription
            # Reset path
            Pop-Location
            $global:ErrorActionPreference = $PreErrorActionPreference
            $StatusMsg
        }
    }
}

function ValidateSingleAdapter([array]$PipelineInput, [array]$AdapterName, [ref]$ErrorMessages)
{
    $Result = $false

    do
    {
        if ($PipelineInput.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        if ($AdapterName.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        $Result = $true
    } while ($false)

    return $Result
}

function IsWin2022ServerOrLater()
{
    $WIN2022_BUILD = 20298
    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iBuildNumber = $OSVersion.Build

    if (($iMajorVersion -eq 10) -and ($iBuildNumber -ge $WIN2022_BUILD) -and -not (IsOperatingSystemClientBased))
    {
        return $true
    }

    return $false
}

function IsOperatingSystemClientBased()
{
    $VER_NT_WORKSTATION = 1
    return ($VER_NT_WORKSTATION -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
}

function CheckPropertyExists($TestObject, $strPropertyName)
{
    $bReturnValue = $true
    try
    {
        return (($TestObject | Get-Member -MemberType "Property" -ErrorAction Stop).Name -contains $strPropertyName)
    }
    catch
    {
        $bReturnValue = $false
    }
    return $bReturnValue
}

function GetAdapterSettings($AdapterName, $DisplayName, $RegistryKeyword)
{
    $AdapterSettings = @()
    $SettingsRet = @()
    if ([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword))
    {
        $AdapterSettings = GetAllAdapterSettings $AdapterName
        $SettingsRet += InitializeProfileSetting $AdapterName
    }
    else
    {
        # Need to handle Profile differently, this is an EthernetCmdlets created setting
        if ($DisplayName -eq 'Profile' -or $RegistryKeyword -eq 'PerformanceProfile')
        {
            $SettingsRet += InitializeProfileSetting $AdapterName

            # Remove Profile from passed in array to avoid GatherSettingEnumOutput (InitializeProfileSetting handles it uniquely)
            if ($null -ne $DisplayName)
            {
                $DisplayName = $DisplayName -ne 'Profile'
            }
            if ($null -ne $RegistryKeyword)
            {
                $RegistryKeyword = $RegistryKeyword -ne 'PerformanceProfile'
            }
        }

        $AdapterSettings = GetSettings $AdapterName $DisplayName $RegistryKeyword
    }

    foreach ($Setting in $AdapterSettings)
    {
        switch ([int]($Setting.DisplayParameterType))
        {
            {$_ -in [int][DisplayParameterType]::int,
            [int][DisplayParameterType]::long,
            [int][DisplayParameterType]::word,
            [int][DisplayParameterType]::dword}
            { $SettingsRet += GatherSettingIntOutput $a $Setting; break }
            {$_ -in [int][DisplayParameterType]::enum,
            [int][DisplayParameterType]::edit}
            {$SettingsRet += GatherSettingEnumOutput $a $Setting; break}
            default {$SettingsRet += GatherSettingEnumOutput $a $Setting; break}
        }
    }

    return $SettingsRet
}

function GetAllAdapterSettings($AdapterName)
{
    return $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName})
}

function GetSettings($AdapterName, $DisplayName, $RegistryKeyword)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            if ($DisplayName -eq 'Profile')
            {
                continue
            }
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName})
            if (-not $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            if ($RegistryKeyword -eq 'PerformanceProfile')
            {
                continue
            }
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword})
            if (-not $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    return $SettingArray
}

function GatherSettingEnumOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting.RegistryKeyword)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DescriptionMap  = $Setting.ValidDisplayValues
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        PossibleValues  = $Setting.ValidRegistryValues
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
    }
}

function GatherSettingIntOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting.RegistryKeyword)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingInt'
        Base            = $Setting.NumericParameterBaseValue
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
        Min             = $Setting.NumericParameterMinValue
        Max             = $Setting.NumericParameterMaxValue
        Step            = $Setting.NumericParameterStepValue
    }
}

function GetSettingMinihelp($RegistryKeyword)
{
    $SettingMiniHelp = $MiniHelp.$($RegistryKeyword)
    if ([string]::IsNullOrEmpty($SettingMiniHelp))
    {
        $SettingMiniHelp = $Messages.MiniHelpNotFound
    }

    return $SettingMiniHelp
}

function InitializeProfileSetting($AdapterName)
{
    # for our supported adapters, by default its safe to add: Standard Server, Web Server, and Low Latency
    # PossibleValues and DescriptionMap for Profile
    $ProfileValuesMap = @{ '2' = $Messages.StandardServer; '3' = $Messages.WebServer; '6' = $Messages.LowLatency }
    $DefaultValue = '2'

    $IntelDCBInstalled = $false
    $HyperVInstalled = $false

    $IntelDCB = Get-Service | Where-Object {$_.Name -eq "IntelDCB"}
    if ($null -ne $IntelDCB)
    {
        $IntelDCBInstalled = $true
    }

    # ID 20 represents Hyper-V
    $HyperV = Get-CimInstance -Namespace "root\cimv2" -ClassName "Win32_ServerFeature" | Where-Object {$_.ID -eq '20'}
    if ($null -ne $HyperV)
    {
        $HyperVInstalled = $true
    }

    if ($IntelDCBInstalled)
    {
        $ProfileValuesMap.Add('4', $Messages.StorageServer)
        $DefaultValue = '4'
    }
    if ($HyperVInstalled)
    {
        $ProfileValuesMap.Add('5', $Messages.VirtServer)
        $DefaultValue = '5'
    }
    if ($IntelDCBInstalled -and $HyperVInstalled)
    {
        $ProfileValuesMap.Add('7', $Messages.StorageVirt)
        $DefaultValue = '7'
    }

    $CurrentProfile = GetCurrentProfileValue $AdapterName
    $CurrentProfileRegistryValue = $null
    $CurrentProfileDisplayValue = $null
    if ($null -ne $CurrentProfile)
    {
        $CurrentProfileRegistryValue = $CurrentProfile.RegistryValue
        $CurrentProfileDisplayValue = $CurrentProfile.DisplayValue
        if ($CurrentProfileRegistryValue -eq 1)
        {
            $ProfileValuesMap.Add($CurrentProfileRegistryValue, $CurrentProfileDisplayValue)
        }
    }
    $ProfileValuesMap = $ProfileValuesMap.GetEnumerator() | Sort-Object -Property Name
    $SettingMiniHelp = GetSettingMinihelp("PerformanceProfile")
    $ParentId = $script:MSNetAdvProperty.Where({ $_.InterfaceDescription -eq $AdapterName}).InstanceID.split(':')[0]

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = 'PerformanceProfile'
        CurrentValue    = $CurrentProfile.RegistryValue
        DefaultValue    = $DefaultValue
        Description     = $Messages.Profile
        DescriptionMap  = $ProfileValuesMap.Value
        DisplayName     = $Messages.Profile
        DisplayValue    = $CurrentProfileDisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $ParentId
        PossibleValues  = $ProfileValuesMap.GetEnumerator().Name
        RegistryKeyword = 'PerformanceProfile'
        RegistryValue   = $CurrentProfileRegistryValue
    }
}

function GetCurrentProfileValue($AdapterName)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName

    if ($null -ne $AdapterRegKey.PerformanceProfile)
    {
        $ProfileRegValue = $AdapterRegKey.PerformanceProfile
        # Profile is already custom settings or in case one of the associated Profile settings got changed
        if ($ProfileRegValue -eq '1' -or ($false -eq (IsProfileStillApplied $ProfileRegValue $AdapterName)))
        {
            $ProfileRegValue = '1'
            $ProfileDisplayVal = $Messages.CustomSettings
        }
        else
        {
            $ProfileDisplayVal = $PROFILE_VALUE_TO_NAME[$ProfileRegValue]
        }

        return [PsCustomObject] @{
            # return the key from the hashtable
            DisplayValue = $ProfileDisplayVal
            RegistryValue = $ProfileRegValue }
    }
}

function IsProfileStillApplied($RegistryValue, $AdapterName)
{
    $TmpProfileName = $PROFILE_VALUE_TO_NAME[$RegistryValue]
    if ($null -ne $TmpProfileName)
    {
        $DriverFamily = $script:SupportedAdapters.Where({ $_.Name -eq $AdapterName }).Service
        $tmp = GetProfileSettingsFromXml $TmpProfileName $DriverFamily
        foreach ($ProfileSetting in $tmp.GetEnumerator())
        {
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq $ProfileSetting.Key})
            if ($TmpSetting.RegistryValue -ne $ProfileSetting.Value)
            {
                return $false
            }
        }
    }

    return $true
}

function GetProfileSettingsFromXml()
{
    Param
    (
        [parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [string]
        $ProfileToSet,

        [parameter(Mandatory=$true)]
        [ValidateNotNullOrEmpty()]
        [ValidateSet("i40ea", "i40eb", "icea", "scea")]
        [string]
        $AdapterFamily
    )

    $Profile_to_XmlProfile = @{
        "Standard Server" = "standard"
        "Web Server" = "web"
        "Storage Server" = "storage"
        "Virtualization Server (Hyper-V*)" = "hyperv"
        "Low Latency" = "lowlatency"
        "Storage + Virtualization" = "storagevirtualization"
    }

    $OSBuild = [Environment]::OSVersion.Version.Build

    # Each Key treated as a range of Build Numbers (14393 to 16299 is NDIS v6.60)
    $OSBuild_to_NDIS = [ordered]@{
        14393 = "ndis660"
        16299 = "ndis680"
    }

    $Adapter_to_ProfileString = @{
        i40ea = "40gig"
        i40eb = "40gig"
        icea  = "E8XX"
        scea  = "E8XX"
    }

    $Adapter_to_ProfileDeviceString = @{
        i40ea = "X710"
        i40eb = "X722"
    }

    [xml]$Xml = Get-Content -Path $PSScriptRoot\PerformanceProfiles.xml
    # validate XML against XSD before using it:
    try
    {
        $Xml.Schemas.Add('', "$PSScriptRoot\PerformanceProfiles.xsd") | Out-Null
        $Xml.Validate($null)
    }
    catch [System.Xml.Schema.XmlSchemaValidationException]
    {
        $script:ErrorMessagesProfile += $Messages.ProfileXmlError
        return
    }

    $PerfProfiles = $Xml.SelectNodes("//ProfileList/Profile")

    # Convert cmdlet parameters to strings from xml
    $AdapterFamilyProfile = $Adapter_to_ProfileString[$AdapterFamily]

    foreach ($CurrentBuildNumber in $OSBuild_to_NDIS.Keys)
    {
        if ($OSBuild -gt $CurrentBuildNumber)
        {
            $NDISVersion = $OSBuild_to_NDIS.$CurrentBuildNumber
        }
    }

    $ProfileName = $Profile_to_XmlProfile[$ProfileToSet]

    # Construct profile names
    # Top-level, eg 40gig_standard
    $TopLevelProfile = $AdapterFamilyProfile + "_" + $ProfileName
    Write-Verbose $TopLevelProfile

    # + NDIS version, eg 40gig_standard.ndis660
    $NdisProfile = $AdapterFamilyProfile + "_" + $ProfileName + "." + $NDISVersion
    Write-Verbose $NdisProfile

    # + device name, eg 40gig_standard.ndis660.X722
    # adding '*' at the end so it matches .no_npar too
    $DeviceProfile = $AdapterFamilyProfile + "_" + $ProfileName + "." + $NDISVersion + "." + $Adapter_to_ProfileDeviceString[$AdapterFamily] + "*"
    Write-Verbose $DeviceProfile

    $TopLevelProfile = $PerfProfiles | Where-Object { $_.Id -like  $TopLevelProfile }
    $NdisProfile     = $PerfProfiles | Where-Object { $_.Id -like  $NdisProfile }
    $DeviceProfile   = $PerfProfiles | Where-Object { $_.Id -like  $DeviceProfile }

    # assemble it all together - from generic to detailed profile
    # ie DeviceProfile settings should take precedence over those from TopLevelProfile
    $Profiles = @($TopLevelProfile, $NdisProfile, $DeviceProfile)
    $RetVal = @{}

    foreach ($Profile in $Profiles)
    {
        foreach($Setting in $Profile.Setting)
        {
            $RetVal[$Setting.id] = $Setting.'#text'
        }
    }

    return $RetVal
}

function GetAdapterPropertiesFromRegistry($AdapterName)
{
    # Individual Adapter GUID  - (Get-NetAdapter -InterfaceDescription $AdapterName | Where-Object {$_.InterfaceDescription -eq $AdapterName}).InterfaceGuid
    $AdapterInstanceID = ($script:MSNetAdapters.Where({$_.InterfaceDescription -eq $AdapterName})).InterfaceGuid
    $972Key = Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -ErrorAction SilentlyContinue
    $AdapterRegKey = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterInstanceID})
    return $AdapterRegKey
}

function SetAdapterPropertyInRegistry($AdapterName, $Property, $Value)
{
    $AdapterRegKey = GetAdapterPropertiesFromRegistry $AdapterName
    Set-ItemProperty -Path $AdapterRegKey.PSPath -Name $Property $Value -ErrorAction SilentlyContinue
}

# SIG # Begin signature block
# MIIocwYJKoZIhvcNAQcCoIIoZDCCKGACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDHTpMmFSI/lkdn
# F+7QxBBCxdWOoc/RHC+hwO4bd7j5dKCCEfMwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlwwggTEoAMCAQICEQC0WMhOLa9BaZ9kSX5iJ3F/MA0G
# CSqGSIb3DQEBCwUAMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLjAsBgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBF
# ViBSMzYwHhcNMjQwMjI2MDAwMDAwWhcNMjUwMjI1MjM1OTU5WjCBuzEQMA4GA1UE
# BRMHMjE4OTA3NDETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDxMUUHJpdmF0ZSBPcmdhbml6YXRpb24xCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIDApDYWxpZm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jw
# b3JhdGlvbjEaMBgGA1UEAwwRSW50ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3
# DQEBAQUAA4IBjwAwggGKAoIBgQDBCfpjptqBxrQLJGyUHE47EvbngKTbZ0xMZoUj
# CJVmRhCCzWtZeKwlwhuI3bJyq4sSeejZxY7IMjroOoditsPm5xYohctw0UO+j1Th
# L71qce9bigWpDFDBBqksK5+011j/XPA+kRu/gJBolI50N8tIHHsH31NzD09/sN7U
# V242zTBy0TnMwanTXLMux/kVJbIloWSHRn0wIZmGuWESmWDrsLQEtSIo4zyUlzvQ
# UmJrtHMmJc3Rw/5TE7rC9Zq4Yt6s+BNu8i5howcK7yEOtiw/sKIlbACFJqpp6EUT
# Kwi7RRLKkuoL7G/+50XrJlCQqDbYxQAm7Tc2oFBVZW9xf4gUz3f48iflabLvDmc0
# pVWgDF0OmX+SzsHf94GYG3slCw8JJKfU66TfJEModuiDPwfgA6ripNWdBHqaDoY7
# JQPt6T6wierKjp64ABBHwyYSD55RIMUm/w33oe0i44tAlvUTkujJzwUQKpjXQ9av
# FyA2VqPea77rc3yiCRNeGQTpyO0CAwEAAaOCAbwwggG4MB8GA1UdIwQYMBaAFIEy
# kkErKM1GyMSixio5EuxIqT8UMB0GA1UdDgQWBBSC0NSIL647v94GegQBXPynnV+p
# cDAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEF
# BQcDAzBJBgNVHSAEQjBAMDUGDCsGAQQBsjEBAgEGATAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAHBgVngQwBAzBLBgNVHR8ERDBCMECgPqA8
# hjpodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmlu
# Z0NBRVZSMzYuY3JsMHsGCCsGAQUFBwEBBG8wbTBGBggrBgEFBQcwAoY6aHR0cDov
# L2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQUVWUjM2
# LmNydDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wLgYDVR0R
# BCcwJaAjBggrBgEFBQcIA6AXMBUME1VTLURFTEFXQVJFLTIxODkwNzQwDQYJKoZI
# hvcNAQELBQADggGBADRT5U3ne/vFqaxPbuhkYXvhfhBMHNi+fHEkOCjPhEyqOkeU
# 1a7bjucUQh+Jb6yOlEKzb8KppbyTDNZzlo5NpkpBtFimyPY/h7pboom7mDOFXS9/
# NUuESrc/niHOTRkTzipQa125g5y/hgZ8a6+XoMjKi/rWOvllhdyUiJi6KY9x5+IN
# nXSgYZu/7xnGge/UybwaVrCVqmirark7p8I3vPOmIQeeGupn7qyzFdiMK5EEpPUI
# uO4po7YGOTQDgpdPjUQGmmGqbkrGgvH2fT2W/Ti8IZSgBM+3i3Rtqo50gOTOe9py
# fG30f9aFUtFHFc9BAA3kvG+Xqr4MLOdFYgQRGFXNjN5IA6zc0admMuG8m/hVasJN
# p+ACnv8HeWID2O6oTGPhwHZkvfgqL05qEO6ZiThnzwWDukiduuceeYxIVqyYW253
# hzgZCKnjWVdDT3gUWoO2TJNR7sZuP/gP7I2hyotU8uRTl3SvlFfbaVGHj+xVqR1k
# taptv3zLnJYUhbTyNjGCFdYwghXSAgEBMGwwVzELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29k
# ZSBTaWduaW5nIENBIEVWIFIzNgIRALRYyE4tr0Fpn2RJfmIncX8wDQYJYIZIAWUD
# BAIBBQCgajAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgEL
# MQ4wDAYKKwYBBAGCNwIBFjAvBgkqhkiG9w0BCQQxIgQghjHg4mYrLGxA2ET5JaLE
# DdMtF9YJ0IaT2nCutbK1PZ8wDQYJKoZIhvcNAQEBBQAEggGAejjzOjzXt8enMXH0
# XlKJgzdO8P9wsMLGRZWxvEndHFSG2iBRrdhEdWZc7i5civdBfCiZ6Zpx+n36mUTE
# OtDZyNryKE8u93R7u5G6szQ0Qmd2zsVOm2/Y2dDZ82jPO2TjHSAN7YC00P26gcGd
# QD4OnALWGpPX3QazcN48vf+Ku2QsStsofc1qWykXcjcCmyjVYeXSizP8ctz0WddP
# uGiWOACR3L1h1fY7jFJInJbUleD9v1ELs9pft/lk4al5bTBwpsUpRcfCL0ANyQFi
# kA4Fh0ejOShRsqfbwWBzm+2O7T0fiSpAM9VQWylJTc2yJc7b0yB0bV3xNv/+OSi8
# UquMBovq3yWcCtr23w7UvcfOre4VXmIZtsbfm7Delrh6UfQUqlWPk7VaCn699hrU
# P/+f8/ViCFNnFAEDAwn9riBtmjHuQlJjbgM7uRRzf2AKCFhoNsZiA+X8FvBv1ZkX
# qJ3eiRt/aZOxS6G6cZxdckLchpGqKcF39jDj0kKPh3/NjYBdoYITTzCCE0sGCisG
# AQQBgjcDAwExghM7MIITNwYJKoZIhvcNAQcCoIITKDCCEyQCAQMxDzANBglghkgB
# ZQMEAgIFADCB8AYLKoZIhvcNAQkQAQSggeAEgd0wgdoCAQEGCisGAQQBsjECAQEw
# MTANBglghkgBZQMEAgEFAAQgp5AAoZXtR9mlkfp0mowTe5+T+M0G50b8KB8JMb4p
# Z9gCFQCYj3LOiU9ordQD2QJSvbJYErKQWRgPMjAyNDA3MTExOTMwMThaoG6kbDBq
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IFNpZ25lciAjNKCCDekwggb1MIIE3aADAgECAhA5TCXhfKBtJ6hl4jvZHSLUMA0G
# CSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1h
# bmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAeFw0y
# MzA1MDMwMDAwMDBaFw0zNDA4MDIyMzU5NTlaMGoxCzAJBgNVBAYTAkdCMRMwEQYD
# VQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNV
# BAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICM0MIICIjANBgkq
# hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEApJMoUkvPJ4d2pCkcmTjA5w7U0RzsaMsB
# ZOSKzXewcWWCvJ/8i7u7lZj7JRGOWogJZhEUWLK6Ilvm9jLxXS3AeqIO4OBWZO2h
# 5YEgciBkQWzHwwj6831d7yGawn7XLMO6EZge/NMgCEKzX79/iFgyqzCz2Ix6lkoZ
# E1ys/Oer6RwWLrCwOJVKz4VQq2cDJaG7OOkPb6lampEoEzW5H/M94STIa7GZ6A3v
# u03lPYxUA5HQ/C3PVTM4egkcB9Ei4GOGp7790oNzEhSbmkwJRr00vOFLUHty4Fv9
# GbsfPGoZe267LUQqvjxMzKyKBJPGV4agczYrgZf6G5t+iIfYUnmJ/m53N9e7UJ/6
# GCVPE/JefKmxIFopq6NCh3fg9EwCSN1YpVOmo6DtGZZlFSnF7TMwJeaWg4Ga9mBm
# kFgHgM1Cdaz7tJHQxd0BQGq2qBDu9o16t551r9OlSxihDJ9XsF4lR5F0zXUS0Zxv
# 5F4Nm+x1Ju7+0/WSL1KF6NpEUSqizADKh2ZDoxsA76K1lp1irScL8htKycOUQjeI
# IISoh67DuiNye/hU7/hrJ7CF9adDhdgrOXTbWncC0aT69c2cPcwfrlHQe2zYHS0R
# QlNxdMLlNaotUhLZJc/w09CRQxLXMn2YbON3Qcj/HyRU726txj5Ve/Fchzpk8WBL
# BU/vuS/sCRMCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7R
# NV8GEgRVMB0GA1UdDgQWBBQDDzHIkSqTvWPz0V1NpDQP0pUBGDAOBgNVHQ8BAf8E
# BAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBKBgNV
# HSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3Nl
# Y3RpZ28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDegNYYzaHR0cDov
# L2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3JsMHQG
# CCsGAQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5zZWN0aWdvLmNv
# bS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUFBzABhhdodHRw
# Oi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEATJtlWPrgec/v
# FcMybd4zket3WOLrvctKPHXefpRtwyLHBJXfZWlhEwz2DJ71iSBewYfHAyTKx6Xw
# Jt/4+DFlDeDrbVFXpoyEUghGHCrC3vLaikXzvvf2LsR+7fjtaL96VkjpYeWaOXe8
# vrqRZIh1/12FFjQn0inL/+0t2v++kwzsbaINzMPxbr0hkRojAFKtl9RieCqEeajX
# Pawhj3DDJHk6l/ENo6NbU9irALpY+zWAT18ocWwZXsKDcpCu4MbY8pn76rSSZXwH
# fDVEHa1YGGti+95sxAqpbNMhRnDcL411TCPCQdB6ljvDS93NkiZ0dlw3oJoknk5f
# TtOPD+UTT1lEZUtDZM9I+GdnuU2/zA2xOjDQoT1IrXpl5Ozf4AHwsypKOazBpPmp
# fTXQMkCgsRkqGCGyyH0FcRpLJzaq4Jgcg3Xnx35LhEPNQ/uQl3YqEqxAwXBbmQpA
# +oBtlGF7yG65yGdnJFxQjQEg3gf3AdT4LhHNnYPl+MolHEQ9J+WwhkcqCxuEdn17
# aE+Nt/cTtO2gLe5zD9kQup2ZLHzXdR+PEMSU5n4k5ZVKiIwn1oVmHfmuZHaR6Ej+
# yFUK7SnDH944psAU+zI9+KmDYjbIw74Ahxyr+kpCHIkD3PVcfHDZXXhO7p9eIOYJ
# anwrCKNI9RX8BE/fzSEceuX1jhrUuUAwggbsMIIE1KADAgECAhAwD2+s3WaYdHyp
# RjaneC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# TmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBV
# U0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZp
# Y2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAxMTgyMzU5NTla
# MH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAO
# BgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UE
# AxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEB
# BQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nPoD5T77NawHbW
# GgSlzkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEFkmZvIavVcRB5
# h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQFcN/nqJc1zjx
# FrJw06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1VdEXOzQhe5cB
# lK9/gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpWcdEtlEzIjDzT
# FKKcvSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ffnz82Pg4wZytG
# tzWvlr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV6xlV/9A5VC0m
# 7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7wW33pdmmTIBxK
# lyx3GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFkyYiSWcuoRsDJ
# 8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXDTYLQjdprt9v3
# GEBXc/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70atZ2h3pVBeqeD
# AgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAd
# BgNVHQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/BAQDAgGGMBIG
# A1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAow
# CDAGBgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0
# LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDB2Bggr
# BgEFBQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNlcnRydXN0LmNv
# bS9VU0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcwAYYZaHR0cDov
# L29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEAbVSBpTNdFuG1
# U4GRdd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNviffmfAO9Aw63T
# 4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzVWHi/SHbhRGdt
# j7BDEA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJqO0I3fBZ4OZOT
# lJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytCiiGS/qhvWiVw
# YoFzY16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6NIxi5o8JHHfBd
# 1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9JkWtgqIsJLnfE5
# XkzeSD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw0NP8srMftpmW
# JvQ8eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUgOw9Bqh/poUGd
# /rG4oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20OAkrSQeq/eww
# 2vGnL8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8XwgDCp3rrWiAoa6
# Ke60WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQsMIIEKAIBATCBkTB9MQswCQYDVQQG
# EwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxm
# b3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28g
# UlNBIFRpbWUgU3RhbXBpbmcgQ0ECEDlMJeF8oG0nqGXiO9kdItQwDQYJYIZIAWUD
# BAICBQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0B
# CQUxDxcNMjQwNzExMTkzMDE4WjA/BgkqhkiG9w0BCQQxMgQwUT5lSfFST9dHHzOM
# XkDSzO5gi1ISUSO1FTqZpZSBVFTyAMKaoJFXWJ2piR2p1uLBMIHtBgsqhkiG9w0B
# CRACDDGB3TCB2jCB1zAWBBSuYq91Cgy9R9ZGH3Vo4ryM58pPlDCBvAQUAtZbleKD
# cMFXAJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
# EwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhl
# IFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRp
# ZmljYXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEB
# AQUABIICAET8IQVdD5lO9cZr5+0tBZmWeoFobkweLe85SIEZH7UOe87StObVbkFl
# 0qogcZkJXYS6ebF29tonuw5+E4ADlJCMnhK9CRXdluknQdbMOG19/fjy2/bWJIho
# HszhkXbCXxDhzK1esNstLsSqmoY7syXdyL6e+tlTMJNrISXepms1FNdX73HWh9LN
# mPEbfMPJUiJUDfCWzdxyf/EMdnTltgEMwbn5bdQyuAT1lqZRwrF6HwnqeHFC/Oxz
# eTfsPCzdB10cJk5+f2PEo2PcxhjJECLQRCZYzfMEU+UMjgqcKNGFSwhtwpwbpDsg
# qcrIfDIfpzstcwHjoJbVYcXDcTEPCetMdpFoD6SYhFmfpttWXGZ9jflyCBcTkPRP
# YkFsERbU1BYnFfl4iGXsntMinkzw1GEQA1EY+0SOjj8EoiCTyYkccRsC1b1RoR8M
# VM8T8zQuItvTDG0TJkbW7QhcHsg8oXgVrTO+MNX9K/TYD+MceWpMjInqQEA380WA
# RjoQ2dzj0a8d3HVruBl+xGoQv2zNv/x2AiVwRkyeeDoarhFhgqjAhyBrTMpDRaxN
# jqiv7CxJt/laoZcPz9vzfdUAAGwiHAtadPcDJSKpOgCsUqAIjjzVwmrp9/JB5Tw/
# Ou585oeUk7ZctGSblgdfU/WVux01gzStCyaCYXyb9TuNMRGdsrA7
# SIG # End signature block
