﻿using Intel.Cst.Client;
using SampleApp;
using System;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace WALWOASampleApp.CallBackFunctions
{
    class FeatureCallback : IFeatureCallback
    {
        Logger logger = new Logger();

        public void OnError(FeatureType featureType, Error state)
        {
            try
            {
                this.logger.Error("Received error " + state.Description + " from service");
                switch (state.ErrorType)
                {
                    case ErrorType.ServiceStopped:
                        {
                            SampleAppProgram.walwoaForm.SetServiceSdkVersion("Waiting for connection...");
                            SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                            {
                                SampleAppProgram.walwoaForm.ClearFeatureCheckboxes();
                                MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                SampleAppProgram.walwoaForm.EnableFeatureCheckboxes();
                            }));
                        }
                        break;
                    case ErrorType.SetOptionError:
                        {
                            MessageBox.Show(state.Description, "Setting Option for " + featureType + "Failed", MessageBoxButtons.OK, MessageBoxIcon.Error);
                            this.logger.Error(state.Description);
                        }
                        break;
                    case ErrorType.IpfUnavailable:
                        {
                            this.logger.Info("CST service lost IPF connection.");
                            SampleAppProgram.walwoaForm.SetIpfConnectionState(false);
                            new Thread(() =>
                            {
                                MessageBox.Show("CST service lost IPF connection. CST will not take action until restored.", "IPF Communication Error", MessageBoxButtons.OK);
                            }).Start();
                        }
                        break;
                    default:
                        {
                            switch (featureType)
                            {
                                case FeatureType.LOCK:
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        SampleAppProgram.walwoaForm.walCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxWAL_CheckedChanged;
                                        if (SampleAppProgram.walwoaForm.walCheckBox.CheckState == CheckState.Checked)
                                        {
                                            SampleAppProgram.walwoaForm.walCheckBox.CheckState = CheckState.Unchecked;
                                            SampleAppProgram.walwoaForm.UpdateWalControlsVisibility(false);
                                            this.logger.Error(state.Description);
                                        }

                                        SampleAppProgram.walwoaForm.walCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxWAL_CheckedChanged;
                                        MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    }));
                                    break;
                                case FeatureType.WAKE:
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        SampleAppProgram.walwoaForm.woaCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxWOA_CheckedChanged;
                                        if (SampleAppProgram.walwoaForm.woaCheckBox.CheckState == CheckState.Checked)
                                        {
                                            SampleAppProgram.walwoaForm.woaCheckBox.CheckState = CheckState.Unchecked;
                                            SampleAppProgram.walwoaForm.UpdateWoaControlsVisibility(false);
                                            this.logger.Error(state.Description);
                                        }

                                        SampleAppProgram.walwoaForm.woaCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxWOA_CheckedChanged;
                                        MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    }));
                                    break;
                                case FeatureType.NOLOCKONPRESENCE:
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        SampleAppProgram.walwoaForm.nlopCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxNlop_CheckedChanged;
                                        if (SampleAppProgram.walwoaForm.nlopCheckBox.CheckState == CheckState.Checked)
                                        {
                                            SampleAppProgram.walwoaForm.nlopCheckBox.CheckState = CheckState.Unchecked;
                                            SampleAppProgram.walwoaForm.UpdateNLoPControlsVisibility(false);
                                            this.logger.Error(state.Description);
                                        }

                                        SampleAppProgram.walwoaForm.nlopCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxNlop_CheckedChanged;
                                        MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    }));
                                    break;
                                case FeatureType.ADAPTIVEDIMMING:
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        SampleAppProgram.walwoaForm.adCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxAd_CheckedChanged;
                                        if (SampleAppProgram.walwoaForm.adCheckBox.CheckState == CheckState.Checked)
                                        {
                                            SampleAppProgram.walwoaForm.adCheckBox.CheckState = CheckState.Unchecked;
                                            SampleAppProgram.walwoaForm.UpdateAdControlsVisibility(false);
                                            this.logger.Error(state.Description);
                                        }

                                        SampleAppProgram.walwoaForm.adCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxAd_CheckedChanged;
                                        MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    }));
                                    break;
                                case FeatureType.ONLOOKERDETECTION:
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.onlookerDetectionCheckBox_CheckedChanged;
                                        if (SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckState == CheckState.Checked)
                                        {
                                            SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckState = CheckState.Unchecked;
                                            SampleAppProgram.walwoaForm.UpdateOnlookerDetectionControlsVisibility(false);
                                            this.logger.Error(state.Description);
                                        }

                                        SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.onlookerDetectionCheckBox_CheckedChanged;
                                        MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    }));
                                    break;
                                case FeatureType.APPLICATION:
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        this.logger.Error(state.Description);
                                        MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
                                    }));
                                    break;
                            }
                        }
                        break;
                }
            }
            catch (Exception e)
            {
                this.logger.Debug(e.Message);
                this.logger.Debug("Exception Thrown in WAL OnError");
            }
        }

        public void OnEvent(FeatureType featureType, EventType eventType, object eventPayload)
        {
            try
            {
                this.logger.Info("Event " + eventType + " for feature " + featureType + " received from service. Event payload " + eventPayload?.ToString());
                switch (eventType)
                {
                    case EventType.EVENT_EXTERNAL_MONITOR:
                        {
                            if (eventPayload is bool externalMonitorConnected)
                            {
                                string message = externalMonitorConnected ? "connected" : "disconnected";

                                new Thread(() =>
                                {
                                    MessageBox.Show($"External Monitor {message}", "External Monitor", MessageBoxButtons.OK);
                                }).Start();
                                SampleAppProgram.walwoaForm.HandleExternalMonitorEvent(externalMonitorConnected);
                            }
                            else
                            {
                                this.logger.Error("Invalid external monitor status!");
                            }
                        }
                        break;
                    case EventType.EVENT_WAL:
                        {
                            //logger.Info("OnEvent: WAL in action " + eventType.ToString());
                            new Thread(() =>
                            {
                                MessageBox.Show("WAL Event", "WAL", MessageBoxButtons.OK);
                            }).Start();
                        }
                        break;
                    case EventType.EVENT_WOA:
                        {
                            //logger.Info("OnEvent: WOA in action " + eventType.ToString());
                            new Thread(() =>
                            {
                                MessageBox.Show("WOA Event", "WOA", MessageBoxButtons.OK);
                            }).Start();
                        }
                        break;
                    case EventType.PROXIMITY:
                        {
                            if (eventPayload is float proximityDistanceMm)
                            {
                                SampleAppProgram.walwoaForm.proximityDistanceValueLabel.Text = proximityDistanceMm.ToString();
                            }
                            else
                            {
                                this.logger.Error("Invalid proximity distance!");
                            }
                        }
                        break;
                    case EventType.EVENT_SERVICE_RECOVERED:
                        {
                            this.logger.Debug("CST service recovered, refreshing features");
                            if (eventPayload is CstConnectionInfo connectionInfo)
                            {
                                SampleAppProgram.walwoaForm.SetServiceSdkVersion(connectionInfo.ServiceSdkVersion);

                                SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                {
                                    SampleAppProgram.walwoaForm.EnableFeatureCheckboxes(true);
                                    _ = SampleAppProgram.walwoaForm.RefreshFeaturesViaGetOptions();
                                    _ = Task.Run(async () => await SampleAppProgram.walwoaForm.GetAllFeatureCapabilities());
                                }));
                            }
                            else
                            {
                                this.logger.Error("Invalid CstConnectionInfo!");
                            }
                        }
                        break;
                    case EventType.EVENT_ONLOOKER_DETECTED:
                        {
                            if (eventPayload is bool onlookerPresent)
                            {
                                // This check is to prevent redundant pop-ups if an onlooker event happens before the initial state has been pulled in via capabilities
                                // You usually don't need this kind of check if you aren't using pop-ups.
                                if (SampleAppProgram.walwoaForm.CheckedInitialOnlookerState)
                                {
                                    string message = onlookerPresent ? "detected" : "clear";

                                    if (onlookerPresent)
                                    {
                                        SampleAppProgram.walwoaForm.ShowOnlookerPopup();
                                    }
                                    else
                                    {
                                        SampleAppProgram.walwoaForm.HandleOnlookerClear(message);
                                    }
                                }
                            }
                            else
                            {
                                this.logger.Error("Invalid Onlooker Detection status!");
                            }
                        }
                        break;
                    case EventType.EVENT_ADAPTIVE_DIMMING:
                        {
                            if (eventPayload is bool adaptiveDimmingStatus)
                            {
                                if (SampleAppProgram.walwoaForm.CheckedInitialAdaptiveDimmingState)
                                {
                                    SampleAppProgram.walwoaForm.ShowAdaptiveDimmingPopup(adaptiveDimmingStatus);
                                }
                            }
                            else
                            {
                                this.logger.Error("Invalid Adaptive Dimming status!");
                            }
                        }
                        break;
                    case EventType.EVENT_CST_CAPABILITY_CHANGED:
                    case EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED:
                        {
                            try
                            {
                                new Thread(() =>
                                {
                                    // Query for status
                                    SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
                                    {
                                        _ = Task.Run(async () => await SampleAppProgram.walwoaForm.RefreshFeatureCapability(featureType, eventType));
                                    }));
                                }).Start();
                            }
                            catch (Exception ex)
                            {
                                this.logger.Debug("Unable to get feature capability for feature " + featureType + " received from service. Exception = " + ex.Message);
                            }
                        }
                        break;
                    case EventType.EVENT_IPF_STARTED:
                        {
                            this.logger.Info("CST service's IPF connection recovered.");
                            SampleAppProgram.walwoaForm.SetIpfConnectionState(true);
                            new Thread(() =>
                            {
                                MessageBox.Show("CST service's IPF connection recovered.", "IPF Communication Restored", MessageBoxButtons.OK);
                            }).Start();

                            // Refresh feature capabilities
                            _ = Task.Run(async () => await SampleAppProgram.walwoaForm.GetAllFeatureCapabilities());
                        }
                        break;
                    case EventType.EVENT_SNOOZE_COMPLETE:
                        {
                            this.logger.Info("Received Snooze Complete event.");
                            SampleAppProgram.walwoaForm.HandleSnoozeComplete();
                        }
                        break;
                    case EventType.EVENT_BIOMETRIC_SENSOR_STATUS_CHANGED:
                        {
                            if (eventPayload is bool sensorReady)
                            {
                                string message = sensorReady ? "ready" : "not ready";

                                new Thread(() =>
                                {
                                    MessageBox.Show($"Biometric presence sensor is now {message}", "Biometric Presence Sensor Status", MessageBoxButtons.OK);
                                }).Start();
                            }
                            else
                            {
                                this.logger.Error("Invalid biometric sensor status!");
                            }

                            break;
                        }
                    case EventType.EVENT_HUMAN_PRESENCE_CHANGED:
                        {
                            if (eventPayload is HumanPresenceStatusPacket updatedPresence)
                            {
                                if (!SampleAppProgram.walwoaForm.HumanPresencePollEnabled)
                                {
                                    _ = Task.Run(async () => await SampleAppProgram.walwoaForm.HumanPresenceChannelWriter.WriteAsync(updatedPresence));
                                }
                            }
                            else
                            {
                                this.logger.Error("Invalid HumanPresenceStatusPacket!");
                            }

                            break;
                        }
                }
            }
            catch (Exception e)
            {
                this.logger.Debug(e.Message);
                this.logger.Error("Exception on handling OnEvent");
            }
        }

        public void OnSuccess(FeatureType featureType, ResponseType responseType)
        {
            try
            {
                this.logger.Info("OnSuccess with response " + responseType.ToString() + " from service, for feature " + featureType.ToString());
                switch (featureType)
                {
                    case FeatureType.LOCK:
                        SampleAppProgram.walwoaForm.walCheckBox.Enabled = true;
                        break;
                    case FeatureType.WAKE:
                        SampleAppProgram.walwoaForm.woaCheckBox.Enabled = true;
                        break;
                    case FeatureType.NOLOCKONPRESENCE:
                        SampleAppProgram.walwoaForm.nlopCheckBox.Enabled = true;
                        break;
                    case FeatureType.ADAPTIVEDIMMING:
                        SampleAppProgram.walwoaForm.adCheckBox.Enabled = true;
                        break;
                    case FeatureType.ONLOOKERDETECTION:
                        SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.Enabled = true;
                        break;
                    case FeatureType.APPLICATION:
                        this.logger.Info("Enable/Disable success for Privacy " + responseType.ToString());
                        String successMsg = "";
                        switch (responseType)
                        {
                            case ResponseType.NO_CHANGE:
                                successMsg = "No face detection sensor found on the system. Sensor switch did not occur.";
                                break;
                            case ResponseType.ENABLE_SUCCESS:
                                successMsg = "Enabled privacy";
                                break;
                            case ResponseType.DISABLE_SUCCESS:
                                successMsg = "Disabled privacy";
                                break;
                        }

                        new Thread(() =>
                        {
                            MessageBox.Show(successMsg, "Privacy Switch", MessageBoxButtons.OK);
                        }).Start();
                        break;
                }
            }
            catch (Exception e)
            {
                this.logger.Debug(e.Message);
                this.logger.Error("Exception on handling OnSuccess");
            }
        }
    }
}
