# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2023 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseDeclaredVarsMoreThanAssignments", "")] Param()

#region CSharp_code
$Dflat = @"
using System;
using System.Runtime.InteropServices;

namespace IntelEthernetCmdlets
{
    class Util
    {
        public static bool CheckBit(UInt16 Value, UInt16 Bit)
        {
            if ((Value & Bit) == Bit)
            {
                return true;
            }
            return false;
        }
    }

    public class DcbConstants
    {
        public const Int32 MAX_TRAFFIC_CLASS = 8;
        public const Int32 MAX_USER_PRIORITY = 8;
        public const Int32 DCBX_MAX_APPS = 32;
        public const Int32 DCBX_APPS_SIZE = DCBX_MAX_APPS * 4;
    }

    public class DcbDataRegFlagValue
    {
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_CAP = 0x0001;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_EN = 0x0002;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_FW_INC = 0x0004;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_NPAR = 0x0008;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_DCB_ISCSI = 0x0010;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_MAIN_VSI = 0x0020;
        public const UInt16 GET_DCB_DATA_REGS_FLAG_FCOE_VSI = 0x0040;
    }

    public class DcbxModeValue
    {
        public const Byte DCBX_MODE_CEE = 0x1;
        public const Byte DCBX_MODE_IEEE = 0x2;
    }

    public enum RegVer
    {
        VER_INVALID,
        I40E,
        ICE
    }

    public class IntlLan_GetDcbData
    {
        public IntlLan_GetDcbData()
        {
            LocalConfig = new IntlLan_GetDcbDataDcbxConfig();
            LocalConfig.ConfigName = "LocalConfig";

            RemoteConfig = new IntlLan_GetDcbDataDcbxConfig();
            RemoteConfig.ConfigName = "RemoteConfig";

            DesiredConfig = new IntlLan_GetDcbDataDcbxConfig();
            DesiredConfig.ConfigName = "DesiredConfig";
        }
        public String Name { get; set; }
        public UInt16 Flags { get; set; }
        public Boolean DcbCapable { get; set; }
        public Boolean DcbEnabled { get; set; }

        public UInt16 DcbxStatus { get; set; }

        public IntlLan_GetDcbDataDcbxConfig LocalConfig { get; set; }
        public IntlLan_GetDcbDataDcbxConfig RemoteConfig { get; set; }
        public IntlLan_GetDcbDataDcbxConfig DesiredConfig { get; set; }

        public IIntlLan_GetDcbDataRegisters DcbRegs { get; set; }
        public RegVer RegVer { get; set; }
    }

    public class IntlLan_GetDcbDataDcbxConfig
    {
        public IntlLan_GetDcbDataDcbxConfig()
        {
            EtsCfg = new IntlLan_GetDcbDataEts();
            EtsCfg.TableName = "EtsCfg";
            EtsRec = new IntlLan_GetDcbDataEts();
            EtsRec.TableName = "EtsRec";
            Pfc = new IntlLan_GetDcbDataPfc();
        }
        public String Name { get; set; }
        public Byte DcbxMode { get; set; }
        public Byte AppMode { get; set; }
        public UInt32 CeeTlvStatus { get; set; }
        public UInt32 NumApps { get; set; }

        public String ConfigName { get; set; }

        public IntlLan_GetDcbDataEts EtsCfg { get; set; }
        public IntlLan_GetDcbDataEts EtsRec { get; set; }
        public IntlLan_GetDcbDataPfc Pfc { get; set; }
        public IntlLan_GetDcbDataAppPriority[] AppPriority { get; set; }
    }

    public class IntlLan_GetDcbDataEts
    {
        public IntlLan_GetDcbDataEts()
        {
            UP = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            BW = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            TSA = new Byte[DcbConstants.MAX_TRAFFIC_CLASS];
            PfcEnableBits = new Boolean[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public String TableName { get; set; }
        public Boolean Willing { get; set; }
        public Byte CBS { get; set; }
        public Byte MaxTCs { get; set; }
        public Boolean[] PfcEnableBits { get; set; }
        public Byte[] UP { get; set; }
        public Byte[] BW { get; set; }
        public Byte[] TSA { get; set; }
    }

    public class IntlLan_GetDcbDataPfc
    {
        public IntlLan_GetDcbDataPfc()
        {
            PfcEnableBits = new Boolean[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public Boolean Willing { get; set; }
        public Byte MBC { get; set; }
        public Byte PFCcap { get; set; }
        public Byte PfcEnable { get; set; }
        public Boolean[] PfcEnableBits { get; set; }
    }

    public class IntlLan_GetDcbDataAppPriority
    {
        public Byte Priority { get; set; }
        public Byte Selector { get; set; }
        public UInt16 ProtocolId { get; set; }
    }

    public interface IIntlLan_GetDcbDataRegisters { }

    public class IntlLan_GetDcbDataRegistersI40e : IIntlLan_GetDcbDataRegisters
    {
        public IntlLan_GetDcbDataRegistersI40e()
        {
            I40E_PRT_TCTUPR = new UInt32[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public UInt32 I40E_PRTDCB_GENC { get; set; }
        public UInt32 I40E_PRTDCB_TC2PFC { get; set; }
        public UInt32 I40E_PRTDCB_MFLCN { get; set; }
        public UInt32 I40E_PRTDCB_TUP2TC { get; set; }
        public UInt32 I40E_PRTDCB_RUP2TC { get; set; }
        public UInt32[] I40E_PRT_TCTUPR { get; set; }
        public UInt32 I40E_VSI_TUPR_MainVsi { get; set; }
        public UInt32 I40E_VSI_RUPR_MainVsi { get; set; }
        public UInt32 I40E_VSI_TUPR_FCoEVsi { get; set; }
        public UInt32 I40E_VSI_RUPR_FCoEVsi { get; set; }
    }

    public class IntlLan_GetDcbDataRegistersIce : IIntlLan_GetDcbDataRegisters
    {
        public IntlLan_GetDcbDataRegistersIce()
        {
            PrtTcTuprVal = new UInt32[DcbConstants.MAX_TRAFFIC_CLASS];
        }
        public String Name { get; set; }
        public UInt32 PrtDcbGencVal { get; set; }
        public UInt32 GlDcbTc2PfcVal { get; set; }
        public UInt32 PrtMacHsecCtlTxPauseEnableVal { get; set; }
        public UInt32 PrtMacHsecCtlRxPauseEnableVal { get; set; }
        public UInt32 PrtDcbTup2TcVal { get; set; }
        public UInt32 PrtDcbRup2TcVal { get; set; }
        public UInt32[] PrtTcTuprVal { get; set; }
        public UInt32 VsiTuprMainVsiVal { get; set; }
        public UInt32 VsiRuprMainVsiVal { get; set; }
    }

    public unsafe interface IGetDcb
    {
        IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes);
    }

    #region 100G DCB
    public unsafe class GetDcbIce : IGetDcb
    {
        private const Int32 ICE_DSCP_NUM_VAL = 64;
        private const Int32 DSCP_MAPPED_SIZE = 8;


        private struct ice_dcb_ets_cfg
        {
            public Byte willing;
            public Byte cbs;
            public Byte maxtcs;
            public fixed Byte prio_table[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tcbwtable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tsatable[DcbConstants.MAX_TRAFFIC_CLASS];
        }


        private struct ice_dcb_pfc_cfg
        {
            public Byte willing;
            public Byte mbc;
            public Byte pfccap;
            public Byte pfcenable;
        }


        private struct ice_dcb_app_priority_table
        {
            public UInt16 prot_id;
            public Byte priority;
            public Byte selector;
        }


        private struct ice_dcbx_cfg
        {
            public Byte dcbx_mode;
            public Byte app_mode;
            public UInt32 numapps;
            public UInt32 tlv_status;
            public ice_dcb_ets_cfg etscfg;
            public ice_dcb_ets_cfg etsrec;
            public ice_dcb_pfc_cfg pfc;
            public fixed Byte app[DcbConstants.DCBX_APPS_SIZE];
        }


        private struct DcbRegisters
        {
            public UInt32 PrtDcbGencVal;
            public UInt32 GlDcbTc2PfcVal;
            public UInt32 PrtMacHsecCtlTxPauseEnableVal;
            public UInt32 PrtMacHsecCtlRxPauseEnableVal;
            public UInt32 PrtDcbTup2TcVal;
            public UInt32 PrtDcbRup2TcVal;
            public fixed UInt32 PrtTcTuprVal[8];
            public UInt32 VsiTuprMainVsiVal;
            public UInt32 VsiRuprMainVsiVal;
        }


        private struct DcbGetGenData
        {
            public UInt16 Flags;
            public UInt16 DcbxStatus;
            public ice_dcbx_cfg LocalConfig;
            public ice_dcbx_cfg RemoteConfig;
            public ice_dcbx_cfg DesiredConfig;
            public DcbRegisters DcbRegs;
        }

        public IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes)
        {
            var pData = GCHandle.Alloc(bytes, GCHandleType.Pinned);
            var result = (DcbGetGenData)Marshal.PtrToStructure(pData.AddrOfPinnedObject(), typeof(DcbGetGenData));
            pData.Free();
            DcbGetGenData* pDcbData = (DcbGetGenData*)&result;
            return CreatePsObject(strPortName, *pDcbData);
        }

        private IntlLan_GetDcbData CreatePsObject(String strPortName, DcbGetGenData DcbData)
        {
            IntlLan_GetDcbData GetDcbData = new IntlLan_GetDcbData
            {
                Name = strPortName,
                Flags = DcbData.Flags,
                DcbCapable = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_CAP)),
                DcbEnabled = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_EN)),
                DcbxStatus = DcbData.DcbxStatus,
            };

            CopyDcbxConfig(strPortName, DcbData.LocalConfig, GetDcbData.LocalConfig);
            CopyDcbxConfig(strPortName, DcbData.RemoteConfig, GetDcbData.RemoteConfig);
            CopyDcbxConfig(strPortName, DcbData.DesiredConfig, GetDcbData.DesiredConfig);

            GetDcbData.DcbRegs = new IntlLan_GetDcbDataRegistersIce();
            GetDcbData.RegVer = RegVer.ICE;
            CopyDcbRegs(strPortName, DcbData.DcbRegs, GetDcbData.DcbRegs);
            return GetDcbData;
        }


        private void CopyDcbxConfig(String strPortName, ice_dcbx_cfg FromDcbxConfig, IntlLan_GetDcbDataDcbxConfig ToDcbxConfig)
        {
            ToDcbxConfig.Name = strPortName;
            ToDcbxConfig.DcbxMode = FromDcbxConfig.dcbx_mode;
            ToDcbxConfig.AppMode = FromDcbxConfig.app_mode;
            ToDcbxConfig.CeeTlvStatus = FromDcbxConfig.tlv_status;
            ToDcbxConfig.NumApps = FromDcbxConfig.numapps;
            if (ToDcbxConfig.NumApps > DcbConstants.DCBX_MAX_APPS)
            {
                ToDcbxConfig.NumApps = DcbConstants.DCBX_MAX_APPS;
            }

            CopyPfcConfig(strPortName, FromDcbxConfig.pfc, ToDcbxConfig.Pfc);

            CopyEtsConfig(strPortName, FromDcbxConfig.etscfg, ToDcbxConfig.EtsCfg);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsCfg.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            CopyEtsConfig(strPortName, FromDcbxConfig.etsrec, ToDcbxConfig.EtsRec);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsRec.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            ToDcbxConfig.AppPriority = new IntlLan_GetDcbDataAppPriority[ToDcbxConfig.NumApps];

            CopyApps(FromDcbxConfig.app, ToDcbxConfig.NumApps, ToDcbxConfig.AppPriority);
        }


        private void CopyEtsConfig(String strPortName, ice_dcb_ets_cfg FromEtsConfig, IntlLan_GetDcbDataEts ToEtsConfig)
        {
            ToEtsConfig.Name = strPortName;
            ToEtsConfig.Willing = (FromEtsConfig.willing == 1);
            ToEtsConfig.CBS = FromEtsConfig.cbs;
            ToEtsConfig.MaxTCs = FromEtsConfig.maxtcs;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToEtsConfig.UP[i] = FromEtsConfig.prio_table[i];
                ToEtsConfig.BW[i] = FromEtsConfig.tcbwtable[i];
                ToEtsConfig.TSA[i] = FromEtsConfig.tsatable[i];
            }
        }


        private void CopyPfcConfig(String strPortName, ice_dcb_pfc_cfg FromPfcConfig, IntlLan_GetDcbDataPfc ToPfcConfig)
        {
            ToPfcConfig.Name = strPortName;
            ToPfcConfig.Willing = (FromPfcConfig.willing == 1);
            ToPfcConfig.MBC = FromPfcConfig.mbc;
            ToPfcConfig.PFCcap = FromPfcConfig.pfccap;
            ToPfcConfig.PfcEnable = FromPfcConfig.pfcenable;
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToPfcConfig.PfcEnableBits[i] = ((ToPfcConfig.PfcEnable & (1 << i)) == (1 << i));
            }
        }


        private void CopyApps(Byte* FromApps, UInt32 NumApps, IntlLan_GetDcbDataAppPriority[] ToApps)
        {
            ice_dcb_app_priority_table* AppTable = (ice_dcb_app_priority_table*)FromApps;
            for (int i = 0; i < NumApps; i++)
            {
                // Since this array's elements are a class, and classes are ref types, we need
                // to explicitly allocate the elements.
                ToApps[i] = new IntlLan_GetDcbDataAppPriority();

                if (AppTable->prot_id > 0)
                {
                    ToApps[i].Priority = AppTable->priority;
                    ToApps[i].Selector = AppTable->selector;
                    ToApps[i].ProtocolId = AppTable->prot_id;
                }

                AppTable++;
            }
        }


        private void CopyDcbRegs(String strPortName, DcbRegisters FromDcbRegs, IIntlLan_GetDcbDataRegisters ToDcbRegs)
        {
            IntlLan_GetDcbDataRegistersIce toReg = (IntlLan_GetDcbDataRegistersIce)ToDcbRegs;

            toReg.Name = strPortName;
            toReg.PrtDcbGencVal = FromDcbRegs.PrtDcbGencVal;
            toReg.GlDcbTc2PfcVal = FromDcbRegs.GlDcbTc2PfcVal;
            toReg.PrtMacHsecCtlTxPauseEnableVal = FromDcbRegs.PrtMacHsecCtlTxPauseEnableVal;
            toReg.PrtMacHsecCtlRxPauseEnableVal = FromDcbRegs.PrtMacHsecCtlRxPauseEnableVal;
            toReg.PrtDcbTup2TcVal = FromDcbRegs.PrtDcbTup2TcVal;
            toReg.PrtDcbRup2TcVal = FromDcbRegs.PrtDcbRup2TcVal;
            toReg.VsiTuprMainVsiVal = FromDcbRegs.VsiTuprMainVsiVal;
            toReg.VsiRuprMainVsiVal = FromDcbRegs.VsiRuprMainVsiVal;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                toReg.PrtTcTuprVal[i] = FromDcbRegs.PrtTcTuprVal[i];
            }
        }
    }
    #endregion

    #region 40G DCB
    public unsafe class GetDcbI40e : IGetDcb
    {
        private struct i40e_dcb_ets_config
        {
            public Byte willing;
            public Byte cbs;
            public Byte maxtcs;
            public fixed Byte prioritytable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tcbwtable[DcbConstants.MAX_TRAFFIC_CLASS];
            public fixed Byte tsatable[DcbConstants.MAX_TRAFFIC_CLASS];
        }

        private struct i40e_dcb_pfc_config
        {
            public Byte willing;
            public Byte mbc;
            public Byte pfccap;
            public Byte pfcenable;
        }

        private struct i40e_dcb_app_priority_table
        {
            public Byte priority;
            public Byte selector;
            public UInt16 protocolid;
        }

        private struct i40e_dcbx_config
        {
            public Byte dcbx_mode;
            public Byte app_mode;
            public UInt32 numapps;
            public UInt32 tlv_status;
            public i40e_dcb_ets_config etscfg;
            public i40e_dcb_ets_config etsrec;
            public i40e_dcb_pfc_config pfc;
            public fixed Byte app[DcbConstants.DCBX_APPS_SIZE];
        }

        private struct DcbRegisters
        {
            public UInt32 I40E_PRTDCB_GENC;
            public UInt32 I40E_PRTDCB_TC2PFC;
            public UInt32 I40E_PRTDCB_MFLCN;
            public UInt32 I40E_PRTDCB_TUP2TC;
            public UInt32 I40E_PRTDCB_RUP2TC;
            public fixed UInt32 I40E_PRT_TCTUPR[8];
            public UInt32 I40E_VSI_TUPR_MainVsi;
            public UInt32 I40E_VSI_RUPR_MainVsi;
            public UInt32 I40E_VSI_TUPR_FCoEVsi;
            public UInt32 I40E_VSI_RUPR_FCoEVsi;
        }

        private struct DcbGetGenData
        {
            public UInt16 Flags;
            public UInt16 DcbxStatus;
            public i40e_dcbx_config LocalConfig;
            public i40e_dcbx_config RemoteConfig;
            public i40e_dcbx_config DesiredConfig;
            public DcbRegisters DcbRegs;
        }

        public IntlLan_GetDcbData GetDcbConfigs(String strPortName, byte[] bytes)
        {
            var pData = GCHandle.Alloc(bytes, GCHandleType.Pinned);
            var result = (DcbGetGenData)Marshal.PtrToStructure(pData.AddrOfPinnedObject(), typeof(DcbGetGenData));
            pData.Free();
            DcbGetGenData* pDcbData = (DcbGetGenData*)&result;
            return CreatePsObject(strPortName, *pDcbData);
        }

        private IntlLan_GetDcbData CreatePsObject(String strPortName, DcbGetGenData DcbData)
        {
            IntlLan_GetDcbData GetDcbData = new IntlLan_GetDcbData
            {
                Name = strPortName,
                Flags = DcbData.Flags,
                DcbCapable = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_CAP)),
                DcbEnabled = (Util.CheckBit(DcbData.Flags, DcbDataRegFlagValue.GET_DCB_DATA_REGS_FLAG_DCB_EN)),
                DcbxStatus = DcbData.DcbxStatus,
            };

            CopyDcbxConfig(strPortName, DcbData.LocalConfig, GetDcbData.LocalConfig);
            CopyDcbxConfig(strPortName, DcbData.RemoteConfig, GetDcbData.RemoteConfig);
            CopyDcbxConfig(strPortName, DcbData.DesiredConfig, GetDcbData.DesiredConfig);

            GetDcbData.DcbRegs = new IntlLan_GetDcbDataRegistersI40e();
            GetDcbData.RegVer = RegVer.I40E;
            CopyDcbRegs(strPortName, DcbData.DcbRegs, GetDcbData.DcbRegs);
            return GetDcbData;
        }

        private void CopyDcbxConfig(String strPortName, i40e_dcbx_config FromDcbxConfig, IntlLan_GetDcbDataDcbxConfig ToDcbxConfig)
        {
            ToDcbxConfig.Name = strPortName;
            ToDcbxConfig.DcbxMode = FromDcbxConfig.dcbx_mode;
            ToDcbxConfig.AppMode = FromDcbxConfig.app_mode;
            ToDcbxConfig.CeeTlvStatus = FromDcbxConfig.tlv_status;
            ToDcbxConfig.NumApps = FromDcbxConfig.numapps;
            if (ToDcbxConfig.NumApps > DcbConstants.DCBX_MAX_APPS)
            {
                ToDcbxConfig.NumApps = DcbConstants.DCBX_MAX_APPS;
            }

            CopyPfcConfig(strPortName, FromDcbxConfig.pfc, ToDcbxConfig.Pfc);

            CopyEtsConfig(strPortName, FromDcbxConfig.etscfg, ToDcbxConfig.EtsCfg);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsCfg.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            CopyEtsConfig(strPortName, FromDcbxConfig.etsrec, ToDcbxConfig.EtsRec);
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToDcbxConfig.EtsRec.PfcEnableBits[i] = ToDcbxConfig.Pfc.PfcEnableBits[i];
            }

            ToDcbxConfig.AppPriority = new IntlLan_GetDcbDataAppPriority[ToDcbxConfig.NumApps];

            CopyApps(FromDcbxConfig.app, ToDcbxConfig.NumApps, ToDcbxConfig.AppPriority);
        }


        private void CopyEtsConfig(String strPortName, i40e_dcb_ets_config FromEtsConfig, IntlLan_GetDcbDataEts ToEtsConfig)
        {
            ToEtsConfig.Name = strPortName;
            ToEtsConfig.Willing = (FromEtsConfig.willing == 1);
            ToEtsConfig.CBS = FromEtsConfig.cbs;
            ToEtsConfig.MaxTCs = FromEtsConfig.maxtcs;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToEtsConfig.UP[i] = FromEtsConfig.prioritytable[i];
                ToEtsConfig.BW[i] = FromEtsConfig.tcbwtable[i];
                ToEtsConfig.TSA[i] = FromEtsConfig.tsatable[i];
            }
        }


        private void CopyPfcConfig(String strPortName, i40e_dcb_pfc_config FromPfcConfig, IntlLan_GetDcbDataPfc ToPfcConfig)
        {
            ToPfcConfig.Name = strPortName;
            ToPfcConfig.Willing = (FromPfcConfig.willing == 1);
            ToPfcConfig.MBC = FromPfcConfig.mbc;
            ToPfcConfig.PFCcap = FromPfcConfig.pfccap;
            ToPfcConfig.PfcEnable = FromPfcConfig.pfcenable;
            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                ToPfcConfig.PfcEnableBits[i] = ((ToPfcConfig.PfcEnable & (1 << i)) == (1 << i));
            }
        }


        private void CopyApps(Byte* FromApps, UInt32 NumApps, IntlLan_GetDcbDataAppPriority[] ToApps)
        {
            i40e_dcb_app_priority_table* AppTable = (i40e_dcb_app_priority_table*)FromApps;
            for (int i = 0; i < NumApps; i++)
            {
                // Since this array's elements are a class, which and classes are ref types, we need
                // to explicitly allocate the elements.
                ToApps[i] = new IntlLan_GetDcbDataAppPriority();

                if (AppTable->protocolid > 0)
                {
                    ToApps[i].Priority = AppTable->priority;
                    ToApps[i].Selector = AppTable->selector;
                    ToApps[i].ProtocolId = AppTable->protocolid;
                }

                AppTable++;
            }
        }


        private void CopyDcbRegs(String strPortName, DcbRegisters FromDcbRegs, IIntlLan_GetDcbDataRegisters ToDcbRegs)
        {
            IntlLan_GetDcbDataRegistersI40e toReg = (IntlLan_GetDcbDataRegistersI40e)ToDcbRegs;

            toReg.Name = strPortName;
            toReg.I40E_PRTDCB_GENC = FromDcbRegs.I40E_PRTDCB_GENC;
            toReg.I40E_PRTDCB_TC2PFC = FromDcbRegs.I40E_PRTDCB_TC2PFC;
            toReg.I40E_PRTDCB_MFLCN = FromDcbRegs.I40E_PRTDCB_MFLCN;
            toReg.I40E_PRTDCB_TUP2TC = FromDcbRegs.I40E_PRTDCB_TUP2TC;
            toReg.I40E_PRTDCB_RUP2TC = FromDcbRegs.I40E_PRTDCB_RUP2TC;

            for (Byte i = 0; i < DcbConstants.MAX_TRAFFIC_CLASS; i++)
            {
                toReg.I40E_PRT_TCTUPR[i] = FromDcbRegs.I40E_PRT_TCTUPR[i];
            }

            toReg.I40E_VSI_TUPR_MainVsi = FromDcbRegs.I40E_VSI_TUPR_MainVsi;
            toReg.I40E_VSI_RUPR_MainVsi = FromDcbRegs.I40E_VSI_RUPR_MainVsi;
            toReg.I40E_VSI_TUPR_FCoEVsi = FromDcbRegs.I40E_VSI_TUPR_FCoEVsi;
            toReg.I40E_VSI_RUPR_FCoEVsi = FromDcbRegs.I40E_VSI_RUPR_FCoEVsi;
        }
    }
    #endregion
}
"@
#endregion CSharp_code

# SIG # Begin signature block
# MIIocAYJKoZIhvcNAQcCoIIoYTCCKF0CAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCM5Jlr7XMXI4xy
# ovQHeqTjsb8STCSzX/UJOOHIVbKPIqCCEfIwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYcMIIEBKADAgECAhAz1wio
# kUBTGeKlu9M5ua1uMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFcxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLjAs
# BgNVBAMTJVNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBFViBSMzYwggGi
# MA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC70f4et0JbePWQp64sg/GNIdMw
# hoV739PN2RZLrIXFuwHP4owoEXIEdiyBxasSekBKxRDogRQ5G19PB/YwMDB/NSXl
# wHM9QAmU6Kj46zkLVdW2DIseJ/jePiLBv+9l7nPuZd0o3bsffZsyf7eZVReqskmo
# PBBqOsMhspmoQ9c7gqgZYbU+alpduLyeE9AKnvVbj2k4aOqlH1vKI+4L7bzQHkND
# brBTjMJzKkQxbr6PuMYC9ruCBBV5DFIg6JgncWHvL+T4AvszWbX0w1Xn3/YIIq62
# 0QlZ7AGfc4m3Q0/V8tm9VlkJ3bcX9sR0gLqHRqwG29sEDdVOuu6MCTQZlRvmcBME
# Jd+PuNeEM4xspgzraLqVT3xE6NRpjSV5wyHxNXf4T7YSVZXQVugYAtXueciGoWnx
# G06UE2oHYvDQa5mll1CeHDOhHu5hiwVoHI717iaQg9b+cYWnmvINFD42tRKtd3V6
# zOdGNmqQU8vGlHHeBzoh+dYyZ+CcblSGoGSgg8sCAwEAAaOCAWMwggFfMB8GA1Ud
# IwQYMBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBSBMpJBKyjNRsjE
# osYqORLsSKk/FDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADAT
# BgNVHSUEDDAKBggrBgEFBQcDAzAaBgNVHSAEEzARMAYGBFUdIAAwBwYFZ4EMAQMw
# SwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRvMG0wRgYI
# KwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0Nv
# ZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9vY3NwLnNl
# Y3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBfNqz7+fZyWhS38Asd3tj9lwHS
# /QHumS2G6Pa38Dn/1oFKWqdCSgotFZ3mlP3FaUqy10vxFhJM9r6QZmWLLXTUqwj3
# ahEDCHd8vmnhsNufJIkD1t5cpOCy1rTP4zjVuW3MJ9bOZBHoEHJ20/ng6SyJ6UnT
# s5eWBgrh9grIQZqRXYHYNneYyoBBl6j4kT9jn6rNVFRLgOr1F2bTlHH9nv1HMePp
# GoYd074g0j+xUl+yk72MlQmYco+VAfSYQ6VK+xQmqp02v3Kw/Ny9hA3s7TSoXpUr
# OBZjBXXZ9jEuFWvilLIq0nQ1tZiao/74Ky+2F0snbFrmuXZe2obdq2TWauqDGIgb
# MYL1iLOUJcAhLwhpAuNMu0wqETDrgXkG4UGVKtQg9guT5Hx2DJ0dJmtfhAH2KpnN
# r97H8OQYok6bLyoMZqaSdSa+2UA1E2+upjcaeuitHFFjBypWBmztfhj24+xkc6Zt
# CDaLrw+ZrnVrFyvCTWrDUUZBVumPwo3/E3Gb2u2e05+r5UWmEsUUWlJBl6MGAAjF
# 5hzqJ4I8O9vmRsTvLQA1E802fZ3lqicIBczOwDYOSxlP0GOabb/FKVMxItt1UHeG
# 0PL4au5rBhs+hSMrl8h+eplBDN1Yfw6owxI9OjWb4J0sjBeBVESoeh2YnZZ/WVim
# VGX/UUIL+Efrz/jlvzCCBlswggTDoAMCAQICEDB3Np9sRenync55S1/V2zEwDQYJ
# KoZIhvcNAQELBQAwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIEVW
# IFIzNjAeFw0yNDAxMTYwMDAwMDBaFw0yNTAxMTUyMzU5NTlaMIG7MRAwDgYDVQQF
# EwcyMTg5MDc0MRMwEQYLKwYBBAGCNzwCAQMTAlVTMRkwFwYLKwYBBAGCNzwCAQIT
# CERlbGF3YXJlMR0wGwYDVQQPExRQcml2YXRlIE9yZ2FuaXphdGlvbjELMAkGA1UE
# BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExGjAYBgNVBAoMEUludGVsIENvcnBv
# cmF0aW9uMRowGAYDVQQDDBFJbnRlbCBDb3Jwb3JhdGlvbjCCAaIwDQYJKoZIhvcN
# AQEBBQADggGPADCCAYoCggGBALUx6h43Sbt+ZfuTa/V0GFWgAAfUIWg9ruenPoc8
# FTqJxPdbqYRlGKEK2vCSQyBTn7tGV54Y/hMGvRTBQce4dMFe7R5wL9p92J3R4jkR
# MadZxJ4HuWalG13CpjQZP1Jg61Wx0KjsCPd2VdbBTpTiLRI+PKjokE6I/TLuA2bA
# tqy1phe2/82SRlx/fVGuIzI1BOQCaC9Olao3kJ9JwKt6VZidIgmzXAYLpzZ2VhW9
# HTRLaqTLTKxgO2mtpMTbfqc7RR/oYHG2cbGCoW3KB5Tlgjhm85ZsMEMV48O7JWeB
# wQoNy9admwqsdUzr5HdDbM7/EuiNHvP+sIauOmHEkLtqxM37sSgDBQX3V+vqiR5b
# H6xTcfIceaa4ukwKuVmfWrB35TGSEkWFzZu3ZUYGKRCI9WZ8GixV/5PHm571xSrD
# jbrigtkHtPahVVC8EUojnIGiDXovnm7ZG0FEXKuVybNOSpD3BzE2Pw3j/evKHuIr
# hWCzdPBOkbbEoYFBxts6UtJNFwIDAQABo4IBvDCCAbgwHwYDVR0jBBgwFoAUgTKS
# QSsozUbIxKLGKjkS7EipPxQwHQYDVR0OBBYEFKRLkSq7xH/Ru5kWUgmDt9ZZsJHj
# MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBMGA1UdJQQMMAoGCCsGAQUF
# BwMDMEkGA1UdIARCMEAwNQYMKwYBBAGyMQECAQYBMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAcGBWeBDAEDMEsGA1UdHwREMEIwQKA+oDyG
# Omh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1YmxpY0NvZGVTaWduaW5n
# Q0FFVlIzNi5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsGAQUFBzAChjpodHRwOi8v
# Y3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2RlU2lnbmluZ0NBRVZSMzYu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTAuBgNVHREE
# JzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVMQVdBUkUtMjE4OTA3NDANBgkqhkiG
# 9w0BAQsFAAOCAYEAI6NvLiKQmXol9qHiDAxrF41RjhawR5g4ZQuWIxU2uMkjL9Eh
# NV16cKJjZ+KSPebFErInVlf/NLvKCfSUpf+2olf7Phu1pbb2p2R1mFtGYIA1Bcwa
# UmIsA/XsQ+ZHYlVdqBcpJCdc/jTzfAQen0gv1hFOIfk0vOKjjAAuTIgQNkG3c5Sw
# FMZjhu+wrSXx9Qvv85BhIX/xF1xYkKN4iBwqqebxAhmaE2tZUI7X/kKDx9QexONP
# Wo5Vw9b3P9I9pP8pWS/Txa4AC8x5AcZ8f6FUkl5Y750Ma6XEeHJZlUGkf1d/Ph9M
# c6bg5/P4wuXceTd4iqu3sK4ZxbiXWiFD+PtZ9UbzDarUNCqskSwudIzEcgwt7glZ
# YUNC+gD64uddohltoXXIasQNs/LK+qzOuqZKhFkRf+v4+Q28EQYuVIcJDHspW1CV
# k3Y4zhEb1dhCAPZ9jyXyz827Uji6HD5dnGO2lPCcEvM/aWEjYYM285g0CZix3LeB
# XQydSyjK95klPc99MYIV1DCCFdACAQEwazBXMQswCQYDVQQGEwJHQjEYMBYGA1UE
# ChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1YmxpYyBDb2Rl
# IFNpZ25pbmcgQ0EgRVYgUjM2AhAwdzafbEXp8p3OeUtf1dsxMA0GCWCGSAFlAwQC
# AQUAoGowGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARYwLwYJKoZIhvcNAQkEMSIEILtfUM0X1mjWTTiEYKCdX8y4
# jyNGTJYIkChhlOmstCwpMA0GCSqGSIb3DQEBAQUABIIBgDsIfD9CVlzBET73AHS2
# VMCdDm3z247x4qDQO1tA8FE9boUzcN+dXPUT+V+oW2atb0qpioyqm5nrBhRswmJO
# fOxEcSZ5OpYt01F5VqVBRDy5Ht+hrYLJXXAvO4OP6nTVqOA8SU0NEUHuL5ZEUtqS
# 4woRNXp3VDc9VbqPylq6+H/N3ZKZfOMaDE+sGLnWl5MsHVBBYtx03UPEMFJebg+R
# LgxasvTFKztfveZFGy0I3TkUxqPhSopdojjCQj8a4AIopkcIGk4TqD67oI47xRyE
# M7IeKWJMtbIHlM6AJ8s7pCPa059AyZzDsv4sy/1qPG5TFmG8K679qDzW3Tdjtvqq
# tz3VVjqtDWz30sXjBGEfQI7kKEqiiFnt8V8Y/DEqwybaz3yutMvwSAW3b2FpY+zV
# 19cM3WRb4vvG7QsPeDfhRC2D8rRIIItZ7xSvYPPsnQOxbJkS8VoIXGcKNGfWEL1W
# zckWx3BJ8ymGIAf9qc5ZcGDbhE1Smuma02a59JLyD2VGl6GCE04wghNKBgorBgEE
# AYI3AwMBMYITOjCCEzYGCSqGSIb3DQEHAqCCEycwghMjAgEDMQ8wDQYJYIZIAWUD
# BAICBQAwge8GCyqGSIb3DQEJEAEEoIHfBIHcMIHZAgEBBgorBgEEAbIxAgEBMDEw
# DQYJYIZIAWUDBAIBBQAEIJtmYBkvHZ7EnOPaPsfzPB/442HrRivPNSz7n1L0sm6Y
# AhRitO1B0H6XqVoIHO+S1JQJ3eo6ihgPMjAyNDAzMTgxODIzNDZaoG6kbDBqMQsw
# CQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2VjdGln
# byBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIFNp
# Z25lciAjNKCCDekwggb1MIIE3aADAgECAhA5TCXhfKBtJ6hl4jvZHSLUMA0GCSqG
# SIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNo
# ZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTAeFw0yMzA1
# MDMwMDAwMDBaFw0zNDA4MDIyMzU5NTlaMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQI
# EwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMM
# I1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICM0MIICIjANBgkqhkiG
# 9w0BAQEFAAOCAg8AMIICCgKCAgEApJMoUkvPJ4d2pCkcmTjA5w7U0RzsaMsBZOSK
# zXewcWWCvJ/8i7u7lZj7JRGOWogJZhEUWLK6Ilvm9jLxXS3AeqIO4OBWZO2h5YEg
# ciBkQWzHwwj6831d7yGawn7XLMO6EZge/NMgCEKzX79/iFgyqzCz2Ix6lkoZE1ys
# /Oer6RwWLrCwOJVKz4VQq2cDJaG7OOkPb6lampEoEzW5H/M94STIa7GZ6A3vu03l
# PYxUA5HQ/C3PVTM4egkcB9Ei4GOGp7790oNzEhSbmkwJRr00vOFLUHty4Fv9Gbsf
# PGoZe267LUQqvjxMzKyKBJPGV4agczYrgZf6G5t+iIfYUnmJ/m53N9e7UJ/6GCVP
# E/JefKmxIFopq6NCh3fg9EwCSN1YpVOmo6DtGZZlFSnF7TMwJeaWg4Ga9mBmkFgH
# gM1Cdaz7tJHQxd0BQGq2qBDu9o16t551r9OlSxihDJ9XsF4lR5F0zXUS0Zxv5F4N
# m+x1Ju7+0/WSL1KF6NpEUSqizADKh2ZDoxsA76K1lp1irScL8htKycOUQjeIIISo
# h67DuiNye/hU7/hrJ7CF9adDhdgrOXTbWncC0aT69c2cPcwfrlHQe2zYHS0RQlNx
# dMLlNaotUhLZJc/w09CRQxLXMn2YbON3Qcj/HyRU726txj5Ve/Fchzpk8WBLBU/v
# uS/sCRMCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7RNV8G
# EgRVMB0GA1UdDgQWBBQDDzHIkSqTvWPz0V1NpDQP0pUBGDAOBgNVHQ8BAf8EBAMC
# BsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBKBgNVHSAE
# QzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3Rp
# Z28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDegNYYzaHR0cDovL2Ny
# bC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3JsMHQGCCsG
# AQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5zZWN0aWdvLmNvbS9T
# ZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8v
# b2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEATJtlWPrgec/vFcMy
# bd4zket3WOLrvctKPHXefpRtwyLHBJXfZWlhEwz2DJ71iSBewYfHAyTKx6XwJt/4
# +DFlDeDrbVFXpoyEUghGHCrC3vLaikXzvvf2LsR+7fjtaL96VkjpYeWaOXe8vrqR
# ZIh1/12FFjQn0inL/+0t2v++kwzsbaINzMPxbr0hkRojAFKtl9RieCqEeajXPawh
# j3DDJHk6l/ENo6NbU9irALpY+zWAT18ocWwZXsKDcpCu4MbY8pn76rSSZXwHfDVE
# Ha1YGGti+95sxAqpbNMhRnDcL411TCPCQdB6ljvDS93NkiZ0dlw3oJoknk5fTtOP
# D+UTT1lEZUtDZM9I+GdnuU2/zA2xOjDQoT1IrXpl5Ozf4AHwsypKOazBpPmpfTXQ
# MkCgsRkqGCGyyH0FcRpLJzaq4Jgcg3Xnx35LhEPNQ/uQl3YqEqxAwXBbmQpA+oBt
# lGF7yG65yGdnJFxQjQEg3gf3AdT4LhHNnYPl+MolHEQ9J+WwhkcqCxuEdn17aE+N
# t/cTtO2gLe5zD9kQup2ZLHzXdR+PEMSU5n4k5ZVKiIwn1oVmHfmuZHaR6Ej+yFUK
# 7SnDH944psAU+zI9+KmDYjbIw74Ahxyr+kpCHIkD3PVcfHDZXXhO7p9eIOYJanwr
# CKNI9RX8BE/fzSEceuX1jhrUuUAwggbsMIIE1KADAgECAhAwD2+s3WaYdHypRjan
# eC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzETMBEGA1UECBMKTmV3
# IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNVBAoTFVRoZSBVU0VS
# VFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJTQSBDZXJ0aWZpY2F0
# aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAxMTgyMzU5NTlaMH0x
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMc
# U2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJKoZIhvcNAQEBBQAD
# ggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nPoD5T77NawHbWGgSl
# zkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEFkmZvIavVcRB5h/RG
# Ks3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQFcN/nqJc1zjxFrJw
# 06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1VdEXOzQhe5cBlK9/
# gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpWcdEtlEzIjDzTFKKc
# vSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ffnz82Pg4wZytGtzWv
# lr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV6xlV/9A5VC0m7kUa
# Om/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7wW33pdmmTIBxKlyx3
# GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFkyYiSWcuoRsDJ8qb/
# fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXDTYLQjdprt9v3GEBX
# c/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70atZ2h3pVBeqeDAgMB
# AAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNV
# HQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/BAQDAgGGMBIGA1Ud
# EwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwEQYDVR0gBAowCDAG
# BgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwudXNlcnRydXN0LmNv
# bS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDB2BggrBgEF
# BQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNlcnRydXN0LmNvbS9V
# U0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcwAYYZaHR0cDovL29j
# c3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEAbVSBpTNdFuG1U4GR
# dd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNviffmfAO9Aw63T4uRW
# +VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzVWHi/SHbhRGdtj7BD
# EA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJqO0I3fBZ4OZOTlJux
# 3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytCiiGS/qhvWiVwYoFz
# Y16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6NIxi5o8JHHfBd1qFz
# VwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9JkWtgqIsJLnfE5Xkze
# SD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw0NP8srMftpmWJvQ8
# eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUgOw9Bqh/poUGd/rG4
# oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20OAkrSQeq/eww2vGn
# L8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8XwgDCp3rrWiAoa6Ke60
# WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQsMIIEKAIBATCBkTB9MQswCQYDVQQGEwJH
# QjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3Jk
# MRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNB
# IFRpbWUgU3RhbXBpbmcgQ0ECEDlMJeF8oG0nqGXiO9kdItQwDQYJYIZIAWUDBAIC
# BQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUx
# DxcNMjQwMzE4MTgyMzQ2WjA/BgkqhkiG9w0BCQQxMgQwNvC8CigUhU4s+uSgH6vS
# rDL5Y0XooiTRq05pm/QwLNDxLxbhtGBarwrTS6R7vmi0MIHtBgsqhkiG9w0BCRAC
# DDGB3TCB2jCB1zAWBBSuYq91Cgy9R9ZGH3Vo4ryM58pPlDCBvAQUAtZbleKDcMFX
# AJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpO
# ZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVT
# RVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBAQUA
# BIICAHA7amdO4OG0NOaumCxbWWP0odmb9k1VtuBwlxtOHJRYXx6RlCj28Gydf1E+
# MfhsYSkq6Abj3LTO7jdTlHWIMHq20r7XwLGLE1opJDHyh/LzpDca7zvf+I7NbPZS
# ILUAHYHn2eSGoOWCVfeYKD9CAerYh7DVz+lptqe+ScjNvl462t+yxESm1+LeB6Ji
# Ua3mLWcRx/Zm+QtFPVrcS54rI40X/Ue17wcmp4ZgBsZ3DsvRzmZmmMX7f/p9PFmp
# j09vcZ1j5TaAcjGWkSltd9RqOW9V0vaF8cM/7y4D1BuyO+qioEIWJ2NwX7vsYh6H
# f/HCPvbXwEgCGL4TEXlQuXokBRcsbexRKSvG8LDlcB8Wfea0l3d4hxXrLGSgDAMh
# 8r4vCyKlYV8fpbGtGkNvwi4+fPCB0VeosvsNeNS56/5ApAtafdXjKZTbbj7oTu+C
# GQMG/aAR7uwWJYiSvWRAuBpbuK46MmkXvZTnaw9aei+mCoh6/t+auVazMx3CFsNo
# PwbJ3ZSmBEDc6RxiYpUXl6zMTptMC53SCGqqTb5F82rKT4XwPfPQy8XuPNB2Nw5n
# mN5UvVytvW5wl0E1dJ+uNlG4rXfz0Awwhz8B4aikUeYzPEsDwVkWctbWFYV0PVOF
# ruMLyEZb/DRe/EtyDilGW9L8UkrE158eYBt8wsXWu2kHTRlD
# SIG # End signature block
