/**************************************************************************

Copyright (c) 2012 - 2023, Intel Corporation. All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

***************************************************************************/
#ifndef EEPROM_CONFIG_H_
#define EEPROM_CONFIG_H_

#include "I40e.h"
#include "Hii/FormsetStd/HiiConfigData.h"
/* Typedefs */
enum ValidMinSrev {FWApiNotOK, FWApiOK };

/* Defines */

// Autogenerated pointers section offsets
#define I40E_AUTOGEN_PTR_PFGEN_PORTNUM_SECTION         0x15
#define I40E_AUTOGEN_PTR_PFGEN_PORTNUM_OFFSET          0x16
#define I40E_AUTOGEN_PTR_GLPCI_CAPSUP_SECTION          0xF
#define I40E_AUTOGEN_PTR_GLPCI_CAPSUP_OFFSET           0x10
#define I40E_AUTOGEN_PTR_PFPCI_FUNC2_SECTION           0x17
#define I40E_AUTOGEN_PTR_PFPCI_FUNC2_OFFSET            0x18
#define I40E_AUTOGEN_PTR_PF_VT_PFALLOC_PCIE_SECTION    0x1B
#define I40E_AUTOGEN_PTR_PF_VT_PFALLOC_PCIE_OFFSET     0x1C
#define I40E_AUTOGEN_PTR_PF_VT_PFALLOC_SECTION         0x1D
#define I40E_AUTOGEN_PTR_PF_VT_PFALLOC_OFFSET          0x1E

// PHY Family ID
#define EEPROM_PHY_FAMILY_ID_WORD 0x1A

// EEPROM capabilities word
#define EEPROM_CAPABILITIES_WORD 0x33
#define EEPROM_CAPABILITIES_SIG  0x4000

// FPK Flat NVM
#define EEPROM_FPK_CTRL_WORD_1                  0x0000
#define EEPROM_FPK_CTRL_WORD_1_FLAT_MAP_MASK    0x1000 /* bit 12 */

// NVM version
#define EEPROM_VERSION_WORD 0x18

// EEPROM size
#define EEPROM_SIZE_32K_WORDS    32768
#define EEPROM_SIGNATURE         0xA5A5

#define EEPROM_SIZE_WORD                 0x12
#define EEPROM_HEPSIZE_MASK              0x000F
#define EEPROM_SIZE_SIGNATURE_MASK       0xC000
#define EEPROM_SIZE_MASK_82570           0x3C00
#define EEPROM_SIZE_SIGNATURE_VALID      0x4000

/* Compatibility Word in EEPROM
 The only thing we need this define for it to determine if the device is a LOM */
#define EEPROM_COMPATIBILITY_WORD          0x03
#define EEPROM_COMPATIBILITY_WORD_OPLIN    0x10
#define EEPROM_COMPATABILITY_LOM_BIT       0x0800

#define NVM_PXE_SETUP_OPTIONS_MODULE_PTR     0x30
#define NVM_PF_PXE_SETUP_OPTIONS_OFFSET(_PF) (0x01 + (_PF))
#define NVM_PXE_SETUP_LINK_SPEED_MASK        (0x7 << 10)
#define NVM_PXE_SETUP_DSM_SHIFT              8
#define NVM_PXE_SETUP_DSM_MASK               (0x1 << NVM_PXE_SETUP_DSM_SHIFT)
#define NVM_PXE_SETUP_PROMPT_TIME_SHIFT      6
#define NVM_PXE_SETUP_PROMPT_TIME_MASK       (0x3 << NVM_PXE_SETUP_PROMPT_TIME_SHIFT)

#define NVM_PXE_CONFIG_CUSTOMIZATION_OPTIONS_MODULE_PTR   0x31

// POR Registers Auto-Load Module Pointer, default value is 0x7FFF
#define POR_REGS_AUTO_LOAD_PTR          0x0038

// APM values inside POR Registers Auto-Load Module
#define NVM_PFPM_APM_DATA_LOW(_PF)      (0x0091 + ((_PF) * 2)) /* _PF = 0...15 */
#define NVM_PFPM_APM_DATA_HIGH(_PF)     (0x0092 + ((_PF) * 2)) /* _PF = 0...15 */

// Definitions for various MAC Address types locations
#define NVM_SAN_MAC_ADDRESS_MODULE_PTR           0x28
#define NVM_SAN_MAC_ADDRESS_OFFSET(_PF)          (0x01 + (_PF) * 3)

// EMP Settings in Shared RAM mapped area
#define NVM_EMP_SR_SETTINGS_MODULE_PTR            0x48
#define NVM_EMP_SR_OEM_CAPABILITIES_OFFSET        0x0003
#define NVM_EMP_SR_OEM_CAPS_FLEX10_MASK           (0x1 << 0)
#define NVM_EMP_SR_OEM_CAPS_CEM_MASK              (0x1 << 1)
#define NVM_EMP_SR_OEM_CAPS_NPAR_MASK             (0x1 << 2)
#define NVM_EMP_SR_OEM_CAPS_NPAR2_MASK            (0x1 << 3)

#define NVM_EMP_SR_OEM_TECHN_ENA_OFFSET           0x0004
#define NVM_EMP_SR_OEM_TECHN_ENA_FLEX10_MASK      (0x1 << 0)
#define NVM_EMP_SR_OEM_TECHN_ENA_CEM_MASK         (0x1 << 1)
#define NVM_EMP_SR_OEM_TECHN_ENA_NPAR_MASK        (0x1 << 2)
#define NVM_EMP_SR_OEM_TECHN_ENA_NPAR_VER_SHIFT   3
#define NVM_EMP_SR_OEM_TECHN_ENA_NPAR_VER_MASK    (0x3 << NVM_EMP_SR_OEM_TECHN_ENA_NPAR_VER_SHIFT)
#define NVM_EMP_SR_OEM_TECHN_ENA_NPAR_1_1         0x0
#define NVM_EMP_SR_OEM_TECHN_ENA_NPAR_2_0         0x1
#define NVM_EMP_SR_OEM_TECHN_ENA_SRIOV_MASK       (0x1 << 5)
#define NVM_EMP_SR_OEM_TECHN_ENA_NPAR_EP_MASK     (0x1 << 6)
#define NVM_EMP_SR_OEM_TECHN_ENA_NCPC_MASK        (0x1 << 7)

#define NVM_EMP_SR_MAX_PF_VF_PER_PORT_WORD_OFFSET 0x0006
#define NVM_EMP_SR_MAX_PF_VF_PER_PORT_NPAREP_MASK  (1 << 7)

#define NVM_EMP_SR_MAX_PF_PER_PORT_OFFSET         0
#define NVM_EMP_SR_MAX_PF_PER_PORT_MASK           (0x7 << NVM_EMP_SR_MAX_PF_PER_PORT_OFFSET)
#define NVM_1_PF_PER_PORT                         0x0
#define NVM_2_PF_PER_PORT                         0x1
#define NVM_4_PF_PER_PORT                         0x2
#define NVM_8_PF_PER_PORT                         0x3
#define NVM_16_PF_PER_PORT                        0x4

#define NVM_EMP_SR_PF_MAC_PTR                     0x0018
#define NVM_EMP_SR_PF_MAC_SAL0(_PF)               (0x0001 + (_PF) * 4)
#define NVM_EMP_SR_PF_MAC_SAL1(_PF)               (0x0002 + (_PF) * 4)
#define NVM_EMP_SR_PF_MAC_SAH0(_PF)               (0x0003 + (_PF) * 4)
#define NVM_EMP_SR_PF_MAC_SAH1(_PF)               (0x0004 + (_PF) * 4)

#define NVM_EMP_SR_PHY_CAPS_LAN_PTR_OFFSET(_PORT) (0x0019 + (_PORT))
#define NVM_EMP_SR_PHY_CAPS_LAN_AUTO_FEC_ENA      (1 << 11)
#define NVM_EMP_SR_PHY_CAPS_LAN_KR_FEC_ENA        (1 << 12)
#define NVM_EMP_SR_PHY_CAPS_LAN_RS_FEC_ENA        (1 << 13)
#define NVM_EMP_SR_PHY_CAPS_LAN_KR_FEC_REQ        (1 << 14)
#define NVM_EMP_SR_PHY_CAPS_LAN_RS_FEC_REQ        (1 << 15)
#define NVM_EMP_SR_PHY_CAPS_LAN_FEC_MASK          0xF800

#define NVM_EMP_SR_PFGEN_STATE(_PF)               (0x001D + (_PF))

#define NVM_PHY_CAPS_DATA_MISC3_OFFSET            0xC

#define NVM_EMP_SR_FEATURES_ENA_OFFSET            0x2A
#define NVM_EMP_SR_FEATURES_ENA_PORT_SHDN         (1 << 4)

// Current Settings section offsets
#define NVM_EMP_SR_CURRENT_SETTING_PTR_FPK        0x31
#define NVM_EMP_SR_CURRENT_SETTING_PTR_FVL        0x2B

// Manageability Module
#define NVM_MNG_MODULE_OEM_STRUC_PTR_OFFSET       0x0A // FVL EAS 6.4 Ch. 6.2.42

#define NVM_DELL_EXT_CAP_PTR_OFFSET               0x03 // FVL EAS 6.4 Ch. 6.2.46
#define NVM_DELL_CAP_FLAGS_OFFSET                 0x02 // FVL EAS 6.4 Ch. 20.4.2.3

#define EEE                                       BIT (8)
#define ON_CHIP_THERMAL_SENSOR                    BIT (9)
#define NIC_PARTITIONING                          BIT (24)

#define CURRENT_SETTINGS_LLDP_ADMIN_STATUS        0x1
#define CURRENT_SETTINGS_LINK_BEHAVIOR            0x2D

#define CURRENT_SETTINGS_LINK_BEHAVIOR_IGNORE_MFP (1 << 0)
#define CURRENT_SETTINGS_LINK_BEHAVIOR_OS_FORCED  (1 << 1)

// Definitions for reading PBA number from NVM
#define NVM_PBA_FLAGS_OFFSET                     0x15
#define NVM_PBA_FLAG_STRING_MODE                 0xFAFA
#define NVM_PBA_BLOCK_MODULE_POINTER             0x16

// Maximum BW is 10Gb/s so it is 200 in 50Mb/s units
#define MAX_BANDWIDTH_IN_50Mb_UNITS   200

#define PF_PERSONALITY_NIC                    0x01
#define PF_PERSONALITY_FCOE                   0x02
#define PF_PERSONALITY_ISCSI                  0x03

// Alternate MAC addresses locations
#define NVM_LAN_ALT_MAC_MODULE_PTR            0x37
#define NVM_LAN_ALT_MAC_PORT_OFFSET(Port)     (0x0001 + (Port) * 3)
#define NVM_LAN_ALT_MAC_PF_OFFSET(_PF)        (0x000D + (_PF) * 3)

// Defines for setings located in Auto-Load modules
#define NVM_POR_REG_AL_MODULE_PTR             0x38
#define NVM_POR_PFPCI_FUNC_DATA_LOW(_PF)      (0x00E7 + (_PF) * 2)
#define NVM_POR_PFPCI_FUNC_DATA_HIGH(_PF)     (0x00E8 + (_PF) * 2)
#define NVM_POR_PFPCI_CLASS_DATA_LOW(_PF)     (0x010A + (_PF) * 8)
#define NVM_POR_PFPCI_CLASS_DATA_HIGH(_PF)    (0x010B + (_PF) * 8)

#define NVM_GLOBR_REG_AL_MODULE_PTR           0x3B
#define NVM_GLOBR_PRTGL_SAL_DATA_LOW(_PF)     (0x0051 + (_PF) * 2)
#define NVM_GLOBR_PRTGL_SAL_DATA_HIGH(_PF)    (0x0052 + (_PF) * 2)
#define NVM_GLOBR_PRTGL_SAH_DATA_LOW(_PF)     (0x005C + (_PF) * 2)
#define NVM_GLOBR_PRTGL_SAH_DATA_HIGH(_PF)    (0x005D + (_PF) * 2)

#define NVM_PCIE_ALT_AL_MODULE_PTR            0x3E
//#define NVM_GLOBR_PFPCI_CLASS_DATA_LOW(_PF)   (0x0007 + (_PF) * 8)
//#define NVM_GLOBR_PFPCI_CLASS_DATA_HIGH(_PF)  (0x0008 + (_PF) * 8)

#define I40E_VF_PER_DEVICE_MAX                128


/* NVM HW module Type 1 entry definition */
#pragma pack(1)
typedef struct NVM_HW_MODULE_TYPE_1_S {
  UINT32 Type       :4;
  UINT32 Address    :28;
  UINT32 Data;
} NVM_HW_MODULE_TYPE_1;
#pragma pack()

/* NVM HW module Type 2 entry definition. It's only a header
   for the remaining part of data */
#pragma pack(1)
typedef struct NVM_HW_MODULE_TYPE_2_S {
  UINT32 Type       :4;
  UINT32 Address    :28;
  UINT16 Attributes;
} NVM_HW_MODULE_TYPE_2;
#pragma pack()

/* MAX PFs supported by FVL/FPK in SFP mode == MAX number of ports */
#define I40E_MAX_PF_SFP_MODE                          4

#define POR_REGS_AUTOLOAD_PFPCI_FUNC_OFFSET           0x10D
#define POR_REGS_AUTOLOAD_PFPCI_FUNC_SIZE_SFP_BYTES   (sizeof (NVM_HW_MODULE_TYPE_2) + (I40E_MAX_PF_SFP_MODE * 4))

#define PFPCI_FUNC2(Pf) (UINT32)(I40E_PFPCI_FUNC2 + ((Pf) * 4))

/* Interface function declarations */

/** Write SR buffer using shared code implementation.

   @param[in]   AdapterInfo    Points to the driver information
   @param[in]   ModulePointer  Pointer to module in words with respect to NVM beginning
   @param[in]   Offset         offset in words from module start
   @param[in]   Words          Number of words to write
   @param[in]   Data           Pointer to location with data to be written

   @retval    EFI_SUCCESS        Buffer successfully written
   @retval    EFI_ACCESS_DENIED  Access to desired NVM memory range is denied
   @retval    EFI_DEVICE_ERROR   Failed to write buffer
   @retval    EFI_DEVICE_ERROR   Waiting for ARQ response timeout
**/
EFI_STATUS
I40eWriteNvmBuffer (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT8        ModulePointer,
  IN UINT32       Offset,
  IN UINT16       Words,
  IN VOID        *Data
  );

/** Writes data buffer to nvm using __i40e_write_nvm_buffer shared code function.

   Function works around the situation when the buffer spreads over two sectors.
   The entire buffer must be located inside the Shared RAM.

   @param[in]   AdapterInfo   Points to the driver information
   @param[in]   Offset        Buffer offset from the start of NVM
   @param[in]   Words         Number of words to write
   @param[in]   Data          Pointer to location with data to be written

   @retval   EFI_SUCCESS       NVM buffer written successfully
   @retval   EFI_DEVICE_ERROR  Failed to write buffer (or either of the sectors)
**/
EFI_STATUS
I40eWriteNvmBufferExt (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT32       Offset,
  IN UINT16       Words,
  IN VOID        *Data
  );

/** Reads SR buffer.

   @param[in]   UndiPrivateData  Points to the driver information.
   @param[in]   Offset           Offset in words from module start.
   @param[in]   Length           Number of words to read.
   @param[out]  Data             Pointer to location with data to be read to.

   @retval    EFI_SUCCESS    Buffer successfully read.
   @retval    !EFI_SUCCESS   Buffer not read, failure of underlying function.
**/
EFI_STATUS
ReadSrBuffer16 (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN  UINT16             Offset,
  IN  UINT16             Length,
  OUT UINT16             *Data
  );

/** Reads SR word.

   @param[in]   UndiPrivateData  Points to the driver information.
   @param[in]   Offset           Offset in words from module start.
   @param[out]  Data             Pointer to location with data to be read to.

   @retval    EFI_SUCCESS     Word successfully read.
   @retval    !EFI_SUCCESS    Word not read, failure of underlying function.
**/
EFI_STATUS
ReadSr16 (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN  UINT16             Offset,
  OUT UINT16             *Data
  );

/** Writes SR buffer.

   @param[in]   UndiPrivateData  Points to the driver information.
   @param[in]   Offset           Offset in words from module start.
   @param[in]   Length           Number of words to write.
   @param[in]   Data             Pointer to location with words to be written.

   @retval    EFI_SUCCESS    Buffer successfully written.
   @retval    !EFI_SUCCESS   Buffer not written, failure of underlying function.
**/
EFI_STATUS
WriteSrBuffer16 (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN  UINT16             Offset,
  IN  UINT16             Length,
  IN  UINT16             *Data
  );

/** Writes SR word.

   @param[in]   UndiPrivateData  Points to the driver information.
   @param[in]   Offset           Offset in words from module start.
   @param[in]   Data             Word to be written.

   @retval    EFI_SUCCESS    Word successfully written.
   @retval    !EFI_SUCCESS   Word not written, failure of underlying function.
**/
EFI_STATUS
WriteSr16 (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN  UINT16             Offset,
  IN  UINT16             Data
  );

/** Returns the offset in the NVM of the CSR register initialization data
   in the module pointed by the ModulePointer.

   Function looks for the register based on its address in CSR register space.

   @param[in]   AdapterInfo           Points to the driver information
   @param[in]   AutoGeneratedSection  Offset of module pointer in autogenerated
                                      pointer section
   @param[in]   AutoGeneratedOffset   Offset of module length pointer in
                                      autogenerated pointer section

   @return      offset of register data in NVM
   @return      0 when register is not found
**/
UINT16
GetRegisterInitializationDataOffset (
  IN DRIVER_DATA *AdapterInfo,
  IN UINT16       AutoGeneratedSection,
  IN UINT16       AutoGeneratedOffset
  );

/** Notify FW of change in alternate RAM structures.

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval   EFI_SUCCESS       Successfully notified of changes
   @retval   EFI_DEVICE_ERROR  Failed to notify
**/
EFI_STATUS
ApplyAlternateSettings (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Restores the factory default MAC address for currently managed PF.

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval      EFI_SUCCESS       New MAC address set successfully
   @retval      !EFI_SUCCESS      Failed to restore factory default MAC address
**/
EFI_STATUS
RestoreDefaultMacAddress (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData
  );

/** Gets factory MAC addresses for PF0.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  FactoryMacAddress    Pointer to buffer for resulting factory
                                     MAC address

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      !EFI_SUCCESS      Failure of underlying function
**/
EFI_STATUS
GetFactoryMacAddressForPf0 (
  IN   UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT  UINT8              *FactoryMacAddress
  );

/** Gets factory MAC address for specified partition.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  FactoryMacAddress    Pointer to buffer for resulting factory
                                     MAC address
   @param[in]   Partition            Partition index

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Partition number exceeds maximum supported on port
   @retval      !EFI_SUCCESS      Failed to read factory MAC address
**/
EFI_STATUS
GetPartitionFactoryMacAddress (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT UINT8              *FactoryMacAddress,
  IN  UINT8              Partition
  );

/** Gets alternate MAC address for specified partition.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  AlternateMacAddress  Pointer to buffer for resulting alternate
                                     MAC address
   @param[in]   Partition            Partition index

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      EFI_NOT_FOUND     Partition number exceeds maximum supported on port
   @retval      EFI_DEVICE_ERROR  Failed to read alternate MAC addr from Alt. RAM
**/
EFI_STATUS
GetPartitionAlternateMacAddress (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT UINT8              *AlternateMacAddress,
  IN  UINT8              Partition
  );

/** Sets alternate MAC address for currently managed PF.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[in]   AlternateMacAddress  Value to set the MAC address to

   @retval      EFI_SUCCESS          New MAC address set successfully
   @retval      !EFI_SUCCESS         Failure to set alternate MAC address
**/
EFI_STATUS
SetAlternateMacAddress (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN  UINT8              *AlternateMacAddress
  );

/** Reads factory default MAC address.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  FactoryMacAddress    Factory default MAC address of the adapter

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      !EFI_SUCCESS      Failure of underlying function
**/
EFI_STATUS
GetFactoryMacAddress (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT UINT8              *FactoryMacAddress
  );

/** Gets alternate MAC address of currently managed PF.

   @param[in]   UndiPrivateData      Pointer to driver private data structure
   @param[out]  AlternateMacAddress  Pointer to buffer for resulting alternate
                                     MAC address

   @retval      EFI_SUCCESS       MAC addresses read successfully
   @retval      !EFI_SUCCESS      Failure of underlying function
**/
EFI_STATUS
GetAlternateMacAddress (
  IN   UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT  UINT8              *AlternateMacAddress
  );


/** Returns Flat NVM status.

   @param[in]   UndiPrivateData    Points to the driver instance private data
   @param[out]  IsFlatNvm          FlatNvm Status:
                                   0 - Structured NVM or N/A (FVL)
                                   1 - FPK Flat NVM

   @retval   EFI_SUCCESS       Ctrl word successfully read
   @retval   EFI_DEVICE_ERROR  Failed to read ctrl word
**/
EFI_STATUS
EepromIsFlatNvm (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT8             *IsFlatNvm
  );

/** Returns EEPROM capabilities word (0x33) for current adapter

   @param[in]   UndiPrivateData    Points to the driver instance private data
   @param[out]  CapabilitiesWord   EEPROM capabilities word (0x33) for current adapter

   @retval   EFI_SUCCESS       Capabilities word successfully read
   @retval   EFI_DEVICE_ERROR  Failed to read capabilities word
**/
EFI_STATUS
EepromGetCapabilitiesWord (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16            *CapabilitiesWord
  );

/** Checks if Phy Family Id is supported by Device

   @param[in]  UndiPrivateData    Points to the driver instance private data
   @param[in]  PhyFamilyId        PhyFamilyId

   @retval   EFI_SUCCESS        Phy Family Id is supported
   @retval   EFI_NOT_SUPPORTED  Phy Family Id is not supported
   @retval   EFI_DEVICE_ERROR   Failed to read capabilities word
**/
EFI_STATUS
EepromCheckPhyFamilyId (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  IN  UINT16            PhyFamilyId
  );

/** Returns NVM version

   @param[in]   UndiPrivateData  Points to the driver instance private data
   @param[out]  MajorVer         NVM major version
   @param[out]  MinorVer         NVM minor version

   @retval   EFI_SUCCESS       NVM version read successfully
   @retval   EFI_DEVICE_ERROR  Failed to read EEPROM version word
**/
EFI_STATUS
EepromGetNvmVersion (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData,
  OUT UINT16            *MajorVer,
  OUT UINT16            *MinorVer
  );



/** Gets maximum number of PFs per port.

   @param[in]   AdapterInfo      Points to the driver information
   @param[out]  PfPerPortNumber  Max. number of PFs per port

   @retval      EFI_SUCCESS       PFs per port number successfully returned
   @retval      EFI_DEVICE_ERROR  Failed to read NVM word with max PF per port number
   @retval      EFI_DEVICE_ERROR  Invalid configuration read from NVM
**/
EFI_STATUS
GetMaxPfPerPortNumber (
  IN  DRIVER_DATA *AdapterInfo,
  OUT UINT8       *PfPerPortNumber
  );

/** Checks if is LOM device

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval      FALSE   Is not LOM device (always returned)
**/
BOOLEAN
EepromIsLomDevice (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Updates NVM checksum

   @param[in]   UndiPrivateData   Pointer to driver private data structure

   @retval      EFI_SUCCESS       Checksum successfully updated
   @retval      EFI_DEVICE_ERROR  Failed to update NVM checksum
**/
EFI_STATUS
UpdateNvmChecksum (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Reads PBA string from NVM.

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[out]  PbaNumberStr     Output string buffer for PBA string

   @retval   EFI_SUCCESS            PBA string successfully read
   @retval   EFI_SUCCESS            PBA string unsupported by adapter
   @retval   EFI_DEVICE_ERROR       Failed to read PBA flags word
   @retval   EFI_DEVICE_ERROR       Failed to read PBA module pointer
   @retval   EFI_DEVICE_ERROR       Failed to read PBA length
   @retval   EFI_DEVICE_ERROR       Failed to read PBA number word
**/
EFI_STATUS
GetPbaStr (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT EFI_STRING         PbaNumberStr
  );

/** Reads port numbers

   @param[in]   AdapterInfo   Points to the driver information
   @param[out]  PortNumbers   Pointer to buffer for resultant port numberes
   @param[in]   ArraySize     Size of buffer with port numbers

   @retval   EFI_SUCCESS        Port numbers successfully read
   @retval   EFI_DEVICE_ERROR   Failed to get port numbers section address with
                                offset
   @retval   EFI_DEVICE_ERROR   Failed to read port number value
**/
EFI_STATUS
EepromReadPortnumValues (
  IN  DRIVER_DATA *AdapterInfo,
  OUT UINT16      *PortNumbers,
  IN  UINT16       ArraySize
  );


/** Checks if FW LLDP Agent status is supported.

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[out]  Supported        Tells whether LLDP Agent is supported

   @retval      EFI_SUCCESS  LLDP Agent support information retrieved.
**/
EFI_STATUS
IsLldpAgentSupported (
  IN   UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT  BOOLEAN            *Supported
  );

/** Reads current FW LLDP Agent status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[out]  LldpAgentEna     Pointer to variable which will store read LLDP Admin status

   @retval      EFI_SUCCESS       LLDP Agent status read successfully
   @retval      EFI_DEVICE_ERROR  Failed to read LLDP Agent status
   @retval      EFI_DEVICE_ERROR  Out of range value read from NVM
**/
EFI_STATUS
GetLldpAgentStatus (
  IN   UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT  BOOLEAN            *LldpAgentEna
  );

/** Sets FW LLDP Agent status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure
   @param[in]   LldpAgentEna     Requested LLDP Agent status

   @retval      EFI_SUCCESS         LLDP Agent Status written successfully
   @retval      EFI_DEVICE_ERROR    Failed to read current LLDP Agent status
   @retval      EFI_SUCCESS         Requested LLDP agent status matches current
   @retval      EFI_DEVICE_ERROR    Failed to start or stop LLDP Agent
   @retval      EFI_DEVICE_ERROR    Failed to set DCB parameters
**/
EFI_STATUS
SetLldpAgentStatus (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  IN  BOOLEAN            *LldpAgentEna
  );

/** Restore factory LLDP Agent status.

   @param[in]   UndiPrivateData  Pointer to driver private data structure.

   @retval      EFI_SUCCESS      LLDP Agent reset successful.
   @retval      EFI_DEVICE_ERROR Failed to reset LLDP Agent.
**/
EFI_STATUS
ResetLldpAgentStatus (
  IN  UNDI_PRIVATE_DATA *UndiPrivateData
  );

/** Get default LLDP Agent status.

   @param[in]   UndiPrivateData      Pointer to driver private data structure.
   @param[out]  DefaultLldpAgentEna  Pointer to variable which should store default value for LLDP Agent.

   @retval      EFI_SUCCESS          LLDP Agent get default successful.
   @retval      EFI_DEVICE_ERROR     Failed to get default LLDP Agent.
**/
EFI_STATUS
GetDfltLldpAgentStatus (
  IN  UNDI_PRIVATE_DATA  *UndiPrivateData,
  OUT BOOLEAN            *DefaultLldpAgentEna
  );


#endif /* EEPROM_CONFIG_H_ */
